/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.iam.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the last time that an IAM role was used. This includes the date and time and the Region in
 * which the role was last used. Activity is only reported for the trailing 400 days. This period can be shorter if your
 * Region began supporting these features within the last year. The role might have been used more than 400 days ago.
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#access-advisor_tracking-period"
 * >Regions Where Data Is Tracked</a> in the <i>IAM User Guide</i>.
 * </p>
 * <p>
 * This data type is returned as a response element in the <a>GetRole</a> and <a>GetAccountAuthorizationDetails</a>
 * operations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RoleLastUsed implements SdkPojo, Serializable, ToCopyableBuilder<RoleLastUsed.Builder, RoleLastUsed> {
    private static final SdkField<Instant> LAST_USED_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUsedDate").getter(getter(RoleLastUsed::lastUsedDate)).setter(setter(Builder::lastUsedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUsedDate").build()).build();

    private static final SdkField<String> REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Region")
            .getter(getter(RoleLastUsed::region)).setter(setter(Builder::region))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Region").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LAST_USED_DATE_FIELD,
            REGION_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant lastUsedDate;

    private final String region;

    private RoleLastUsed(BuilderImpl builder) {
        this.lastUsedDate = builder.lastUsedDate;
        this.region = builder.region;
    }

    /**
     * <p>
     * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a> that the role was
     * last used.
     * </p>
     * <p>
     * This field is null if the role has not been used within the IAM tracking period. For more information about the
     * tracking period, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#access-advisor_tracking-period"
     * >Regions Where Data Is Tracked</a> in the <i>IAM User Guide</i>.
     * </p>
     * 
     * @return The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a> that the
     *         role was last used.</p>
     *         <p>
     *         This field is null if the role has not been used within the IAM tracking period. For more information
     *         about the tracking period, see <a href=
     *         "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#access-advisor_tracking-period"
     *         >Regions Where Data Is Tracked</a> in the <i>IAM User Guide</i>.
     */
    public Instant lastUsedDate() {
        return lastUsedDate;
    }

    /**
     * <p>
     * The name of the AWS Region in which the role was last used.
     * </p>
     * 
     * @return The name of the AWS Region in which the role was last used.
     */
    public String region() {
        return region;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(lastUsedDate());
        hashCode = 31 * hashCode + Objects.hashCode(region());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RoleLastUsed)) {
            return false;
        }
        RoleLastUsed other = (RoleLastUsed) obj;
        return Objects.equals(lastUsedDate(), other.lastUsedDate()) && Objects.equals(region(), other.region());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("RoleLastUsed").add("LastUsedDate", lastUsedDate()).add("Region", region()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LastUsedDate":
            return Optional.ofNullable(clazz.cast(lastUsedDate()));
        case "Region":
            return Optional.ofNullable(clazz.cast(region()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RoleLastUsed, T> g) {
        return obj -> g.apply((RoleLastUsed) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RoleLastUsed> {
        /**
         * <p>
         * The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a> that the role
         * was last used.
         * </p>
         * <p>
         * This field is null if the role has not been used within the IAM tracking period. For more information about
         * the tracking period, see <a href=
         * "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#access-advisor_tracking-period"
         * >Regions Where Data Is Tracked</a> in the <i>IAM User Guide</i>.
         * </p>
         * 
         * @param lastUsedDate
         *        The date and time, in <a href="http://www.iso.org/iso/iso8601">ISO 8601 date-time format</a> that the
         *        role was last used.</p>
         *        <p>
         *        This field is null if the role has not been used within the IAM tracking period. For more information
         *        about the tracking period, see <a href=
         *        "https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_access-advisor.html#access-advisor_tracking-period"
         *        >Regions Where Data Is Tracked</a> in the <i>IAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUsedDate(Instant lastUsedDate);

        /**
         * <p>
         * The name of the AWS Region in which the role was last used.
         * </p>
         * 
         * @param region
         *        The name of the AWS Region in which the role was last used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder region(String region);
    }

    static final class BuilderImpl implements Builder {
        private Instant lastUsedDate;

        private String region;

        private BuilderImpl() {
        }

        private BuilderImpl(RoleLastUsed model) {
            lastUsedDate(model.lastUsedDate);
            region(model.region);
        }

        public final Instant getLastUsedDate() {
            return lastUsedDate;
        }

        @Override
        public final Builder lastUsedDate(Instant lastUsedDate) {
            this.lastUsedDate = lastUsedDate;
            return this;
        }

        public final void setLastUsedDate(Instant lastUsedDate) {
            this.lastUsedDate = lastUsedDate;
        }

        public final String getRegion() {
            return region;
        }

        @Override
        public final Builder region(String region) {
            this.region = region;
            return this;
        }

        public final void setRegion(String region) {
            this.region = region;
        }

        @Override
        public RoleLastUsed build() {
            return new RoleLastUsed(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
