/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Additional options for the Amazon Kinesis streaming data source.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KinesisStreamingSourceOptions implements SdkPojo, Serializable,
        ToCopyableBuilder<KinesisStreamingSourceOptions.Builder, KinesisStreamingSourceOptions> {
    private static final SdkField<String> ENDPOINT_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EndpointUrl").getter(getter(KinesisStreamingSourceOptions::endpointUrl))
            .setter(setter(Builder::endpointUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointUrl").build()).build();

    private static final SdkField<String> STREAM_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamName").getter(getter(KinesisStreamingSourceOptions::streamName))
            .setter(setter(Builder::streamName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamName").build()).build();

    private static final SdkField<String> CLASSIFICATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Classification").getter(getter(KinesisStreamingSourceOptions::classification))
            .setter(setter(Builder::classification))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Classification").build()).build();

    private static final SdkField<String> DELIMITER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Delimiter").getter(getter(KinesisStreamingSourceOptions::delimiter)).setter(setter(Builder::delimiter))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Delimiter").build()).build();

    private static final SdkField<String> STARTING_POSITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartingPosition").getter(getter(KinesisStreamingSourceOptions::startingPositionAsString))
            .setter(setter(Builder::startingPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartingPosition").build()).build();

    private static final SdkField<Long> MAX_FETCH_TIME_IN_MS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaxFetchTimeInMs").getter(getter(KinesisStreamingSourceOptions::maxFetchTimeInMs))
            .setter(setter(Builder::maxFetchTimeInMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxFetchTimeInMs").build()).build();

    private static final SdkField<Long> MAX_FETCH_RECORDS_PER_SHARD_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaxFetchRecordsPerShard").getter(getter(KinesisStreamingSourceOptions::maxFetchRecordsPerShard))
            .setter(setter(Builder::maxFetchRecordsPerShard))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxFetchRecordsPerShard").build())
            .build();

    private static final SdkField<Long> MAX_RECORD_PER_READ_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaxRecordPerRead").getter(getter(KinesisStreamingSourceOptions::maxRecordPerRead))
            .setter(setter(Builder::maxRecordPerRead))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRecordPerRead").build()).build();

    private static final SdkField<Boolean> ADD_IDLE_TIME_BETWEEN_READS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN).memberName("AddIdleTimeBetweenReads")
            .getter(getter(KinesisStreamingSourceOptions::addIdleTimeBetweenReads))
            .setter(setter(Builder::addIdleTimeBetweenReads))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddIdleTimeBetweenReads").build())
            .build();

    private static final SdkField<Long> IDLE_TIME_BETWEEN_READS_IN_MS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("IdleTimeBetweenReadsInMs").getter(getter(KinesisStreamingSourceOptions::idleTimeBetweenReadsInMs))
            .setter(setter(Builder::idleTimeBetweenReadsInMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IdleTimeBetweenReadsInMs").build())
            .build();

    private static final SdkField<Long> DESCRIBE_SHARD_INTERVAL_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("DescribeShardInterval").getter(getter(KinesisStreamingSourceOptions::describeShardInterval))
            .setter(setter(Builder::describeShardInterval))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DescribeShardInterval").build())
            .build();

    private static final SdkField<Integer> NUM_RETRIES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("NumRetries").getter(getter(KinesisStreamingSourceOptions::numRetries))
            .setter(setter(Builder::numRetries))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumRetries").build()).build();

    private static final SdkField<Long> RETRY_INTERVAL_MS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("RetryIntervalMs").getter(getter(KinesisStreamingSourceOptions::retryIntervalMs))
            .setter(setter(Builder::retryIntervalMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RetryIntervalMs").build()).build();

    private static final SdkField<Long> MAX_RETRY_INTERVAL_MS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaxRetryIntervalMs").getter(getter(KinesisStreamingSourceOptions::maxRetryIntervalMs))
            .setter(setter(Builder::maxRetryIntervalMs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxRetryIntervalMs").build())
            .build();

    private static final SdkField<Boolean> AVOID_EMPTY_BATCHES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AvoidEmptyBatches").getter(getter(KinesisStreamingSourceOptions::avoidEmptyBatches))
            .setter(setter(Builder::avoidEmptyBatches))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvoidEmptyBatches").build()).build();

    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamArn").getter(getter(KinesisStreamingSourceOptions::streamArn)).setter(setter(Builder::streamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamArn").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleArn").getter(getter(KinesisStreamingSourceOptions::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<String> ROLE_SESSION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RoleSessionName").getter(getter(KinesisStreamingSourceOptions::roleSessionName))
            .setter(setter(Builder::roleSessionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleSessionName").build()).build();

    private static final SdkField<String> ADD_RECORD_TIMESTAMP_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AddRecordTimestamp").getter(getter(KinesisStreamingSourceOptions::addRecordTimestamp))
            .setter(setter(Builder::addRecordTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddRecordTimestamp").build())
            .build();

    private static final SdkField<String> EMIT_CONSUMER_LAG_METRICS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EmitConsumerLagMetrics").getter(getter(KinesisStreamingSourceOptions::emitConsumerLagMetrics))
            .setter(setter(Builder::emitConsumerLagMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EmitConsumerLagMetrics").build())
            .build();

    private static final SdkField<Instant> STARTING_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartingTimestamp")
            .getter(getter(KinesisStreamingSourceOptions::startingTimestamp))
            .setter(setter(Builder::startingTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartingTimestamp").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_URL_FIELD,
            STREAM_NAME_FIELD, CLASSIFICATION_FIELD, DELIMITER_FIELD, STARTING_POSITION_FIELD, MAX_FETCH_TIME_IN_MS_FIELD,
            MAX_FETCH_RECORDS_PER_SHARD_FIELD, MAX_RECORD_PER_READ_FIELD, ADD_IDLE_TIME_BETWEEN_READS_FIELD,
            IDLE_TIME_BETWEEN_READS_IN_MS_FIELD, DESCRIBE_SHARD_INTERVAL_FIELD, NUM_RETRIES_FIELD, RETRY_INTERVAL_MS_FIELD,
            MAX_RETRY_INTERVAL_MS_FIELD, AVOID_EMPTY_BATCHES_FIELD, STREAM_ARN_FIELD, ROLE_ARN_FIELD, ROLE_SESSION_NAME_FIELD,
            ADD_RECORD_TIMESTAMP_FIELD, EMIT_CONSUMER_LAG_METRICS_FIELD, STARTING_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String endpointUrl;

    private final String streamName;

    private final String classification;

    private final String delimiter;

    private final String startingPosition;

    private final Long maxFetchTimeInMs;

    private final Long maxFetchRecordsPerShard;

    private final Long maxRecordPerRead;

    private final Boolean addIdleTimeBetweenReads;

    private final Long idleTimeBetweenReadsInMs;

    private final Long describeShardInterval;

    private final Integer numRetries;

    private final Long retryIntervalMs;

    private final Long maxRetryIntervalMs;

    private final Boolean avoidEmptyBatches;

    private final String streamArn;

    private final String roleArn;

    private final String roleSessionName;

    private final String addRecordTimestamp;

    private final String emitConsumerLagMetrics;

    private final Instant startingTimestamp;

    private KinesisStreamingSourceOptions(BuilderImpl builder) {
        this.endpointUrl = builder.endpointUrl;
        this.streamName = builder.streamName;
        this.classification = builder.classification;
        this.delimiter = builder.delimiter;
        this.startingPosition = builder.startingPosition;
        this.maxFetchTimeInMs = builder.maxFetchTimeInMs;
        this.maxFetchRecordsPerShard = builder.maxFetchRecordsPerShard;
        this.maxRecordPerRead = builder.maxRecordPerRead;
        this.addIdleTimeBetweenReads = builder.addIdleTimeBetweenReads;
        this.idleTimeBetweenReadsInMs = builder.idleTimeBetweenReadsInMs;
        this.describeShardInterval = builder.describeShardInterval;
        this.numRetries = builder.numRetries;
        this.retryIntervalMs = builder.retryIntervalMs;
        this.maxRetryIntervalMs = builder.maxRetryIntervalMs;
        this.avoidEmptyBatches = builder.avoidEmptyBatches;
        this.streamArn = builder.streamArn;
        this.roleArn = builder.roleArn;
        this.roleSessionName = builder.roleSessionName;
        this.addRecordTimestamp = builder.addRecordTimestamp;
        this.emitConsumerLagMetrics = builder.emitConsumerLagMetrics;
        this.startingTimestamp = builder.startingTimestamp;
    }

    /**
     * <p>
     * The URL of the Kinesis endpoint.
     * </p>
     * 
     * @return The URL of the Kinesis endpoint.
     */
    public final String endpointUrl() {
        return endpointUrl;
    }

    /**
     * <p>
     * The name of the Kinesis data stream.
     * </p>
     * 
     * @return The name of the Kinesis data stream.
     */
    public final String streamName() {
        return streamName;
    }

    /**
     * <p>
     * An optional classification.
     * </p>
     * 
     * @return An optional classification.
     */
    public final String classification() {
        return classification;
    }

    /**
     * <p>
     * Specifies the delimiter character.
     * </p>
     * 
     * @return Specifies the delimiter character.
     */
    public final String delimiter() {
        return delimiter;
    }

    /**
     * <p>
     * The starting position in the Kinesis data stream to read data from. The possible values are <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in UTC format in the pattern
     * <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC timezone offset with a +/-. For example:
     * "2023-04-04T08:00:00-04:00"). The default value is <code>"latest"</code>.
     * </p>
     * <p>
     * Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only for Glue
     * version 4.0 or later.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #startingPosition}
     * will return {@link StartingPosition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #startingPositionAsString}.
     * </p>
     * 
     * @return The starting position in the Kinesis data stream to read data from. The possible values are
     *         <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in UTC
     *         format in the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC timezone
     *         offset with a +/-. For example: "2023-04-04T08:00:00-04:00"). The default value is <code>"latest"</code>
     *         .</p>
     *         <p>
     *         Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only for
     *         Glue version 4.0 or later.
     * @see StartingPosition
     */
    public final StartingPosition startingPosition() {
        return StartingPosition.fromValue(startingPosition);
    }

    /**
     * <p>
     * The starting position in the Kinesis data stream to read data from. The possible values are <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in UTC format in the pattern
     * <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC timezone offset with a +/-. For example:
     * "2023-04-04T08:00:00-04:00"). The default value is <code>"latest"</code>.
     * </p>
     * <p>
     * Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only for Glue
     * version 4.0 or later.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #startingPosition}
     * will return {@link StartingPosition#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #startingPositionAsString}.
     * </p>
     * 
     * @return The starting position in the Kinesis data stream to read data from. The possible values are
     *         <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in UTC
     *         format in the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC timezone
     *         offset with a +/-. For example: "2023-04-04T08:00:00-04:00"). The default value is <code>"latest"</code>
     *         .</p>
     *         <p>
     *         Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only for
     *         Glue version 4.0 or later.
     * @see StartingPosition
     */
    public final String startingPositionAsString() {
        return startingPosition;
    }

    /**
     * <p>
     * The maximum time spent for the job executor to read records for the current batch from the Kinesis data stream,
     * specified in milliseconds (ms). Multiple <code>GetRecords</code> API calls may be made within this time. The
     * default value is <code>1000</code>.
     * </p>
     * 
     * @return The maximum time spent for the job executor to read records for the current batch from the Kinesis data
     *         stream, specified in milliseconds (ms). Multiple <code>GetRecords</code> API calls may be made within
     *         this time. The default value is <code>1000</code>.
     */
    public final Long maxFetchTimeInMs() {
        return maxFetchTimeInMs;
    }

    /**
     * <p>
     * The maximum number of records to fetch per shard in the Kinesis data stream per microbatch. Note: The client can
     * exceed this limit if the streaming job has already read extra records from Kinesis (in the same get-records
     * call). If <code>MaxFetchRecordsPerShard</code> needs to be strict then it needs to be a multiple of
     * <code>MaxRecordPerRead</code>. The default value is <code>100000</code>.
     * </p>
     * 
     * @return The maximum number of records to fetch per shard in the Kinesis data stream per microbatch. Note: The
     *         client can exceed this limit if the streaming job has already read extra records from Kinesis (in the
     *         same get-records call). If <code>MaxFetchRecordsPerShard</code> needs to be strict then it needs to be a
     *         multiple of <code>MaxRecordPerRead</code>. The default value is <code>100000</code>.
     */
    public final Long maxFetchRecordsPerShard() {
        return maxFetchRecordsPerShard;
    }

    /**
     * <p>
     * The maximum number of records to fetch from the Kinesis data stream in each getRecords operation. The default
     * value is <code>10000</code>.
     * </p>
     * 
     * @return The maximum number of records to fetch from the Kinesis data stream in each getRecords operation. The
     *         default value is <code>10000</code>.
     */
    public final Long maxRecordPerRead() {
        return maxRecordPerRead;
    }

    /**
     * <p>
     * Adds a time delay between two consecutive getRecords operations. The default value is <code>"False"</code>. This
     * option is only configurable for Glue version 2.0 and above.
     * </p>
     * 
     * @return Adds a time delay between two consecutive getRecords operations. The default value is
     *         <code>"False"</code>. This option is only configurable for Glue version 2.0 and above.
     */
    public final Boolean addIdleTimeBetweenReads() {
        return addIdleTimeBetweenReads;
    }

    /**
     * <p>
     * The minimum time delay between two consecutive getRecords operations, specified in ms. The default value is
     * <code>1000</code>. This option is only configurable for Glue version 2.0 and above.
     * </p>
     * 
     * @return The minimum time delay between two consecutive getRecords operations, specified in ms. The default value
     *         is <code>1000</code>. This option is only configurable for Glue version 2.0 and above.
     */
    public final Long idleTimeBetweenReadsInMs() {
        return idleTimeBetweenReadsInMs;
    }

    /**
     * <p>
     * The minimum time interval between two ListShards API calls for your script to consider resharding. The default
     * value is <code>1s</code>.
     * </p>
     * 
     * @return The minimum time interval between two ListShards API calls for your script to consider resharding. The
     *         default value is <code>1s</code>.
     */
    public final Long describeShardInterval() {
        return describeShardInterval;
    }

    /**
     * <p>
     * The maximum number of retries for Kinesis Data Streams API requests. The default value is <code>3</code>.
     * </p>
     * 
     * @return The maximum number of retries for Kinesis Data Streams API requests. The default value is <code>3</code>.
     */
    public final Integer numRetries() {
        return numRetries;
    }

    /**
     * <p>
     * The cool-off time period (specified in ms) before retrying the Kinesis Data Streams API call. The default value
     * is <code>1000</code>.
     * </p>
     * 
     * @return The cool-off time period (specified in ms) before retrying the Kinesis Data Streams API call. The default
     *         value is <code>1000</code>.
     */
    public final Long retryIntervalMs() {
        return retryIntervalMs;
    }

    /**
     * <p>
     * The maximum cool-off time period (specified in ms) between two retries of a Kinesis Data Streams API call. The
     * default value is <code>10000</code>.
     * </p>
     * 
     * @return The maximum cool-off time period (specified in ms) between two retries of a Kinesis Data Streams API
     *         call. The default value is <code>10000</code>.
     */
    public final Long maxRetryIntervalMs() {
        return maxRetryIntervalMs;
    }

    /**
     * <p>
     * Avoids creating an empty microbatch job by checking for unread data in the Kinesis data stream before the batch
     * is started. The default value is <code>"False"</code>.
     * </p>
     * 
     * @return Avoids creating an empty microbatch job by checking for unread data in the Kinesis data stream before the
     *         batch is started. The default value is <code>"False"</code>.
     */
    public final Boolean avoidEmptyBatches() {
        return avoidEmptyBatches;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Kinesis data stream.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Kinesis data stream.
     */
    public final String streamArn() {
        return streamArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the role to assume using AWS Security Token Service (AWS STS). This role must
     * have permissions for describe or read record operations for the Kinesis data stream. You must use this parameter
     * when accessing a data stream in a different account. Used in conjunction with <code>"awsSTSSessionName"</code>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the role to assume using AWS Security Token Service (AWS STS). This
     *         role must have permissions for describe or read record operations for the Kinesis data stream. You must
     *         use this parameter when accessing a data stream in a different account. Used in conjunction with
     *         <code>"awsSTSSessionName"</code>.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * An identifier for the session assuming the role using AWS STS. You must use this parameter when accessing a data
     * stream in a different account. Used in conjunction with <code>"awsSTSRoleARN"</code>.
     * </p>
     * 
     * @return An identifier for the session assuming the role using AWS STS. You must use this parameter when accessing
     *         a data stream in a different account. Used in conjunction with <code>"awsSTSRoleARN"</code>.
     */
    public final String roleSessionName() {
        return roleSessionName;
    }

    /**
     * <p>
     * When this option is set to 'true', the data output will contain an additional column named "__src_timestamp" that
     * indicates the time when the corresponding record received by the stream. The default value is 'false'. This
     * option is supported in Glue version 4.0 or later.
     * </p>
     * 
     * @return When this option is set to 'true', the data output will contain an additional column named
     *         "__src_timestamp" that indicates the time when the corresponding record received by the stream. The
     *         default value is 'false'. This option is supported in Glue version 4.0 or later.
     */
    public final String addRecordTimestamp() {
        return addRecordTimestamp;
    }

    /**
     * <p>
     * When this option is set to 'true', for each batch, it will emit the metrics for the duration between the oldest
     * record received by the stream and the time it arrives in Glue to CloudWatch. The metric's name is
     * "glue.driver.streaming.maxConsumerLagInMs". The default value is 'false'. This option is supported in Glue
     * version 4.0 or later.
     * </p>
     * 
     * @return When this option is set to 'true', for each batch, it will emit the metrics for the duration between the
     *         oldest record received by the stream and the time it arrives in Glue to CloudWatch. The metric's name is
     *         "glue.driver.streaming.maxConsumerLagInMs". The default value is 'false'. This option is supported in
     *         Glue version 4.0 or later.
     */
    public final String emitConsumerLagMetrics() {
        return emitConsumerLagMetrics;
    }

    /**
     * <p>
     * The timestamp of the record in the Kinesis data stream to start reading data from. The possible values are a
     * timestamp string in UTC format of the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where Z represents a UTC
     * timezone offset with a +/-. For example: "2023-04-04T08:00:00+08:00").
     * </p>
     * 
     * @return The timestamp of the record in the Kinesis data stream to start reading data from. The possible values
     *         are a timestamp string in UTC format of the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where Z represents
     *         a UTC timezone offset with a +/-. For example: "2023-04-04T08:00:00+08:00").
     */
    public final Instant startingTimestamp() {
        return startingTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpointUrl());
        hashCode = 31 * hashCode + Objects.hashCode(streamName());
        hashCode = 31 * hashCode + Objects.hashCode(classification());
        hashCode = 31 * hashCode + Objects.hashCode(delimiter());
        hashCode = 31 * hashCode + Objects.hashCode(startingPositionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxFetchTimeInMs());
        hashCode = 31 * hashCode + Objects.hashCode(maxFetchRecordsPerShard());
        hashCode = 31 * hashCode + Objects.hashCode(maxRecordPerRead());
        hashCode = 31 * hashCode + Objects.hashCode(addIdleTimeBetweenReads());
        hashCode = 31 * hashCode + Objects.hashCode(idleTimeBetweenReadsInMs());
        hashCode = 31 * hashCode + Objects.hashCode(describeShardInterval());
        hashCode = 31 * hashCode + Objects.hashCode(numRetries());
        hashCode = 31 * hashCode + Objects.hashCode(retryIntervalMs());
        hashCode = 31 * hashCode + Objects.hashCode(maxRetryIntervalMs());
        hashCode = 31 * hashCode + Objects.hashCode(avoidEmptyBatches());
        hashCode = 31 * hashCode + Objects.hashCode(streamArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleSessionName());
        hashCode = 31 * hashCode + Objects.hashCode(addRecordTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(emitConsumerLagMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(startingTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KinesisStreamingSourceOptions)) {
            return false;
        }
        KinesisStreamingSourceOptions other = (KinesisStreamingSourceOptions) obj;
        return Objects.equals(endpointUrl(), other.endpointUrl()) && Objects.equals(streamName(), other.streamName())
                && Objects.equals(classification(), other.classification()) && Objects.equals(delimiter(), other.delimiter())
                && Objects.equals(startingPositionAsString(), other.startingPositionAsString())
                && Objects.equals(maxFetchTimeInMs(), other.maxFetchTimeInMs())
                && Objects.equals(maxFetchRecordsPerShard(), other.maxFetchRecordsPerShard())
                && Objects.equals(maxRecordPerRead(), other.maxRecordPerRead())
                && Objects.equals(addIdleTimeBetweenReads(), other.addIdleTimeBetweenReads())
                && Objects.equals(idleTimeBetweenReadsInMs(), other.idleTimeBetweenReadsInMs())
                && Objects.equals(describeShardInterval(), other.describeShardInterval())
                && Objects.equals(numRetries(), other.numRetries()) && Objects.equals(retryIntervalMs(), other.retryIntervalMs())
                && Objects.equals(maxRetryIntervalMs(), other.maxRetryIntervalMs())
                && Objects.equals(avoidEmptyBatches(), other.avoidEmptyBatches())
                && Objects.equals(streamArn(), other.streamArn()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(roleSessionName(), other.roleSessionName())
                && Objects.equals(addRecordTimestamp(), other.addRecordTimestamp())
                && Objects.equals(emitConsumerLagMetrics(), other.emitConsumerLagMetrics())
                && Objects.equals(startingTimestamp(), other.startingTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("KinesisStreamingSourceOptions").add("EndpointUrl", endpointUrl())
                .add("StreamName", streamName()).add("Classification", classification()).add("Delimiter", delimiter())
                .add("StartingPosition", startingPositionAsString()).add("MaxFetchTimeInMs", maxFetchTimeInMs())
                .add("MaxFetchRecordsPerShard", maxFetchRecordsPerShard()).add("MaxRecordPerRead", maxRecordPerRead())
                .add("AddIdleTimeBetweenReads", addIdleTimeBetweenReads())
                .add("IdleTimeBetweenReadsInMs", idleTimeBetweenReadsInMs())
                .add("DescribeShardInterval", describeShardInterval()).add("NumRetries", numRetries())
                .add("RetryIntervalMs", retryIntervalMs()).add("MaxRetryIntervalMs", maxRetryIntervalMs())
                .add("AvoidEmptyBatches", avoidEmptyBatches()).add("StreamArn", streamArn()).add("RoleArn", roleArn())
                .add("RoleSessionName", roleSessionName()).add("AddRecordTimestamp", addRecordTimestamp())
                .add("EmitConsumerLagMetrics", emitConsumerLagMetrics()).add("StartingTimestamp", startingTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EndpointUrl":
            return Optional.ofNullable(clazz.cast(endpointUrl()));
        case "StreamName":
            return Optional.ofNullable(clazz.cast(streamName()));
        case "Classification":
            return Optional.ofNullable(clazz.cast(classification()));
        case "Delimiter":
            return Optional.ofNullable(clazz.cast(delimiter()));
        case "StartingPosition":
            return Optional.ofNullable(clazz.cast(startingPositionAsString()));
        case "MaxFetchTimeInMs":
            return Optional.ofNullable(clazz.cast(maxFetchTimeInMs()));
        case "MaxFetchRecordsPerShard":
            return Optional.ofNullable(clazz.cast(maxFetchRecordsPerShard()));
        case "MaxRecordPerRead":
            return Optional.ofNullable(clazz.cast(maxRecordPerRead()));
        case "AddIdleTimeBetweenReads":
            return Optional.ofNullable(clazz.cast(addIdleTimeBetweenReads()));
        case "IdleTimeBetweenReadsInMs":
            return Optional.ofNullable(clazz.cast(idleTimeBetweenReadsInMs()));
        case "DescribeShardInterval":
            return Optional.ofNullable(clazz.cast(describeShardInterval()));
        case "NumRetries":
            return Optional.ofNullable(clazz.cast(numRetries()));
        case "RetryIntervalMs":
            return Optional.ofNullable(clazz.cast(retryIntervalMs()));
        case "MaxRetryIntervalMs":
            return Optional.ofNullable(clazz.cast(maxRetryIntervalMs()));
        case "AvoidEmptyBatches":
            return Optional.ofNullable(clazz.cast(avoidEmptyBatches()));
        case "StreamArn":
            return Optional.ofNullable(clazz.cast(streamArn()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "RoleSessionName":
            return Optional.ofNullable(clazz.cast(roleSessionName()));
        case "AddRecordTimestamp":
            return Optional.ofNullable(clazz.cast(addRecordTimestamp()));
        case "EmitConsumerLagMetrics":
            return Optional.ofNullable(clazz.cast(emitConsumerLagMetrics()));
        case "StartingTimestamp":
            return Optional.ofNullable(clazz.cast(startingTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KinesisStreamingSourceOptions, T> g) {
        return obj -> g.apply((KinesisStreamingSourceOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KinesisStreamingSourceOptions> {
        /**
         * <p>
         * The URL of the Kinesis endpoint.
         * </p>
         * 
         * @param endpointUrl
         *        The URL of the Kinesis endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointUrl(String endpointUrl);

        /**
         * <p>
         * The name of the Kinesis data stream.
         * </p>
         * 
         * @param streamName
         *        The name of the Kinesis data stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamName(String streamName);

        /**
         * <p>
         * An optional classification.
         * </p>
         * 
         * @param classification
         *        An optional classification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classification(String classification);

        /**
         * <p>
         * Specifies the delimiter character.
         * </p>
         * 
         * @param delimiter
         *        Specifies the delimiter character.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder delimiter(String delimiter);

        /**
         * <p>
         * The starting position in the Kinesis data stream to read data from. The possible values are
         * <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in UTC
         * format in the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC timezone
         * offset with a +/-. For example: "2023-04-04T08:00:00-04:00"). The default value is <code>"latest"</code>.
         * </p>
         * <p>
         * Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only for
         * Glue version 4.0 or later.
         * </p>
         * 
         * @param startingPosition
         *        The starting position in the Kinesis data stream to read data from. The possible values are
         *        <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in
         *        UTC format in the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC
         *        timezone offset with a +/-. For example: "2023-04-04T08:00:00-04:00"). The default value is
         *        <code>"latest"</code>.</p>
         *        <p>
         *        Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only
         *        for Glue version 4.0 or later.
         * @see StartingPosition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StartingPosition
         */
        Builder startingPosition(String startingPosition);

        /**
         * <p>
         * The starting position in the Kinesis data stream to read data from. The possible values are
         * <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in UTC
         * format in the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC timezone
         * offset with a +/-. For example: "2023-04-04T08:00:00-04:00"). The default value is <code>"latest"</code>.
         * </p>
         * <p>
         * Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only for
         * Glue version 4.0 or later.
         * </p>
         * 
         * @param startingPosition
         *        The starting position in the Kinesis data stream to read data from. The possible values are
         *        <code>"latest"</code>, <code>"trim_horizon"</code>, <code>"earliest"</code>, or a timestamp string in
         *        UTC format in the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where <code>Z</code> represents a UTC
         *        timezone offset with a +/-. For example: "2023-04-04T08:00:00-04:00"). The default value is
         *        <code>"latest"</code>.</p>
         *        <p>
         *        Note: Using a value that is a timestamp string in UTC format for "startingPosition" is supported only
         *        for Glue version 4.0 or later.
         * @see StartingPosition
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StartingPosition
         */
        Builder startingPosition(StartingPosition startingPosition);

        /**
         * <p>
         * The maximum time spent for the job executor to read records for the current batch from the Kinesis data
         * stream, specified in milliseconds (ms). Multiple <code>GetRecords</code> API calls may be made within this
         * time. The default value is <code>1000</code>.
         * </p>
         * 
         * @param maxFetchTimeInMs
         *        The maximum time spent for the job executor to read records for the current batch from the Kinesis
         *        data stream, specified in milliseconds (ms). Multiple <code>GetRecords</code> API calls may be made
         *        within this time. The default value is <code>1000</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxFetchTimeInMs(Long maxFetchTimeInMs);

        /**
         * <p>
         * The maximum number of records to fetch per shard in the Kinesis data stream per microbatch. Note: The client
         * can exceed this limit if the streaming job has already read extra records from Kinesis (in the same
         * get-records call). If <code>MaxFetchRecordsPerShard</code> needs to be strict then it needs to be a multiple
         * of <code>MaxRecordPerRead</code>. The default value is <code>100000</code>.
         * </p>
         * 
         * @param maxFetchRecordsPerShard
         *        The maximum number of records to fetch per shard in the Kinesis data stream per microbatch. Note: The
         *        client can exceed this limit if the streaming job has already read extra records from Kinesis (in the
         *        same get-records call). If <code>MaxFetchRecordsPerShard</code> needs to be strict then it needs to be
         *        a multiple of <code>MaxRecordPerRead</code>. The default value is <code>100000</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxFetchRecordsPerShard(Long maxFetchRecordsPerShard);

        /**
         * <p>
         * The maximum number of records to fetch from the Kinesis data stream in each getRecords operation. The default
         * value is <code>10000</code>.
         * </p>
         * 
         * @param maxRecordPerRead
         *        The maximum number of records to fetch from the Kinesis data stream in each getRecords operation. The
         *        default value is <code>10000</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRecordPerRead(Long maxRecordPerRead);

        /**
         * <p>
         * Adds a time delay between two consecutive getRecords operations. The default value is <code>"False"</code>.
         * This option is only configurable for Glue version 2.0 and above.
         * </p>
         * 
         * @param addIdleTimeBetweenReads
         *        Adds a time delay between two consecutive getRecords operations. The default value is
         *        <code>"False"</code>. This option is only configurable for Glue version 2.0 and above.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addIdleTimeBetweenReads(Boolean addIdleTimeBetweenReads);

        /**
         * <p>
         * The minimum time delay between two consecutive getRecords operations, specified in ms. The default value is
         * <code>1000</code>. This option is only configurable for Glue version 2.0 and above.
         * </p>
         * 
         * @param idleTimeBetweenReadsInMs
         *        The minimum time delay between two consecutive getRecords operations, specified in ms. The default
         *        value is <code>1000</code>. This option is only configurable for Glue version 2.0 and above.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder idleTimeBetweenReadsInMs(Long idleTimeBetweenReadsInMs);

        /**
         * <p>
         * The minimum time interval between two ListShards API calls for your script to consider resharding. The
         * default value is <code>1s</code>.
         * </p>
         * 
         * @param describeShardInterval
         *        The minimum time interval between two ListShards API calls for your script to consider resharding. The
         *        default value is <code>1s</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder describeShardInterval(Long describeShardInterval);

        /**
         * <p>
         * The maximum number of retries for Kinesis Data Streams API requests. The default value is <code>3</code>.
         * </p>
         * 
         * @param numRetries
         *        The maximum number of retries for Kinesis Data Streams API requests. The default value is
         *        <code>3</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numRetries(Integer numRetries);

        /**
         * <p>
         * The cool-off time period (specified in ms) before retrying the Kinesis Data Streams API call. The default
         * value is <code>1000</code>.
         * </p>
         * 
         * @param retryIntervalMs
         *        The cool-off time period (specified in ms) before retrying the Kinesis Data Streams API call. The
         *        default value is <code>1000</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder retryIntervalMs(Long retryIntervalMs);

        /**
         * <p>
         * The maximum cool-off time period (specified in ms) between two retries of a Kinesis Data Streams API call.
         * The default value is <code>10000</code>.
         * </p>
         * 
         * @param maxRetryIntervalMs
         *        The maximum cool-off time period (specified in ms) between two retries of a Kinesis Data Streams API
         *        call. The default value is <code>10000</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxRetryIntervalMs(Long maxRetryIntervalMs);

        /**
         * <p>
         * Avoids creating an empty microbatch job by checking for unread data in the Kinesis data stream before the
         * batch is started. The default value is <code>"False"</code>.
         * </p>
         * 
         * @param avoidEmptyBatches
         *        Avoids creating an empty microbatch job by checking for unread data in the Kinesis data stream before
         *        the batch is started. The default value is <code>"False"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder avoidEmptyBatches(Boolean avoidEmptyBatches);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Kinesis data stream.
         * </p>
         * 
         * @param streamArn
         *        The Amazon Resource Name (ARN) of the Kinesis data stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamArn(String streamArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the role to assume using AWS Security Token Service (AWS STS). This role
         * must have permissions for describe or read record operations for the Kinesis data stream. You must use this
         * parameter when accessing a data stream in a different account. Used in conjunction with
         * <code>"awsSTSSessionName"</code>.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the role to assume using AWS Security Token Service (AWS STS). This
         *        role must have permissions for describe or read record operations for the Kinesis data stream. You
         *        must use this parameter when accessing a data stream in a different account. Used in conjunction with
         *        <code>"awsSTSSessionName"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * An identifier for the session assuming the role using AWS STS. You must use this parameter when accessing a
         * data stream in a different account. Used in conjunction with <code>"awsSTSRoleARN"</code>.
         * </p>
         * 
         * @param roleSessionName
         *        An identifier for the session assuming the role using AWS STS. You must use this parameter when
         *        accessing a data stream in a different account. Used in conjunction with <code>"awsSTSRoleARN"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleSessionName(String roleSessionName);

        /**
         * <p>
         * When this option is set to 'true', the data output will contain an additional column named "__src_timestamp"
         * that indicates the time when the corresponding record received by the stream. The default value is 'false'.
         * This option is supported in Glue version 4.0 or later.
         * </p>
         * 
         * @param addRecordTimestamp
         *        When this option is set to 'true', the data output will contain an additional column named
         *        "__src_timestamp" that indicates the time when the corresponding record received by the stream. The
         *        default value is 'false'. This option is supported in Glue version 4.0 or later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder addRecordTimestamp(String addRecordTimestamp);

        /**
         * <p>
         * When this option is set to 'true', for each batch, it will emit the metrics for the duration between the
         * oldest record received by the stream and the time it arrives in Glue to CloudWatch. The metric's name is
         * "glue.driver.streaming.maxConsumerLagInMs". The default value is 'false'. This option is supported in Glue
         * version 4.0 or later.
         * </p>
         * 
         * @param emitConsumerLagMetrics
         *        When this option is set to 'true', for each batch, it will emit the metrics for the duration between
         *        the oldest record received by the stream and the time it arrives in Glue to CloudWatch. The metric's
         *        name is "glue.driver.streaming.maxConsumerLagInMs". The default value is 'false'. This option is
         *        supported in Glue version 4.0 or later.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder emitConsumerLagMetrics(String emitConsumerLagMetrics);

        /**
         * <p>
         * The timestamp of the record in the Kinesis data stream to start reading data from. The possible values are a
         * timestamp string in UTC format of the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where Z represents a UTC
         * timezone offset with a +/-. For example: "2023-04-04T08:00:00+08:00").
         * </p>
         * 
         * @param startingTimestamp
         *        The timestamp of the record in the Kinesis data stream to start reading data from. The possible values
         *        are a timestamp string in UTC format of the pattern <code>yyyy-mm-ddTHH:MM:SSZ</code> (where Z
         *        represents a UTC timezone offset with a +/-. For example: "2023-04-04T08:00:00+08:00").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startingTimestamp(Instant startingTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String endpointUrl;

        private String streamName;

        private String classification;

        private String delimiter;

        private String startingPosition;

        private Long maxFetchTimeInMs;

        private Long maxFetchRecordsPerShard;

        private Long maxRecordPerRead;

        private Boolean addIdleTimeBetweenReads;

        private Long idleTimeBetweenReadsInMs;

        private Long describeShardInterval;

        private Integer numRetries;

        private Long retryIntervalMs;

        private Long maxRetryIntervalMs;

        private Boolean avoidEmptyBatches;

        private String streamArn;

        private String roleArn;

        private String roleSessionName;

        private String addRecordTimestamp;

        private String emitConsumerLagMetrics;

        private Instant startingTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(KinesisStreamingSourceOptions model) {
            endpointUrl(model.endpointUrl);
            streamName(model.streamName);
            classification(model.classification);
            delimiter(model.delimiter);
            startingPosition(model.startingPosition);
            maxFetchTimeInMs(model.maxFetchTimeInMs);
            maxFetchRecordsPerShard(model.maxFetchRecordsPerShard);
            maxRecordPerRead(model.maxRecordPerRead);
            addIdleTimeBetweenReads(model.addIdleTimeBetweenReads);
            idleTimeBetweenReadsInMs(model.idleTimeBetweenReadsInMs);
            describeShardInterval(model.describeShardInterval);
            numRetries(model.numRetries);
            retryIntervalMs(model.retryIntervalMs);
            maxRetryIntervalMs(model.maxRetryIntervalMs);
            avoidEmptyBatches(model.avoidEmptyBatches);
            streamArn(model.streamArn);
            roleArn(model.roleArn);
            roleSessionName(model.roleSessionName);
            addRecordTimestamp(model.addRecordTimestamp);
            emitConsumerLagMetrics(model.emitConsumerLagMetrics);
            startingTimestamp(model.startingTimestamp);
        }

        public final String getEndpointUrl() {
            return endpointUrl;
        }

        public final void setEndpointUrl(String endpointUrl) {
            this.endpointUrl = endpointUrl;
        }

        @Override
        public final Builder endpointUrl(String endpointUrl) {
            this.endpointUrl = endpointUrl;
            return this;
        }

        public final String getStreamName() {
            return streamName;
        }

        public final void setStreamName(String streamName) {
            this.streamName = streamName;
        }

        @Override
        public final Builder streamName(String streamName) {
            this.streamName = streamName;
            return this;
        }

        public final String getClassification() {
            return classification;
        }

        public final void setClassification(String classification) {
            this.classification = classification;
        }

        @Override
        public final Builder classification(String classification) {
            this.classification = classification;
            return this;
        }

        public final String getDelimiter() {
            return delimiter;
        }

        public final void setDelimiter(String delimiter) {
            this.delimiter = delimiter;
        }

        @Override
        public final Builder delimiter(String delimiter) {
            this.delimiter = delimiter;
            return this;
        }

        public final String getStartingPosition() {
            return startingPosition;
        }

        public final void setStartingPosition(String startingPosition) {
            this.startingPosition = startingPosition;
        }

        @Override
        public final Builder startingPosition(String startingPosition) {
            this.startingPosition = startingPosition;
            return this;
        }

        @Override
        public final Builder startingPosition(StartingPosition startingPosition) {
            this.startingPosition(startingPosition == null ? null : startingPosition.toString());
            return this;
        }

        public final Long getMaxFetchTimeInMs() {
            return maxFetchTimeInMs;
        }

        public final void setMaxFetchTimeInMs(Long maxFetchTimeInMs) {
            this.maxFetchTimeInMs = maxFetchTimeInMs;
        }

        @Override
        public final Builder maxFetchTimeInMs(Long maxFetchTimeInMs) {
            this.maxFetchTimeInMs = maxFetchTimeInMs;
            return this;
        }

        public final Long getMaxFetchRecordsPerShard() {
            return maxFetchRecordsPerShard;
        }

        public final void setMaxFetchRecordsPerShard(Long maxFetchRecordsPerShard) {
            this.maxFetchRecordsPerShard = maxFetchRecordsPerShard;
        }

        @Override
        public final Builder maxFetchRecordsPerShard(Long maxFetchRecordsPerShard) {
            this.maxFetchRecordsPerShard = maxFetchRecordsPerShard;
            return this;
        }

        public final Long getMaxRecordPerRead() {
            return maxRecordPerRead;
        }

        public final void setMaxRecordPerRead(Long maxRecordPerRead) {
            this.maxRecordPerRead = maxRecordPerRead;
        }

        @Override
        public final Builder maxRecordPerRead(Long maxRecordPerRead) {
            this.maxRecordPerRead = maxRecordPerRead;
            return this;
        }

        public final Boolean getAddIdleTimeBetweenReads() {
            return addIdleTimeBetweenReads;
        }

        public final void setAddIdleTimeBetweenReads(Boolean addIdleTimeBetweenReads) {
            this.addIdleTimeBetweenReads = addIdleTimeBetweenReads;
        }

        @Override
        public final Builder addIdleTimeBetweenReads(Boolean addIdleTimeBetweenReads) {
            this.addIdleTimeBetweenReads = addIdleTimeBetweenReads;
            return this;
        }

        public final Long getIdleTimeBetweenReadsInMs() {
            return idleTimeBetweenReadsInMs;
        }

        public final void setIdleTimeBetweenReadsInMs(Long idleTimeBetweenReadsInMs) {
            this.idleTimeBetweenReadsInMs = idleTimeBetweenReadsInMs;
        }

        @Override
        public final Builder idleTimeBetweenReadsInMs(Long idleTimeBetweenReadsInMs) {
            this.idleTimeBetweenReadsInMs = idleTimeBetweenReadsInMs;
            return this;
        }

        public final Long getDescribeShardInterval() {
            return describeShardInterval;
        }

        public final void setDescribeShardInterval(Long describeShardInterval) {
            this.describeShardInterval = describeShardInterval;
        }

        @Override
        public final Builder describeShardInterval(Long describeShardInterval) {
            this.describeShardInterval = describeShardInterval;
            return this;
        }

        public final Integer getNumRetries() {
            return numRetries;
        }

        public final void setNumRetries(Integer numRetries) {
            this.numRetries = numRetries;
        }

        @Override
        public final Builder numRetries(Integer numRetries) {
            this.numRetries = numRetries;
            return this;
        }

        public final Long getRetryIntervalMs() {
            return retryIntervalMs;
        }

        public final void setRetryIntervalMs(Long retryIntervalMs) {
            this.retryIntervalMs = retryIntervalMs;
        }

        @Override
        public final Builder retryIntervalMs(Long retryIntervalMs) {
            this.retryIntervalMs = retryIntervalMs;
            return this;
        }

        public final Long getMaxRetryIntervalMs() {
            return maxRetryIntervalMs;
        }

        public final void setMaxRetryIntervalMs(Long maxRetryIntervalMs) {
            this.maxRetryIntervalMs = maxRetryIntervalMs;
        }

        @Override
        public final Builder maxRetryIntervalMs(Long maxRetryIntervalMs) {
            this.maxRetryIntervalMs = maxRetryIntervalMs;
            return this;
        }

        public final Boolean getAvoidEmptyBatches() {
            return avoidEmptyBatches;
        }

        public final void setAvoidEmptyBatches(Boolean avoidEmptyBatches) {
            this.avoidEmptyBatches = avoidEmptyBatches;
        }

        @Override
        public final Builder avoidEmptyBatches(Boolean avoidEmptyBatches) {
            this.avoidEmptyBatches = avoidEmptyBatches;
            return this;
        }

        public final String getStreamArn() {
            return streamArn;
        }

        public final void setStreamArn(String streamArn) {
            this.streamArn = streamArn;
        }

        @Override
        public final Builder streamArn(String streamArn) {
            this.streamArn = streamArn;
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final String getRoleSessionName() {
            return roleSessionName;
        }

        public final void setRoleSessionName(String roleSessionName) {
            this.roleSessionName = roleSessionName;
        }

        @Override
        public final Builder roleSessionName(String roleSessionName) {
            this.roleSessionName = roleSessionName;
            return this;
        }

        public final String getAddRecordTimestamp() {
            return addRecordTimestamp;
        }

        public final void setAddRecordTimestamp(String addRecordTimestamp) {
            this.addRecordTimestamp = addRecordTimestamp;
        }

        @Override
        public final Builder addRecordTimestamp(String addRecordTimestamp) {
            this.addRecordTimestamp = addRecordTimestamp;
            return this;
        }

        public final String getEmitConsumerLagMetrics() {
            return emitConsumerLagMetrics;
        }

        public final void setEmitConsumerLagMetrics(String emitConsumerLagMetrics) {
            this.emitConsumerLagMetrics = emitConsumerLagMetrics;
        }

        @Override
        public final Builder emitConsumerLagMetrics(String emitConsumerLagMetrics) {
            this.emitConsumerLagMetrics = emitConsumerLagMetrics;
            return this;
        }

        public final Instant getStartingTimestamp() {
            return startingTimestamp;
        }

        public final void setStartingTimestamp(Instant startingTimestamp) {
            this.startingTimestamp = startingTimestamp;
        }

        @Override
        public final Builder startingTimestamp(Instant startingTimestamp) {
            this.startingTimestamp = startingTimestamp;
            return this;
        }

        @Override
        public KinesisStreamingSourceOptions build() {
            return new KinesisStreamingSourceOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
