/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the metric based observation generated based on evaluated data quality metrics.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricBasedObservation implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricBasedObservation.Builder, MetricBasedObservation> {
    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(MetricBasedObservation::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<DataQualityMetricValues> METRIC_VALUES_FIELD = SdkField
            .<DataQualityMetricValues> builder(MarshallingType.SDK_POJO).memberName("MetricValues")
            .getter(getter(MetricBasedObservation::metricValues)).setter(setter(Builder::metricValues))
            .constructor(DataQualityMetricValues::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricValues").build()).build();

    private static final SdkField<List<String>> NEW_RULES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NewRules")
            .getter(getter(MetricBasedObservation::newRules))
            .setter(setter(Builder::newRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NewRules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_NAME_FIELD,
            METRIC_VALUES_FIELD, NEW_RULES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricName;

    private final DataQualityMetricValues metricValues;

    private final List<String> newRules;

    private MetricBasedObservation(BuilderImpl builder) {
        this.metricName = builder.metricName;
        this.metricValues = builder.metricValues;
        this.newRules = builder.newRules;
    }

    /**
     * <p>
     * The name of the data quality metric used for generating the observation.
     * </p>
     * 
     * @return The name of the data quality metric used for generating the observation.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * <p>
     * An object of type <code>DataQualityMetricValues</code> representing the analysis of the data quality metric
     * value.
     * </p>
     * 
     * @return An object of type <code>DataQualityMetricValues</code> representing the analysis of the data quality
     *         metric value.
     */
    public final DataQualityMetricValues metricValues() {
        return metricValues;
    }

    /**
     * For responses, this returns true if the service returned a value for the NewRules property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNewRules() {
        return newRules != null && !(newRules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of new data quality rules generated as part of the observation based on the data quality metric value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNewRules} method.
     * </p>
     * 
     * @return A list of new data quality rules generated as part of the observation based on the data quality metric
     *         value.
     */
    public final List<String> newRules() {
        return newRules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(metricValues());
        hashCode = 31 * hashCode + Objects.hashCode(hasNewRules() ? newRules() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricBasedObservation)) {
            return false;
        }
        MetricBasedObservation other = (MetricBasedObservation) obj;
        return Objects.equals(metricName(), other.metricName()) && Objects.equals(metricValues(), other.metricValues())
                && hasNewRules() == other.hasNewRules() && Objects.equals(newRules(), other.newRules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricBasedObservation").add("MetricName", metricName()).add("MetricValues", metricValues())
                .add("NewRules", hasNewRules() ? newRules() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "MetricValues":
            return Optional.ofNullable(clazz.cast(metricValues()));
        case "NewRules":
            return Optional.ofNullable(clazz.cast(newRules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricBasedObservation, T> g) {
        return obj -> g.apply((MetricBasedObservation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricBasedObservation> {
        /**
         * <p>
         * The name of the data quality metric used for generating the observation.
         * </p>
         * 
         * @param metricName
         *        The name of the data quality metric used for generating the observation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * An object of type <code>DataQualityMetricValues</code> representing the analysis of the data quality metric
         * value.
         * </p>
         * 
         * @param metricValues
         *        An object of type <code>DataQualityMetricValues</code> representing the analysis of the data quality
         *        metric value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricValues(DataQualityMetricValues metricValues);

        /**
         * <p>
         * An object of type <code>DataQualityMetricValues</code> representing the analysis of the data quality metric
         * value.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataQualityMetricValues.Builder} avoiding
         * the need to create one manually via {@link DataQualityMetricValues#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataQualityMetricValues.Builder#build()} is called immediately
         * and its result is passed to {@link #metricValues(DataQualityMetricValues)}.
         * 
         * @param metricValues
         *        a consumer that will call methods on {@link DataQualityMetricValues.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricValues(DataQualityMetricValues)
         */
        default Builder metricValues(Consumer<DataQualityMetricValues.Builder> metricValues) {
            return metricValues(DataQualityMetricValues.builder().applyMutation(metricValues).build());
        }

        /**
         * <p>
         * A list of new data quality rules generated as part of the observation based on the data quality metric value.
         * </p>
         * 
         * @param newRules
         *        A list of new data quality rules generated as part of the observation based on the data quality metric
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newRules(Collection<String> newRules);

        /**
         * <p>
         * A list of new data quality rules generated as part of the observation based on the data quality metric value.
         * </p>
         * 
         * @param newRules
         *        A list of new data quality rules generated as part of the observation based on the data quality metric
         *        value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder newRules(String... newRules);
    }

    static final class BuilderImpl implements Builder {
        private String metricName;

        private DataQualityMetricValues metricValues;

        private List<String> newRules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricBasedObservation model) {
            metricName(model.metricName);
            metricValues(model.metricValues);
            newRules(model.newRules);
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final DataQualityMetricValues.Builder getMetricValues() {
            return metricValues != null ? metricValues.toBuilder() : null;
        }

        public final void setMetricValues(DataQualityMetricValues.BuilderImpl metricValues) {
            this.metricValues = metricValues != null ? metricValues.build() : null;
        }

        @Override
        public final Builder metricValues(DataQualityMetricValues metricValues) {
            this.metricValues = metricValues;
            return this;
        }

        public final Collection<String> getNewRules() {
            if (newRules instanceof SdkAutoConstructList) {
                return null;
            }
            return newRules;
        }

        public final void setNewRules(Collection<String> newRules) {
            this.newRules = NewRulesCopier.copy(newRules);
        }

        @Override
        public final Builder newRules(Collection<String> newRules) {
            this.newRules = NewRulesCopier.copy(newRules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder newRules(String... newRules) {
            newRules(Arrays.asList(newRules));
            return this;
        }

        @Override
        public MetricBasedObservation build() {
            return new MetricBasedObservation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
