/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a crawler program that examines a data source and uses classifiers to try to determine its schema. If
 * successful, the crawler records metadata concerning the data source in the AWS Glue Data Catalog.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Crawler implements SdkPojo, Serializable, ToCopyableBuilder<Crawler.Builder, Crawler> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Crawler::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Role")
            .getter(getter(Crawler::role)).setter(setter(Builder::role))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Role").build()).build();

    private static final SdkField<CrawlerTargets> TARGETS_FIELD = SdkField.<CrawlerTargets> builder(MarshallingType.SDK_POJO)
            .memberName("Targets").getter(getter(Crawler::targets)).setter(setter(Builder::targets))
            .constructor(CrawlerTargets::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Targets").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DatabaseName").getter(getter(Crawler::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Crawler::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<String>> CLASSIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Classifiers")
            .getter(getter(Crawler::classifiers))
            .setter(setter(Builder::classifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Classifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RecrawlPolicy> RECRAWL_POLICY_FIELD = SdkField
            .<RecrawlPolicy> builder(MarshallingType.SDK_POJO).memberName("RecrawlPolicy").getter(getter(Crawler::recrawlPolicy))
            .setter(setter(Builder::recrawlPolicy)).constructor(RecrawlPolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecrawlPolicy").build()).build();

    private static final SdkField<SchemaChangePolicy> SCHEMA_CHANGE_POLICY_FIELD = SdkField
            .<SchemaChangePolicy> builder(MarshallingType.SDK_POJO).memberName("SchemaChangePolicy")
            .getter(getter(Crawler::schemaChangePolicy)).setter(setter(Builder::schemaChangePolicy))
            .constructor(SchemaChangePolicy::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SchemaChangePolicy").build())
            .build();

    private static final SdkField<LineageConfiguration> LINEAGE_CONFIGURATION_FIELD = SdkField
            .<LineageConfiguration> builder(MarshallingType.SDK_POJO).memberName("LineageConfiguration")
            .getter(getter(Crawler::lineageConfiguration)).setter(setter(Builder::lineageConfiguration))
            .constructor(LineageConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LineageConfiguration").build())
            .build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(Crawler::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> TABLE_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TablePrefix").getter(getter(Crawler::tablePrefix)).setter(setter(Builder::tablePrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TablePrefix").build()).build();

    private static final SdkField<Schedule> SCHEDULE_FIELD = SdkField.<Schedule> builder(MarshallingType.SDK_POJO)
            .memberName("Schedule").getter(getter(Crawler::schedule)).setter(setter(Builder::schedule))
            .constructor(Schedule::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Schedule").build()).build();

    private static final SdkField<Long> CRAWL_ELAPSED_TIME_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("CrawlElapsedTime").getter(getter(Crawler::crawlElapsedTime)).setter(setter(Builder::crawlElapsedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlElapsedTime").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Crawler::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdated").getter(getter(Crawler::lastUpdated)).setter(setter(Builder::lastUpdated))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdated").build()).build();

    private static final SdkField<LastCrawlInfo> LAST_CRAWL_FIELD = SdkField.<LastCrawlInfo> builder(MarshallingType.SDK_POJO)
            .memberName("LastCrawl").getter(getter(Crawler::lastCrawl)).setter(setter(Builder::lastCrawl))
            .constructor(LastCrawlInfo::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastCrawl").build()).build();

    private static final SdkField<Long> VERSION_FIELD = SdkField.<Long> builder(MarshallingType.LONG).memberName("Version")
            .getter(getter(Crawler::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> CONFIGURATION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Configuration").getter(getter(Crawler::configuration)).setter(setter(Builder::configuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configuration").build()).build();

    private static final SdkField<String> CRAWLER_SECURITY_CONFIGURATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CrawlerSecurityConfiguration")
            .getter(getter(Crawler::crawlerSecurityConfiguration))
            .setter(setter(Builder::crawlerSecurityConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CrawlerSecurityConfiguration")
                    .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ROLE_FIELD,
            TARGETS_FIELD, DATABASE_NAME_FIELD, DESCRIPTION_FIELD, CLASSIFIERS_FIELD, RECRAWL_POLICY_FIELD,
            SCHEMA_CHANGE_POLICY_FIELD, LINEAGE_CONFIGURATION_FIELD, STATE_FIELD, TABLE_PREFIX_FIELD, SCHEDULE_FIELD,
            CRAWL_ELAPSED_TIME_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_FIELD, LAST_CRAWL_FIELD, VERSION_FIELD,
            CONFIGURATION_FIELD, CRAWLER_SECURITY_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String role;

    private final CrawlerTargets targets;

    private final String databaseName;

    private final String description;

    private final List<String> classifiers;

    private final RecrawlPolicy recrawlPolicy;

    private final SchemaChangePolicy schemaChangePolicy;

    private final LineageConfiguration lineageConfiguration;

    private final String state;

    private final String tablePrefix;

    private final Schedule schedule;

    private final Long crawlElapsedTime;

    private final Instant creationTime;

    private final Instant lastUpdated;

    private final LastCrawlInfo lastCrawl;

    private final Long version;

    private final String configuration;

    private final String crawlerSecurityConfiguration;

    private Crawler(BuilderImpl builder) {
        this.name = builder.name;
        this.role = builder.role;
        this.targets = builder.targets;
        this.databaseName = builder.databaseName;
        this.description = builder.description;
        this.classifiers = builder.classifiers;
        this.recrawlPolicy = builder.recrawlPolicy;
        this.schemaChangePolicy = builder.schemaChangePolicy;
        this.lineageConfiguration = builder.lineageConfiguration;
        this.state = builder.state;
        this.tablePrefix = builder.tablePrefix;
        this.schedule = builder.schedule;
        this.crawlElapsedTime = builder.crawlElapsedTime;
        this.creationTime = builder.creationTime;
        this.lastUpdated = builder.lastUpdated;
        this.lastCrawl = builder.lastCrawl;
        this.version = builder.version;
        this.configuration = builder.configuration;
        this.crawlerSecurityConfiguration = builder.crawlerSecurityConfiguration;
    }

    /**
     * <p>
     * The name of the crawler.
     * </p>
     * 
     * @return The name of the crawler.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of an IAM role that's used to access customer resources, such as Amazon Simple
     * Storage Service (Amazon S3) data.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of an IAM role that's used to access customer resources, such as Amazon
     *         Simple Storage Service (Amazon S3) data.
     */
    public final String role() {
        return role;
    }

    /**
     * <p>
     * A collection of targets to crawl.
     * </p>
     * 
     * @return A collection of targets to crawl.
     */
    public final CrawlerTargets targets() {
        return targets;
    }

    /**
     * <p>
     * The name of the database in which the crawler's output is stored.
     * </p>
     * 
     * @return The name of the database in which the crawler's output is stored.
     */
    public final String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * A description of the crawler.
     * </p>
     * 
     * @return A description of the crawler.
     */
    public final String description() {
        return description;
    }

    /**
     * Returns true if the Classifiers property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasClassifiers() {
        return classifiers != null && !(classifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of UTF-8 strings that specify the custom classifiers that are associated with the crawler.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasClassifiers()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of UTF-8 strings that specify the custom classifiers that are associated with the crawler.
     */
    public final List<String> classifiers() {
        return classifiers;
    }

    /**
     * <p>
     * A policy that specifies whether to crawl the entire dataset again, or to crawl only folders that were added since
     * the last crawler run.
     * </p>
     * 
     * @return A policy that specifies whether to crawl the entire dataset again, or to crawl only folders that were
     *         added since the last crawler run.
     */
    public final RecrawlPolicy recrawlPolicy() {
        return recrawlPolicy;
    }

    /**
     * <p>
     * The policy that specifies update and delete behaviors for the crawler.
     * </p>
     * 
     * @return The policy that specifies update and delete behaviors for the crawler.
     */
    public final SchemaChangePolicy schemaChangePolicy() {
        return schemaChangePolicy;
    }

    /**
     * <p>
     * A configuration that specifies whether data lineage is enabled for the crawler.
     * </p>
     * 
     * @return A configuration that specifies whether data lineage is enabled for the crawler.
     */
    public final LineageConfiguration lineageConfiguration() {
        return lineageConfiguration;
    }

    /**
     * <p>
     * Indicates whether the crawler is running, or whether a run is pending.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CrawlerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Indicates whether the crawler is running, or whether a run is pending.
     * @see CrawlerState
     */
    public final CrawlerState state() {
        return CrawlerState.fromValue(state);
    }

    /**
     * <p>
     * Indicates whether the crawler is running, or whether a run is pending.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CrawlerState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return Indicates whether the crawler is running, or whether a run is pending.
     * @see CrawlerState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The prefix added to the names of tables that are created.
     * </p>
     * 
     * @return The prefix added to the names of tables that are created.
     */
    public final String tablePrefix() {
        return tablePrefix;
    }

    /**
     * <p>
     * For scheduled crawlers, the schedule when the crawler runs.
     * </p>
     * 
     * @return For scheduled crawlers, the schedule when the crawler runs.
     */
    public final Schedule schedule() {
        return schedule;
    }

    /**
     * <p>
     * If the crawler is running, contains the total time elapsed since the last crawl began.
     * </p>
     * 
     * @return If the crawler is running, contains the total time elapsed since the last crawl began.
     */
    public final Long crawlElapsedTime() {
        return crawlElapsedTime;
    }

    /**
     * <p>
     * The time that the crawler was created.
     * </p>
     * 
     * @return The time that the crawler was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time that the crawler was last updated.
     * </p>
     * 
     * @return The time that the crawler was last updated.
     */
    public final Instant lastUpdated() {
        return lastUpdated;
    }

    /**
     * <p>
     * The status of the last crawl, and potentially error information if an error occurred.
     * </p>
     * 
     * @return The status of the last crawl, and potentially error information if an error occurred.
     */
    public final LastCrawlInfo lastCrawl() {
        return lastCrawl;
    }

    /**
     * <p>
     * The version of the crawler.
     * </p>
     * 
     * @return The version of the crawler.
     */
    public final Long version() {
        return version;
    }

    /**
     * <p>
     * Crawler configuration information. This versioned JSON string allows users to specify aspects of a crawler's
     * behavior. For more information, see <a
     * href="https://docs.aws.amazon.com/glue/latest/dg/crawler-configuration.html">Configuring a Crawler</a>.
     * </p>
     * 
     * @return Crawler configuration information. This versioned JSON string allows users to specify aspects of a
     *         crawler's behavior. For more information, see <a
     *         href="https://docs.aws.amazon.com/glue/latest/dg/crawler-configuration.html">Configuring a Crawler</a>.
     */
    public final String configuration() {
        return configuration;
    }

    /**
     * <p>
     * The name of the <code>SecurityConfiguration</code> structure to be used by this crawler.
     * </p>
     * 
     * @return The name of the <code>SecurityConfiguration</code> structure to be used by this crawler.
     */
    public final String crawlerSecurityConfiguration() {
        return crawlerSecurityConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(role());
        hashCode = 31 * hashCode + Objects.hashCode(targets());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasClassifiers() ? classifiers() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recrawlPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(schemaChangePolicy());
        hashCode = 31 * hashCode + Objects.hashCode(lineageConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tablePrefix());
        hashCode = 31 * hashCode + Objects.hashCode(schedule());
        hashCode = 31 * hashCode + Objects.hashCode(crawlElapsedTime());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdated());
        hashCode = 31 * hashCode + Objects.hashCode(lastCrawl());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(crawlerSecurityConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Crawler)) {
            return false;
        }
        Crawler other = (Crawler) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(role(), other.role())
                && Objects.equals(targets(), other.targets()) && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(description(), other.description()) && hasClassifiers() == other.hasClassifiers()
                && Objects.equals(classifiers(), other.classifiers()) && Objects.equals(recrawlPolicy(), other.recrawlPolicy())
                && Objects.equals(schemaChangePolicy(), other.schemaChangePolicy())
                && Objects.equals(lineageConfiguration(), other.lineageConfiguration())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(tablePrefix(), other.tablePrefix())
                && Objects.equals(schedule(), other.schedule()) && Objects.equals(crawlElapsedTime(), other.crawlElapsedTime())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(lastUpdated(), other.lastUpdated())
                && Objects.equals(lastCrawl(), other.lastCrawl()) && Objects.equals(version(), other.version())
                && Objects.equals(configuration(), other.configuration())
                && Objects.equals(crawlerSecurityConfiguration(), other.crawlerSecurityConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Crawler").add("Name", name()).add("Role", role()).add("Targets", targets())
                .add("DatabaseName", databaseName()).add("Description", description())
                .add("Classifiers", hasClassifiers() ? classifiers() : null).add("RecrawlPolicy", recrawlPolicy())
                .add("SchemaChangePolicy", schemaChangePolicy()).add("LineageConfiguration", lineageConfiguration())
                .add("State", stateAsString()).add("TablePrefix", tablePrefix()).add("Schedule", schedule())
                .add("CrawlElapsedTime", crawlElapsedTime()).add("CreationTime", creationTime())
                .add("LastUpdated", lastUpdated()).add("LastCrawl", lastCrawl()).add("Version", version())
                .add("Configuration", configuration()).add("CrawlerSecurityConfiguration", crawlerSecurityConfiguration())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Role":
            return Optional.ofNullable(clazz.cast(role()));
        case "Targets":
            return Optional.ofNullable(clazz.cast(targets()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Classifiers":
            return Optional.ofNullable(clazz.cast(classifiers()));
        case "RecrawlPolicy":
            return Optional.ofNullable(clazz.cast(recrawlPolicy()));
        case "SchemaChangePolicy":
            return Optional.ofNullable(clazz.cast(schemaChangePolicy()));
        case "LineageConfiguration":
            return Optional.ofNullable(clazz.cast(lineageConfiguration()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "TablePrefix":
            return Optional.ofNullable(clazz.cast(tablePrefix()));
        case "Schedule":
            return Optional.ofNullable(clazz.cast(schedule()));
        case "CrawlElapsedTime":
            return Optional.ofNullable(clazz.cast(crawlElapsedTime()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastUpdated":
            return Optional.ofNullable(clazz.cast(lastUpdated()));
        case "LastCrawl":
            return Optional.ofNullable(clazz.cast(lastCrawl()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "CrawlerSecurityConfiguration":
            return Optional.ofNullable(clazz.cast(crawlerSecurityConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Crawler, T> g) {
        return obj -> g.apply((Crawler) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Crawler> {
        /**
         * <p>
         * The name of the crawler.
         * </p>
         * 
         * @param name
         *        The name of the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of an IAM role that's used to access customer resources, such as Amazon Simple
         * Storage Service (Amazon S3) data.
         * </p>
         * 
         * @param role
         *        The Amazon Resource Name (ARN) of an IAM role that's used to access customer resources, such as Amazon
         *        Simple Storage Service (Amazon S3) data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder role(String role);

        /**
         * <p>
         * A collection of targets to crawl.
         * </p>
         * 
         * @param targets
         *        A collection of targets to crawl.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targets(CrawlerTargets targets);

        /**
         * <p>
         * A collection of targets to crawl.
         * </p>
         * This is a convenience that creates an instance of the {@link CrawlerTargets.Builder} avoiding the need to
         * create one manually via {@link CrawlerTargets#builder()}.
         *
         * When the {@link Consumer} completes, {@link CrawlerTargets.Builder#build()} is called immediately and its
         * result is passed to {@link #targets(CrawlerTargets)}.
         * 
         * @param targets
         *        a consumer that will call methods on {@link CrawlerTargets.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targets(CrawlerTargets)
         */
        default Builder targets(Consumer<CrawlerTargets.Builder> targets) {
            return targets(CrawlerTargets.builder().applyMutation(targets).build());
        }

        /**
         * <p>
         * The name of the database in which the crawler's output is stored.
         * </p>
         * 
         * @param databaseName
         *        The name of the database in which the crawler's output is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * A description of the crawler.
         * </p>
         * 
         * @param description
         *        A description of the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of UTF-8 strings that specify the custom classifiers that are associated with the crawler.
         * </p>
         * 
         * @param classifiers
         *        A list of UTF-8 strings that specify the custom classifiers that are associated with the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifiers(Collection<String> classifiers);

        /**
         * <p>
         * A list of UTF-8 strings that specify the custom classifiers that are associated with the crawler.
         * </p>
         * 
         * @param classifiers
         *        A list of UTF-8 strings that specify the custom classifiers that are associated with the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder classifiers(String... classifiers);

        /**
         * <p>
         * A policy that specifies whether to crawl the entire dataset again, or to crawl only folders that were added
         * since the last crawler run.
         * </p>
         * 
         * @param recrawlPolicy
         *        A policy that specifies whether to crawl the entire dataset again, or to crawl only folders that were
         *        added since the last crawler run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recrawlPolicy(RecrawlPolicy recrawlPolicy);

        /**
         * <p>
         * A policy that specifies whether to crawl the entire dataset again, or to crawl only folders that were added
         * since the last crawler run.
         * </p>
         * This is a convenience that creates an instance of the {@link RecrawlPolicy.Builder} avoiding the need to
         * create one manually via {@link RecrawlPolicy#builder()}.
         *
         * When the {@link Consumer} completes, {@link RecrawlPolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #recrawlPolicy(RecrawlPolicy)}.
         * 
         * @param recrawlPolicy
         *        a consumer that will call methods on {@link RecrawlPolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recrawlPolicy(RecrawlPolicy)
         */
        default Builder recrawlPolicy(Consumer<RecrawlPolicy.Builder> recrawlPolicy) {
            return recrawlPolicy(RecrawlPolicy.builder().applyMutation(recrawlPolicy).build());
        }

        /**
         * <p>
         * The policy that specifies update and delete behaviors for the crawler.
         * </p>
         * 
         * @param schemaChangePolicy
         *        The policy that specifies update and delete behaviors for the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schemaChangePolicy(SchemaChangePolicy schemaChangePolicy);

        /**
         * <p>
         * The policy that specifies update and delete behaviors for the crawler.
         * </p>
         * This is a convenience that creates an instance of the {@link SchemaChangePolicy.Builder} avoiding the need to
         * create one manually via {@link SchemaChangePolicy#builder()}.
         *
         * When the {@link Consumer} completes, {@link SchemaChangePolicy.Builder#build()} is called immediately and its
         * result is passed to {@link #schemaChangePolicy(SchemaChangePolicy)}.
         * 
         * @param schemaChangePolicy
         *        a consumer that will call methods on {@link SchemaChangePolicy.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schemaChangePolicy(SchemaChangePolicy)
         */
        default Builder schemaChangePolicy(Consumer<SchemaChangePolicy.Builder> schemaChangePolicy) {
            return schemaChangePolicy(SchemaChangePolicy.builder().applyMutation(schemaChangePolicy).build());
        }

        /**
         * <p>
         * A configuration that specifies whether data lineage is enabled for the crawler.
         * </p>
         * 
         * @param lineageConfiguration
         *        A configuration that specifies whether data lineage is enabled for the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lineageConfiguration(LineageConfiguration lineageConfiguration);

        /**
         * <p>
         * A configuration that specifies whether data lineage is enabled for the crawler.
         * </p>
         * This is a convenience that creates an instance of the {@link LineageConfiguration.Builder} avoiding the need
         * to create one manually via {@link LineageConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link LineageConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #lineageConfiguration(LineageConfiguration)}.
         * 
         * @param lineageConfiguration
         *        a consumer that will call methods on {@link LineageConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lineageConfiguration(LineageConfiguration)
         */
        default Builder lineageConfiguration(Consumer<LineageConfiguration.Builder> lineageConfiguration) {
            return lineageConfiguration(LineageConfiguration.builder().applyMutation(lineageConfiguration).build());
        }

        /**
         * <p>
         * Indicates whether the crawler is running, or whether a run is pending.
         * </p>
         * 
         * @param state
         *        Indicates whether the crawler is running, or whether a run is pending.
         * @see CrawlerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CrawlerState
         */
        Builder state(String state);

        /**
         * <p>
         * Indicates whether the crawler is running, or whether a run is pending.
         * </p>
         * 
         * @param state
         *        Indicates whether the crawler is running, or whether a run is pending.
         * @see CrawlerState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CrawlerState
         */
        Builder state(CrawlerState state);

        /**
         * <p>
         * The prefix added to the names of tables that are created.
         * </p>
         * 
         * @param tablePrefix
         *        The prefix added to the names of tables that are created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tablePrefix(String tablePrefix);

        /**
         * <p>
         * For scheduled crawlers, the schedule when the crawler runs.
         * </p>
         * 
         * @param schedule
         *        For scheduled crawlers, the schedule when the crawler runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder schedule(Schedule schedule);

        /**
         * <p>
         * For scheduled crawlers, the schedule when the crawler runs.
         * </p>
         * This is a convenience that creates an instance of the {@link Schedule.Builder} avoiding the need to create
         * one manually via {@link Schedule#builder()}.
         *
         * When the {@link Consumer} completes, {@link Schedule.Builder#build()} is called immediately and its result is
         * passed to {@link #schedule(Schedule)}.
         * 
         * @param schedule
         *        a consumer that will call methods on {@link Schedule.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #schedule(Schedule)
         */
        default Builder schedule(Consumer<Schedule.Builder> schedule) {
            return schedule(Schedule.builder().applyMutation(schedule).build());
        }

        /**
         * <p>
         * If the crawler is running, contains the total time elapsed since the last crawl began.
         * </p>
         * 
         * @param crawlElapsedTime
         *        If the crawler is running, contains the total time elapsed since the last crawl began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlElapsedTime(Long crawlElapsedTime);

        /**
         * <p>
         * The time that the crawler was created.
         * </p>
         * 
         * @param creationTime
         *        The time that the crawler was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time that the crawler was last updated.
         * </p>
         * 
         * @param lastUpdated
         *        The time that the crawler was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdated(Instant lastUpdated);

        /**
         * <p>
         * The status of the last crawl, and potentially error information if an error occurred.
         * </p>
         * 
         * @param lastCrawl
         *        The status of the last crawl, and potentially error information if an error occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastCrawl(LastCrawlInfo lastCrawl);

        /**
         * <p>
         * The status of the last crawl, and potentially error information if an error occurred.
         * </p>
         * This is a convenience that creates an instance of the {@link LastCrawlInfo.Builder} avoiding the need to
         * create one manually via {@link LastCrawlInfo#builder()}.
         *
         * When the {@link Consumer} completes, {@link LastCrawlInfo.Builder#build()} is called immediately and its
         * result is passed to {@link #lastCrawl(LastCrawlInfo)}.
         * 
         * @param lastCrawl
         *        a consumer that will call methods on {@link LastCrawlInfo.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastCrawl(LastCrawlInfo)
         */
        default Builder lastCrawl(Consumer<LastCrawlInfo.Builder> lastCrawl) {
            return lastCrawl(LastCrawlInfo.builder().applyMutation(lastCrawl).build());
        }

        /**
         * <p>
         * The version of the crawler.
         * </p>
         * 
         * @param version
         *        The version of the crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(Long version);

        /**
         * <p>
         * Crawler configuration information. This versioned JSON string allows users to specify aspects of a crawler's
         * behavior. For more information, see <a
         * href="https://docs.aws.amazon.com/glue/latest/dg/crawler-configuration.html">Configuring a Crawler</a>.
         * </p>
         * 
         * @param configuration
         *        Crawler configuration information. This versioned JSON string allows users to specify aspects of a
         *        crawler's behavior. For more information, see <a
         *        href="https://docs.aws.amazon.com/glue/latest/dg/crawler-configuration.html">Configuring a
         *        Crawler</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(String configuration);

        /**
         * <p>
         * The name of the <code>SecurityConfiguration</code> structure to be used by this crawler.
         * </p>
         * 
         * @param crawlerSecurityConfiguration
         *        The name of the <code>SecurityConfiguration</code> structure to be used by this crawler.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder crawlerSecurityConfiguration(String crawlerSecurityConfiguration);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String role;

        private CrawlerTargets targets;

        private String databaseName;

        private String description;

        private List<String> classifiers = DefaultSdkAutoConstructList.getInstance();

        private RecrawlPolicy recrawlPolicy;

        private SchemaChangePolicy schemaChangePolicy;

        private LineageConfiguration lineageConfiguration;

        private String state;

        private String tablePrefix;

        private Schedule schedule;

        private Long crawlElapsedTime;

        private Instant creationTime;

        private Instant lastUpdated;

        private LastCrawlInfo lastCrawl;

        private Long version;

        private String configuration;

        private String crawlerSecurityConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(Crawler model) {
            name(model.name);
            role(model.role);
            targets(model.targets);
            databaseName(model.databaseName);
            description(model.description);
            classifiers(model.classifiers);
            recrawlPolicy(model.recrawlPolicy);
            schemaChangePolicy(model.schemaChangePolicy);
            lineageConfiguration(model.lineageConfiguration);
            state(model.state);
            tablePrefix(model.tablePrefix);
            schedule(model.schedule);
            crawlElapsedTime(model.crawlElapsedTime);
            creationTime(model.creationTime);
            lastUpdated(model.lastUpdated);
            lastCrawl(model.lastCrawl);
            version(model.version);
            configuration(model.configuration);
            crawlerSecurityConfiguration(model.crawlerSecurityConfiguration);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getRole() {
            return role;
        }

        @Override
        public final Builder role(String role) {
            this.role = role;
            return this;
        }

        public final void setRole(String role) {
            this.role = role;
        }

        public final CrawlerTargets.Builder getTargets() {
            return targets != null ? targets.toBuilder() : null;
        }

        @Override
        public final Builder targets(CrawlerTargets targets) {
            this.targets = targets;
            return this;
        }

        public final void setTargets(CrawlerTargets.BuilderImpl targets) {
            this.targets = targets != null ? targets.build() : null;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<String> getClassifiers() {
            if (classifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return classifiers;
        }

        @Override
        public final Builder classifiers(Collection<String> classifiers) {
            this.classifiers = ClassifierNameListCopier.copy(classifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder classifiers(String... classifiers) {
            classifiers(Arrays.asList(classifiers));
            return this;
        }

        public final void setClassifiers(Collection<String> classifiers) {
            this.classifiers = ClassifierNameListCopier.copy(classifiers);
        }

        public final RecrawlPolicy.Builder getRecrawlPolicy() {
            return recrawlPolicy != null ? recrawlPolicy.toBuilder() : null;
        }

        @Override
        public final Builder recrawlPolicy(RecrawlPolicy recrawlPolicy) {
            this.recrawlPolicy = recrawlPolicy;
            return this;
        }

        public final void setRecrawlPolicy(RecrawlPolicy.BuilderImpl recrawlPolicy) {
            this.recrawlPolicy = recrawlPolicy != null ? recrawlPolicy.build() : null;
        }

        public final SchemaChangePolicy.Builder getSchemaChangePolicy() {
            return schemaChangePolicy != null ? schemaChangePolicy.toBuilder() : null;
        }

        @Override
        public final Builder schemaChangePolicy(SchemaChangePolicy schemaChangePolicy) {
            this.schemaChangePolicy = schemaChangePolicy;
            return this;
        }

        public final void setSchemaChangePolicy(SchemaChangePolicy.BuilderImpl schemaChangePolicy) {
            this.schemaChangePolicy = schemaChangePolicy != null ? schemaChangePolicy.build() : null;
        }

        public final LineageConfiguration.Builder getLineageConfiguration() {
            return lineageConfiguration != null ? lineageConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder lineageConfiguration(LineageConfiguration lineageConfiguration) {
            this.lineageConfiguration = lineageConfiguration;
            return this;
        }

        public final void setLineageConfiguration(LineageConfiguration.BuilderImpl lineageConfiguration) {
            this.lineageConfiguration = lineageConfiguration != null ? lineageConfiguration.build() : null;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CrawlerState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getTablePrefix() {
            return tablePrefix;
        }

        @Override
        public final Builder tablePrefix(String tablePrefix) {
            this.tablePrefix = tablePrefix;
            return this;
        }

        public final void setTablePrefix(String tablePrefix) {
            this.tablePrefix = tablePrefix;
        }

        public final Schedule.Builder getSchedule() {
            return schedule != null ? schedule.toBuilder() : null;
        }

        @Override
        public final Builder schedule(Schedule schedule) {
            this.schedule = schedule;
            return this;
        }

        public final void setSchedule(Schedule.BuilderImpl schedule) {
            this.schedule = schedule != null ? schedule.build() : null;
        }

        public final Long getCrawlElapsedTime() {
            return crawlElapsedTime;
        }

        @Override
        public final Builder crawlElapsedTime(Long crawlElapsedTime) {
            this.crawlElapsedTime = crawlElapsedTime;
            return this;
        }

        public final void setCrawlElapsedTime(Long crawlElapsedTime) {
            this.crawlElapsedTime = crawlElapsedTime;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getLastUpdated() {
            return lastUpdated;
        }

        @Override
        public final Builder lastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
            return this;
        }

        public final void setLastUpdated(Instant lastUpdated) {
            this.lastUpdated = lastUpdated;
        }

        public final LastCrawlInfo.Builder getLastCrawl() {
            return lastCrawl != null ? lastCrawl.toBuilder() : null;
        }

        @Override
        public final Builder lastCrawl(LastCrawlInfo lastCrawl) {
            this.lastCrawl = lastCrawl;
            return this;
        }

        public final void setLastCrawl(LastCrawlInfo.BuilderImpl lastCrawl) {
            this.lastCrawl = lastCrawl != null ? lastCrawl.build() : null;
        }

        public final Long getVersion() {
            return version;
        }

        @Override
        public final Builder version(Long version) {
            this.version = version;
            return this;
        }

        public final void setVersion(Long version) {
            this.version = version;
        }

        public final String getConfiguration() {
            return configuration;
        }

        @Override
        public final Builder configuration(String configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(String configuration) {
            this.configuration = configuration;
        }

        public final String getCrawlerSecurityConfiguration() {
            return crawlerSecurityConfiguration;
        }

        @Override
        public final Builder crawlerSecurityConfiguration(String crawlerSecurityConfiguration) {
            this.crawlerSecurityConfiguration = crawlerSecurityConfiguration;
            return this;
        }

        public final void setCrawlerSecurityConfiguration(String crawlerSecurityConfiguration) {
            this.crawlerSecurityConfiguration = crawlerSecurityConfiguration;
        }

        @Override
        public Crawler build() {
            return new Crawler(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
