/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A workflow represents a flow in which AWS Glue components should be executed to complete a logical task.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Workflow implements SdkPojo, Serializable, ToCopyableBuilder<Workflow.Builder, Workflow> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Workflow::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Workflow::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Map<String, String>> DEFAULT_RUN_PROPERTIES_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Workflow::defaultRunProperties))
            .setter(setter(Builder::defaultRunProperties))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DefaultRunProperties").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Workflow::createdOn)).setter(setter(Builder::createdOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedOn").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_ON_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Workflow::lastModifiedOn)).setter(setter(Builder::lastModifiedOn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedOn").build()).build();

    private static final SdkField<WorkflowRun> LAST_RUN_FIELD = SdkField.<WorkflowRun> builder(MarshallingType.SDK_POJO)
            .getter(getter(Workflow::lastRun)).setter(setter(Builder::lastRun)).constructor(WorkflowRun::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastRun").build()).build();

    private static final SdkField<WorkflowGraph> GRAPH_FIELD = SdkField.<WorkflowGraph> builder(MarshallingType.SDK_POJO)
            .getter(getter(Workflow::graph)).setter(setter(Builder::graph)).constructor(WorkflowGraph::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Graph").build()).build();

    private static final SdkField<Integer> MAX_CONCURRENT_RUNS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Workflow::maxConcurrentRuns)).setter(setter(Builder::maxConcurrentRuns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxConcurrentRuns").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, DESCRIPTION_FIELD,
            DEFAULT_RUN_PROPERTIES_FIELD, CREATED_ON_FIELD, LAST_MODIFIED_ON_FIELD, LAST_RUN_FIELD, GRAPH_FIELD,
            MAX_CONCURRENT_RUNS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String description;

    private final Map<String, String> defaultRunProperties;

    private final Instant createdOn;

    private final Instant lastModifiedOn;

    private final WorkflowRun lastRun;

    private final WorkflowGraph graph;

    private final Integer maxConcurrentRuns;

    private Workflow(BuilderImpl builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.defaultRunProperties = builder.defaultRunProperties;
        this.createdOn = builder.createdOn;
        this.lastModifiedOn = builder.lastModifiedOn;
        this.lastRun = builder.lastRun;
        this.graph = builder.graph;
        this.maxConcurrentRuns = builder.maxConcurrentRuns;
    }

    /**
     * <p>
     * The name of the workflow representing the flow.
     * </p>
     * 
     * @return The name of the workflow representing the flow.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * A description of the workflow.
     * </p>
     * 
     * @return A description of the workflow.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the DefaultRunProperties property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasDefaultRunProperties() {
        return defaultRunProperties != null && !(defaultRunProperties instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A collection of properties to be used as part of each execution of the workflow.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDefaultRunProperties()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A collection of properties to be used as part of each execution of the workflow.
     */
    public Map<String, String> defaultRunProperties() {
        return defaultRunProperties;
    }

    /**
     * <p>
     * The date and time when the workflow was created.
     * </p>
     * 
     * @return The date and time when the workflow was created.
     */
    public Instant createdOn() {
        return createdOn;
    }

    /**
     * <p>
     * The date and time when the workflow was last modified.
     * </p>
     * 
     * @return The date and time when the workflow was last modified.
     */
    public Instant lastModifiedOn() {
        return lastModifiedOn;
    }

    /**
     * <p>
     * The information about the last execution of the workflow.
     * </p>
     * 
     * @return The information about the last execution of the workflow.
     */
    public WorkflowRun lastRun() {
        return lastRun;
    }

    /**
     * <p>
     * The graph representing all the AWS Glue components that belong to the workflow as nodes and directed connections
     * between them as edges.
     * </p>
     * 
     * @return The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
     *         connections between them as edges.
     */
    public WorkflowGraph graph() {
        return graph;
    }

    /**
     * <p>
     * You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some cases, to
     * prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you leave this parameter
     * blank, there is no limit to the number of concurrent workflow runs.
     * </p>
     * 
     * @return You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some
     *         cases, to prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you
     *         leave this parameter blank, there is no limit to the number of concurrent workflow runs.
     */
    public Integer maxConcurrentRuns() {
        return maxConcurrentRuns;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(defaultRunProperties());
        hashCode = 31 * hashCode + Objects.hashCode(createdOn());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedOn());
        hashCode = 31 * hashCode + Objects.hashCode(lastRun());
        hashCode = 31 * hashCode + Objects.hashCode(graph());
        hashCode = 31 * hashCode + Objects.hashCode(maxConcurrentRuns());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Workflow)) {
            return false;
        }
        Workflow other = (Workflow) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(description(), other.description())
                && Objects.equals(defaultRunProperties(), other.defaultRunProperties())
                && Objects.equals(createdOn(), other.createdOn()) && Objects.equals(lastModifiedOn(), other.lastModifiedOn())
                && Objects.equals(lastRun(), other.lastRun()) && Objects.equals(graph(), other.graph())
                && Objects.equals(maxConcurrentRuns(), other.maxConcurrentRuns());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Workflow").add("Name", name()).add("Description", description())
                .add("DefaultRunProperties", defaultRunProperties()).add("CreatedOn", createdOn())
                .add("LastModifiedOn", lastModifiedOn()).add("LastRun", lastRun()).add("Graph", graph())
                .add("MaxConcurrentRuns", maxConcurrentRuns()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DefaultRunProperties":
            return Optional.ofNullable(clazz.cast(defaultRunProperties()));
        case "CreatedOn":
            return Optional.ofNullable(clazz.cast(createdOn()));
        case "LastModifiedOn":
            return Optional.ofNullable(clazz.cast(lastModifiedOn()));
        case "LastRun":
            return Optional.ofNullable(clazz.cast(lastRun()));
        case "Graph":
            return Optional.ofNullable(clazz.cast(graph()));
        case "MaxConcurrentRuns":
            return Optional.ofNullable(clazz.cast(maxConcurrentRuns()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Workflow, T> g) {
        return obj -> g.apply((Workflow) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Workflow> {
        /**
         * <p>
         * The name of the workflow representing the flow.
         * </p>
         * 
         * @param name
         *        The name of the workflow representing the flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * A description of the workflow.
         * </p>
         * 
         * @param description
         *        A description of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A collection of properties to be used as part of each execution of the workflow.
         * </p>
         * 
         * @param defaultRunProperties
         *        A collection of properties to be used as part of each execution of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder defaultRunProperties(Map<String, String> defaultRunProperties);

        /**
         * <p>
         * The date and time when the workflow was created.
         * </p>
         * 
         * @param createdOn
         *        The date and time when the workflow was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdOn(Instant createdOn);

        /**
         * <p>
         * The date and time when the workflow was last modified.
         * </p>
         * 
         * @param lastModifiedOn
         *        The date and time when the workflow was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedOn(Instant lastModifiedOn);

        /**
         * <p>
         * The information about the last execution of the workflow.
         * </p>
         * 
         * @param lastRun
         *        The information about the last execution of the workflow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRun(WorkflowRun lastRun);

        /**
         * <p>
         * The information about the last execution of the workflow.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowRun.Builder} avoiding the need to create
         * one manually via {@link WorkflowRun#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowRun.Builder#build()} is called immediately and its result
         * is passed to {@link #lastRun(WorkflowRun)}.
         * 
         * @param lastRun
         *        a consumer that will call methods on {@link WorkflowRun.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lastRun(WorkflowRun)
         */
        default Builder lastRun(Consumer<WorkflowRun.Builder> lastRun) {
            return lastRun(WorkflowRun.builder().applyMutation(lastRun).build());
        }

        /**
         * <p>
         * The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
         * connections between them as edges.
         * </p>
         * 
         * @param graph
         *        The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
         *        connections between them as edges.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder graph(WorkflowGraph graph);

        /**
         * <p>
         * The graph representing all the AWS Glue components that belong to the workflow as nodes and directed
         * connections between them as edges.
         * </p>
         * This is a convenience that creates an instance of the {@link WorkflowGraph.Builder} avoiding the need to
         * create one manually via {@link WorkflowGraph#builder()}.
         *
         * When the {@link Consumer} completes, {@link WorkflowGraph.Builder#build()} is called immediately and its
         * result is passed to {@link #graph(WorkflowGraph)}.
         * 
         * @param graph
         *        a consumer that will call methods on {@link WorkflowGraph.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #graph(WorkflowGraph)
         */
        default Builder graph(Consumer<WorkflowGraph.Builder> graph) {
            return graph(WorkflowGraph.builder().applyMutation(graph).build());
        }

        /**
         * <p>
         * You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some cases,
         * to prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you leave this
         * parameter blank, there is no limit to the number of concurrent workflow runs.
         * </p>
         * 
         * @param maxConcurrentRuns
         *        You can use this parameter to prevent unwanted multiple updates to data, to control costs, or in some
         *        cases, to prevent exceeding the maximum number of concurrent runs of any of the component jobs. If you
         *        leave this parameter blank, there is no limit to the number of concurrent workflow runs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxConcurrentRuns(Integer maxConcurrentRuns);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String description;

        private Map<String, String> defaultRunProperties = DefaultSdkAutoConstructMap.getInstance();

        private Instant createdOn;

        private Instant lastModifiedOn;

        private WorkflowRun lastRun;

        private WorkflowGraph graph;

        private Integer maxConcurrentRuns;

        private BuilderImpl() {
        }

        private BuilderImpl(Workflow model) {
            name(model.name);
            description(model.description);
            defaultRunProperties(model.defaultRunProperties);
            createdOn(model.createdOn);
            lastModifiedOn(model.lastModifiedOn);
            lastRun(model.lastRun);
            graph(model.graph);
            maxConcurrentRuns(model.maxConcurrentRuns);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Map<String, String> getDefaultRunProperties() {
            return defaultRunProperties;
        }

        @Override
        public final Builder defaultRunProperties(Map<String, String> defaultRunProperties) {
            this.defaultRunProperties = WorkflowRunPropertiesCopier.copy(defaultRunProperties);
            return this;
        }

        public final void setDefaultRunProperties(Map<String, String> defaultRunProperties) {
            this.defaultRunProperties = WorkflowRunPropertiesCopier.copy(defaultRunProperties);
        }

        public final Instant getCreatedOn() {
            return createdOn;
        }

        @Override
        public final Builder createdOn(Instant createdOn) {
            this.createdOn = createdOn;
            return this;
        }

        public final void setCreatedOn(Instant createdOn) {
            this.createdOn = createdOn;
        }

        public final Instant getLastModifiedOn() {
            return lastModifiedOn;
        }

        @Override
        public final Builder lastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
            return this;
        }

        public final void setLastModifiedOn(Instant lastModifiedOn) {
            this.lastModifiedOn = lastModifiedOn;
        }

        public final WorkflowRun.Builder getLastRun() {
            return lastRun != null ? lastRun.toBuilder() : null;
        }

        @Override
        public final Builder lastRun(WorkflowRun lastRun) {
            this.lastRun = lastRun;
            return this;
        }

        public final void setLastRun(WorkflowRun.BuilderImpl lastRun) {
            this.lastRun = lastRun != null ? lastRun.build() : null;
        }

        public final WorkflowGraph.Builder getGraph() {
            return graph != null ? graph.toBuilder() : null;
        }

        @Override
        public final Builder graph(WorkflowGraph graph) {
            this.graph = graph;
            return this;
        }

        public final void setGraph(WorkflowGraph.BuilderImpl graph) {
            this.graph = graph != null ? graph.build() : null;
        }

        public final Integer getMaxConcurrentRuns() {
            return maxConcurrentRuns;
        }

        @Override
        public final Builder maxConcurrentRuns(Integer maxConcurrentRuns) {
            this.maxConcurrentRuns = maxConcurrentRuns;
            return this;
        }

        public final void setMaxConcurrentRuns(Integer maxConcurrentRuns) {
            this.maxConcurrentRuns = maxConcurrentRuns;
        }

        @Override
        public Workflow build() {
            return new Workflow(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
