/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The data structure used by the Data Catalog to encrypt the password as part of <code>CreateConnection</code> or
 * <code>UpdateConnection</code> and store it in the <code>ENCRYPTED_PASSWORD</code> field in the connection properties.
 * You can enable catalog encryption or only password encryption.
 * </p>
 * <p>
 * When a <code>CreationConnection</code> request arrives containing a password, the Data Catalog first encrypts the
 * password using your AWS KMS key. It then encrypts the whole connection object again if catalog encryption is also
 * enabled.
 * </p>
 * <p>
 * This encryption requires that you set AWS KMS key permissions to enable or restrict access on the password key
 * according to your security requirements. For example, you might want only administrators to have decrypt permission
 * on the password key.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConnectionPasswordEncryption implements SdkPojo, Serializable,
        ToCopyableBuilder<ConnectionPasswordEncryption.Builder, ConnectionPasswordEncryption> {
    private static final SdkField<Boolean> RETURN_CONNECTION_PASSWORD_ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(ConnectionPasswordEncryption::returnConnectionPasswordEncrypted))
            .setter(setter(Builder::returnConnectionPasswordEncrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReturnConnectionPasswordEncrypted")
                    .build()).build();

    private static final SdkField<String> AWS_KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ConnectionPasswordEncryption::awsKmsKeyId)).setter(setter(Builder::awsKmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsKmsKeyId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            RETURN_CONNECTION_PASSWORD_ENCRYPTED_FIELD, AWS_KMS_KEY_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean returnConnectionPasswordEncrypted;

    private final String awsKmsKeyId;

    private ConnectionPasswordEncryption(BuilderImpl builder) {
        this.returnConnectionPasswordEncrypted = builder.returnConnectionPasswordEncrypted;
        this.awsKmsKeyId = builder.awsKmsKeyId;
    }

    /**
     * <p>
     * When the <code>ReturnConnectionPasswordEncrypted</code> flag is set to "true", passwords remain encrypted in the
     * responses of <code>GetConnection</code> and <code>GetConnections</code>. This encryption takes effect
     * independently from catalog encryption.
     * </p>
     * 
     * @return When the <code>ReturnConnectionPasswordEncrypted</code> flag is set to "true", passwords remain encrypted
     *         in the responses of <code>GetConnection</code> and <code>GetConnections</code>. This encryption takes
     *         effect independently from catalog encryption.
     */
    public Boolean returnConnectionPasswordEncrypted() {
        return returnConnectionPasswordEncrypted;
    }

    /**
     * <p>
     * An AWS KMS key that is used to encrypt the connection password.
     * </p>
     * <p>
     * If connection password protection is enabled, the caller of <code>CreateConnection</code> and
     * <code>UpdateConnection</code> needs at least <code>kms:Encrypt</code> permission on the specified AWS KMS key, to
     * encrypt passwords before storing them in the Data Catalog.
     * </p>
     * <p>
     * You can set the decrypt permission to enable or restrict access on the password key according to your security
     * requirements.
     * </p>
     * 
     * @return An AWS KMS key that is used to encrypt the connection password. </p>
     *         <p>
     *         If connection password protection is enabled, the caller of <code>CreateConnection</code> and
     *         <code>UpdateConnection</code> needs at least <code>kms:Encrypt</code> permission on the specified AWS KMS
     *         key, to encrypt passwords before storing them in the Data Catalog.
     *         </p>
     *         <p>
     *         You can set the decrypt permission to enable or restrict access on the password key according to your
     *         security requirements.
     */
    public String awsKmsKeyId() {
        return awsKmsKeyId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(returnConnectionPasswordEncrypted());
        hashCode = 31 * hashCode + Objects.hashCode(awsKmsKeyId());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConnectionPasswordEncryption)) {
            return false;
        }
        ConnectionPasswordEncryption other = (ConnectionPasswordEncryption) obj;
        return Objects.equals(returnConnectionPasswordEncrypted(), other.returnConnectionPasswordEncrypted())
                && Objects.equals(awsKmsKeyId(), other.awsKmsKeyId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ConnectionPasswordEncryption")
                .add("ReturnConnectionPasswordEncrypted", returnConnectionPasswordEncrypted()).add("AwsKmsKeyId", awsKmsKeyId())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReturnConnectionPasswordEncrypted":
            return Optional.ofNullable(clazz.cast(returnConnectionPasswordEncrypted()));
        case "AwsKmsKeyId":
            return Optional.ofNullable(clazz.cast(awsKmsKeyId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ConnectionPasswordEncryption, T> g) {
        return obj -> g.apply((ConnectionPasswordEncryption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConnectionPasswordEncryption> {
        /**
         * <p>
         * When the <code>ReturnConnectionPasswordEncrypted</code> flag is set to "true", passwords remain encrypted in
         * the responses of <code>GetConnection</code> and <code>GetConnections</code>. This encryption takes effect
         * independently from catalog encryption.
         * </p>
         * 
         * @param returnConnectionPasswordEncrypted
         *        When the <code>ReturnConnectionPasswordEncrypted</code> flag is set to "true", passwords remain
         *        encrypted in the responses of <code>GetConnection</code> and <code>GetConnections</code>. This
         *        encryption takes effect independently from catalog encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder returnConnectionPasswordEncrypted(Boolean returnConnectionPasswordEncrypted);

        /**
         * <p>
         * An AWS KMS key that is used to encrypt the connection password.
         * </p>
         * <p>
         * If connection password protection is enabled, the caller of <code>CreateConnection</code> and
         * <code>UpdateConnection</code> needs at least <code>kms:Encrypt</code> permission on the specified AWS KMS
         * key, to encrypt passwords before storing them in the Data Catalog.
         * </p>
         * <p>
         * You can set the decrypt permission to enable or restrict access on the password key according to your
         * security requirements.
         * </p>
         * 
         * @param awsKmsKeyId
         *        An AWS KMS key that is used to encrypt the connection password. </p>
         *        <p>
         *        If connection password protection is enabled, the caller of <code>CreateConnection</code> and
         *        <code>UpdateConnection</code> needs at least <code>kms:Encrypt</code> permission on the specified AWS
         *        KMS key, to encrypt passwords before storing them in the Data Catalog.
         *        </p>
         *        <p>
         *        You can set the decrypt permission to enable or restrict access on the password key according to your
         *        security requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsKmsKeyId(String awsKmsKeyId);
    }

    static final class BuilderImpl implements Builder {
        private Boolean returnConnectionPasswordEncrypted;

        private String awsKmsKeyId;

        private BuilderImpl() {
        }

        private BuilderImpl(ConnectionPasswordEncryption model) {
            returnConnectionPasswordEncrypted(model.returnConnectionPasswordEncrypted);
            awsKmsKeyId(model.awsKmsKeyId);
        }

        public final Boolean getReturnConnectionPasswordEncrypted() {
            return returnConnectionPasswordEncrypted;
        }

        @Override
        public final Builder returnConnectionPasswordEncrypted(Boolean returnConnectionPasswordEncrypted) {
            this.returnConnectionPasswordEncrypted = returnConnectionPasswordEncrypted;
            return this;
        }

        public final void setReturnConnectionPasswordEncrypted(Boolean returnConnectionPasswordEncrypted) {
            this.returnConnectionPasswordEncrypted = returnConnectionPasswordEncrypted;
        }

        public final String getAwsKmsKeyId() {
            return awsKmsKeyId;
        }

        @Override
        public final Builder awsKmsKeyId(String awsKmsKeyId) {
            this.awsKmsKeyId = awsKmsKeyId;
            return this;
        }

        public final void setAwsKmsKeyId(String awsKmsKeyId) {
            this.awsKmsKeyId = awsKmsKeyId;
        }

        @Override
        public ConnectionPasswordEncryption build() {
            return new ConnectionPasswordEncryption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
