/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glacier.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.services.glacier.transform.DefaultAccountIdSupplier;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides options to upload a part of an archive in a multipart upload operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UploadMultipartPartRequest extends GlacierRequest implements
        ToCopyableBuilder<UploadMultipartPartRequest.Builder, UploadMultipartPartRequest> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("accountId")
            .getter(getter(UploadMultipartPartRequest::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("accountId").build(),
                    DefaultValueTrait.create(DefaultAccountIdSupplier.getInstance())).build();

    private static final SdkField<String> VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("vaultName").getter(getter(UploadMultipartPartRequest::vaultName)).setter(setter(Builder::vaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("vaultName").build()).build();

    private static final SdkField<String> UPLOAD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("uploadId").getter(getter(UploadMultipartPartRequest::uploadId)).setter(setter(Builder::uploadId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("uploadId").build()).build();

    private static final SdkField<String> CHECKSUM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("checksum").getter(getter(UploadMultipartPartRequest::checksum)).setter(setter(Builder::checksum))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("x-amz-sha256-tree-hash").build())
            .build();

    private static final SdkField<String> RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("range")
            .getter(getter(UploadMultipartPartRequest::range)).setter(setter(Builder::range))
            .traits(LocationTrait.builder().location(MarshallLocation.HEADER).locationName("Content-Range").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            VAULT_NAME_FIELD, UPLOAD_ID_FIELD, CHECKSUM_FIELD, RANGE_FIELD));

    private final String accountId;

    private final String vaultName;

    private final String uploadId;

    private final String checksum;

    private final String range;

    private UploadMultipartPartRequest(BuilderImpl builder) {
        super(builder);
        this.accountId = builder.accountId;
        this.vaultName = builder.vaultName;
        this.uploadId = builder.uploadId;
        this.checksum = builder.checksum;
        this.range = builder.range;
    }

    /**
     * <p>
     * The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can either specify
     * an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case Amazon S3 Glacier uses the AWS
     * account ID associated with the credentials used to sign the request. If you use an account ID, do not include any
     * hyphens ('-') in the ID.
     * </p>
     * 
     * @return The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can either
     *         specify an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case Amazon S3
     *         Glacier uses the AWS account ID associated with the credentials used to sign the request. If you use an
     *         account ID, do not include any hyphens ('-') in the ID.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The name of the vault.
     * </p>
     * 
     * @return The name of the vault.
     */
    public final String vaultName() {
        return vaultName;
    }

    /**
     * <p>
     * The upload ID of the multipart upload.
     * </p>
     * 
     * @return The upload ID of the multipart upload.
     */
    public final String uploadId() {
        return uploadId;
    }

    /**
     * <p>
     * The SHA256 tree hash of the data being uploaded.
     * </p>
     * 
     * @return The SHA256 tree hash of the data being uploaded.
     */
    public final String checksum() {
        return checksum;
    }

    /**
     * <p>
     * Identifies the range of bytes in the assembled archive that will be uploaded in this part. Amazon S3 Glacier uses
     * this information to assemble the archive in the proper sequence. The format of this header follows RFC 2616. An
     * example header is Content-Range:bytes 0-4194303/*.
     * </p>
     * 
     * @return Identifies the range of bytes in the assembled archive that will be uploaded in this part. Amazon S3
     *         Glacier uses this information to assemble the archive in the proper sequence. The format of this header
     *         follows RFC 2616. An example header is Content-Range:bytes 0-4194303/*.
     */
    public final String range() {
        return range;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(vaultName());
        hashCode = 31 * hashCode + Objects.hashCode(uploadId());
        hashCode = 31 * hashCode + Objects.hashCode(checksum());
        hashCode = 31 * hashCode + Objects.hashCode(range());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UploadMultipartPartRequest)) {
            return false;
        }
        UploadMultipartPartRequest other = (UploadMultipartPartRequest) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(vaultName(), other.vaultName())
                && Objects.equals(uploadId(), other.uploadId()) && Objects.equals(checksum(), other.checksum())
                && Objects.equals(range(), other.range());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UploadMultipartPartRequest").add("AccountId", accountId()).add("VaultName", vaultName())
                .add("UploadId", uploadId()).add("Checksum", checksum()).add("Range", range()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "vaultName":
            return Optional.ofNullable(clazz.cast(vaultName()));
        case "uploadId":
            return Optional.ofNullable(clazz.cast(uploadId()));
        case "checksum":
            return Optional.ofNullable(clazz.cast(checksum()));
        case "range":
            return Optional.ofNullable(clazz.cast(range()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UploadMultipartPartRequest, T> g) {
        return obj -> g.apply((UploadMultipartPartRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends GlacierRequest.Builder, SdkPojo, CopyableBuilder<Builder, UploadMultipartPartRequest> {
        /**
         * <p>
         * The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can either
         * specify an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case Amazon S3 Glacier
         * uses the AWS account ID associated with the credentials used to sign the request. If you use an account ID,
         * do not include any hyphens ('-') in the ID.
         * </p>
         * 
         * @param accountId
         *        The <code>AccountId</code> value is the AWS account ID of the account that owns the vault. You can
         *        either specify an AWS account ID or optionally a single '<code>-</code>' (hyphen), in which case
         *        Amazon S3 Glacier uses the AWS account ID associated with the credentials used to sign the request. If
         *        you use an account ID, do not include any hyphens ('-') in the ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The name of the vault.
         * </p>
         * 
         * @param vaultName
         *        The name of the vault.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vaultName(String vaultName);

        /**
         * <p>
         * The upload ID of the multipart upload.
         * </p>
         * 
         * @param uploadId
         *        The upload ID of the multipart upload.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uploadId(String uploadId);

        /**
         * <p>
         * The SHA256 tree hash of the data being uploaded.
         * </p>
         * 
         * @param checksum
         *        The SHA256 tree hash of the data being uploaded.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder checksum(String checksum);

        /**
         * <p>
         * Identifies the range of bytes in the assembled archive that will be uploaded in this part. Amazon S3 Glacier
         * uses this information to assemble the archive in the proper sequence. The format of this header follows RFC
         * 2616. An example header is Content-Range:bytes 0-4194303/*.
         * </p>
         * 
         * @param range
         *        Identifies the range of bytes in the assembled archive that will be uploaded in this part. Amazon S3
         *        Glacier uses this information to assemble the archive in the proper sequence. The format of this
         *        header follows RFC 2616. An example header is Content-Range:bytes 0-4194303/*.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder range(String range);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GlacierRequest.BuilderImpl implements Builder {
        private String accountId;

        private String vaultName;

        private String uploadId;

        private String checksum;

        private String range;

        private BuilderImpl() {
        }

        private BuilderImpl(UploadMultipartPartRequest model) {
            super(model);
            accountId(model.accountId);
            vaultName(model.vaultName);
            uploadId(model.uploadId);
            checksum(model.checksum);
            range(model.range);
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getVaultName() {
            return vaultName;
        }

        @Override
        public final Builder vaultName(String vaultName) {
            this.vaultName = vaultName;
            return this;
        }

        public final void setVaultName(String vaultName) {
            this.vaultName = vaultName;
        }

        public final String getUploadId() {
            return uploadId;
        }

        @Override
        public final Builder uploadId(String uploadId) {
            this.uploadId = uploadId;
            return this;
        }

        public final void setUploadId(String uploadId) {
            this.uploadId = uploadId;
        }

        public final String getChecksum() {
            return checksum;
        }

        @Override
        public final Builder checksum(String checksum) {
            this.checksum = checksum;
            return this;
        }

        public final void setChecksum(String checksum) {
            this.checksum = checksum;
        }

        public final String getRange() {
            return range;
        }

        @Override
        public final Builder range(String range) {
            this.range = range;
            return this;
        }

        public final void setRange(String range) {
            this.range = range;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UploadMultipartPartRequest build() {
            return new UploadMultipartPartRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
