/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glacier.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a vault's notification configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VaultNotificationConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<VaultNotificationConfig.Builder, VaultNotificationConfig> {
    private static final SdkField<String> SNS_TOPIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(VaultNotificationConfig::snsTopic)).setter(setter(Builder::snsTopic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SNSTopic").build()).build();

    private static final SdkField<List<String>> EVENTS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(VaultNotificationConfig::events))
            .setter(setter(Builder::events))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Events").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections
            .unmodifiableList(Arrays.asList(SNS_TOPIC_FIELD, EVENTS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String snsTopic;

    private final List<String> events;

    private VaultNotificationConfig(BuilderImpl builder) {
        this.snsTopic = builder.snsTopic;
        this.events = builder.events;
    }

    /**
     * <p>
     * The Amazon Simple Notification Service (Amazon SNS) topic Amazon Resource Name (ARN).
     * </p>
     * 
     * @return The Amazon Simple Notification Service (Amazon SNS) topic Amazon Resource Name (ARN).
     */
    public String snsTopic() {
        return snsTopic;
    }

    /**
     * Returns true if the Events property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasEvents() {
        return events != null && !(events instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of one or more events for which Amazon S3 Glacier will send a notification to the specified Amazon SNS
     * topic.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEvents()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of one or more events for which Amazon S3 Glacier will send a notification to the specified Amazon
     *         SNS topic.
     */
    public List<String> events() {
        return events;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(snsTopic());
        hashCode = 31 * hashCode + Objects.hashCode(events());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VaultNotificationConfig)) {
            return false;
        }
        VaultNotificationConfig other = (VaultNotificationConfig) obj;
        return Objects.equals(snsTopic(), other.snsTopic()) && Objects.equals(events(), other.events());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("VaultNotificationConfig").add("SNSTopic", snsTopic()).add("Events", events()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SNSTopic":
            return Optional.ofNullable(clazz.cast(snsTopic()));
        case "Events":
            return Optional.ofNullable(clazz.cast(events()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VaultNotificationConfig, T> g) {
        return obj -> g.apply((VaultNotificationConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VaultNotificationConfig> {
        /**
         * <p>
         * The Amazon Simple Notification Service (Amazon SNS) topic Amazon Resource Name (ARN).
         * </p>
         * 
         * @param snsTopic
         *        The Amazon Simple Notification Service (Amazon SNS) topic Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snsTopic(String snsTopic);

        /**
         * <p>
         * A list of one or more events for which Amazon S3 Glacier will send a notification to the specified Amazon SNS
         * topic.
         * </p>
         * 
         * @param events
         *        A list of one or more events for which Amazon S3 Glacier will send a notification to the specified
         *        Amazon SNS topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(Collection<String> events);

        /**
         * <p>
         * A list of one or more events for which Amazon S3 Glacier will send a notification to the specified Amazon SNS
         * topic.
         * </p>
         * 
         * @param events
         *        A list of one or more events for which Amazon S3 Glacier will send a notification to the specified
         *        Amazon SNS topic.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(String... events);
    }

    static final class BuilderImpl implements Builder {
        private String snsTopic;

        private List<String> events = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(VaultNotificationConfig model) {
            snsTopic(model.snsTopic);
            events(model.events);
        }

        public final String getSnsTopic() {
            return snsTopic;
        }

        @Override
        public final Builder snsTopic(String snsTopic) {
            this.snsTopic = snsTopic;
            return this;
        }

        public final void setSnsTopic(String snsTopic) {
            this.snsTopic = snsTopic;
        }

        public final Collection<String> getEvents() {
            return events;
        }

        @Override
        public final Builder events(Collection<String> events) {
            this.events = NotificationEventListCopier.copy(events);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder events(String... events) {
            events(Arrays.asList(events));
            return this;
        }

        public final void setEvents(Collection<String> events) {
            this.events = NotificationEventListCopier.copy(events);
        }

        @Override
        public VaultNotificationConfig build() {
            return new VaultNotificationConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
