/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Rule that controls how a fleet is scaled. Scaling policies are uniquely identified by the combination of name and
 * fleet ID.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>DescribeFleetCapacity</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>UpdateFleetCapacity</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeEC2InstanceLimits</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * Manage scaling policies:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>PutScalingPolicy</a> (auto-scaling)
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeScalingPolicies</a> (auto-scaling)
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DeleteScalingPolicy</a> (auto-scaling)
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * Manage fleet actions:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>StartFleetActions</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>StopFleetActions</a>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScalingPolicy implements SdkPojo, Serializable, ToCopyableBuilder<ScalingPolicy.Builder, ScalingPolicy> {
    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FleetId").getter(getter(ScalingPolicy::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(ScalingPolicy::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(ScalingPolicy::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Integer> SCALING_ADJUSTMENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ScalingAdjustment").getter(getter(ScalingPolicy::scalingAdjustment))
            .setter(setter(Builder::scalingAdjustment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingAdjustment").build()).build();

    private static final SdkField<String> SCALING_ADJUSTMENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalingAdjustmentType").getter(getter(ScalingPolicy::scalingAdjustmentTypeAsString))
            .setter(setter(Builder::scalingAdjustmentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingAdjustmentType").build())
            .build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComparisonOperator").getter(getter(ScalingPolicy::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<Double> THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("Threshold").getter(getter(ScalingPolicy::threshold)).setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Threshold").build()).build();

    private static final SdkField<Integer> EVALUATION_PERIODS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("EvaluationPeriods").getter(getter(ScalingPolicy::evaluationPeriods))
            .setter(setter(Builder::evaluationPeriods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationPeriods").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(ScalingPolicy::metricNameAsString)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<String> POLICY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PolicyType").getter(getter(ScalingPolicy::policyTypeAsString)).setter(setter(Builder::policyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PolicyType").build()).build();

    private static final SdkField<TargetConfiguration> TARGET_CONFIGURATION_FIELD = SdkField
            .<TargetConfiguration> builder(MarshallingType.SDK_POJO).memberName("TargetConfiguration")
            .getter(getter(ScalingPolicy::targetConfiguration)).setter(setter(Builder::targetConfiguration))
            .constructor(TargetConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FLEET_ID_FIELD, NAME_FIELD,
            STATUS_FIELD, SCALING_ADJUSTMENT_FIELD, SCALING_ADJUSTMENT_TYPE_FIELD, COMPARISON_OPERATOR_FIELD, THRESHOLD_FIELD,
            EVALUATION_PERIODS_FIELD, METRIC_NAME_FIELD, POLICY_TYPE_FIELD, TARGET_CONFIGURATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String fleetId;

    private final String name;

    private final String status;

    private final Integer scalingAdjustment;

    private final String scalingAdjustmentType;

    private final String comparisonOperator;

    private final Double threshold;

    private final Integer evaluationPeriods;

    private final String metricName;

    private final String policyType;

    private final TargetConfiguration targetConfiguration;

    private ScalingPolicy(BuilderImpl builder) {
        this.fleetId = builder.fleetId;
        this.name = builder.name;
        this.status = builder.status;
        this.scalingAdjustment = builder.scalingAdjustment;
        this.scalingAdjustmentType = builder.scalingAdjustmentType;
        this.comparisonOperator = builder.comparisonOperator;
        this.threshold = builder.threshold;
        this.evaluationPeriods = builder.evaluationPeriods;
        this.metricName = builder.metricName;
        this.policyType = builder.policyType;
        this.targetConfiguration = builder.targetConfiguration;
    }

    /**
     * <p>
     * A unique identifier for a fleet that is associated with this scaling policy.
     * </p>
     * 
     * @return A unique identifier for a fleet that is associated with this scaling policy.
     */
    public final String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * A descriptive label that is associated with a scaling policy. Policy names do not need to be unique.
     * </p>
     * 
     * @return A descriptive label that is associated with a scaling policy. Policy names do not need to be unique.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
     * status. Scaling policies can be suspended for individual fleets (see <a>StopFleetActions</a>; if suspended for a
     * fleet, the policy status does not change. View a fleet's stopped actions by calling <a>DescribeFleetCapacity</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATING</b> -- A change is being made to the scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETING</b> -- The scaling policy is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETED</b> -- The scaling policy has been deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ScalingStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the scaling policy. The scaling policy can be in force only when in an
     *         <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets (see
     *         <a>StopFleetActions</a>; if suspended for a fleet, the policy status does not change. View a fleet's
     *         stopped actions by calling <a>DescribeFleetCapacity</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATING</b> -- A change is being made to the scaling policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETING</b> -- The scaling policy is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETED</b> -- The scaling policy has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     *         </p>
     *         </li>
     * @see ScalingStatusType
     */
    public final ScalingStatusType status() {
        return ScalingStatusType.fromValue(status);
    }

    /**
     * <p>
     * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
     * status. Scaling policies can be suspended for individual fleets (see <a>StopFleetActions</a>; if suspended for a
     * fleet, the policy status does not change. View a fleet's stopped actions by calling <a>DescribeFleetCapacity</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>UPDATING</b> -- A change is being made to the scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETING</b> -- The scaling policy is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>DELETED</b> -- The scaling policy has been deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ScalingStatusType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the scaling policy. The scaling policy can be in force only when in an
     *         <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets (see
     *         <a>StopFleetActions</a>; if suspended for a fleet, the policy status does not change. View a fleet's
     *         stopped actions by calling <a>DescribeFleetCapacity</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>UPDATING</b> -- A change is being made to the scaling policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETING</b> -- The scaling policy is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>DELETED</b> -- The scaling policy has been deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
     *         </p>
     *         </li>
     * @see ScalingStatusType
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Amount of adjustment to make, based on the scaling adjustment type.
     * </p>
     * 
     * @return Amount of adjustment to make, based on the scaling adjustment type.
     */
    public final Integer scalingAdjustment() {
        return scalingAdjustment;
    }

    /**
     * <p>
     * The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
     * Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment, read
     * as a percentage. Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scalingAdjustmentType} will return {@link ScalingAdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #scalingAdjustmentTypeAsString}.
     * </p>
     * 
     * @return The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
     *         count. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
     *         adjustment, read as a percentage. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     * @see ScalingAdjustmentType
     */
    public final ScalingAdjustmentType scalingAdjustmentType() {
        return ScalingAdjustmentType.fromValue(scalingAdjustmentType);
    }

    /**
     * <p>
     * The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
     * Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment, read
     * as a percentage. Positive values scale up while negative values scale down.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #scalingAdjustmentType} will return {@link ScalingAdjustmentType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #scalingAdjustmentTypeAsString}.
     * </p>
     * 
     * @return The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
     *         count. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
     *         adjustment, read as a percentage. Positive values scale up while negative values scale down.
     *         </p>
     *         </li>
     * @see ScalingAdjustmentType
     */
    public final String scalingAdjustmentTypeAsString() {
        return scalingAdjustmentType;
    }

    /**
     * <p>
     * Comparison operator to use when measuring a metric against the threshold value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperatorType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return Comparison operator to use when measuring a metric against the threshold value.
     * @see ComparisonOperatorType
     */
    public final ComparisonOperatorType comparisonOperator() {
        return ComparisonOperatorType.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * Comparison operator to use when measuring a metric against the threshold value.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperatorType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return Comparison operator to use when measuring a metric against the threshold value.
     * @see ComparisonOperatorType
     */
    public final String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * Metric value used to trigger a scaling event.
     * </p>
     * 
     * @return Metric value used to trigger a scaling event.
     */
    public final Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is triggered.
     * </p>
     * 
     * @return Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is
     *         triggered.
     */
    public final Integer evaluationPeriods() {
        return evaluationPeriods;
    }

    /**
     * <p>
     * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     * descriptions of fleet metrics, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
     * GameLift with Amazon CloudWatch</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
     * capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
     * sessions that are not currently accepting players. Reserved player slots are not included.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
     * reserved for a player.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet could
     * host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
     * sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
     * top-priority destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the current
     * fleet is the top-priority destination.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     *         descriptions of fleet metrics, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
     *         Amazon GameLift with Amazon CloudWatch</a>. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
     *         current capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
     *         game sessions that are not currently accepting players. Reserved player slots are not included.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or
     *         are reserved for a player.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
     *         fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
     *         policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero
     *         game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
     *         the top-priority destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
     *         current fleet is the top-priority destination.
     *         </p>
     *         </li>
     * @see MetricName
     */
    public final MetricName metricName() {
        return MetricName.fromValue(metricName);
    }

    /**
     * <p>
     * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     * descriptions of fleet metrics, see <a
     * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
     * GameLift with Amazon CloudWatch</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
     * capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
     * sessions that are not currently accepting players. Reserved player slots are not included.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
     * reserved for a player.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet could
     * host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
     * sessions.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
     * top-priority destination.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the current
     * fleet is the top-priority destination.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #metricName} will
     * return {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #metricNameAsString}.
     * </p>
     * 
     * @return Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
     *         descriptions of fleet metrics, see <a
     *         href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
     *         Amazon GameLift with Amazon CloudWatch</a>. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveGameSessions</b> -- Game sessions that are currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
     *         current capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
     *         game sessions that are not currently accepting players. Reserved player slots are not included.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or
     *         are reserved for a player.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
     *         fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
     *         policy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero
     *         game sessions.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
     *         the top-priority destination.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
     *         current fleet is the top-priority destination.
     *         </p>
     *         </li>
     * @see MetricName
     */
    public final String metricNameAsString() {
        return metricName;
    }

    /**
     * <p>
     * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>, <i>EvaluationPeriods</i>,
     * <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     *         'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     *         following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
     *         <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * @see PolicyType
     */
    public final PolicyType policyType() {
        return PolicyType.fromValue(policyType);
    }

    /**
     * <p>
     * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>, <i>EvaluationPeriods</i>,
     * <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #policyType} will
     * return {@link PolicyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #policyTypeAsString}.
     * </p>
     * 
     * @return The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
     *         'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
     *         following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
     *         <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
     * @see PolicyType
     */
    public final String policyTypeAsString() {
        return policyType;
    }

    /**
     * <p>
     * The settings for a target-based scaling policy.
     * </p>
     * 
     * @return The settings for a target-based scaling policy.
     */
    public final TargetConfiguration targetConfiguration() {
        return targetConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scalingAdjustment());
        hashCode = 31 * hashCode + Objects.hashCode(scalingAdjustmentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationPeriods());
        hashCode = 31 * hashCode + Objects.hashCode(metricNameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(policyTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(targetConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingPolicy)) {
            return false;
        }
        ScalingPolicy other = (ScalingPolicy) obj;
        return Objects.equals(fleetId(), other.fleetId()) && Objects.equals(name(), other.name())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(scalingAdjustment(), other.scalingAdjustment())
                && Objects.equals(scalingAdjustmentTypeAsString(), other.scalingAdjustmentTypeAsString())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(threshold(), other.threshold())
                && Objects.equals(evaluationPeriods(), other.evaluationPeriods())
                && Objects.equals(metricNameAsString(), other.metricNameAsString())
                && Objects.equals(policyTypeAsString(), other.policyTypeAsString())
                && Objects.equals(targetConfiguration(), other.targetConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScalingPolicy").add("FleetId", fleetId()).add("Name", name()).add("Status", statusAsString())
                .add("ScalingAdjustment", scalingAdjustment()).add("ScalingAdjustmentType", scalingAdjustmentTypeAsString())
                .add("ComparisonOperator", comparisonOperatorAsString()).add("Threshold", threshold())
                .add("EvaluationPeriods", evaluationPeriods()).add("MetricName", metricNameAsString())
                .add("PolicyType", policyTypeAsString()).add("TargetConfiguration", targetConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "ScalingAdjustment":
            return Optional.ofNullable(clazz.cast(scalingAdjustment()));
        case "ScalingAdjustmentType":
            return Optional.ofNullable(clazz.cast(scalingAdjustmentTypeAsString()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "Threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "EvaluationPeriods":
            return Optional.ofNullable(clazz.cast(evaluationPeriods()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricNameAsString()));
        case "PolicyType":
            return Optional.ofNullable(clazz.cast(policyTypeAsString()));
        case "TargetConfiguration":
            return Optional.ofNullable(clazz.cast(targetConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScalingPolicy, T> g) {
        return obj -> g.apply((ScalingPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScalingPolicy> {
        /**
         * <p>
         * A unique identifier for a fleet that is associated with this scaling policy.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for a fleet that is associated with this scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * A descriptive label that is associated with a scaling policy. Policy names do not need to be unique.
         * </p>
         * 
         * @param name
         *        A descriptive label that is associated with a scaling policy. Policy names do not need to be unique.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
         * status. Scaling policies can be suspended for individual fleets (see <a>StopFleetActions</a>; if suspended
         * for a fleet, the policy status does not change. View a fleet's stopped actions by calling
         * <a>DescribeFleetCapacity</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATING</b> -- A change is being made to the scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETING</b> -- The scaling policy is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETED</b> -- The scaling policy has been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the scaling policy. The scaling policy can be in force only when in an
         *        <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets (see
         *        <a>StopFleetActions</a>; if suspended for a fleet, the policy status does not change. View a fleet's
         *        stopped actions by calling <a>DescribeFleetCapacity</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATING</b> -- A change is being made to the scaling policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETING</b> -- The scaling policy is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETED</b> -- The scaling policy has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         *        </p>
         *        </li>
         * @see ScalingStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStatusType
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the scaling policy. The scaling policy can be in force only when in an <code>ACTIVE</code>
         * status. Scaling policies can be suspended for individual fleets (see <a>StopFleetActions</a>; if suspended
         * for a fleet, the policy status does not change. View a fleet's stopped actions by calling
         * <a>DescribeFleetCapacity</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>UPDATING</b> -- A change is being made to the scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETING</b> -- The scaling policy is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>DELETED</b> -- The scaling policy has been deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the scaling policy. The scaling policy can be in force only when in an
         *        <code>ACTIVE</code> status. Scaling policies can be suspended for individual fleets (see
         *        <a>StopFleetActions</a>; if suspended for a fleet, the policy status does not change. View a fleet's
         *        stopped actions by calling <a>DescribeFleetCapacity</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The scaling policy can be used for auto-scaling a fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATE_REQUESTED</b> -- A request to update the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>UPDATING</b> -- A change is being made to the scaling policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETE_REQUESTED</b> -- A request to delete the scaling policy has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETING</b> -- The scaling policy is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>DELETED</b> -- The scaling policy has been deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ERROR</b> -- An error occurred in creating the policy. It should be removed and recreated.
         *        </p>
         *        </li>
         * @see ScalingStatusType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingStatusType
         */
        Builder status(ScalingStatusType status);

        /**
         * <p>
         * Amount of adjustment to make, based on the scaling adjustment type.
         * </p>
         * 
         * @param scalingAdjustment
         *        Amount of adjustment to make, based on the scaling adjustment type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingAdjustment(Integer scalingAdjustment);

        /**
         * <p>
         * The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
         * Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment,
         * read as a percentage. Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalingAdjustmentType
         *        The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
         *        count. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
         *        adjustment, read as a percentage. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         * @see ScalingAdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingAdjustmentType
         */
        Builder scalingAdjustmentType(String scalingAdjustmentType);

        /**
         * <p>
         * The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance count.
         * Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling adjustment,
         * read as a percentage. Positive values scale up while negative values scale down.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalingAdjustmentType
         *        The type of adjustment to make to a fleet's instance count (see <a>FleetCapacity</a>):</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ChangeInCapacity</b> -- add (or subtract) the scaling adjustment value from the current instance
         *        count. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ExactCapacity</b> -- set the instance count to the scaling adjustment value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentChangeInCapacity</b> -- increase or reduce the current instance count by the scaling
         *        adjustment, read as a percentage. Positive values scale up while negative values scale down.
         *        </p>
         *        </li>
         * @see ScalingAdjustmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingAdjustmentType
         */
        Builder scalingAdjustmentType(ScalingAdjustmentType scalingAdjustmentType);

        /**
         * <p>
         * Comparison operator to use when measuring a metric against the threshold value.
         * </p>
         * 
         * @param comparisonOperator
         *        Comparison operator to use when measuring a metric against the threshold value.
         * @see ComparisonOperatorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperatorType
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * Comparison operator to use when measuring a metric against the threshold value.
         * </p>
         * 
         * @param comparisonOperator
         *        Comparison operator to use when measuring a metric against the threshold value.
         * @see ComparisonOperatorType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperatorType
         */
        Builder comparisonOperator(ComparisonOperatorType comparisonOperator);

        /**
         * <p>
         * Metric value used to trigger a scaling event.
         * </p>
         * 
         * @param threshold
         *        Metric value used to trigger a scaling event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is
         * triggered.
         * </p>
         * 
         * @param evaluationPeriods
         *        Length of time (in minutes) the metric must be at or beyond the threshold before a scaling event is
         *        triggered.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationPeriods(Integer evaluationPeriods);

        /**
         * <p>
         * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         * descriptions of fleet metrics, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
         * GameLift with Amazon CloudWatch</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
         * capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
         * sessions that are not currently accepting players. Reserved player slots are not included.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
         * reserved for a player.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet
         * could host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
         * sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
         * top-priority destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
         * current fleet is the top-priority destination.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         *        descriptions of fleet metrics, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
         *        Amazon GameLift with Amazon CloudWatch</a>. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
         *        current capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
         *        game sessions that are not currently accepting players. Reserved player slots are not included.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player
         *        or are reserved for a player.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
         *        fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
         *        policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting
         *        zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
         *        the top-priority destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where
         *        the current fleet is the top-priority destination.
         *        </p>
         *        </li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         * descriptions of fleet metrics, see <a
         * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor Amazon
         * GameLift with Amazon CloudWatch</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given current
         * capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes game
         * sessions that are not currently accepting players. Reserved player slots are not included.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player or are
         * reserved for a player.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a fleet
         * could host simultaneously, given current capacity. Use this metric for a target-based scaling policy.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting zero game
         * sessions.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is the
         * top-priority destination.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where the
         * current fleet is the top-priority destination.
         * </p>
         * </li>
         * </ul>
         * 
         * @param metricName
         *        Name of the Amazon GameLift-defined metric that is used to trigger a scaling adjustment. For detailed
         *        descriptions of fleet metrics, see <a
         *        href="https://docs.aws.amazon.com/gamelift/latest/developerguide/monitoring-cloudwatch.html">Monitor
         *        Amazon GameLift with Amazon CloudWatch</a>. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>ActivatingGameSessions</b> -- Game sessions in the process of being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveGameSessions</b> -- Game sessions that are currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ActiveInstances</b> -- Fleet instances that are currently running at least one game session.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailableGameSessions</b> -- Additional game sessions that fleet could host simultaneously, given
         *        current capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>AvailablePlayerSessions</b> -- Empty player slots in currently active game sessions. This includes
         *        game sessions that are not currently accepting players. Reserved player slots are not included.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>CurrentPlayerSessions</b> -- Player slots in active game sessions that are being used by a player
         *        or are reserved for a player.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>IdleInstances</b> -- Active instances that are currently hosting zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentAvailableGameSessions</b> -- Unused percentage of the total number of game sessions that a
         *        fleet could host simultaneously, given current capacity. Use this metric for a target-based scaling
         *        policy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>PercentIdleInstances</b> -- Percentage of the total number of active instances that are hosting
         *        zero game sessions.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>QueueDepth</b> -- Pending game session placement requests, in any queue, where the current fleet is
         *        the top-priority destination.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>WaitTime</b> -- Current wait time for pending game session placement requests, in any queue, where
         *        the current fleet is the top-priority destination.
         *        </p>
         *        </li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder metricName(MetricName metricName);

        /**
         * <p>
         * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
         * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
         * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         * <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * </p>
         * 
         * @param policyType
         *        The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i>
         *        to 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy
         *        set the following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         *        <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(String policyType);

        /**
         * <p>
         * The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i> to
         * 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy set the
         * following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         * <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * </p>
         * 
         * @param policyType
         *        The type of scaling policy to create. For a target-based policy, set the parameter <i>MetricName</i>
         *        to 'PercentAvailableGameSessions' and specify a <i>TargetConfiguration</i>. For a rule-based policy
         *        set the following parameters: <i>MetricName</i>, <i>ComparisonOperator</i>, <i>Threshold</i>,
         *        <i>EvaluationPeriods</i>, <i>ScalingAdjustmentType</i>, and <i>ScalingAdjustment</i>.
         * @see PolicyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PolicyType
         */
        Builder policyType(PolicyType policyType);

        /**
         * <p>
         * The settings for a target-based scaling policy.
         * </p>
         * 
         * @param targetConfiguration
         *        The settings for a target-based scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetConfiguration(TargetConfiguration targetConfiguration);

        /**
         * <p>
         * The settings for a target-based scaling policy.
         * </p>
         * This is a convenience that creates an instance of the {@link TargetConfiguration.Builder} avoiding the need
         * to create one manually via {@link TargetConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link TargetConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #targetConfiguration(TargetConfiguration)}.
         * 
         * @param targetConfiguration
         *        a consumer that will call methods on {@link TargetConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #targetConfiguration(TargetConfiguration)
         */
        default Builder targetConfiguration(Consumer<TargetConfiguration.Builder> targetConfiguration) {
            return targetConfiguration(TargetConfiguration.builder().applyMutation(targetConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String fleetId;

        private String name;

        private String status;

        private Integer scalingAdjustment;

        private String scalingAdjustmentType;

        private String comparisonOperator;

        private Double threshold;

        private Integer evaluationPeriods;

        private String metricName;

        private String policyType;

        private TargetConfiguration targetConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingPolicy model) {
            fleetId(model.fleetId);
            name(model.name);
            status(model.status);
            scalingAdjustment(model.scalingAdjustment);
            scalingAdjustmentType(model.scalingAdjustmentType);
            comparisonOperator(model.comparisonOperator);
            threshold(model.threshold);
            evaluationPeriods(model.evaluationPeriods);
            metricName(model.metricName);
            policyType(model.policyType);
            targetConfiguration(model.targetConfiguration);
        }

        public final String getFleetId() {
            return fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ScalingStatusType status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Integer getScalingAdjustment() {
            return scalingAdjustment;
        }

        @Override
        public final Builder scalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
            return this;
        }

        public final void setScalingAdjustment(Integer scalingAdjustment) {
            this.scalingAdjustment = scalingAdjustment;
        }

        public final String getScalingAdjustmentType() {
            return scalingAdjustmentType;
        }

        @Override
        public final Builder scalingAdjustmentType(String scalingAdjustmentType) {
            this.scalingAdjustmentType = scalingAdjustmentType;
            return this;
        }

        @Override
        public final Builder scalingAdjustmentType(ScalingAdjustmentType scalingAdjustmentType) {
            this.scalingAdjustmentType(scalingAdjustmentType == null ? null : scalingAdjustmentType.toString());
            return this;
        }

        public final void setScalingAdjustmentType(String scalingAdjustmentType) {
            this.scalingAdjustmentType = scalingAdjustmentType;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperatorType comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        public final Double getThreshold() {
            return threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        public final Integer getEvaluationPeriods() {
            return evaluationPeriods;
        }

        @Override
        public final Builder evaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
            return this;
        }

        public final void setEvaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        @Override
        public final Builder metricName(MetricName metricName) {
            this.metricName(metricName == null ? null : metricName.toString());
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final String getPolicyType() {
            return policyType;
        }

        @Override
        public final Builder policyType(String policyType) {
            this.policyType = policyType;
            return this;
        }

        @Override
        public final Builder policyType(PolicyType policyType) {
            this.policyType(policyType == null ? null : policyType.toString());
            return this;
        }

        public final void setPolicyType(String policyType) {
            this.policyType = policyType;
        }

        public final TargetConfiguration.Builder getTargetConfiguration() {
            return targetConfiguration != null ? targetConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder targetConfiguration(TargetConfiguration targetConfiguration) {
            this.targetConfiguration = targetConfiguration;
            return this;
        }

        public final void setTargetConfiguration(TargetConfiguration.BuilderImpl targetConfiguration) {
            this.targetConfiguration = targetConfiguration != null ? targetConfiguration.build() : null;
        }

        @Override
        public ScalingPolicy build() {
            return new ScalingPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
