/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * <b>This data type is part of Amazon GameLift FleetIQ with game server groups, which is in preview release and is
 * subject to change.</b>
 * </p>
 * <p>
 * Properties describing a game server group resource. A game server group manages certain properties of a corresponding
 * EC2 Auto Scaling group.
 * </p>
 * <p>
 * A game server group is created by a successful call to <a>CreateGameServerGroup</a> and deleted by calling
 * <a>DeleteGameServerGroup</a>. Game server group activity can be temporarily suspended and resumed by calling
 * <a>SuspendGameServerGroup</a> and <a>ResumeGameServerGroup</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GameServerGroup implements SdkPojo, Serializable, ToCopyableBuilder<GameServerGroup.Builder, GameServerGroup> {
    private static final SdkField<String> GAME_SERVER_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::gameServerGroupName)).setter(setter(Builder::gameServerGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupName").build())
            .build();

    private static final SdkField<String> GAME_SERVER_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::gameServerGroupArn)).setter(setter(Builder::gameServerGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerGroupArn").build())
            .build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RoleArn").build()).build();

    private static final SdkField<List<InstanceDefinition>> INSTANCE_DEFINITIONS_FIELD = SdkField
            .<List<InstanceDefinition>> builder(MarshallingType.LIST)
            .getter(getter(GameServerGroup::instanceDefinitions))
            .setter(setter(Builder::instanceDefinitions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceDefinitions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceDefinition> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceDefinition::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> BALANCING_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::balancingStrategyAsString)).setter(setter(Builder::balancingStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BalancingStrategy").build()).build();

    private static final SdkField<String> GAME_SERVER_PROTECTION_POLICY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::gameServerProtectionPolicyAsString))
            .setter(setter(Builder::gameServerProtectionPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameServerProtectionPolicy").build())
            .build();

    private static final SdkField<String> AUTO_SCALING_GROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::autoScalingGroupArn)).setter(setter(Builder::autoScalingGroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingGroupArn").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(GameServerGroup::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<List<String>> SUSPENDED_ACTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(GameServerGroup::suspendedActionsAsStrings))
            .setter(setter(Builder::suspendedActionsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SuspendedActions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameServerGroup::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> LAST_UPDATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(GameServerGroup::lastUpdatedTime)).setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GAME_SERVER_GROUP_NAME_FIELD,
            GAME_SERVER_GROUP_ARN_FIELD, ROLE_ARN_FIELD, INSTANCE_DEFINITIONS_FIELD, BALANCING_STRATEGY_FIELD,
            GAME_SERVER_PROTECTION_POLICY_FIELD, AUTO_SCALING_GROUP_ARN_FIELD, STATUS_FIELD, STATUS_REASON_FIELD,
            SUSPENDED_ACTIONS_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String gameServerGroupName;

    private final String gameServerGroupArn;

    private final String roleArn;

    private final List<InstanceDefinition> instanceDefinitions;

    private final String balancingStrategy;

    private final String gameServerProtectionPolicy;

    private final String autoScalingGroupArn;

    private final String status;

    private final String statusReason;

    private final List<String> suspendedActions;

    private final Instant creationTime;

    private final Instant lastUpdatedTime;

    private GameServerGroup(BuilderImpl builder) {
        this.gameServerGroupName = builder.gameServerGroupName;
        this.gameServerGroupArn = builder.gameServerGroupArn;
        this.roleArn = builder.roleArn;
        this.instanceDefinitions = builder.instanceDefinitions;
        this.balancingStrategy = builder.balancingStrategy;
        this.gameServerProtectionPolicy = builder.gameServerProtectionPolicy;
        this.autoScalingGroupArn = builder.autoScalingGroupArn;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.suspendedActions = builder.suspendedActions;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
    }

    /**
     * <p>
     * A developer-defined identifier for the game server group. The name is unique per Region per AWS account.
     * </p>
     * 
     * @return A developer-defined identifier for the game server group. The name is unique per Region per AWS account.
     */
    public String gameServerGroupName() {
        return gameServerGroupName;
    }

    /**
     * <p>
     * A generated unique ID for the game server group.
     * </p>
     * 
     * @return A generated unique ID for the game server group.
     */
    public String gameServerGroupArn() {
        return gameServerGroupArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>)
     * for an IAM role that allows Amazon GameLift to access your EC2 Auto Scaling groups. The submitted role is
     * validated to ensure that it contains the necessary permissions for game server groups.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) for an IAM role that
     *         allows Amazon GameLift to access your EC2 Auto Scaling groups. The submitted role is validated to ensure
     *         that it contains the necessary permissions for game server groups.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * Returns true if the InstanceDefinitions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasInstanceDefinitions() {
        return instanceDefinitions != null && !(instanceDefinitions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances in the
     * group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstanceDefinitions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances in
     *         the group.
     */
    public List<InstanceDefinition> instanceDefinitions() {
        return instanceDefinitions;
    }

    /**
     * <p>
     * The fallback balancing method to use for the game server group when Spot instances in a Region become unavailable
     * or are not viable for game hosting. Once triggered, this method remains active until Spot instances can once
     * again be used. Method options include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No new
     * instances are started, and the existing nonviable Spot instances are terminated (once current gameplay ends) and
     * not replaced.
     * </p>
     * </li>
     * <li>
     * <p>
     * SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide hosting capacity
     * by using On-Demand instances. Existing nonviable Spot instances are terminated (once current gameplay ends) and
     * replaced with new On-Demand instances.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #balancingStrategy}
     * will return {@link BalancingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #balancingStrategyAsString}.
     * </p>
     * 
     * @return The fallback balancing method to use for the game server group when Spot instances in a Region become
     *         unavailable or are not viable for game hosting. Once triggered, this method remains active until Spot
     *         instances can once again be used. Method options include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No
     *         new instances are started, and the existing nonviable Spot instances are terminated (once current
     *         gameplay ends) and not replaced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide hosting
     *         capacity by using On-Demand instances. Existing nonviable Spot instances are terminated (once current
     *         gameplay ends) and replaced with new On-Demand instances.
     *         </p>
     *         </li>
     * @see BalancingStrategy
     */
    public BalancingStrategy balancingStrategy() {
        return BalancingStrategy.fromValue(balancingStrategy);
    }

    /**
     * <p>
     * The fallback balancing method to use for the game server group when Spot instances in a Region become unavailable
     * or are not viable for game hosting. Once triggered, this method remains active until Spot instances can once
     * again be used. Method options include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No new
     * instances are started, and the existing nonviable Spot instances are terminated (once current gameplay ends) and
     * not replaced.
     * </p>
     * </li>
     * <li>
     * <p>
     * SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide hosting capacity
     * by using On-Demand instances. Existing nonviable Spot instances are terminated (once current gameplay ends) and
     * replaced with new On-Demand instances.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #balancingStrategy}
     * will return {@link BalancingStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #balancingStrategyAsString}.
     * </p>
     * 
     * @return The fallback balancing method to use for the game server group when Spot instances in a Region become
     *         unavailable or are not viable for game hosting. Once triggered, this method remains active until Spot
     *         instances can once again be used. Method options include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No
     *         new instances are started, and the existing nonviable Spot instances are terminated (once current
     *         gameplay ends) and not replaced.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide hosting
     *         capacity by using On-Demand instances. Existing nonviable Spot instances are terminated (once current
     *         gameplay ends) and replaced with new On-Demand instances.
     *         </p>
     *         </li>
     * @see BalancingStrategy
     */
    public String balancingStrategyAsString() {
        return balancingStrategy;
    }

    /**
     * <p>
     * A flag that indicates whether instances in the game server group are protected from early termination.
     * Unprotected instances that have active game servers running may be terminated during a scale-down event, causing
     * players to be dropped from the game. Protected instances cannot be terminated while there are active game servers
     * running except in the event of a forced game server group deletion (see <a>DeleteGameServerGroup</a>). An
     * exception to this is Spot Instances, which may be terminated by AWS regardless of protection status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #gameServerProtectionPolicy} will return {@link GameServerProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #gameServerProtectionPolicyAsString}.
     * </p>
     * 
     * @return A flag that indicates whether instances in the game server group are protected from early termination.
     *         Unprotected instances that have active game servers running may be terminated during a scale-down event,
     *         causing players to be dropped from the game. Protected instances cannot be terminated while there are
     *         active game servers running except in the event of a forced game server group deletion (see
     *         <a>DeleteGameServerGroup</a>). An exception to this is Spot Instances, which may be terminated by AWS
     *         regardless of protection status.
     * @see GameServerProtectionPolicy
     */
    public GameServerProtectionPolicy gameServerProtectionPolicy() {
        return GameServerProtectionPolicy.fromValue(gameServerProtectionPolicy);
    }

    /**
     * <p>
     * A flag that indicates whether instances in the game server group are protected from early termination.
     * Unprotected instances that have active game servers running may be terminated during a scale-down event, causing
     * players to be dropped from the game. Protected instances cannot be terminated while there are active game servers
     * running except in the event of a forced game server group deletion (see <a>DeleteGameServerGroup</a>). An
     * exception to this is Spot Instances, which may be terminated by AWS regardless of protection status.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #gameServerProtectionPolicy} will return {@link GameServerProtectionPolicy#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #gameServerProtectionPolicyAsString}.
     * </p>
     * 
     * @return A flag that indicates whether instances in the game server group are protected from early termination.
     *         Unprotected instances that have active game servers running may be terminated during a scale-down event,
     *         causing players to be dropped from the game. Protected instances cannot be terminated while there are
     *         active game servers running except in the event of a forced game server group deletion (see
     *         <a>DeleteGameServerGroup</a>). An exception to this is Spot Instances, which may be terminated by AWS
     *         regardless of protection status.
     * @see GameServerProtectionPolicy
     */
    public String gameServerProtectionPolicyAsString() {
        return gameServerProtectionPolicy;
    }

    /**
     * <p>
     * A generated unique ID for the EC2 Auto Scaling group with is associated with this game server group.
     * </p>
     * 
     * @return A generated unique ID for the EC2 Auto Scaling group with is associated with this game server group.
     */
    public String autoScalingGroupArn() {
        return autoScalingGroupArn;
    }

    /**
     * <p>
     * The current status of the game server group. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an autoscaling group in
     * your AWS account.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE - The game server group has been successfully created.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETE_SCHEDULED - A request to delete the game server group has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is processing
     * it. GameLift FleetIQ must first complete and release hosts before it deletes the autoscaling group and the game
     * server group.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETED - The game server group has been successfully deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting in an
     * error state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameServerGroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the game server group. Possible statuses include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an autoscaling
     *         group in your AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE - The game server group has been successfully created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETE_SCHEDULED - A request to delete the game server group has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
     *         processing it. GameLift FleetIQ must first complete and release hosts before it deletes the autoscaling
     *         group and the game server group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETED - The game server group has been successfully deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting in
     *         an error state.
     *         </p>
     *         </li>
     * @see GameServerGroupStatus
     */
    public GameServerGroupStatus status() {
        return GameServerGroupStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the game server group. Possible statuses include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an autoscaling group in
     * your AWS account.
     * </p>
     * </li>
     * <li>
     * <p>
     * ACTIVE - The game server group has been successfully created.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETE_SCHEDULED - A request to delete the game server group has been received.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is processing
     * it. GameLift FleetIQ must first complete and release hosts before it deletes the autoscaling group and the game
     * server group.
     * </p>
     * </li>
     * <li>
     * <p>
     * DELETED - The game server group has been successfully deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting in an
     * error state.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link GameServerGroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the game server group. Possible statuses include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an autoscaling
     *         group in your AWS account.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ACTIVE - The game server group has been successfully created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETE_SCHEDULED - A request to delete the game server group has been received.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
     *         processing it. GameLift FleetIQ must first complete and release hosts before it deletes the autoscaling
     *         group and the game server group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DELETED - The game server group has been successfully deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting in
     *         an error state.
     *         </p>
     *         </li>
     * @see GameServerGroupStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * Additional information about the current game server group status. This information may provide additional
     * insight on groups that in ERROR status.
     * </p>
     * 
     * @return Additional information about the current game server group status. This information may provide
     *         additional insight on groups that in ERROR status.
     */
    public String statusReason() {
        return statusReason;
    }

    /**
     * <p>
     * A list of activities that are currently suspended for this game server group. If this property is empty, all
     * activities are occurring.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSuspendedActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of activities that are currently suspended for this game server group. If this property is empty,
     *         all activities are occurring.
     */
    public List<GameServerGroupAction> suspendedActions() {
        return GameServerGroupActionsCopier.copyStringToEnum(suspendedActions);
    }

    /**
     * Returns true if the SuspendedActions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSuspendedActions() {
        return suspendedActions != null && !(suspendedActions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of activities that are currently suspended for this game server group. If this property is empty, all
     * activities are occurring.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSuspendedActions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of activities that are currently suspended for this game server group. If this property is empty,
     *         all activities are occurring.
     */
    public List<String> suspendedActionsAsStrings() {
        return suspendedActions;
    }

    /**
     * <p>
     * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example "1469498468.057").
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * A time stamp indicating when this game server group was last updated.
     * </p>
     * 
     * @return A time stamp indicating when this game server group was last updated.
     */
    public Instant lastUpdatedTime() {
        return lastUpdatedTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(instanceDefinitions());
        hashCode = 31 * hashCode + Objects.hashCode(balancingStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(gameServerProtectionPolicyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingGroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(suspendedActionsAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GameServerGroup)) {
            return false;
        }
        GameServerGroup other = (GameServerGroup) obj;
        return Objects.equals(gameServerGroupName(), other.gameServerGroupName())
                && Objects.equals(gameServerGroupArn(), other.gameServerGroupArn()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(instanceDefinitions(), other.instanceDefinitions())
                && Objects.equals(balancingStrategyAsString(), other.balancingStrategyAsString())
                && Objects.equals(gameServerProtectionPolicyAsString(), other.gameServerProtectionPolicyAsString())
                && Objects.equals(autoScalingGroupArn(), other.autoScalingGroupArn())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(suspendedActionsAsStrings(), other.suspendedActionsAsStrings())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GameServerGroup").add("GameServerGroupName", gameServerGroupName())
                .add("GameServerGroupArn", gameServerGroupArn()).add("RoleArn", roleArn())
                .add("InstanceDefinitions", instanceDefinitions()).add("BalancingStrategy", balancingStrategyAsString())
                .add("GameServerProtectionPolicy", gameServerProtectionPolicyAsString())
                .add("AutoScalingGroupArn", autoScalingGroupArn()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("SuspendedActions", suspendedActionsAsStrings())
                .add("CreationTime", creationTime()).add("LastUpdatedTime", lastUpdatedTime()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GameServerGroupName":
            return Optional.ofNullable(clazz.cast(gameServerGroupName()));
        case "GameServerGroupArn":
            return Optional.ofNullable(clazz.cast(gameServerGroupArn()));
        case "RoleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "InstanceDefinitions":
            return Optional.ofNullable(clazz.cast(instanceDefinitions()));
        case "BalancingStrategy":
            return Optional.ofNullable(clazz.cast(balancingStrategyAsString()));
        case "GameServerProtectionPolicy":
            return Optional.ofNullable(clazz.cast(gameServerProtectionPolicyAsString()));
        case "AutoScalingGroupArn":
            return Optional.ofNullable(clazz.cast(autoScalingGroupArn()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "SuspendedActions":
            return Optional.ofNullable(clazz.cast(suspendedActionsAsStrings()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GameServerGroup, T> g) {
        return obj -> g.apply((GameServerGroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GameServerGroup> {
        /**
         * <p>
         * A developer-defined identifier for the game server group. The name is unique per Region per AWS account.
         * </p>
         * 
         * @param gameServerGroupName
         *        A developer-defined identifier for the game server group. The name is unique per Region per AWS
         *        account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupName(String gameServerGroupName);

        /**
         * <p>
         * A generated unique ID for the game server group.
         * </p>
         * 
         * @param gameServerGroupArn
         *        A generated unique ID for the game server group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameServerGroupArn(String gameServerGroupArn);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) for an IAM role that
         * allows Amazon GameLift to access your EC2 Auto Scaling groups. The submitted role is validated to ensure that
         * it contains the necessary permissions for game server groups.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/dev/s3-arn-format.html">ARN</a>) for an IAM role
         *        that allows Amazon GameLift to access your EC2 Auto Scaling groups. The submitted role is validated to
         *        ensure that it contains the necessary permissions for game server groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances in the
         * group.
         * </p>
         * 
         * @param instanceDefinitions
         *        The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances
         *        in the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceDefinitions(Collection<InstanceDefinition> instanceDefinitions);

        /**
         * <p>
         * The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances in the
         * group.
         * </p>
         * 
         * @param instanceDefinitions
         *        The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances
         *        in the group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceDefinitions(InstanceDefinition... instanceDefinitions);

        /**
         * <p>
         * The set of EC2 instance types that GameLift FleetIQ can use when rebalancing and autoscaling instances in the
         * group.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceDefinition>.Builder} avoiding the
         * need to create one manually via {@link List<InstanceDefinition>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceDefinition>.Builder#build()} is called immediately
         * and its result is passed to {@link #instanceDefinitions(List<InstanceDefinition>)}.
         * 
         * @param instanceDefinitions
         *        a consumer that will call methods on {@link List<InstanceDefinition>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceDefinitions(List<InstanceDefinition>)
         */
        Builder instanceDefinitions(Consumer<InstanceDefinition.Builder>... instanceDefinitions);

        /**
         * <p>
         * The fallback balancing method to use for the game server group when Spot instances in a Region become
         * unavailable or are not viable for game hosting. Once triggered, this method remains active until Spot
         * instances can once again be used. Method options include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No new
         * instances are started, and the existing nonviable Spot instances are terminated (once current gameplay ends)
         * and not replaced.
         * </p>
         * </li>
         * <li>
         * <p>
         * SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide hosting
         * capacity by using On-Demand instances. Existing nonviable Spot instances are terminated (once current
         * gameplay ends) and replaced with new On-Demand instances.
         * </p>
         * </li>
         * </ul>
         * 
         * @param balancingStrategy
         *        The fallback balancing method to use for the game server group when Spot instances in a Region become
         *        unavailable or are not viable for game hosting. Once triggered, this method remains active until Spot
         *        instances can once again be used. Method options include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No
         *        new instances are started, and the existing nonviable Spot instances are terminated (once current
         *        gameplay ends) and not replaced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide
         *        hosting capacity by using On-Demand instances. Existing nonviable Spot instances are terminated (once
         *        current gameplay ends) and replaced with new On-Demand instances.
         *        </p>
         *        </li>
         * @see BalancingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BalancingStrategy
         */
        Builder balancingStrategy(String balancingStrategy);

        /**
         * <p>
         * The fallback balancing method to use for the game server group when Spot instances in a Region become
         * unavailable or are not viable for game hosting. Once triggered, this method remains active until Spot
         * instances can once again be used. Method options include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No new
         * instances are started, and the existing nonviable Spot instances are terminated (once current gameplay ends)
         * and not replaced.
         * </p>
         * </li>
         * <li>
         * <p>
         * SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide hosting
         * capacity by using On-Demand instances. Existing nonviable Spot instances are terminated (once current
         * gameplay ends) and replaced with new On-Demand instances.
         * </p>
         * </li>
         * </ul>
         * 
         * @param balancingStrategy
         *        The fallback balancing method to use for the game server group when Spot instances in a Region become
         *        unavailable or are not viable for game hosting. Once triggered, this method remains active until Spot
         *        instances can once again be used. Method options include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        SPOT_ONLY -- If Spot instances are unavailable, the game server group provides no hosting capacity. No
         *        new instances are started, and the existing nonviable Spot instances are terminated (once current
         *        gameplay ends) and not replaced.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SPOT_PREFERRED -- If Spot instances are unavailable, the game server group continues to provide
         *        hosting capacity by using On-Demand instances. Existing nonviable Spot instances are terminated (once
         *        current gameplay ends) and replaced with new On-Demand instances.
         *        </p>
         *        </li>
         * @see BalancingStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BalancingStrategy
         */
        Builder balancingStrategy(BalancingStrategy balancingStrategy);

        /**
         * <p>
         * A flag that indicates whether instances in the game server group are protected from early termination.
         * Unprotected instances that have active game servers running may be terminated during a scale-down event,
         * causing players to be dropped from the game. Protected instances cannot be terminated while there are active
         * game servers running except in the event of a forced game server group deletion (see
         * <a>DeleteGameServerGroup</a>). An exception to this is Spot Instances, which may be terminated by AWS
         * regardless of protection status.
         * </p>
         * 
         * @param gameServerProtectionPolicy
         *        A flag that indicates whether instances in the game server group are protected from early termination.
         *        Unprotected instances that have active game servers running may be terminated during a scale-down
         *        event, causing players to be dropped from the game. Protected instances cannot be terminated while
         *        there are active game servers running except in the event of a forced game server group deletion (see
         *        <a>DeleteGameServerGroup</a>). An exception to this is Spot Instances, which may be terminated by AWS
         *        regardless of protection status.
         * @see GameServerProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerProtectionPolicy
         */
        Builder gameServerProtectionPolicy(String gameServerProtectionPolicy);

        /**
         * <p>
         * A flag that indicates whether instances in the game server group are protected from early termination.
         * Unprotected instances that have active game servers running may be terminated during a scale-down event,
         * causing players to be dropped from the game. Protected instances cannot be terminated while there are active
         * game servers running except in the event of a forced game server group deletion (see
         * <a>DeleteGameServerGroup</a>). An exception to this is Spot Instances, which may be terminated by AWS
         * regardless of protection status.
         * </p>
         * 
         * @param gameServerProtectionPolicy
         *        A flag that indicates whether instances in the game server group are protected from early termination.
         *        Unprotected instances that have active game servers running may be terminated during a scale-down
         *        event, causing players to be dropped from the game. Protected instances cannot be terminated while
         *        there are active game servers running except in the event of a forced game server group deletion (see
         *        <a>DeleteGameServerGroup</a>). An exception to this is Spot Instances, which may be terminated by AWS
         *        regardless of protection status.
         * @see GameServerProtectionPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerProtectionPolicy
         */
        Builder gameServerProtectionPolicy(GameServerProtectionPolicy gameServerProtectionPolicy);

        /**
         * <p>
         * A generated unique ID for the EC2 Auto Scaling group with is associated with this game server group.
         * </p>
         * 
         * @param autoScalingGroupArn
         *        A generated unique ID for the EC2 Auto Scaling group with is associated with this game server group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingGroupArn(String autoScalingGroupArn);

        /**
         * <p>
         * The current status of the game server group. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an autoscaling group
         * in your AWS account.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE - The game server group has been successfully created.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETE_SCHEDULED - A request to delete the game server group has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
         * processing it. GameLift FleetIQ must first complete and release hosts before it deletes the autoscaling group
         * and the game server group.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETED - The game server group has been successfully deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting in an
         * error state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the game server group. Possible statuses include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an
         *        autoscaling group in your AWS account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE - The game server group has been successfully created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETE_SCHEDULED - A request to delete the game server group has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
         *        processing it. GameLift FleetIQ must first complete and release hosts before it deletes the
         *        autoscaling group and the game server group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETED - The game server group has been successfully deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting
         *        in an error state.
         *        </p>
         *        </li>
         * @see GameServerGroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerGroupStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the game server group. Possible statuses include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an autoscaling group
         * in your AWS account.
         * </p>
         * </li>
         * <li>
         * <p>
         * ACTIVE - The game server group has been successfully created.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETE_SCHEDULED - A request to delete the game server group has been received.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
         * processing it. GameLift FleetIQ must first complete and release hosts before it deletes the autoscaling group
         * and the game server group.
         * </p>
         * </li>
         * <li>
         * <p>
         * DELETED - The game server group has been successfully deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting in an
         * error state.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The current status of the game server group. Possible statuses include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        NEW - GameLift FleetIQ has validated the <code>CreateGameServerGroup()</code> request.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVATING - GameLift FleetIQ is setting up a game server group, which includes creating an
         *        autoscaling group in your AWS account.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ACTIVE - The game server group has been successfully created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETE_SCHEDULED - A request to delete the game server group has been received.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETING - GameLift FleetIQ has received a valid <code>DeleteGameServerGroup()</code> request and is
         *        processing it. GameLift FleetIQ must first complete and release hosts before it deletes the
         *        autoscaling group and the game server group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DELETED - The game server group has been successfully deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ERROR - The asynchronous processes of activating or deleting a game server group has failed, resulting
         *        in an error state.
         *        </p>
         *        </li>
         * @see GameServerGroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GameServerGroupStatus
         */
        Builder status(GameServerGroupStatus status);

        /**
         * <p>
         * Additional information about the current game server group status. This information may provide additional
         * insight on groups that in ERROR status.
         * </p>
         * 
         * @param statusReason
         *        Additional information about the current game server group status. This information may provide
         *        additional insight on groups that in ERROR status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActionsWithStrings(Collection<String> suspendedActions);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActionsWithStrings(String... suspendedActions);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActions(Collection<GameServerGroupAction> suspendedActions);

        /**
         * <p>
         * A list of activities that are currently suspended for this game server group. If this property is empty, all
         * activities are occurring.
         * </p>
         * 
         * @param suspendedActions
         *        A list of activities that are currently suspended for this game server group. If this property is
         *        empty, all activities are occurring.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder suspendedActions(GameServerGroupAction... suspendedActions);

        /**
         * <p>
         * A time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param creationTime
         *        A time stamp indicating when this data object was created. Format is a number expressed in Unix time
         *        as milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * A time stamp indicating when this game server group was last updated.
         * </p>
         * 
         * @param lastUpdatedTime
         *        A time stamp indicating when this game server group was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(Instant lastUpdatedTime);
    }

    static final class BuilderImpl implements Builder {
        private String gameServerGroupName;

        private String gameServerGroupArn;

        private String roleArn;

        private List<InstanceDefinition> instanceDefinitions = DefaultSdkAutoConstructList.getInstance();

        private String balancingStrategy;

        private String gameServerProtectionPolicy;

        private String autoScalingGroupArn;

        private String status;

        private String statusReason;

        private List<String> suspendedActions = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private Instant lastUpdatedTime;

        private BuilderImpl() {
        }

        private BuilderImpl(GameServerGroup model) {
            gameServerGroupName(model.gameServerGroupName);
            gameServerGroupArn(model.gameServerGroupArn);
            roleArn(model.roleArn);
            instanceDefinitions(model.instanceDefinitions);
            balancingStrategy(model.balancingStrategy);
            gameServerProtectionPolicy(model.gameServerProtectionPolicy);
            autoScalingGroupArn(model.autoScalingGroupArn);
            status(model.status);
            statusReason(model.statusReason);
            suspendedActionsWithStrings(model.suspendedActions);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
        }

        public final String getGameServerGroupName() {
            return gameServerGroupName;
        }

        @Override
        public final Builder gameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
            return this;
        }

        public final void setGameServerGroupName(String gameServerGroupName) {
            this.gameServerGroupName = gameServerGroupName;
        }

        public final String getGameServerGroupArn() {
            return gameServerGroupArn;
        }

        @Override
        public final Builder gameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
            return this;
        }

        public final void setGameServerGroupArn(String gameServerGroupArn) {
            this.gameServerGroupArn = gameServerGroupArn;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final Collection<InstanceDefinition.Builder> getInstanceDefinitions() {
            return instanceDefinitions != null ? instanceDefinitions.stream().map(InstanceDefinition::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder instanceDefinitions(Collection<InstanceDefinition> instanceDefinitions) {
            this.instanceDefinitions = InstanceDefinitionsCopier.copy(instanceDefinitions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceDefinitions(InstanceDefinition... instanceDefinitions) {
            instanceDefinitions(Arrays.asList(instanceDefinitions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceDefinitions(Consumer<InstanceDefinition.Builder>... instanceDefinitions) {
            instanceDefinitions(Stream.of(instanceDefinitions).map(c -> InstanceDefinition.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setInstanceDefinitions(Collection<InstanceDefinition.BuilderImpl> instanceDefinitions) {
            this.instanceDefinitions = InstanceDefinitionsCopier.copyFromBuilder(instanceDefinitions);
        }

        public final String getBalancingStrategy() {
            return balancingStrategy;
        }

        @Override
        public final Builder balancingStrategy(String balancingStrategy) {
            this.balancingStrategy = balancingStrategy;
            return this;
        }

        @Override
        public final Builder balancingStrategy(BalancingStrategy balancingStrategy) {
            this.balancingStrategy(balancingStrategy == null ? null : balancingStrategy.toString());
            return this;
        }

        public final void setBalancingStrategy(String balancingStrategy) {
            this.balancingStrategy = balancingStrategy;
        }

        public final String getGameServerProtectionPolicy() {
            return gameServerProtectionPolicy;
        }

        @Override
        public final Builder gameServerProtectionPolicy(String gameServerProtectionPolicy) {
            this.gameServerProtectionPolicy = gameServerProtectionPolicy;
            return this;
        }

        @Override
        public final Builder gameServerProtectionPolicy(GameServerProtectionPolicy gameServerProtectionPolicy) {
            this.gameServerProtectionPolicy(gameServerProtectionPolicy == null ? null : gameServerProtectionPolicy.toString());
            return this;
        }

        public final void setGameServerProtectionPolicy(String gameServerProtectionPolicy) {
            this.gameServerProtectionPolicy = gameServerProtectionPolicy;
        }

        public final String getAutoScalingGroupArn() {
            return autoScalingGroupArn;
        }

        @Override
        public final Builder autoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
            return this;
        }

        public final void setAutoScalingGroupArn(String autoScalingGroupArn) {
            this.autoScalingGroupArn = autoScalingGroupArn;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(GameServerGroupStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        public final Collection<String> getSuspendedActions() {
            return suspendedActions;
        }

        @Override
        public final Builder suspendedActionsWithStrings(Collection<String> suspendedActions) {
            this.suspendedActions = GameServerGroupActionsCopier.copy(suspendedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder suspendedActionsWithStrings(String... suspendedActions) {
            suspendedActionsWithStrings(Arrays.asList(suspendedActions));
            return this;
        }

        @Override
        public final Builder suspendedActions(Collection<GameServerGroupAction> suspendedActions) {
            this.suspendedActions = GameServerGroupActionsCopier.copyEnumToString(suspendedActions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder suspendedActions(GameServerGroupAction... suspendedActions) {
            suspendedActions(Arrays.asList(suspendedActions));
            return this;
        }

        public final void setSuspendedActions(Collection<String> suspendedActions) {
            this.suspendedActions = GameServerGroupActionsCopier.copy(suspendedActions);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final void setLastUpdatedTime(Instant lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public GameServerGroup build() {
            return new GameServerGroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
