/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.gamelift.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Properties describing a player session. Player session objects are created either by creating a player session for a
 * specific game session, or as part of a game session placement. A player session represents either a player
 * reservation for a game session (status <code>RESERVED</code>) or actual player activity in a game session (status
 * <code>ACTIVE</code>). A player session object (including player data) is automatically passed to a game session when
 * the player connects to the game session and is validated.
 * </p>
 * <p>
 * When a player disconnects, the player session status changes to <code>COMPLETED</code>. Once the session ends, the
 * player session object is retained for 30 days and then removed.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>CreatePlayerSession</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>CreatePlayerSessions</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribePlayerSessions</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * Game session placements
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a>StartGameSessionPlacement</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>DescribeGameSessionPlacement</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a>StopGameSessionPlacement</a>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PlayerSession implements SdkPojo, Serializable, ToCopyableBuilder<PlayerSession.Builder, PlayerSession> {
    private static final SdkField<String> PLAYER_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::playerSessionId)).setter(setter(Builder::playerSessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerSessionId").build()).build();

    private static final SdkField<String> PLAYER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::playerId)).setter(setter(Builder::playerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerId").build()).build();

    private static final SdkField<String> GAME_SESSION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::gameSessionId)).setter(setter(Builder::gameSessionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GameSessionId").build()).build();

    private static final SdkField<String> FLEET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::fleetId)).setter(setter(Builder::fleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetId").build()).build();

    private static final SdkField<String> FLEET_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::fleetArn)).setter(setter(Builder::fleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FleetArn").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(PlayerSession::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> TERMINATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(PlayerSession::terminationTime)).setter(setter(Builder::terminationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TerminationTime").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> IP_ADDRESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::ipAddress)).setter(setter(Builder::ipAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddress").build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(PlayerSession::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> PLAYER_DATA_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(PlayerSession::playerData)).setter(setter(Builder::playerData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlayerData").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PLAYER_SESSION_ID_FIELD,
            PLAYER_ID_FIELD, GAME_SESSION_ID_FIELD, FLEET_ID_FIELD, FLEET_ARN_FIELD, CREATION_TIME_FIELD, TERMINATION_TIME_FIELD,
            STATUS_FIELD, IP_ADDRESS_FIELD, DNS_NAME_FIELD, PORT_FIELD, PLAYER_DATA_FIELD));

    private static final long serialVersionUID = 1L;

    private final String playerSessionId;

    private final String playerId;

    private final String gameSessionId;

    private final String fleetId;

    private final String fleetArn;

    private final Instant creationTime;

    private final Instant terminationTime;

    private final String status;

    private final String ipAddress;

    private final String dnsName;

    private final Integer port;

    private final String playerData;

    private PlayerSession(BuilderImpl builder) {
        this.playerSessionId = builder.playerSessionId;
        this.playerId = builder.playerId;
        this.gameSessionId = builder.gameSessionId;
        this.fleetId = builder.fleetId;
        this.fleetArn = builder.fleetArn;
        this.creationTime = builder.creationTime;
        this.terminationTime = builder.terminationTime;
        this.status = builder.status;
        this.ipAddress = builder.ipAddress;
        this.dnsName = builder.dnsName;
        this.port = builder.port;
        this.playerData = builder.playerData;
    }

    /**
     * <p>
     * A unique identifier for a player session.
     * </p>
     * 
     * @return A unique identifier for a player session.
     */
    public String playerSessionId() {
        return playerSessionId;
    }

    /**
     * <p>
     * A unique identifier for a player that is associated with this player session.
     * </p>
     * 
     * @return A unique identifier for a player that is associated with this player session.
     */
    public String playerId() {
        return playerId;
    }

    /**
     * <p>
     * A unique identifier for the game session that the player session is connected to.
     * </p>
     * 
     * @return A unique identifier for the game session that the player session is connected to.
     */
    public String gameSessionId() {
        return gameSessionId;
    }

    /**
     * <p>
     * A unique identifier for a fleet that the player's game session is running on.
     * </p>
     * 
     * @return A unique identifier for a fleet that the player's game session is running on.
     */
    public String fleetId() {
        return fleetId;
    }

    /**
     * <p>
     * The Amazon Resource Name (<a
     * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated with the
     * GameLift fleet that the player's game session is running on.
     * </p>
     * 
     * @return The Amazon Resource Name (<a
     *         href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated
     *         with the GameLift fleet that the player's game session is running on.
     */
    public String fleetArn() {
        return fleetArn;
    }

    /**
     * <p>
     * Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
     *         milliseconds (for example "1469498468.057").
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * Time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
     * milliseconds (for example "1469498468.057").
     * </p>
     * 
     * @return Time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
     *         milliseconds (for example "1469498468.057").
     */
    public Instant terminationTime() {
        return terminationTime;
    }

    /**
     * <p>
     * Current status of the player session.
     * </p>
     * <p>
     * Possible player session statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected to the
     * server process and/or been validated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>COMPLETED</b> -- The player connection has been dropped.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was not validated
     * within the timeout limit (60 seconds).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PlayerSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the player session.</p>
     *         <p>
     *         Possible player session statuses include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected to
     *         the server process and/or been validated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>COMPLETED</b> -- The player connection has been dropped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was not
     *         validated within the timeout limit (60 seconds).
     *         </p>
     *         </li>
     * @see PlayerSessionStatus
     */
    public PlayerSessionStatus status() {
        return PlayerSessionStatus.fromValue(status);
    }

    /**
     * <p>
     * Current status of the player session.
     * </p>
     * <p>
     * Possible player session statuses include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected to the
     * server process and/or been validated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>COMPLETED</b> -- The player connection has been dropped.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was not validated
     * within the timeout limit (60 seconds).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PlayerSessionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return Current status of the player session.</p>
     *         <p>
     *         Possible player session statuses include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected to
     *         the server process and/or been validated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>COMPLETED</b> -- The player connection has been dropped.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was not
     *         validated within the timeout limit (60 seconds).
     *         </p>
     *         </li>
     * @see PlayerSessionStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * IP address of the instance that is running the game session. When connecting to a Amazon GameLift game server, a
     * client needs to reference an IP address (or DNS name) and port number.
     * </p>
     * 
     * @return IP address of the instance that is running the game session. When connecting to a Amazon GameLift game
     *         server, a client needs to reference an IP address (or DNS name) and port number.
     */
    public String ipAddress() {
        return ipAddress;
    }

    /**
     * <p>
     * DNS identifier assigned to the instance that is running the game session. Values have the following format:
     * </p>
     * <ul>
     * <li>
     * <p>
     * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     * >Amazon EC2 Instance IP Addressing</a>.)
     * </p>
     * </li>
     * </ul>
     * <p>
     * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the IP
     * address.
     * </p>
     * 
     * @return DNS identifier assigned to the instance that is running the game session. Values have the following
     *         format:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
     *         >Amazon EC2 Instance IP Addressing</a>.)
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not
     *         the IP address.
     */
    public String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * Port number for the game session. To connect to a Amazon GameLift server process, an app needs both the IP
     * address and port number.
     * </p>
     * 
     * @return Port number for the game session. To connect to a Amazon GameLift server process, an app needs both the
     *         IP address and port number.
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * Developer-defined information related to a player. Amazon GameLift does not use this data, so it can be formatted
     * as needed for use in the game.
     * </p>
     * 
     * @return Developer-defined information related to a player. Amazon GameLift does not use this data, so it can be
     *         formatted as needed for use in the game.
     */
    public String playerData() {
        return playerData;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(playerSessionId());
        hashCode = 31 * hashCode + Objects.hashCode(playerId());
        hashCode = 31 * hashCode + Objects.hashCode(gameSessionId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetId());
        hashCode = 31 * hashCode + Objects.hashCode(fleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(terminationTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddress());
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(playerData());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PlayerSession)) {
            return false;
        }
        PlayerSession other = (PlayerSession) obj;
        return Objects.equals(playerSessionId(), other.playerSessionId()) && Objects.equals(playerId(), other.playerId())
                && Objects.equals(gameSessionId(), other.gameSessionId()) && Objects.equals(fleetId(), other.fleetId())
                && Objects.equals(fleetArn(), other.fleetArn()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(terminationTime(), other.terminationTime())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(ipAddress(), other.ipAddress())
                && Objects.equals(dnsName(), other.dnsName()) && Objects.equals(port(), other.port())
                && Objects.equals(playerData(), other.playerData());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("PlayerSession").add("PlayerSessionId", playerSessionId()).add("PlayerId", playerId())
                .add("GameSessionId", gameSessionId()).add("FleetId", fleetId()).add("FleetArn", fleetArn())
                .add("CreationTime", creationTime()).add("TerminationTime", terminationTime()).add("Status", statusAsString())
                .add("IpAddress", ipAddress()).add("DnsName", dnsName()).add("Port", port()).add("PlayerData", playerData())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "PlayerSessionId":
            return Optional.ofNullable(clazz.cast(playerSessionId()));
        case "PlayerId":
            return Optional.ofNullable(clazz.cast(playerId()));
        case "GameSessionId":
            return Optional.ofNullable(clazz.cast(gameSessionId()));
        case "FleetId":
            return Optional.ofNullable(clazz.cast(fleetId()));
        case "FleetArn":
            return Optional.ofNullable(clazz.cast(fleetArn()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "TerminationTime":
            return Optional.ofNullable(clazz.cast(terminationTime()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "IpAddress":
            return Optional.ofNullable(clazz.cast(ipAddress()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "PlayerData":
            return Optional.ofNullable(clazz.cast(playerData()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PlayerSession, T> g) {
        return obj -> g.apply((PlayerSession) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PlayerSession> {
        /**
         * <p>
         * A unique identifier for a player session.
         * </p>
         * 
         * @param playerSessionId
         *        A unique identifier for a player session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerSessionId(String playerSessionId);

        /**
         * <p>
         * A unique identifier for a player that is associated with this player session.
         * </p>
         * 
         * @param playerId
         *        A unique identifier for a player that is associated with this player session.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerId(String playerId);

        /**
         * <p>
         * A unique identifier for the game session that the player session is connected to.
         * </p>
         * 
         * @param gameSessionId
         *        A unique identifier for the game session that the player session is connected to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder gameSessionId(String gameSessionId);

        /**
         * <p>
         * A unique identifier for a fleet that the player's game session is running on.
         * </p>
         * 
         * @param fleetId
         *        A unique identifier for a fleet that the player's game session is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetId(String fleetId);

        /**
         * <p>
         * The Amazon Resource Name (<a
         * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated with
         * the GameLift fleet that the player's game session is running on.
         * </p>
         * 
         * @param fleetArn
         *        The Amazon Resource Name (<a
         *        href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">ARN</a>) associated
         *        with the GameLift fleet that the player's game session is running on.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fleetArn(String fleetArn);

        /**
         * <p>
         * Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param creationTime
         *        Time stamp indicating when this data object was created. Format is a number expressed in Unix time as
         *        milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * Time stamp indicating when this data object was terminated. Format is a number expressed in Unix time as
         * milliseconds (for example "1469498468.057").
         * </p>
         * 
         * @param terminationTime
         *        Time stamp indicating when this data object was terminated. Format is a number expressed in Unix time
         *        as milliseconds (for example "1469498468.057").
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder terminationTime(Instant terminationTime);

        /**
         * <p>
         * Current status of the player session.
         * </p>
         * <p>
         * Possible player session statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected to the
         * server process and/or been validated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>COMPLETED</b> -- The player connection has been dropped.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was not
         * validated within the timeout limit (60 seconds).
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the player session.</p>
         *        <p>
         *        Possible player session statuses include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected
         *        to the server process and/or been validated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>COMPLETED</b> -- The player connection has been dropped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was
         *        not validated within the timeout limit (60 seconds).
         *        </p>
         *        </li>
         * @see PlayerSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlayerSessionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Current status of the player session.
         * </p>
         * <p>
         * Possible player session statuses include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected to the
         * server process and/or been validated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>COMPLETED</b> -- The player connection has been dropped.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was not
         * validated within the timeout limit (60 seconds).
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        Current status of the player session.</p>
         *        <p>
         *        Possible player session statuses include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b>RESERVED</b> -- The player session request has been received, but the player has not yet connected
         *        to the server process and/or been validated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>ACTIVE</b> -- The player has been validated by the server process and is currently connected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>COMPLETED</b> -- The player connection has been dropped.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b>TIMEDOUT</b> -- A player session request was received, but the player did not connect and/or was
         *        not validated within the timeout limit (60 seconds).
         *        </p>
         *        </li>
         * @see PlayerSessionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlayerSessionStatus
         */
        Builder status(PlayerSessionStatus status);

        /**
         * <p>
         * IP address of the instance that is running the game session. When connecting to a Amazon GameLift game
         * server, a client needs to reference an IP address (or DNS name) and port number.
         * </p>
         * 
         * @param ipAddress
         *        IP address of the instance that is running the game session. When connecting to a Amazon GameLift game
         *        server, a client needs to reference an IP address (or DNS name) and port number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipAddress(String ipAddress);

        /**
         * <p>
         * DNS identifier assigned to the instance that is running the game session. Values have the following format:
         * </p>
         * <ul>
         * <li>
         * <p>
         * TLS-enabled fleets: <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         * >Amazon EC2 Instance IP Addressing</a>.)
         * </p>
         * </li>
         * </ul>
         * <p>
         * When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name, not the
         * IP address.
         * </p>
         * 
         * @param dnsName
         *        DNS identifier assigned to the instance that is running the game session. Values have the following
         *        format:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        TLS-enabled fleets:
         *        <code>&lt;unique identifier&gt;.&lt;region identifier&gt;.amazongamelift.com</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Non-TLS-enabled fleets: <code>ec2-&lt;unique identifier&gt;.compute.amazonaws.com</code>. (See <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/using-instance-addressing.html#concepts-public-addresses"
         *        >Amazon EC2 Instance IP Addressing</a>.)
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        When connecting to a game session that is running on a TLS-enabled fleet, you must use the DNS name,
         *        not the IP address.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * Port number for the game session. To connect to a Amazon GameLift server process, an app needs both the IP
         * address and port number.
         * </p>
         * 
         * @param port
         *        Port number for the game session. To connect to a Amazon GameLift server process, an app needs both
         *        the IP address and port number.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * Developer-defined information related to a player. Amazon GameLift does not use this data, so it can be
         * formatted as needed for use in the game.
         * </p>
         * 
         * @param playerData
         *        Developer-defined information related to a player. Amazon GameLift does not use this data, so it can
         *        be formatted as needed for use in the game.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder playerData(String playerData);
    }

    static final class BuilderImpl implements Builder {
        private String playerSessionId;

        private String playerId;

        private String gameSessionId;

        private String fleetId;

        private String fleetArn;

        private Instant creationTime;

        private Instant terminationTime;

        private String status;

        private String ipAddress;

        private String dnsName;

        private Integer port;

        private String playerData;

        private BuilderImpl() {
        }

        private BuilderImpl(PlayerSession model) {
            playerSessionId(model.playerSessionId);
            playerId(model.playerId);
            gameSessionId(model.gameSessionId);
            fleetId(model.fleetId);
            fleetArn(model.fleetArn);
            creationTime(model.creationTime);
            terminationTime(model.terminationTime);
            status(model.status);
            ipAddress(model.ipAddress);
            dnsName(model.dnsName);
            port(model.port);
            playerData(model.playerData);
        }

        public final String getPlayerSessionId() {
            return playerSessionId;
        }

        @Override
        public final Builder playerSessionId(String playerSessionId) {
            this.playerSessionId = playerSessionId;
            return this;
        }

        public final void setPlayerSessionId(String playerSessionId) {
            this.playerSessionId = playerSessionId;
        }

        public final String getPlayerId() {
            return playerId;
        }

        @Override
        public final Builder playerId(String playerId) {
            this.playerId = playerId;
            return this;
        }

        public final void setPlayerId(String playerId) {
            this.playerId = playerId;
        }

        public final String getGameSessionId() {
            return gameSessionId;
        }

        @Override
        public final Builder gameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
            return this;
        }

        public final void setGameSessionId(String gameSessionId) {
            this.gameSessionId = gameSessionId;
        }

        public final String getFleetId() {
            return fleetId;
        }

        @Override
        public final Builder fleetId(String fleetId) {
            this.fleetId = fleetId;
            return this;
        }

        public final void setFleetId(String fleetId) {
            this.fleetId = fleetId;
        }

        public final String getFleetArn() {
            return fleetArn;
        }

        @Override
        public final Builder fleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
            return this;
        }

        public final void setFleetArn(String fleetArn) {
            this.fleetArn = fleetArn;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getTerminationTime() {
            return terminationTime;
        }

        @Override
        public final Builder terminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
            return this;
        }

        public final void setTerminationTime(Instant terminationTime) {
            this.terminationTime = terminationTime;
        }

        public final String getStatusAsString() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PlayerSessionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getIpAddress() {
            return ipAddress;
        }

        @Override
        public final Builder ipAddress(String ipAddress) {
            this.ipAddress = ipAddress;
            return this;
        }

        public final void setIpAddress(String ipAddress) {
            this.ipAddress = ipAddress;
        }

        public final String getDnsName() {
            return dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getPlayerData() {
            return playerData;
        }

        @Override
        public final Builder playerData(String playerData) {
            this.playerData = playerData;
            return this;
        }

        public final void setPlayerData(String playerData) {
            this.playerData = playerData;
        }

        @Override
        public PlayerSession build() {
            return new PlayerSession(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
