/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies who can mount an OpenZFS file system and the options available while mounting the file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpenZFSClientConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<OpenZFSClientConfiguration.Builder, OpenZFSClientConfiguration> {
    private static final SdkField<String> CLIENTS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Clients")
            .getter(getter(OpenZFSClientConfiguration::clients)).setter(setter(Builder::clients))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Clients").build()).build();

    private static final SdkField<List<String>> OPTIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Options")
            .getter(getter(OpenZFSClientConfiguration::options))
            .setter(setter(Builder::options))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Options").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENTS_FIELD, OPTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String clients;

    private final List<String> options;

    private OpenZFSClientConfiguration(BuilderImpl builder) {
        this.clients = builder.clients;
        this.options = builder.options;
    }

    /**
     * <p>
     * A value that specifies who can mount the file system. You can provide a wildcard character (<code>*</code>), an
     * IP address (<code>0.0.0.0</code>), or a CIDR address (<code>192.0.2.0/24</code>). By default, Amazon FSx uses the
     * wildcard character when specifying the client.
     * </p>
     * 
     * @return A value that specifies who can mount the file system. You can provide a wildcard character (
     *         <code>*</code>), an IP address (<code>0.0.0.0</code>), or a CIDR address (<code>192.0.2.0/24</code>). By
     *         default, Amazon FSx uses the wildcard character when specifying the client.
     */
    public final String clients() {
        return clients;
    }

    /**
     * For responses, this returns true if the service returned a value for the Options property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOptions() {
        return options != null && !(options instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The options to use when mounting the file system. For a list of options that you can use with Network File System
     * (NFS), see the <a href="https://linux.die.net/man/5/exports">exports(5) - Linux man page</a>. When choosing your
     * options, consider the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>crossmnt</code> is used by default. If you don't specify <code>crossmnt</code> when changing the client
     * configuration, you won't be able to see or access snapshots in your file system's snapshot directory.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sync</code> is used by default. If you instead specify <code>async</code>, the system acknowledges writes
     * before writing to disk. If the system crashes before the writes are finished, you lose the unwritten data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOptions} method.
     * </p>
     * 
     * @return The options to use when mounting the file system. For a list of options that you can use with Network
     *         File System (NFS), see the <a href="https://linux.die.net/man/5/exports">exports(5) - Linux man page</a>.
     *         When choosing your options, consider the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>crossmnt</code> is used by default. If you don't specify <code>crossmnt</code> when changing the
     *         client configuration, you won't be able to see or access snapshots in your file system's snapshot
     *         directory.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sync</code> is used by default. If you instead specify <code>async</code>, the system acknowledges
     *         writes before writing to disk. If the system crashes before the writes are finished, you lose the
     *         unwritten data.
     *         </p>
     *         </li>
     */
    public final List<String> options() {
        return options;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(clients());
        hashCode = 31 * hashCode + Objects.hashCode(hasOptions() ? options() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpenZFSClientConfiguration)) {
            return false;
        }
        OpenZFSClientConfiguration other = (OpenZFSClientConfiguration) obj;
        return Objects.equals(clients(), other.clients()) && hasOptions() == other.hasOptions()
                && Objects.equals(options(), other.options());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpenZFSClientConfiguration").add("Clients", clients())
                .add("Options", hasOptions() ? options() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Clients":
            return Optional.ofNullable(clazz.cast(clients()));
        case "Options":
            return Optional.ofNullable(clazz.cast(options()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Clients", CLIENTS_FIELD);
        map.put("Options", OPTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OpenZFSClientConfiguration, T> g) {
        return obj -> g.apply((OpenZFSClientConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpenZFSClientConfiguration> {
        /**
         * <p>
         * A value that specifies who can mount the file system. You can provide a wildcard character (<code>*</code>),
         * an IP address (<code>0.0.0.0</code>), or a CIDR address (<code>192.0.2.0/24</code>). By default, Amazon FSx
         * uses the wildcard character when specifying the client.
         * </p>
         * 
         * @param clients
         *        A value that specifies who can mount the file system. You can provide a wildcard character (
         *        <code>*</code>), an IP address (<code>0.0.0.0</code>), or a CIDR address (<code>192.0.2.0/24</code>).
         *        By default, Amazon FSx uses the wildcard character when specifying the client.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clients(String clients);

        /**
         * <p>
         * The options to use when mounting the file system. For a list of options that you can use with Network File
         * System (NFS), see the <a href="https://linux.die.net/man/5/exports">exports(5) - Linux man page</a>. When
         * choosing your options, consider the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>crossmnt</code> is used by default. If you don't specify <code>crossmnt</code> when changing the client
         * configuration, you won't be able to see or access snapshots in your file system's snapshot directory.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sync</code> is used by default. If you instead specify <code>async</code>, the system acknowledges
         * writes before writing to disk. If the system crashes before the writes are finished, you lose the unwritten
         * data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param options
         *        The options to use when mounting the file system. For a list of options that you can use with Network
         *        File System (NFS), see the <a href="https://linux.die.net/man/5/exports">exports(5) - Linux man
         *        page</a>. When choosing your options, consider the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>crossmnt</code> is used by default. If you don't specify <code>crossmnt</code> when changing the
         *        client configuration, you won't be able to see or access snapshots in your file system's snapshot
         *        directory.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sync</code> is used by default. If you instead specify <code>async</code>, the system
         *        acknowledges writes before writing to disk. If the system crashes before the writes are finished, you
         *        lose the unwritten data.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(Collection<String> options);

        /**
         * <p>
         * The options to use when mounting the file system. For a list of options that you can use with Network File
         * System (NFS), see the <a href="https://linux.die.net/man/5/exports">exports(5) - Linux man page</a>. When
         * choosing your options, consider the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>crossmnt</code> is used by default. If you don't specify <code>crossmnt</code> when changing the client
         * configuration, you won't be able to see or access snapshots in your file system's snapshot directory.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sync</code> is used by default. If you instead specify <code>async</code>, the system acknowledges
         * writes before writing to disk. If the system crashes before the writes are finished, you lose the unwritten
         * data.
         * </p>
         * </li>
         * </ul>
         * 
         * @param options
         *        The options to use when mounting the file system. For a list of options that you can use with Network
         *        File System (NFS), see the <a href="https://linux.die.net/man/5/exports">exports(5) - Linux man
         *        page</a>. When choosing your options, consider the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>crossmnt</code> is used by default. If you don't specify <code>crossmnt</code> when changing the
         *        client configuration, you won't be able to see or access snapshots in your file system's snapshot
         *        directory.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sync</code> is used by default. If you instead specify <code>async</code>, the system
         *        acknowledges writes before writing to disk. If the system crashes before the writes are finished, you
         *        lose the unwritten data.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(String... options);
    }

    static final class BuilderImpl implements Builder {
        private String clients;

        private List<String> options = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(OpenZFSClientConfiguration model) {
            clients(model.clients);
            options(model.options);
        }

        public final String getClients() {
            return clients;
        }

        public final void setClients(String clients) {
            this.clients = clients;
        }

        @Override
        public final Builder clients(String clients) {
            this.clients = clients;
            return this;
        }

        public final Collection<String> getOptions() {
            if (options instanceof SdkAutoConstructList) {
                return null;
            }
            return options;
        }

        public final void setOptions(Collection<String> options) {
            this.options = OpenZFSNfsExportOptionsCopier.copy(options);
        }

        @Override
        public final Builder options(Collection<String> options) {
            this.options = OpenZFSNfsExportOptionsCopier.copy(options);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder options(String... options) {
            options(Arrays.asList(options));
            return this;
        }

        @Override
        public OpenZFSClientConfiguration build() {
            return new OpenZFSClientConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
