/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the file system user identity that will be used for authorizing all file access requests that are made
 * using the S3 access point. The identity can be either a UNIX user or a Windows user.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OntapFileSystemIdentity implements SdkPojo, Serializable,
        ToCopyableBuilder<OntapFileSystemIdentity.Builder, OntapFileSystemIdentity> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(OntapFileSystemIdentity::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<OntapUnixFileSystemUser> UNIX_USER_FIELD = SdkField
            .<OntapUnixFileSystemUser> builder(MarshallingType.SDK_POJO).memberName("UnixUser")
            .getter(getter(OntapFileSystemIdentity::unixUser)).setter(setter(Builder::unixUser))
            .constructor(OntapUnixFileSystemUser::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UnixUser").build()).build();

    private static final SdkField<OntapWindowsFileSystemUser> WINDOWS_USER_FIELD = SdkField
            .<OntapWindowsFileSystemUser> builder(MarshallingType.SDK_POJO).memberName("WindowsUser")
            .getter(getter(OntapFileSystemIdentity::windowsUser)).setter(setter(Builder::windowsUser))
            .constructor(OntapWindowsFileSystemUser::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowsUser").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, UNIX_USER_FIELD,
            WINDOWS_USER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final OntapUnixFileSystemUser unixUser;

    private final OntapWindowsFileSystemUser windowsUser;

    private OntapFileSystemIdentity(BuilderImpl builder) {
        this.type = builder.type;
        this.unixUser = builder.unixUser;
        this.windowsUser = builder.windowsUser;
    }

    /**
     * <p>
     * Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and <code>WINDOWS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link OntapFileSystemUserType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and
     *         <code>WINDOWS</code>.
     * @see OntapFileSystemUserType
     */
    public final OntapFileSystemUserType type() {
        return OntapFileSystemUserType.fromValue(type);
    }

    /**
     * <p>
     * Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and <code>WINDOWS</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link OntapFileSystemUserType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and
     *         <code>WINDOWS</code>.
     * @see OntapFileSystemUserType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * Specifies the UNIX user identity for file system operations.
     * </p>
     * 
     * @return Specifies the UNIX user identity for file system operations.
     */
    public final OntapUnixFileSystemUser unixUser() {
        return unixUser;
    }

    /**
     * <p>
     * Specifies the Windows user identity for file system operations.
     * </p>
     * 
     * @return Specifies the Windows user identity for file system operations.
     */
    public final OntapWindowsFileSystemUser windowsUser() {
        return windowsUser;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(unixUser());
        hashCode = 31 * hashCode + Objects.hashCode(windowsUser());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OntapFileSystemIdentity)) {
            return false;
        }
        OntapFileSystemIdentity other = (OntapFileSystemIdentity) obj;
        return Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(unixUser(), other.unixUser())
                && Objects.equals(windowsUser(), other.windowsUser());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OntapFileSystemIdentity").add("Type", typeAsString()).add("UnixUser", unixUser())
                .add("WindowsUser", windowsUser()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "UnixUser":
            return Optional.ofNullable(clazz.cast(unixUser()));
        case "WindowsUser":
            return Optional.ofNullable(clazz.cast(windowsUser()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("UnixUser", UNIX_USER_FIELD);
        map.put("WindowsUser", WINDOWS_USER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<OntapFileSystemIdentity, T> g) {
        return obj -> g.apply((OntapFileSystemIdentity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OntapFileSystemIdentity> {
        /**
         * <p>
         * Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and <code>WINDOWS</code>.
         * </p>
         * 
         * @param type
         *        Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and
         *        <code>WINDOWS</code>.
         * @see OntapFileSystemUserType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OntapFileSystemUserType
         */
        Builder type(String type);

        /**
         * <p>
         * Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and <code>WINDOWS</code>.
         * </p>
         * 
         * @param type
         *        Specifies the FSx for ONTAP user identity type. Valid values are <code>UNIX</code> and
         *        <code>WINDOWS</code>.
         * @see OntapFileSystemUserType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OntapFileSystemUserType
         */
        Builder type(OntapFileSystemUserType type);

        /**
         * <p>
         * Specifies the UNIX user identity for file system operations.
         * </p>
         * 
         * @param unixUser
         *        Specifies the UNIX user identity for file system operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unixUser(OntapUnixFileSystemUser unixUser);

        /**
         * <p>
         * Specifies the UNIX user identity for file system operations.
         * </p>
         * This is a convenience method that creates an instance of the {@link OntapUnixFileSystemUser.Builder} avoiding
         * the need to create one manually via {@link OntapUnixFileSystemUser#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OntapUnixFileSystemUser.Builder#build()} is called immediately
         * and its result is passed to {@link #unixUser(OntapUnixFileSystemUser)}.
         * 
         * @param unixUser
         *        a consumer that will call methods on {@link OntapUnixFileSystemUser.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #unixUser(OntapUnixFileSystemUser)
         */
        default Builder unixUser(Consumer<OntapUnixFileSystemUser.Builder> unixUser) {
            return unixUser(OntapUnixFileSystemUser.builder().applyMutation(unixUser).build());
        }

        /**
         * <p>
         * Specifies the Windows user identity for file system operations.
         * </p>
         * 
         * @param windowsUser
         *        Specifies the Windows user identity for file system operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowsUser(OntapWindowsFileSystemUser windowsUser);

        /**
         * <p>
         * Specifies the Windows user identity for file system operations.
         * </p>
         * This is a convenience method that creates an instance of the {@link OntapWindowsFileSystemUser.Builder}
         * avoiding the need to create one manually via {@link OntapWindowsFileSystemUser#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OntapWindowsFileSystemUser.Builder#build()} is called immediately
         * and its result is passed to {@link #windowsUser(OntapWindowsFileSystemUser)}.
         * 
         * @param windowsUser
         *        a consumer that will call methods on {@link OntapWindowsFileSystemUser.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #windowsUser(OntapWindowsFileSystemUser)
         */
        default Builder windowsUser(Consumer<OntapWindowsFileSystemUser.Builder> windowsUser) {
            return windowsUser(OntapWindowsFileSystemUser.builder().applyMutation(windowsUser).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private OntapUnixFileSystemUser unixUser;

        private OntapWindowsFileSystemUser windowsUser;

        private BuilderImpl() {
        }

        private BuilderImpl(OntapFileSystemIdentity model) {
            type(model.type);
            unixUser(model.unixUser);
            windowsUser(model.windowsUser);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(OntapFileSystemUserType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final OntapUnixFileSystemUser.Builder getUnixUser() {
            return unixUser != null ? unixUser.toBuilder() : null;
        }

        public final void setUnixUser(OntapUnixFileSystemUser.BuilderImpl unixUser) {
            this.unixUser = unixUser != null ? unixUser.build() : null;
        }

        @Override
        public final Builder unixUser(OntapUnixFileSystemUser unixUser) {
            this.unixUser = unixUser;
            return this;
        }

        public final OntapWindowsFileSystemUser.Builder getWindowsUser() {
            return windowsUser != null ? windowsUser.toBuilder() : null;
        }

        public final void setWindowsUser(OntapWindowsFileSystemUser.BuilderImpl windowsUser) {
            this.windowsUser = windowsUser != null ? windowsUser.build() : null;
        }

        @Override
        public final Builder windowsUser(OntapWindowsFileSystemUser windowsUser) {
            this.windowsUser = windowsUser;
            return this;
        }

        @Override
        public OntapFileSystemIdentity build() {
            return new OntapFileSystemIdentity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
