/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the Amazon FSx for NetApp ONTAP storage virtual machine (SVM) configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StorageVirtualMachine implements SdkPojo, Serializable,
        ToCopyableBuilder<StorageVirtualMachine.Builder, StorageVirtualMachine> {
    private static final SdkField<SvmActiveDirectoryConfiguration> ACTIVE_DIRECTORY_CONFIGURATION_FIELD = SdkField
            .<SvmActiveDirectoryConfiguration> builder(MarshallingType.SDK_POJO)
            .memberName("ActiveDirectoryConfiguration")
            .getter(getter(StorageVirtualMachine::activeDirectoryConfiguration))
            .setter(setter(Builder::activeDirectoryConfiguration))
            .constructor(SvmActiveDirectoryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActiveDirectoryConfiguration")
                    .build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(StorageVirtualMachine::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<SvmEndpoints> ENDPOINTS_FIELD = SdkField.<SvmEndpoints> builder(MarshallingType.SDK_POJO)
            .memberName("Endpoints").getter(getter(StorageVirtualMachine::endpoints)).setter(setter(Builder::endpoints))
            .constructor(SvmEndpoints::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Endpoints").build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(StorageVirtualMachine::fileSystemId)).setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(StorageVirtualMachine::lifecycleAsString)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(StorageVirtualMachine::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(StorageVirtualMachine::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<String> STORAGE_VIRTUAL_MACHINE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StorageVirtualMachineId").getter(getter(StorageVirtualMachine::storageVirtualMachineId))
            .setter(setter(Builder::storageVirtualMachineId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageVirtualMachineId").build())
            .build();

    private static final SdkField<String> SUBTYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Subtype")
            .getter(getter(StorageVirtualMachine::subtypeAsString)).setter(setter(Builder::subtype))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subtype").build()).build();

    private static final SdkField<String> UUID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("UUID")
            .getter(getter(StorageVirtualMachine::uuid)).setter(setter(Builder::uuid))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UUID").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(StorageVirtualMachine::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LifecycleTransitionReason> LIFECYCLE_TRANSITION_REASON_FIELD = SdkField
            .<LifecycleTransitionReason> builder(MarshallingType.SDK_POJO).memberName("LifecycleTransitionReason")
            .getter(getter(StorageVirtualMachine::lifecycleTransitionReason)).setter(setter(Builder::lifecycleTransitionReason))
            .constructor(LifecycleTransitionReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LifecycleTransitionReason").build())
            .build();

    private static final SdkField<String> ROOT_VOLUME_SECURITY_STYLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RootVolumeSecurityStyle").getter(getter(StorageVirtualMachine::rootVolumeSecurityStyleAsString))
            .setter(setter(Builder::rootVolumeSecurityStyle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RootVolumeSecurityStyle").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ACTIVE_DIRECTORY_CONFIGURATION_FIELD, CREATION_TIME_FIELD, ENDPOINTS_FIELD, FILE_SYSTEM_ID_FIELD, LIFECYCLE_FIELD,
            NAME_FIELD, RESOURCE_ARN_FIELD, STORAGE_VIRTUAL_MACHINE_ID_FIELD, SUBTYPE_FIELD, UUID_FIELD, TAGS_FIELD,
            LIFECYCLE_TRANSITION_REASON_FIELD, ROOT_VOLUME_SECURITY_STYLE_FIELD));

    private static final long serialVersionUID = 1L;

    private final SvmActiveDirectoryConfiguration activeDirectoryConfiguration;

    private final Instant creationTime;

    private final SvmEndpoints endpoints;

    private final String fileSystemId;

    private final String lifecycle;

    private final String name;

    private final String resourceARN;

    private final String storageVirtualMachineId;

    private final String subtype;

    private final String uuid;

    private final List<Tag> tags;

    private final LifecycleTransitionReason lifecycleTransitionReason;

    private final String rootVolumeSecurityStyle;

    private StorageVirtualMachine(BuilderImpl builder) {
        this.activeDirectoryConfiguration = builder.activeDirectoryConfiguration;
        this.creationTime = builder.creationTime;
        this.endpoints = builder.endpoints;
        this.fileSystemId = builder.fileSystemId;
        this.lifecycle = builder.lifecycle;
        this.name = builder.name;
        this.resourceARN = builder.resourceARN;
        this.storageVirtualMachineId = builder.storageVirtualMachineId;
        this.subtype = builder.subtype;
        this.uuid = builder.uuid;
        this.tags = builder.tags;
        this.lifecycleTransitionReason = builder.lifecycleTransitionReason;
        this.rootVolumeSecurityStyle = builder.rootVolumeSecurityStyle;
    }

    /**
     * <p>
     * Describes the Microsoft Active Directory configuration to which the SVM is joined, if applicable.
     * </p>
     * 
     * @return Describes the Microsoft Active Directory configuration to which the SVM is joined, if applicable.
     */
    public final SvmActiveDirectoryConfiguration activeDirectoryConfiguration() {
        return activeDirectoryConfiguration;
    }

    /**
     * Returns the value of the CreationTime property for this object.
     * 
     * @return The value of the CreationTime property for this object.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The endpoints that are used to access data or to manage the SVM using the NetApp ONTAP CLI, REST API, or NetApp
     * CloudManager. They are the <code>Iscsi</code>, <code>Management</code>, <code>Nfs</code>, and <code>Smb</code>
     * endpoints.
     * </p>
     * 
     * @return The endpoints that are used to access data or to manage the SVM using the NetApp ONTAP CLI, REST API, or
     *         NetApp CloudManager. They are the <code>Iscsi</code>, <code>Management</code>, <code>Nfs</code>, and
     *         <code>Smb</code> endpoints.
     */
    public final SvmEndpoints endpoints() {
        return endpoints;
    }

    /**
     * Returns the value of the FileSystemId property for this object.
     * 
     * @return The value of the FileSystemId property for this object.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * <p>
     * Describes the SVM's lifecycle status.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATED</code> - The SVM is fully available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the new SVM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Amazon FSx was unable to create the SVM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING</code> - Amazon FSx has not started creating the SVM.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link StorageVirtualMachineLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the SVM's lifecycle status.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATED</code> - The SVM is fully available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the new SVM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Amazon FSx was unable to create the SVM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - Amazon FSx has not started creating the SVM.
     *         </p>
     *         </li>
     * @see StorageVirtualMachineLifecycle
     */
    public final StorageVirtualMachineLifecycle lifecycle() {
        return StorageVirtualMachineLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * Describes the SVM's lifecycle status.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATED</code> - The SVM is fully available for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the new SVM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Amazon FSx was unable to create the SVM.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING</code> - Amazon FSx has not started creating the SVM.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link StorageVirtualMachineLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return Describes the SVM's lifecycle status.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATED</code> - The SVM is fully available for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the new SVM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Amazon FSx was unable to create the SVM.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - Amazon FSx has not started creating the SVM.
     *         </p>
     *         </li>
     * @see StorageVirtualMachineLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * <p>
     * The name of the SVM, if provisioned.
     * </p>
     * 
     * @return The name of the SVM, if provisioned.
     */
    public final String name() {
        return name;
    }

    /**
     * Returns the value of the ResourceARN property for this object.
     * 
     * @return The value of the ResourceARN property for this object.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * <p>
     * The SVM's system generated unique ID.
     * </p>
     * 
     * @return The SVM's system generated unique ID.
     */
    public final String storageVirtualMachineId() {
        return storageVirtualMachineId;
    }

    /**
     * <p>
     * Describes the SVM's subtype.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subtype} will
     * return {@link StorageVirtualMachineSubtype#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #subtypeAsString}.
     * </p>
     * 
     * @return Describes the SVM's subtype.
     * @see StorageVirtualMachineSubtype
     */
    public final StorageVirtualMachineSubtype subtype() {
        return StorageVirtualMachineSubtype.fromValue(subtype);
    }

    /**
     * <p>
     * Describes the SVM's subtype.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #subtype} will
     * return {@link StorageVirtualMachineSubtype#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #subtypeAsString}.
     * </p>
     * 
     * @return Describes the SVM's subtype.
     * @see StorageVirtualMachineSubtype
     */
    public final String subtypeAsString() {
        return subtype;
    }

    /**
     * <p>
     * The SVM's UUID (universally unique identifier).
     * </p>
     * 
     * @return The SVM's UUID (universally unique identifier).
     */
    public final String uuid() {
        return uuid;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Describes why the SVM lifecycle state changed.
     * </p>
     * 
     * @return Describes why the SVM lifecycle state changed.
     */
    public final LifecycleTransitionReason lifecycleTransitionReason() {
        return lifecycleTransitionReason;
    }

    /**
     * <p>
     * The security style of the root volume of the SVM.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #rootVolumeSecurityStyle} will return
     * {@link StorageVirtualMachineRootVolumeSecurityStyle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #rootVolumeSecurityStyleAsString}.
     * </p>
     * 
     * @return The security style of the root volume of the SVM.
     * @see StorageVirtualMachineRootVolumeSecurityStyle
     */
    public final StorageVirtualMachineRootVolumeSecurityStyle rootVolumeSecurityStyle() {
        return StorageVirtualMachineRootVolumeSecurityStyle.fromValue(rootVolumeSecurityStyle);
    }

    /**
     * <p>
     * The security style of the root volume of the SVM.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #rootVolumeSecurityStyle} will return
     * {@link StorageVirtualMachineRootVolumeSecurityStyle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #rootVolumeSecurityStyleAsString}.
     * </p>
     * 
     * @return The security style of the root volume of the SVM.
     * @see StorageVirtualMachineRootVolumeSecurityStyle
     */
    public final String rootVolumeSecurityStyleAsString() {
        return rootVolumeSecurityStyle;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(activeDirectoryConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(endpoints());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(storageVirtualMachineId());
        hashCode = 31 * hashCode + Objects.hashCode(subtypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(uuid());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleTransitionReason());
        hashCode = 31 * hashCode + Objects.hashCode(rootVolumeSecurityStyleAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StorageVirtualMachine)) {
            return false;
        }
        StorageVirtualMachine other = (StorageVirtualMachine) obj;
        return Objects.equals(activeDirectoryConfiguration(), other.activeDirectoryConfiguration())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(endpoints(), other.endpoints())
                && Objects.equals(fileSystemId(), other.fileSystemId())
                && Objects.equals(lifecycleAsString(), other.lifecycleAsString()) && Objects.equals(name(), other.name())
                && Objects.equals(resourceARN(), other.resourceARN())
                && Objects.equals(storageVirtualMachineId(), other.storageVirtualMachineId())
                && Objects.equals(subtypeAsString(), other.subtypeAsString()) && Objects.equals(uuid(), other.uuid())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(lifecycleTransitionReason(), other.lifecycleTransitionReason())
                && Objects.equals(rootVolumeSecurityStyleAsString(), other.rootVolumeSecurityStyleAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StorageVirtualMachine").add("ActiveDirectoryConfiguration", activeDirectoryConfiguration())
                .add("CreationTime", creationTime()).add("Endpoints", endpoints()).add("FileSystemId", fileSystemId())
                .add("Lifecycle", lifecycleAsString()).add("Name", name()).add("ResourceARN", resourceARN())
                .add("StorageVirtualMachineId", storageVirtualMachineId()).add("Subtype", subtypeAsString()).add("UUID", uuid())
                .add("Tags", hasTags() ? tags() : null).add("LifecycleTransitionReason", lifecycleTransitionReason())
                .add("RootVolumeSecurityStyle", rootVolumeSecurityStyleAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActiveDirectoryConfiguration":
            return Optional.ofNullable(clazz.cast(activeDirectoryConfiguration()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "Endpoints":
            return Optional.ofNullable(clazz.cast(endpoints()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "StorageVirtualMachineId":
            return Optional.ofNullable(clazz.cast(storageVirtualMachineId()));
        case "Subtype":
            return Optional.ofNullable(clazz.cast(subtypeAsString()));
        case "UUID":
            return Optional.ofNullable(clazz.cast(uuid()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "LifecycleTransitionReason":
            return Optional.ofNullable(clazz.cast(lifecycleTransitionReason()));
        case "RootVolumeSecurityStyle":
            return Optional.ofNullable(clazz.cast(rootVolumeSecurityStyleAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StorageVirtualMachine, T> g) {
        return obj -> g.apply((StorageVirtualMachine) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StorageVirtualMachine> {
        /**
         * <p>
         * Describes the Microsoft Active Directory configuration to which the SVM is joined, if applicable.
         * </p>
         * 
         * @param activeDirectoryConfiguration
         *        Describes the Microsoft Active Directory configuration to which the SVM is joined, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activeDirectoryConfiguration(SvmActiveDirectoryConfiguration activeDirectoryConfiguration);

        /**
         * <p>
         * Describes the Microsoft Active Directory configuration to which the SVM is joined, if applicable.
         * </p>
         * This is a convenience method that creates an instance of the {@link SvmActiveDirectoryConfiguration.Builder}
         * avoiding the need to create one manually via {@link SvmActiveDirectoryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SvmActiveDirectoryConfiguration.Builder#build()} is called
         * immediately and its result is passed to
         * {@link #activeDirectoryConfiguration(SvmActiveDirectoryConfiguration)}.
         * 
         * @param activeDirectoryConfiguration
         *        a consumer that will call methods on {@link SvmActiveDirectoryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #activeDirectoryConfiguration(SvmActiveDirectoryConfiguration)
         */
        default Builder activeDirectoryConfiguration(
                Consumer<SvmActiveDirectoryConfiguration.Builder> activeDirectoryConfiguration) {
            return activeDirectoryConfiguration(SvmActiveDirectoryConfiguration.builder()
                    .applyMutation(activeDirectoryConfiguration).build());
        }

        /**
         * Sets the value of the CreationTime property for this object.
         *
         * @param creationTime
         *        The new value for the CreationTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The endpoints that are used to access data or to manage the SVM using the NetApp ONTAP CLI, REST API, or
         * NetApp CloudManager. They are the <code>Iscsi</code>, <code>Management</code>, <code>Nfs</code>, and
         * <code>Smb</code> endpoints.
         * </p>
         * 
         * @param endpoints
         *        The endpoints that are used to access data or to manage the SVM using the NetApp ONTAP CLI, REST API,
         *        or NetApp CloudManager. They are the <code>Iscsi</code>, <code>Management</code>, <code>Nfs</code>,
         *        and <code>Smb</code> endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoints(SvmEndpoints endpoints);

        /**
         * <p>
         * The endpoints that are used to access data or to manage the SVM using the NetApp ONTAP CLI, REST API, or
         * NetApp CloudManager. They are the <code>Iscsi</code>, <code>Management</code>, <code>Nfs</code>, and
         * <code>Smb</code> endpoints.
         * </p>
         * This is a convenience method that creates an instance of the {@link SvmEndpoints.Builder} avoiding the need
         * to create one manually via {@link SvmEndpoints#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link SvmEndpoints.Builder#build()} is called immediately and its
         * result is passed to {@link #endpoints(SvmEndpoints)}.
         * 
         * @param endpoints
         *        a consumer that will call methods on {@link SvmEndpoints.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #endpoints(SvmEndpoints)
         */
        default Builder endpoints(Consumer<SvmEndpoints.Builder> endpoints) {
            return endpoints(SvmEndpoints.builder().applyMutation(endpoints).build());
        }

        /**
         * Sets the value of the FileSystemId property for this object.
         *
         * @param fileSystemId
         *        The new value for the FileSystemId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * Describes the SVM's lifecycle status.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATED</code> - The SVM is fully available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the new SVM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Amazon FSx was unable to create the SVM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PENDING</code> - Amazon FSx has not started creating the SVM.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the SVM's lifecycle status.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATED</code> - The SVM is fully available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the new SVM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Amazon FSx was unable to create the SVM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - Amazon FSx has not started creating the SVM.
         *        </p>
         *        </li>
         * @see StorageVirtualMachineLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageVirtualMachineLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * Describes the SVM's lifecycle status.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATED</code> - The SVM is fully available for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the new SVM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Amazon FSx was unable to create the SVM.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PENDING</code> - Amazon FSx has not started creating the SVM.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        Describes the SVM's lifecycle status.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATED</code> - The SVM is fully available for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the new SVM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - Amazon FSx is deleting an existing SVM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Amazon FSx was unable to create the SVM.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MISCONFIGURED</code> - The SVM is in a failed but recoverable state.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - Amazon FSx has not started creating the SVM.
         *        </p>
         *        </li>
         * @see StorageVirtualMachineLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageVirtualMachineLifecycle
         */
        Builder lifecycle(StorageVirtualMachineLifecycle lifecycle);

        /**
         * <p>
         * The name of the SVM, if provisioned.
         * </p>
         * 
         * @param name
         *        The name of the SVM, if provisioned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * Sets the value of the ResourceARN property for this object.
         *
         * @param resourceARN
         *        The new value for the ResourceARN property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * The SVM's system generated unique ID.
         * </p>
         * 
         * @param storageVirtualMachineId
         *        The SVM's system generated unique ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageVirtualMachineId(String storageVirtualMachineId);

        /**
         * <p>
         * Describes the SVM's subtype.
         * </p>
         * 
         * @param subtype
         *        Describes the SVM's subtype.
         * @see StorageVirtualMachineSubtype
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageVirtualMachineSubtype
         */
        Builder subtype(String subtype);

        /**
         * <p>
         * Describes the SVM's subtype.
         * </p>
         * 
         * @param subtype
         *        Describes the SVM's subtype.
         * @see StorageVirtualMachineSubtype
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageVirtualMachineSubtype
         */
        Builder subtype(StorageVirtualMachineSubtype subtype);

        /**
         * <p>
         * The SVM's UUID (universally unique identifier).
         * </p>
         * 
         * @param uuid
         *        The SVM's UUID (universally unique identifier).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uuid(String uuid);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.fsx.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.fsx.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.fsx.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Describes why the SVM lifecycle state changed.
         * </p>
         * 
         * @param lifecycleTransitionReason
         *        Describes why the SVM lifecycle state changed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleTransitionReason(LifecycleTransitionReason lifecycleTransitionReason);

        /**
         * <p>
         * Describes why the SVM lifecycle state changed.
         * </p>
         * This is a convenience method that creates an instance of the {@link LifecycleTransitionReason.Builder}
         * avoiding the need to create one manually via {@link LifecycleTransitionReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LifecycleTransitionReason.Builder#build()} is called immediately
         * and its result is passed to {@link #lifecycleTransitionReason(LifecycleTransitionReason)}.
         * 
         * @param lifecycleTransitionReason
         *        a consumer that will call methods on {@link LifecycleTransitionReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycleTransitionReason(LifecycleTransitionReason)
         */
        default Builder lifecycleTransitionReason(Consumer<LifecycleTransitionReason.Builder> lifecycleTransitionReason) {
            return lifecycleTransitionReason(LifecycleTransitionReason.builder().applyMutation(lifecycleTransitionReason).build());
        }

        /**
         * <p>
         * The security style of the root volume of the SVM.
         * </p>
         * 
         * @param rootVolumeSecurityStyle
         *        The security style of the root volume of the SVM.
         * @see StorageVirtualMachineRootVolumeSecurityStyle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageVirtualMachineRootVolumeSecurityStyle
         */
        Builder rootVolumeSecurityStyle(String rootVolumeSecurityStyle);

        /**
         * <p>
         * The security style of the root volume of the SVM.
         * </p>
         * 
         * @param rootVolumeSecurityStyle
         *        The security style of the root volume of the SVM.
         * @see StorageVirtualMachineRootVolumeSecurityStyle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageVirtualMachineRootVolumeSecurityStyle
         */
        Builder rootVolumeSecurityStyle(StorageVirtualMachineRootVolumeSecurityStyle rootVolumeSecurityStyle);
    }

    static final class BuilderImpl implements Builder {
        private SvmActiveDirectoryConfiguration activeDirectoryConfiguration;

        private Instant creationTime;

        private SvmEndpoints endpoints;

        private String fileSystemId;

        private String lifecycle;

        private String name;

        private String resourceARN;

        private String storageVirtualMachineId;

        private String subtype;

        private String uuid;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private LifecycleTransitionReason lifecycleTransitionReason;

        private String rootVolumeSecurityStyle;

        private BuilderImpl() {
        }

        private BuilderImpl(StorageVirtualMachine model) {
            activeDirectoryConfiguration(model.activeDirectoryConfiguration);
            creationTime(model.creationTime);
            endpoints(model.endpoints);
            fileSystemId(model.fileSystemId);
            lifecycle(model.lifecycle);
            name(model.name);
            resourceARN(model.resourceARN);
            storageVirtualMachineId(model.storageVirtualMachineId);
            subtype(model.subtype);
            uuid(model.uuid);
            tags(model.tags);
            lifecycleTransitionReason(model.lifecycleTransitionReason);
            rootVolumeSecurityStyle(model.rootVolumeSecurityStyle);
        }

        public final SvmActiveDirectoryConfiguration.Builder getActiveDirectoryConfiguration() {
            return activeDirectoryConfiguration != null ? activeDirectoryConfiguration.toBuilder() : null;
        }

        public final void setActiveDirectoryConfiguration(SvmActiveDirectoryConfiguration.BuilderImpl activeDirectoryConfiguration) {
            this.activeDirectoryConfiguration = activeDirectoryConfiguration != null ? activeDirectoryConfiguration.build()
                    : null;
        }

        @Override
        public final Builder activeDirectoryConfiguration(SvmActiveDirectoryConfiguration activeDirectoryConfiguration) {
            this.activeDirectoryConfiguration = activeDirectoryConfiguration;
            return this;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final SvmEndpoints.Builder getEndpoints() {
            return endpoints != null ? endpoints.toBuilder() : null;
        }

        public final void setEndpoints(SvmEndpoints.BuilderImpl endpoints) {
            this.endpoints = endpoints != null ? endpoints.build() : null;
        }

        @Override
        public final Builder endpoints(SvmEndpoints endpoints) {
            this.endpoints = endpoints;
            return this;
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(StorageVirtualMachineLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final String getStorageVirtualMachineId() {
            return storageVirtualMachineId;
        }

        public final void setStorageVirtualMachineId(String storageVirtualMachineId) {
            this.storageVirtualMachineId = storageVirtualMachineId;
        }

        @Override
        public final Builder storageVirtualMachineId(String storageVirtualMachineId) {
            this.storageVirtualMachineId = storageVirtualMachineId;
            return this;
        }

        public final String getSubtype() {
            return subtype;
        }

        public final void setSubtype(String subtype) {
            this.subtype = subtype;
        }

        @Override
        public final Builder subtype(String subtype) {
            this.subtype = subtype;
            return this;
        }

        @Override
        public final Builder subtype(StorageVirtualMachineSubtype subtype) {
            this.subtype(subtype == null ? null : subtype.toString());
            return this;
        }

        public final String getUuid() {
            return uuid;
        }

        public final void setUuid(String uuid) {
            this.uuid = uuid;
        }

        @Override
        public final Builder uuid(String uuid) {
            this.uuid = uuid;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagsCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final LifecycleTransitionReason.Builder getLifecycleTransitionReason() {
            return lifecycleTransitionReason != null ? lifecycleTransitionReason.toBuilder() : null;
        }

        public final void setLifecycleTransitionReason(LifecycleTransitionReason.BuilderImpl lifecycleTransitionReason) {
            this.lifecycleTransitionReason = lifecycleTransitionReason != null ? lifecycleTransitionReason.build() : null;
        }

        @Override
        public final Builder lifecycleTransitionReason(LifecycleTransitionReason lifecycleTransitionReason) {
            this.lifecycleTransitionReason = lifecycleTransitionReason;
            return this;
        }

        public final String getRootVolumeSecurityStyle() {
            return rootVolumeSecurityStyle;
        }

        public final void setRootVolumeSecurityStyle(String rootVolumeSecurityStyle) {
            this.rootVolumeSecurityStyle = rootVolumeSecurityStyle;
        }

        @Override
        public final Builder rootVolumeSecurityStyle(String rootVolumeSecurityStyle) {
            this.rootVolumeSecurityStyle = rootVolumeSecurityStyle;
            return this;
        }

        @Override
        public final Builder rootVolumeSecurityStyle(StorageVirtualMachineRootVolumeSecurityStyle rootVolumeSecurityStyle) {
            this.rootVolumeSecurityStyle(rootVolumeSecurityStyle == null ? null : rootVolumeSecurityStyle.toString());
            return this;
        }

        @Override
        public StorageVirtualMachine build() {
            return new StorageVirtualMachine(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
