/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A description of the data repository task. You use data repository tasks to perform bulk transfer operations between
 * your Amazon FSx file system and its linked data repository.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataRepositoryTask implements SdkPojo, Serializable,
        ToCopyableBuilder<DataRepositoryTask.Builder, DataRepositoryTask> {
    private static final SdkField<String> TASK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("TaskId")
            .getter(getter(DataRepositoryTask::taskId)).setter(setter(Builder::taskId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TaskId").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(DataRepositoryTask::lifecycleAsString)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(DataRepositoryTask::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(DataRepositoryTask::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(DataRepositoryTask::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(DataRepositoryTask::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(DataRepositoryTask::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(DataRepositoryTask::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> FILE_SYSTEM_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("FileSystemId").getter(getter(DataRepositoryTask::fileSystemId)).setter(setter(Builder::fileSystemId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemId").build()).build();

    private static final SdkField<List<String>> PATHS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Paths")
            .getter(getter(DataRepositoryTask::paths))
            .setter(setter(Builder::paths))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Paths").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<DataRepositoryTaskFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<DataRepositoryTaskFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(DataRepositoryTask::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(DataRepositoryTaskFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<DataRepositoryTaskStatus> STATUS_FIELD = SdkField
            .<DataRepositoryTaskStatus> builder(MarshallingType.SDK_POJO).memberName("Status")
            .getter(getter(DataRepositoryTask::status)).setter(setter(Builder::status))
            .constructor(DataRepositoryTaskStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<CompletionReport> REPORT_FIELD = SdkField.<CompletionReport> builder(MarshallingType.SDK_POJO)
            .memberName("Report").getter(getter(DataRepositoryTask::report)).setter(setter(Builder::report))
            .constructor(CompletionReport::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Report").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TASK_ID_FIELD,
            LIFECYCLE_FIELD, TYPE_FIELD, CREATION_TIME_FIELD, START_TIME_FIELD, END_TIME_FIELD, RESOURCE_ARN_FIELD, TAGS_FIELD,
            FILE_SYSTEM_ID_FIELD, PATHS_FIELD, FAILURE_DETAILS_FIELD, STATUS_FIELD, REPORT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String taskId;

    private final String lifecycle;

    private final String type;

    private final Instant creationTime;

    private final Instant startTime;

    private final Instant endTime;

    private final String resourceARN;

    private final List<Tag> tags;

    private final String fileSystemId;

    private final List<String> paths;

    private final DataRepositoryTaskFailureDetails failureDetails;

    private final DataRepositoryTaskStatus status;

    private final CompletionReport report;

    private DataRepositoryTask(BuilderImpl builder) {
        this.taskId = builder.taskId;
        this.lifecycle = builder.lifecycle;
        this.type = builder.type;
        this.creationTime = builder.creationTime;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.resourceARN = builder.resourceARN;
        this.tags = builder.tags;
        this.fileSystemId = builder.fileSystemId;
        this.paths = builder.paths;
        this.failureDetails = builder.failureDetails;
        this.status = builder.status;
        this.report = builder.report;
    }

    /**
     * <p>
     * The system-generated, unique 17-digit ID of the data repository task.
     * </p>
     * 
     * @return The system-generated, unique 17-digit ID of the data repository task.
     */
    public final String taskId() {
        return taskId;
    }

    /**
     * <p>
     * The lifecycle status of the data repository task, as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> - Amazon FSx has not started the task.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXECUTING</code> - Amazon FSx is processing the task.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files the task
     * failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more information about task
     * failures.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code> - FSx completed the task successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELING</code> - FSx is in process of canceling the task.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system in the
     * <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository task is finished
     * (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>). You can use the
     * DescribeDataRepositoryTask action to monitor the task status. Contact the FSx team if you need to delete your
     * file system immediately.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryTaskLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the data repository task, as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - Amazon FSx has not started the task.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EXECUTING</code> - Amazon FSx is processing the task.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files the
     *         task failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more information
     *         about task failures.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code> - FSx completed the task successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELING</code> - FSx is in process of canceling the task.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system in
     *         the <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository task is
     *         finished (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>). You
     *         can use the DescribeDataRepositoryTask action to monitor the task status. Contact the FSx team if you
     *         need to delete your file system immediately.
     *         </p>
     * @see DataRepositoryTaskLifecycle
     */
    public final DataRepositoryTaskLifecycle lifecycle() {
        return DataRepositoryTaskLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * The lifecycle status of the data repository task, as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PENDING</code> - Amazon FSx has not started the task.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXECUTING</code> - Amazon FSx is processing the task.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files the task
     * failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more information about task
     * failures.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SUCCEEDED</code> - FSx completed the task successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANCELING</code> - FSx is in process of canceling the task.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system in the
     * <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository task is finished
     * (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>). You can use the
     * DescribeDataRepositoryTask action to monitor the task status. Contact the FSx team if you need to delete your
     * file system immediately.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link DataRepositoryTaskLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the data repository task, as follows:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - Amazon FSx has not started the task.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EXECUTING</code> - Amazon FSx is processing the task.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files the
     *         task failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more information
     *         about task failures.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SUCCEEDED</code> - FSx completed the task successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANCELING</code> - FSx is in process of canceling the task.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system in
     *         the <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository task is
     *         finished (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>). You
     *         can use the DescribeDataRepositoryTask action to monitor the task status. Contact the FSx team if you
     *         need to delete your file system immediately.
     *         </p>
     * @see DataRepositoryTaskLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * <p>
     * The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataRepositoryTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
     * @see DataRepositoryTaskType
     */
    public final DataRepositoryTaskType type() {
        return DataRepositoryTaskType.fromValue(type);
    }

    /**
     * <p>
     * The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link DataRepositoryTaskType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
     * @see DataRepositoryTaskType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * Returns the value of the CreationTime property for this object.
     * 
     * @return The value of the CreationTime property for this object.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The time that Amazon FSx began processing the task.
     * </p>
     * 
     * @return The time that Amazon FSx began processing the task.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time that Amazon FSx completed processing the task, populated after the task is complete.
     * </p>
     * 
     * @return The time that Amazon FSx completed processing the task, populated after the task is complete.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * Returns the value of the ResourceARN property for this object.
     * 
     * @return The value of the ResourceARN property for this object.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * Returns the value of the FileSystemId property for this object.
     * 
     * @return The value of the FileSystemId property for this object.
     */
    public final String fileSystemId() {
        return fileSystemId;
    }

    /**
     * Returns true if the Paths property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasPaths() {
        return paths != null && !(paths instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of paths on the Amazon FSx for Lustre file system that specify the data for the data repository task to
     * process. For example, in an EXPORT_TO_REPOSITORY task, the paths specify which data to export to the linked data
     * repository.
     * </p>
     * <p>
     * (Default) If <code>Paths</code> is not specified, Amazon FSx uses the file system root directory.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPaths()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of paths on the Amazon FSx for Lustre file system that specify the data for the data repository
     *         task to process. For example, in an EXPORT_TO_REPOSITORY task, the paths specify which data to export to
     *         the linked data repository.</p>
     *         <p>
     *         (Default) If <code>Paths</code> is not specified, Amazon FSx uses the file system root directory.
     */
    public final List<String> paths() {
        return paths;
    }

    /**
     * <p>
     * Failure message describing why the task failed, it is populated only when <code>Lifecycle</code> is set to
     * <code>FAILED</code>.
     * </p>
     * 
     * @return Failure message describing why the task failed, it is populated only when <code>Lifecycle</code> is set
     *         to <code>FAILED</code>.
     */
    public final DataRepositoryTaskFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * Provides the status of the number of files that the task has processed successfully and failed to process.
     * </p>
     * 
     * @return Provides the status of the number of files that the task has processed successfully and failed to
     *         process.
     */
    public final DataRepositoryTaskStatus status() {
        return status;
    }

    /**
     * Returns the value of the Report property for this object.
     * 
     * @return The value of the Report property for this object.
     */
    public final CompletionReport report() {
        return report;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(taskId());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemId());
        hashCode = 31 * hashCode + Objects.hashCode(hasPaths() ? paths() : null);
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(report());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataRepositoryTask)) {
            return false;
        }
        DataRepositoryTask other = (DataRepositoryTask) obj;
        return Objects.equals(taskId(), other.taskId()) && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(resourceARN(), other.resourceARN()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(fileSystemId(), other.fileSystemId())
                && hasPaths() == other.hasPaths() && Objects.equals(paths(), other.paths())
                && Objects.equals(failureDetails(), other.failureDetails()) && Objects.equals(status(), other.status())
                && Objects.equals(report(), other.report());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataRepositoryTask").add("TaskId", taskId()).add("Lifecycle", lifecycleAsString())
                .add("Type", typeAsString()).add("CreationTime", creationTime()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("ResourceARN", resourceARN()).add("Tags", hasTags() ? tags() : null)
                .add("FileSystemId", fileSystemId()).add("Paths", hasPaths() ? paths() : null)
                .add("FailureDetails", failureDetails()).add("Status", status()).add("Report", report()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TaskId":
            return Optional.ofNullable(clazz.cast(taskId()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "FileSystemId":
            return Optional.ofNullable(clazz.cast(fileSystemId()));
        case "Paths":
            return Optional.ofNullable(clazz.cast(paths()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Report":
            return Optional.ofNullable(clazz.cast(report()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DataRepositoryTask, T> g) {
        return obj -> g.apply((DataRepositoryTask) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataRepositoryTask> {
        /**
         * <p>
         * The system-generated, unique 17-digit ID of the data repository task.
         * </p>
         * 
         * @param taskId
         *        The system-generated, unique 17-digit ID of the data repository task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder taskId(String taskId);

        /**
         * <p>
         * The lifecycle status of the data repository task, as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> - Amazon FSx has not started the task.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EXECUTING</code> - Amazon FSx is processing the task.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files the task
         * failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more information about task
         * failures.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code> - FSx completed the task successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELING</code> - FSx is in process of canceling the task.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system in the
         * <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository task is finished
         * (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>). You can use the
         * DescribeDataRepositoryTask action to monitor the task status. Contact the FSx team if you need to delete your
         * file system immediately.
         * </p>
         * </note>
         * 
         * @param lifecycle
         *        The lifecycle status of the data repository task, as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - Amazon FSx has not started the task.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EXECUTING</code> - Amazon FSx is processing the task.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files
         *        the task failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more
         *        information about task failures.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code> - FSx completed the task successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELING</code> - FSx is in process of canceling the task.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system
         *        in the <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository
         *        task is finished (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or
         *        <code>FAILED</code>). You can use the DescribeDataRepositoryTask action to monitor the task status.
         *        Contact the FSx team if you need to delete your file system immediately.
         *        </p>
         * @see DataRepositoryTaskLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryTaskLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * The lifecycle status of the data repository task, as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PENDING</code> - Amazon FSx has not started the task.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EXECUTING</code> - Amazon FSx is processing the task.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files the task
         * failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more information about task
         * failures.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SUCCEEDED</code> - FSx completed the task successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANCELING</code> - FSx is in process of canceling the task.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system in the
         * <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository task is finished
         * (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or <code>FAILED</code>). You can use the
         * DescribeDataRepositoryTask action to monitor the task status. Contact the FSx team if you need to delete your
         * file system immediately.
         * </p>
         * </note>
         * 
         * @param lifecycle
         *        The lifecycle status of the data repository task, as follows:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - Amazon FSx has not started the task.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EXECUTING</code> - Amazon FSx is processing the task.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Amazon FSx was not able to complete the task. For example, there may be files
         *        the task failed to process. The <a>DataRepositoryTaskFailureDetails</a> property provides more
         *        information about task failures.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SUCCEEDED</code> - FSx completed the task successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELED</code> - Amazon FSx canceled the task and it did not complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANCELING</code> - FSx is in process of canceling the task.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You cannot delete an FSx for Lustre file system if there are data repository tasks for the file system
         *        in the <code>PENDING</code> or <code>EXECUTING</code> states. Please retry when the data repository
         *        task is finished (with a status of <code>CANCELED</code>, <code>SUCCEEDED</code>, or
         *        <code>FAILED</code>). You can use the DescribeDataRepositoryTask action to monitor the task status.
         *        Contact the FSx team if you need to delete your file system immediately.
         *        </p>
         * @see DataRepositoryTaskLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryTaskLifecycle
         */
        Builder lifecycle(DataRepositoryTaskLifecycle lifecycle);

        /**
         * <p>
         * The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
         * </p>
         * 
         * @param type
         *        The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
         * @see DataRepositoryTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryTaskType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
         * </p>
         * 
         * @param type
         *        The type of data repository task; EXPORT_TO_REPOSITORY is the only type currently supported.
         * @see DataRepositoryTaskType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataRepositoryTaskType
         */
        Builder type(DataRepositoryTaskType type);

        /**
         * Sets the value of the CreationTime property for this object.
         *
         * @param creationTime
         *        The new value for the CreationTime property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The time that Amazon FSx began processing the task.
         * </p>
         * 
         * @param startTime
         *        The time that Amazon FSx began processing the task.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time that Amazon FSx completed processing the task, populated after the task is complete.
         * </p>
         * 
         * @param endTime
         *        The time that Amazon FSx completed processing the task, populated after the task is complete.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * Sets the value of the ResourceARN property for this object.
         *
         * @param resourceARN
         *        The new value for the ResourceARN property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * Sets the value of the Tags property for this object.
         *
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * Sets the value of the FileSystemId property for this object.
         *
         * @param fileSystemId
         *        The new value for the FileSystemId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystemId(String fileSystemId);

        /**
         * <p>
         * An array of paths on the Amazon FSx for Lustre file system that specify the data for the data repository task
         * to process. For example, in an EXPORT_TO_REPOSITORY task, the paths specify which data to export to the
         * linked data repository.
         * </p>
         * <p>
         * (Default) If <code>Paths</code> is not specified, Amazon FSx uses the file system root directory.
         * </p>
         * 
         * @param paths
         *        An array of paths on the Amazon FSx for Lustre file system that specify the data for the data
         *        repository task to process. For example, in an EXPORT_TO_REPOSITORY task, the paths specify which data
         *        to export to the linked data repository.</p>
         *        <p>
         *        (Default) If <code>Paths</code> is not specified, Amazon FSx uses the file system root directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paths(Collection<String> paths);

        /**
         * <p>
         * An array of paths on the Amazon FSx for Lustre file system that specify the data for the data repository task
         * to process. For example, in an EXPORT_TO_REPOSITORY task, the paths specify which data to export to the
         * linked data repository.
         * </p>
         * <p>
         * (Default) If <code>Paths</code> is not specified, Amazon FSx uses the file system root directory.
         * </p>
         * 
         * @param paths
         *        An array of paths on the Amazon FSx for Lustre file system that specify the data for the data
         *        repository task to process. For example, in an EXPORT_TO_REPOSITORY task, the paths specify which data
         *        to export to the linked data repository.</p>
         *        <p>
         *        (Default) If <code>Paths</code> is not specified, Amazon FSx uses the file system root directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder paths(String... paths);

        /**
         * <p>
         * Failure message describing why the task failed, it is populated only when <code>Lifecycle</code> is set to
         * <code>FAILED</code>.
         * </p>
         * 
         * @param failureDetails
         *        Failure message describing why the task failed, it is populated only when <code>Lifecycle</code> is
         *        set to <code>FAILED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(DataRepositoryTaskFailureDetails failureDetails);

        /**
         * <p>
         * Failure message describing why the task failed, it is populated only when <code>Lifecycle</code> is set to
         * <code>FAILED</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link DataRepositoryTaskFailureDetails.Builder}
         * avoiding the need to create one manually via {@link DataRepositoryTaskFailureDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataRepositoryTaskFailureDetails.Builder#build()} is called
         * immediately and its result is passed to {@link #failureDetails(DataRepositoryTaskFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link DataRepositoryTaskFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(DataRepositoryTaskFailureDetails)
         */
        default Builder failureDetails(Consumer<DataRepositoryTaskFailureDetails.Builder> failureDetails) {
            return failureDetails(DataRepositoryTaskFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * Provides the status of the number of files that the task has processed successfully and failed to process.
         * </p>
         * 
         * @param status
         *        Provides the status of the number of files that the task has processed successfully and failed to
         *        process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(DataRepositoryTaskStatus status);

        /**
         * <p>
         * Provides the status of the number of files that the task has processed successfully and failed to process.
         * </p>
         * This is a convenience that creates an instance of the {@link DataRepositoryTaskStatus.Builder} avoiding the
         * need to create one manually via {@link DataRepositoryTaskStatus#builder()}.
         *
         * When the {@link Consumer} completes, {@link DataRepositoryTaskStatus.Builder#build()} is called immediately
         * and its result is passed to {@link #status(DataRepositoryTaskStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link DataRepositoryTaskStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(DataRepositoryTaskStatus)
         */
        default Builder status(Consumer<DataRepositoryTaskStatus.Builder> status) {
            return status(DataRepositoryTaskStatus.builder().applyMutation(status).build());
        }

        /**
         * Sets the value of the Report property for this object.
         *
         * @param report
         *        The new value for the Report property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder report(CompletionReport report);

        /**
         * Sets the value of the Report property for this object.
         *
         * This is a convenience that creates an instance of the {@link CompletionReport.Builder} avoiding the need to
         * create one manually via {@link CompletionReport#builder()}.
         *
         * When the {@link Consumer} completes, {@link CompletionReport.Builder#build()} is called immediately and its
         * result is passed to {@link #report(CompletionReport)}.
         * 
         * @param report
         *        a consumer that will call methods on {@link CompletionReport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #report(CompletionReport)
         */
        default Builder report(Consumer<CompletionReport.Builder> report) {
            return report(CompletionReport.builder().applyMutation(report).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String taskId;

        private String lifecycle;

        private String type;

        private Instant creationTime;

        private Instant startTime;

        private Instant endTime;

        private String resourceARN;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String fileSystemId;

        private List<String> paths = DefaultSdkAutoConstructList.getInstance();

        private DataRepositoryTaskFailureDetails failureDetails;

        private DataRepositoryTaskStatus status;

        private CompletionReport report;

        private BuilderImpl() {
        }

        private BuilderImpl(DataRepositoryTask model) {
            taskId(model.taskId);
            lifecycle(model.lifecycle);
            type(model.type);
            creationTime(model.creationTime);
            startTime(model.startTime);
            endTime(model.endTime);
            resourceARN(model.resourceARN);
            tags(model.tags);
            fileSystemId(model.fileSystemId);
            paths(model.paths);
            failureDetails(model.failureDetails);
            status(model.status);
            report(model.report);
        }

        public final String getTaskId() {
            return taskId;
        }

        @Override
        public final Builder taskId(String taskId) {
            this.taskId = taskId;
            return this;
        }

        public final void setTaskId(String taskId) {
            this.taskId = taskId;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(DataRepositoryTaskLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(DataRepositoryTaskType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final String getFileSystemId() {
            return fileSystemId;
        }

        @Override
        public final Builder fileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
            return this;
        }

        public final void setFileSystemId(String fileSystemId) {
            this.fileSystemId = fileSystemId;
        }

        public final Collection<String> getPaths() {
            if (paths instanceof SdkAutoConstructList) {
                return null;
            }
            return paths;
        }

        @Override
        public final Builder paths(Collection<String> paths) {
            this.paths = DataRepositoryTaskPathsCopier.copy(paths);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder paths(String... paths) {
            paths(Arrays.asList(paths));
            return this;
        }

        public final void setPaths(Collection<String> paths) {
            this.paths = DataRepositoryTaskPathsCopier.copy(paths);
        }

        public final DataRepositoryTaskFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        @Override
        public final Builder failureDetails(DataRepositoryTaskFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final void setFailureDetails(DataRepositoryTaskFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        public final DataRepositoryTaskStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        @Override
        public final Builder status(DataRepositoryTaskStatus status) {
            this.status = status;
            return this;
        }

        public final void setStatus(DataRepositoryTaskStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        public final CompletionReport.Builder getReport() {
            return report != null ? report.toBuilder() : null;
        }

        @Override
        public final Builder report(CompletionReport report) {
            this.report = report;
            return this;
        }

        public final void setReport(CompletionReport.BuilderImpl report) {
            this.report = report != null ? report.build() : null;
        }

        @Override
        public DataRepositoryTask build() {
            return new DataRepositoryTask(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
