/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A backup of an Amazon FSx file system. For more information see:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html">Working with backups for Windows
 * file systems</a>
 * </p>
 * </li>
 * <li>
 * <p>
 * <a href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html">Working with backups for Lustre
 * file systems</a>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Backup implements SdkPojo, Serializable, ToCopyableBuilder<Backup.Builder, Backup> {
    private static final SdkField<String> BACKUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupId").getter(getter(Backup::backupId)).setter(setter(Builder::backupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupId").build()).build();

    private static final SdkField<String> LIFECYCLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Lifecycle").getter(getter(Backup::lifecycleAsString)).setter(setter(Builder::lifecycle))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Lifecycle").build()).build();

    private static final SdkField<BackupFailureDetails> FAILURE_DETAILS_FIELD = SdkField
            .<BackupFailureDetails> builder(MarshallingType.SDK_POJO).memberName("FailureDetails")
            .getter(getter(Backup::failureDetails)).setter(setter(Builder::failureDetails))
            .constructor(BackupFailureDetails::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureDetails").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(Backup::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<Integer> PROGRESS_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ProgressPercent").getter(getter(Backup::progressPercent)).setter(setter(Builder::progressPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProgressPercent").build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationTime").getter(getter(Backup::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId").getter(getter(Backup::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceARN").getter(getter(Backup::resourceARN)).setter(setter(Builder::resourceARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceARN").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Backup::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<FileSystem> FILE_SYSTEM_FIELD = SdkField.<FileSystem> builder(MarshallingType.SDK_POJO)
            .memberName("FileSystem").getter(getter(Backup::fileSystem)).setter(setter(Builder::fileSystem))
            .constructor(FileSystem::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystem").build()).build();

    private static final SdkField<ActiveDirectoryBackupAttributes> DIRECTORY_INFORMATION_FIELD = SdkField
            .<ActiveDirectoryBackupAttributes> builder(MarshallingType.SDK_POJO).memberName("DirectoryInformation")
            .getter(getter(Backup::directoryInformation)).setter(setter(Builder::directoryInformation))
            .constructor(ActiveDirectoryBackupAttributes::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DirectoryInformation").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BACKUP_ID_FIELD,
            LIFECYCLE_FIELD, FAILURE_DETAILS_FIELD, TYPE_FIELD, PROGRESS_PERCENT_FIELD, CREATION_TIME_FIELD, KMS_KEY_ID_FIELD,
            RESOURCE_ARN_FIELD, TAGS_FIELD, FILE_SYSTEM_FIELD, DIRECTORY_INFORMATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String backupId;

    private final String lifecycle;

    private final BackupFailureDetails failureDetails;

    private final String type;

    private final Integer progressPercent;

    private final Instant creationTime;

    private final String kmsKeyId;

    private final String resourceARN;

    private final List<Tag> tags;

    private final FileSystem fileSystem;

    private final ActiveDirectoryBackupAttributes directoryInformation;

    private Backup(BuilderImpl builder) {
        this.backupId = builder.backupId;
        this.lifecycle = builder.lifecycle;
        this.failureDetails = builder.failureDetails;
        this.type = builder.type;
        this.progressPercent = builder.progressPercent;
        this.creationTime = builder.creationTime;
        this.kmsKeyId = builder.kmsKeyId;
        this.resourceARN = builder.resourceARN;
        this.tags = builder.tags;
        this.fileSystem = builder.fileSystem;
        this.directoryInformation = builder.directoryInformation;
    }

    /**
     * <p>
     * The ID of the backup.
     * </p>
     * 
     * @return The ID of the backup.
     */
    public final String backupId() {
        return backupId;
    }

    /**
     * <p>
     * The lifecycle status of the backup.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The backup is fully available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not started
     * creating the backup.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the backup.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is transferring
     * the backup to S3.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Amazon FSx could not complete the backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link BackupLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the backup.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The backup is fully available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not started
     *         creating the backup.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the backup.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is
     *         transferring the backup to S3.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Amazon FSx could not complete the backup.
     *         </p>
     *         </li>
     * @see BackupLifecycle
     */
    public final BackupLifecycle lifecycle() {
        return BackupLifecycle.fromValue(lifecycle);
    }

    /**
     * <p>
     * The lifecycle status of the backup.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AVAILABLE</code> - The backup is fully available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not started
     * creating the backup.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CREATING</code> - Amazon FSx is creating the backup.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is transferring
     * the backup to S3.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Amazon FSx could not complete the backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #lifecycle} will
     * return {@link BackupLifecycle#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #lifecycleAsString}.
     * </p>
     * 
     * @return The lifecycle status of the backup.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>AVAILABLE</code> - The backup is fully available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not started
     *         creating the backup.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - Amazon FSx is creating the backup.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is
     *         transferring the backup to S3.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FAILED</code> - Amazon FSx could not complete the backup.
     *         </p>
     *         </li>
     * @see BackupLifecycle
     */
    public final String lifecycleAsString() {
        return lifecycle;
    }

    /**
     * <p>
     * Details explaining any failures that occur when creating a backup.
     * </p>
     * 
     * @return Details explaining any failures that occur when creating a backup.
     */
    public final BackupFailureDetails failureDetails() {
        return failureDetails;
    }

    /**
     * <p>
     * The type of the file system backup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link BackupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the file system backup.
     * @see BackupType
     */
    public final BackupType type() {
        return BackupType.fromValue(type);
    }

    /**
     * <p>
     * The type of the file system backup.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link BackupType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of the file system backup.
     * @see BackupType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * Returns the value of the ProgressPercent property for this object.
     * 
     * @return The value of the ProgressPercent property for this object.
     */
    public final Integer progressPercent() {
        return progressPercent;
    }

    /**
     * <p>
     * The time when a particular backup was created.
     * </p>
     * 
     * @return The time when a particular backup was created.
     */
    public final Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the backup of the Amazon FSx file system's
     * data at rest.
     * </p>
     * 
     * @return The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the backup of the Amazon FSx file
     *         system's data at rest.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the backup resource.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the backup resource.
     */
    public final String resourceARN() {
        return resourceARN;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags associated with a particular file system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Tags associated with a particular file system.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Metadata of the file system associated with the backup. This metadata is persisted even if the file system is
     * deleted.
     * </p>
     * 
     * @return Metadata of the file system associated with the backup. This metadata is persisted even if the file
     *         system is deleted.
     */
    public final FileSystem fileSystem() {
        return fileSystem;
    }

    /**
     * <p>
     * The configuration of the self-managed Microsoft Active Directory (AD) to which the Windows File Server instance
     * is joined.
     * </p>
     * 
     * @return The configuration of the self-managed Microsoft Active Directory (AD) to which the Windows File Server
     *         instance is joined.
     */
    public final ActiveDirectoryBackupAttributes directoryInformation() {
        return directoryInformation;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(backupId());
        hashCode = 31 * hashCode + Objects.hashCode(lifecycleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureDetails());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(progressPercent());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceARN());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(fileSystem());
        hashCode = 31 * hashCode + Objects.hashCode(directoryInformation());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Backup)) {
            return false;
        }
        Backup other = (Backup) obj;
        return Objects.equals(backupId(), other.backupId()) && Objects.equals(lifecycleAsString(), other.lifecycleAsString())
                && Objects.equals(failureDetails(), other.failureDetails())
                && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(progressPercent(), other.progressPercent())
                && Objects.equals(creationTime(), other.creationTime()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(resourceARN(), other.resourceARN()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(fileSystem(), other.fileSystem())
                && Objects.equals(directoryInformation(), other.directoryInformation());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Backup").add("BackupId", backupId()).add("Lifecycle", lifecycleAsString())
                .add("FailureDetails", failureDetails()).add("Type", typeAsString()).add("ProgressPercent", progressPercent())
                .add("CreationTime", creationTime()).add("KmsKeyId", kmsKeyId()).add("ResourceARN", resourceARN())
                .add("Tags", hasTags() ? tags() : null).add("FileSystem", fileSystem())
                .add("DirectoryInformation", directoryInformation()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BackupId":
            return Optional.ofNullable(clazz.cast(backupId()));
        case "Lifecycle":
            return Optional.ofNullable(clazz.cast(lifecycleAsString()));
        case "FailureDetails":
            return Optional.ofNullable(clazz.cast(failureDetails()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ProgressPercent":
            return Optional.ofNullable(clazz.cast(progressPercent()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "ResourceARN":
            return Optional.ofNullable(clazz.cast(resourceARN()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "FileSystem":
            return Optional.ofNullable(clazz.cast(fileSystem()));
        case "DirectoryInformation":
            return Optional.ofNullable(clazz.cast(directoryInformation()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Backup, T> g) {
        return obj -> g.apply((Backup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Backup> {
        /**
         * <p>
         * The ID of the backup.
         * </p>
         * 
         * @param backupId
         *        The ID of the backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupId(String backupId);

        /**
         * <p>
         * The lifecycle status of the backup.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The backup is fully available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not started
         * creating the backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is
         * transferring the backup to S3.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Amazon FSx could not complete the backup.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the backup.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The backup is fully available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not
         *        started creating the backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is
         *        transferring the backup to S3.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Amazon FSx could not complete the backup.
         *        </p>
         *        </li>
         * @see BackupLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupLifecycle
         */
        Builder lifecycle(String lifecycle);

        /**
         * <p>
         * The lifecycle status of the backup.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>AVAILABLE</code> - The backup is fully available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not started
         * creating the backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CREATING</code> - Amazon FSx is creating the backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is
         * transferring the backup to S3.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FAILED</code> - Amazon FSx could not complete the backup.
         * </p>
         * </li>
         * </ul>
         * 
         * @param lifecycle
         *        The lifecycle status of the backup.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>AVAILABLE</code> - The backup is fully available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>PENDING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx has not
         *        started creating the backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - Amazon FSx is creating the backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TRANSFERRING</code> - For user-initiated backups on Lustre file systems only; Amazon FSx is
         *        transferring the backup to S3.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETED</code> - Amazon FSx deleted the backup and it is no longer available.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FAILED</code> - Amazon FSx could not complete the backup.
         *        </p>
         *        </li>
         * @see BackupLifecycle
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupLifecycle
         */
        Builder lifecycle(BackupLifecycle lifecycle);

        /**
         * <p>
         * Details explaining any failures that occur when creating a backup.
         * </p>
         * 
         * @param failureDetails
         *        Details explaining any failures that occur when creating a backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureDetails(BackupFailureDetails failureDetails);

        /**
         * <p>
         * Details explaining any failures that occur when creating a backup.
         * </p>
         * This is a convenience that creates an instance of the {@link BackupFailureDetails.Builder} avoiding the need
         * to create one manually via {@link BackupFailureDetails#builder()}.
         *
         * When the {@link Consumer} completes, {@link BackupFailureDetails.Builder#build()} is called immediately and
         * its result is passed to {@link #failureDetails(BackupFailureDetails)}.
         * 
         * @param failureDetails
         *        a consumer that will call methods on {@link BackupFailureDetails.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #failureDetails(BackupFailureDetails)
         */
        default Builder failureDetails(Consumer<BackupFailureDetails.Builder> failureDetails) {
            return failureDetails(BackupFailureDetails.builder().applyMutation(failureDetails).build());
        }

        /**
         * <p>
         * The type of the file system backup.
         * </p>
         * 
         * @param type
         *        The type of the file system backup.
         * @see BackupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of the file system backup.
         * </p>
         * 
         * @param type
         *        The type of the file system backup.
         * @see BackupType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupType
         */
        Builder type(BackupType type);

        /**
         * Sets the value of the ProgressPercent property for this object.
         *
         * @param progressPercent
         *        The new value for the ProgressPercent property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder progressPercent(Integer progressPercent);

        /**
         * <p>
         * The time when a particular backup was created.
         * </p>
         * 
         * @param creationTime
         *        The time when a particular backup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the backup of the Amazon FSx file
         * system's data at rest.
         * </p>
         * 
         * @param kmsKeyId
         *        The ID of the AWS Key Management Service (AWS KMS) key used to encrypt the backup of the Amazon FSx
         *        file system's data at rest.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the backup resource.
         * </p>
         * 
         * @param resourceARN
         *        The Amazon Resource Name (ARN) for the backup resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceARN(String resourceARN);

        /**
         * <p>
         * Tags associated with a particular file system.
         * </p>
         * 
         * @param tags
         *        Tags associated with a particular file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Tags associated with a particular file system.
         * </p>
         * 
         * @param tags
         *        Tags associated with a particular file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Tags associated with a particular file system.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Metadata of the file system associated with the backup. This metadata is persisted even if the file system is
         * deleted.
         * </p>
         * 
         * @param fileSystem
         *        Metadata of the file system associated with the backup. This metadata is persisted even if the file
         *        system is deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fileSystem(FileSystem fileSystem);

        /**
         * <p>
         * Metadata of the file system associated with the backup. This metadata is persisted even if the file system is
         * deleted.
         * </p>
         * This is a convenience that creates an instance of the {@link FileSystem.Builder} avoiding the need to create
         * one manually via {@link FileSystem#builder()}.
         *
         * When the {@link Consumer} completes, {@link FileSystem.Builder#build()} is called immediately and its result
         * is passed to {@link #fileSystem(FileSystem)}.
         * 
         * @param fileSystem
         *        a consumer that will call methods on {@link FileSystem.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #fileSystem(FileSystem)
         */
        default Builder fileSystem(Consumer<FileSystem.Builder> fileSystem) {
            return fileSystem(FileSystem.builder().applyMutation(fileSystem).build());
        }

        /**
         * <p>
         * The configuration of the self-managed Microsoft Active Directory (AD) to which the Windows File Server
         * instance is joined.
         * </p>
         * 
         * @param directoryInformation
         *        The configuration of the self-managed Microsoft Active Directory (AD) to which the Windows File Server
         *        instance is joined.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder directoryInformation(ActiveDirectoryBackupAttributes directoryInformation);

        /**
         * <p>
         * The configuration of the self-managed Microsoft Active Directory (AD) to which the Windows File Server
         * instance is joined.
         * </p>
         * This is a convenience that creates an instance of the {@link ActiveDirectoryBackupAttributes.Builder}
         * avoiding the need to create one manually via {@link ActiveDirectoryBackupAttributes#builder()}.
         *
         * When the {@link Consumer} completes, {@link ActiveDirectoryBackupAttributes.Builder#build()} is called
         * immediately and its result is passed to {@link #directoryInformation(ActiveDirectoryBackupAttributes)}.
         * 
         * @param directoryInformation
         *        a consumer that will call methods on {@link ActiveDirectoryBackupAttributes.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #directoryInformation(ActiveDirectoryBackupAttributes)
         */
        default Builder directoryInformation(Consumer<ActiveDirectoryBackupAttributes.Builder> directoryInformation) {
            return directoryInformation(ActiveDirectoryBackupAttributes.builder().applyMutation(directoryInformation).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String backupId;

        private String lifecycle;

        private BackupFailureDetails failureDetails;

        private String type;

        private Integer progressPercent;

        private Instant creationTime;

        private String kmsKeyId;

        private String resourceARN;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private FileSystem fileSystem;

        private ActiveDirectoryBackupAttributes directoryInformation;

        private BuilderImpl() {
        }

        private BuilderImpl(Backup model) {
            backupId(model.backupId);
            lifecycle(model.lifecycle);
            failureDetails(model.failureDetails);
            type(model.type);
            progressPercent(model.progressPercent);
            creationTime(model.creationTime);
            kmsKeyId(model.kmsKeyId);
            resourceARN(model.resourceARN);
            tags(model.tags);
            fileSystem(model.fileSystem);
            directoryInformation(model.directoryInformation);
        }

        public final String getBackupId() {
            return backupId;
        }

        @Override
        public final Builder backupId(String backupId) {
            this.backupId = backupId;
            return this;
        }

        public final void setBackupId(String backupId) {
            this.backupId = backupId;
        }

        public final String getLifecycle() {
            return lifecycle;
        }

        @Override
        public final Builder lifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
            return this;
        }

        @Override
        public final Builder lifecycle(BackupLifecycle lifecycle) {
            this.lifecycle(lifecycle == null ? null : lifecycle.toString());
            return this;
        }

        public final void setLifecycle(String lifecycle) {
            this.lifecycle = lifecycle;
        }

        public final BackupFailureDetails.Builder getFailureDetails() {
            return failureDetails != null ? failureDetails.toBuilder() : null;
        }

        @Override
        public final Builder failureDetails(BackupFailureDetails failureDetails) {
            this.failureDetails = failureDetails;
            return this;
        }

        public final void setFailureDetails(BackupFailureDetails.BuilderImpl failureDetails) {
            this.failureDetails = failureDetails != null ? failureDetails.build() : null;
        }

        public final String getType() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(BackupType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final Integer getProgressPercent() {
            return progressPercent;
        }

        @Override
        public final Builder progressPercent(Integer progressPercent) {
            this.progressPercent = progressPercent;
            return this;
        }

        public final void setProgressPercent(Integer progressPercent) {
            this.progressPercent = progressPercent;
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final String getResourceARN() {
            return resourceARN;
        }

        @Override
        public final Builder resourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
            return this;
        }

        public final void setResourceARN(String resourceARN) {
            this.resourceARN = resourceARN;
        }

        public final Collection<Tag.Builder> getTags() {
            if (tags instanceof SdkAutoConstructList) {
                return null;
            }
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final FileSystem.Builder getFileSystem() {
            return fileSystem != null ? fileSystem.toBuilder() : null;
        }

        @Override
        public final Builder fileSystem(FileSystem fileSystem) {
            this.fileSystem = fileSystem;
            return this;
        }

        public final void setFileSystem(FileSystem.BuilderImpl fileSystem) {
            this.fileSystem = fileSystem != null ? fileSystem.build() : null;
        }

        public final ActiveDirectoryBackupAttributes.Builder getDirectoryInformation() {
            return directoryInformation != null ? directoryInformation.toBuilder() : null;
        }

        @Override
        public final Builder directoryInformation(ActiveDirectoryBackupAttributes directoryInformation) {
            this.directoryInformation = directoryInformation;
            return this;
        }

        public final void setDirectoryInformation(ActiveDirectoryBackupAttributes.BuilderImpl directoryInformation) {
            this.directoryInformation = directoryInformation != null ? directoryInformation.build() : null;
        }

        @Override
        public Backup build() {
            return new Backup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
