/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The request object used to create a new Amazon FSx file system.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateFileSystemRequest extends FSxRequest implements
        ToCopyableBuilder<CreateFileSystemRequest.Builder, CreateFileSystemRequest> {
    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateFileSystemRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> FILE_SYSTEM_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateFileSystemRequest::fileSystemTypeAsString)).setter(setter(Builder::fileSystemType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FileSystemType").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateFileSystemRequest::storageCapacity)).setter(setter(Builder::storageCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacity").build()).build();

    private static final SdkField<String> STORAGE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateFileSystemRequest::storageTypeAsString)).setter(setter(Builder::storageType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageType").build()).build();

    private static final SdkField<List<String>> SUBNET_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateFileSystemRequest::subnetIds))
            .setter(setter(Builder::subnetIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateFileSystemRequest::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(CreateFileSystemRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateFileSystemRequest::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<CreateFileSystemWindowsConfiguration> WINDOWS_CONFIGURATION_FIELD = SdkField
            .<CreateFileSystemWindowsConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateFileSystemRequest::windowsConfiguration)).setter(setter(Builder::windowsConfiguration))
            .constructor(CreateFileSystemWindowsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WindowsConfiguration").build())
            .build();

    private static final SdkField<CreateFileSystemLustreConfiguration> LUSTRE_CONFIGURATION_FIELD = SdkField
            .<CreateFileSystemLustreConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(CreateFileSystemRequest::lustreConfiguration)).setter(setter(Builder::lustreConfiguration))
            .constructor(CreateFileSystemLustreConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LustreConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_REQUEST_TOKEN_FIELD,
            FILE_SYSTEM_TYPE_FIELD, STORAGE_CAPACITY_FIELD, STORAGE_TYPE_FIELD, SUBNET_IDS_FIELD, SECURITY_GROUP_IDS_FIELD,
            TAGS_FIELD, KMS_KEY_ID_FIELD, WINDOWS_CONFIGURATION_FIELD, LUSTRE_CONFIGURATION_FIELD));

    private final String clientRequestToken;

    private final String fileSystemType;

    private final Integer storageCapacity;

    private final String storageType;

    private final List<String> subnetIds;

    private final List<String> securityGroupIds;

    private final List<Tag> tags;

    private final String kmsKeyId;

    private final CreateFileSystemWindowsConfiguration windowsConfiguration;

    private final CreateFileSystemLustreConfiguration lustreConfiguration;

    private CreateFileSystemRequest(BuilderImpl builder) {
        super(builder);
        this.clientRequestToken = builder.clientRequestToken;
        this.fileSystemType = builder.fileSystemType;
        this.storageCapacity = builder.storageCapacity;
        this.storageType = builder.storageType;
        this.subnetIds = builder.subnetIds;
        this.securityGroupIds = builder.securityGroupIds;
        this.tags = builder.tags;
        this.kmsKeyId = builder.kmsKeyId;
        this.windowsConfiguration = builder.windowsConfiguration;
        this.lustreConfiguration = builder.lustreConfiguration;
    }

    /**
     * <p>
     * (Optional) A string of up to 64 ASCII characters that Amazon FSx uses to ensure idempotent creation. This string
     * is automatically filled on your behalf when you use the AWS Command Line Interface (AWS CLI) or an AWS SDK.
     * </p>
     * 
     * @return (Optional) A string of up to 64 ASCII characters that Amazon FSx uses to ensure idempotent creation. This
     *         string is automatically filled on your behalf when you use the AWS Command Line Interface (AWS CLI) or an
     *         AWS SDK.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
     * @see FileSystemType
     */
    public FileSystemType fileSystemType() {
        return FileSystemType.fromValue(fileSystemType);
    }

    /**
     * <p>
     * The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #fileSystemType}
     * will return {@link FileSystemType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #fileSystemTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
     * @see FileSystemType
     */
    public String fileSystemTypeAsString() {
        return fileSystemType;
    }

    /**
     * <p>
     * Sets the storage capacity of the file system that you're creating.
     * </p>
     * <p>
     * For Lustre file systems:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, valid values are 1.2, 2.4, and
     * increments of 2.4 TiB.
     * </p>
     * </li>
     * <li>
     * <p>
     * For <code>SCRATCH_1</code> deployment type, valid values are 1.2, 2.4, and increments of 3.6 TiB.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For Windows file systems:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If <code>StorageType=SSD</code>, valid values are 32 GiB - 65,536 GiB (64 TiB).
     * </p>
     * </li>
     * <li>
     * <p>
     * If <code>StorageType=HDD</code>, valid values are 2000 GiB - 65,536 GiB (64 TiB).
     * </p>
     * </li>
     * </ul>
     * 
     * @return Sets the storage capacity of the file system that you're creating.</p>
     *         <p>
     *         For Lustre file systems:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, valid values are 1.2, 2.4, and
     *         increments of 2.4 TiB.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For <code>SCRATCH_1</code> deployment type, valid values are 1.2, 2.4, and increments of 3.6 TiB.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For Windows file systems:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If <code>StorageType=SSD</code>, valid values are 32 GiB - 65,536 GiB (64 TiB).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If <code>StorageType=HDD</code>, valid values are 2000 GiB - 65,536 GiB (64 TiB).
     *         </p>
     *         </li>
     */
    public Integer storageCapacity() {
        return storageCapacity;
    }

    /**
     * <p>
     * Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
     * <code>SSD</code> and <code>HDD</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment types.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     * <code>MULTI_AZ_1</code> Windows file system deployment types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default value is <code>SSD</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"> Storage
     * Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
     *         <code>SSD</code> and <code>HDD</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment
     *         types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     *         <code>MULTI_AZ_1</code> Windows file system deployment types.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default value is <code>SSD</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
     *         Storage Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
     * @see StorageType
     */
    public StorageType storageType() {
        return StorageType.fromValue(storageType);
    }

    /**
     * <p>
     * Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
     * <code>SSD</code> and <code>HDD</code>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment types.
     * </p>
     * </li>
     * <li>
     * <p>
     * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     * <code>MULTI_AZ_1</code> Windows file system deployment types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default value is <code>SSD</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"> Storage
     * Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #storageType} will
     * return {@link StorageType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #storageTypeAsString}.
     * </p>
     * 
     * @return Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
     *         <code>SSD</code> and <code>HDD</code>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment
     *         types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
     *         <code>MULTI_AZ_1</code> Windows file system deployment types.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default value is <code>SSD</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
     *         Storage Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
     * @see StorageType
     */
    public String storageTypeAsString() {
        return storageType;
    }

    /**
     * Returns true if the SubnetIds property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnetIds() {
        return subnetIds != null && !(subnetIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies the IDs of the subnets that the file system will be accessible from. For Windows
     * <code>MULTI_AZ_1</code> file system deployment types, provide exactly two subnet IDs, one for the preferred file
     * server and one for the standby file server. You specify one of these subnets as the preferred subnet using the
     * <code>WindowsConfiguration &gt; PreferredSubnetID</code> property.
     * </p>
     * <p>
     * For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> file system deployment types and Lustre file
     * systems, provide exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnetIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies the IDs of the subnets that the file system will be accessible from. For Windows
     *         <code>MULTI_AZ_1</code> file system deployment types, provide exactly two subnet IDs, one for the
     *         preferred file server and one for the standby file server. You specify one of these subnets as the
     *         preferred subnet using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> property.</p>
     *         <p>
     *         For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> file system deployment types and Lustre
     *         file systems, provide exactly one subnet ID. The file server is launched in that subnet's Availability
     *         Zone.
     */
    public List<String> subnetIds() {
        return subnetIds;
    }

    /**
     * Returns true if the SecurityGroupIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of IDs specifying the security groups to apply to all network interfaces created for file system access.
     * This list isn't returned in later requests to describe the file system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSecurityGroupIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of IDs specifying the security groups to apply to all network interfaces created for file system
     *         access. This list isn't returned in later requests to describe the file system.
     */
    public List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears in the
     * console as the file system name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears in
     *         the console as the file system name.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * Returns the value of the KmsKeyId property for this object.
     * 
     * @return The value of the KmsKeyId property for this object.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Microsoft Windows configuration for the file system being created.
     * </p>
     * 
     * @return The Microsoft Windows configuration for the file system being created.
     */
    public CreateFileSystemWindowsConfiguration windowsConfiguration() {
        return windowsConfiguration;
    }

    /**
     * Returns the value of the LustreConfiguration property for this object.
     * 
     * @return The value of the LustreConfiguration property for this object.
     */
    public CreateFileSystemLustreConfiguration lustreConfiguration() {
        return lustreConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(fileSystemTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(storageTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subnetIds());
        hashCode = 31 * hashCode + Objects.hashCode(securityGroupIds());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(windowsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(lustreConfiguration());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateFileSystemRequest)) {
            return false;
        }
        CreateFileSystemRequest other = (CreateFileSystemRequest) obj;
        return Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(fileSystemTypeAsString(), other.fileSystemTypeAsString())
                && Objects.equals(storageCapacity(), other.storageCapacity())
                && Objects.equals(storageTypeAsString(), other.storageTypeAsString())
                && Objects.equals(subnetIds(), other.subnetIds()) && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(tags(), other.tags()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(windowsConfiguration(), other.windowsConfiguration())
                && Objects.equals(lustreConfiguration(), other.lustreConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateFileSystemRequest").add("ClientRequestToken", clientRequestToken())
                .add("FileSystemType", fileSystemTypeAsString()).add("StorageCapacity", storageCapacity())
                .add("StorageType", storageTypeAsString()).add("SubnetIds", subnetIds())
                .add("SecurityGroupIds", securityGroupIds()).add("Tags", tags()).add("KmsKeyId", kmsKeyId())
                .add("WindowsConfiguration", windowsConfiguration()).add("LustreConfiguration", lustreConfiguration()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "FileSystemType":
            return Optional.ofNullable(clazz.cast(fileSystemTypeAsString()));
        case "StorageCapacity":
            return Optional.ofNullable(clazz.cast(storageCapacity()));
        case "StorageType":
            return Optional.ofNullable(clazz.cast(storageTypeAsString()));
        case "SubnetIds":
            return Optional.ofNullable(clazz.cast(subnetIds()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "WindowsConfiguration":
            return Optional.ofNullable(clazz.cast(windowsConfiguration()));
        case "LustreConfiguration":
            return Optional.ofNullable(clazz.cast(lustreConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateFileSystemRequest, T> g) {
        return obj -> g.apply((CreateFileSystemRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends FSxRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateFileSystemRequest> {
        /**
         * <p>
         * (Optional) A string of up to 64 ASCII characters that Amazon FSx uses to ensure idempotent creation. This
         * string is automatically filled on your behalf when you use the AWS Command Line Interface (AWS CLI) or an AWS
         * SDK.
         * </p>
         * 
         * @param clientRequestToken
         *        (Optional) A string of up to 64 ASCII characters that Amazon FSx uses to ensure idempotent creation.
         *        This string is automatically filled on your behalf when you use the AWS Command Line Interface (AWS
         *        CLI) or an AWS SDK.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(String fileSystemType);

        /**
         * <p>
         * The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
         * </p>
         * 
         * @param fileSystemType
         *        The type of Amazon FSx file system to create, either <code>WINDOWS</code> or <code>LUSTRE</code>.
         * @see FileSystemType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FileSystemType
         */
        Builder fileSystemType(FileSystemType fileSystemType);

        /**
         * <p>
         * Sets the storage capacity of the file system that you're creating.
         * </p>
         * <p>
         * For Lustre file systems:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, valid values are 1.2, 2.4, and
         * increments of 2.4 TiB.
         * </p>
         * </li>
         * <li>
         * <p>
         * For <code>SCRATCH_1</code> deployment type, valid values are 1.2, 2.4, and increments of 3.6 TiB.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For Windows file systems:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If <code>StorageType=SSD</code>, valid values are 32 GiB - 65,536 GiB (64 TiB).
         * </p>
         * </li>
         * <li>
         * <p>
         * If <code>StorageType=HDD</code>, valid values are 2000 GiB - 65,536 GiB (64 TiB).
         * </p>
         * </li>
         * </ul>
         * 
         * @param storageCapacity
         *        Sets the storage capacity of the file system that you're creating.</p>
         *        <p>
         *        For Lustre file systems:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For <code>SCRATCH_2</code> and <code>PERSISTENT_1</code> deployment types, valid values are 1.2, 2.4,
         *        and increments of 2.4 TiB.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For <code>SCRATCH_1</code> deployment type, valid values are 1.2, 2.4, and increments of 3.6 TiB.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For Windows file systems:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If <code>StorageType=SSD</code>, valid values are 32 GiB - 65,536 GiB (64 TiB).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If <code>StorageType=HDD</code>, valid values are 2000 GiB - 65,536 GiB (64 TiB).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacity(Integer storageCapacity);

        /**
         * <p>
         * Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
         * <code>SSD</code> and <code>HDD</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment types.
         * </p>
         * </li>
         * <li>
         * <p>
         * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
         * <code>MULTI_AZ_1</code> Windows file system deployment types.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default value is <code>SSD</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
         * Storage Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
         * </p>
         * 
         * @param storageType
         *        Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
         *        <code>SSD</code> and <code>HDD</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment
         *        types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code>
         *        and <code>MULTI_AZ_1</code> Windows file system deployment types.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default value is <code>SSD</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"
         *        > Storage Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(String storageType);

        /**
         * <p>
         * Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
         * <code>SSD</code> and <code>HDD</code>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment types.
         * </p>
         * </li>
         * <li>
         * <p>
         * Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code> and
         * <code>MULTI_AZ_1</code> Windows file system deployment types.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default value is <code>SSD</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options">
         * Storage Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
         * </p>
         * 
         * @param storageType
         *        Sets the storage type for the Amazon FSx for Windows file system you're creating. Valid values are
         *        <code>SSD</code> and <code>HDD</code>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Set to <code>SSD</code> to use solid state drive storage. SSD is supported on all Windows deployment
         *        types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Set to <code>HDD</code> to use hard disk drive storage. HDD is supported on <code>SINGLE_AZ_2</code>
         *        and <code>MULTI_AZ_1</code> Windows file system deployment types.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default value is <code>SSD</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/optimize-fsx-costs.html#storage-type-options"
         *        > Storage Type Options</a> in the <i>Amazon FSx for Windows User Guide</i>.
         * @see StorageType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StorageType
         */
        Builder storageType(StorageType storageType);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system will be accessible from. For Windows
         * <code>MULTI_AZ_1</code> file system deployment types, provide exactly two subnet IDs, one for the preferred
         * file server and one for the standby file server. You specify one of these subnets as the preferred subnet
         * using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> property.
         * </p>
         * <p>
         * For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> file system deployment types and Lustre
         * file systems, provide exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system will be accessible from. For Windows
         *        <code>MULTI_AZ_1</code> file system deployment types, provide exactly two subnet IDs, one for the
         *        preferred file server and one for the standby file server. You specify one of these subnets as the
         *        preferred subnet using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> property.</p>
         *        <p>
         *        For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> file system deployment types and
         *        Lustre file systems, provide exactly one subnet ID. The file server is launched in that subnet's
         *        Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(Collection<String> subnetIds);

        /**
         * <p>
         * Specifies the IDs of the subnets that the file system will be accessible from. For Windows
         * <code>MULTI_AZ_1</code> file system deployment types, provide exactly two subnet IDs, one for the preferred
         * file server and one for the standby file server. You specify one of these subnets as the preferred subnet
         * using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> property.
         * </p>
         * <p>
         * For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> file system deployment types and Lustre
         * file systems, provide exactly one subnet ID. The file server is launched in that subnet's Availability Zone.
         * </p>
         * 
         * @param subnetIds
         *        Specifies the IDs of the subnets that the file system will be accessible from. For Windows
         *        <code>MULTI_AZ_1</code> file system deployment types, provide exactly two subnet IDs, one for the
         *        preferred file server and one for the standby file server. You specify one of these subnets as the
         *        preferred subnet using the <code>WindowsConfiguration &gt; PreferredSubnetID</code> property.</p>
         *        <p>
         *        For Windows <code>SINGLE_AZ_1</code> and <code>SINGLE_AZ_2</code> file system deployment types and
         *        Lustre file systems, provide exactly one subnet ID. The file server is launched in that subnet's
         *        Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetIds(String... subnetIds);

        /**
         * <p>
         * A list of IDs specifying the security groups to apply to all network interfaces created for file system
         * access. This list isn't returned in later requests to describe the file system.
         * </p>
         * 
         * @param securityGroupIds
         *        A list of IDs specifying the security groups to apply to all network interfaces created for file
         *        system access. This list isn't returned in later requests to describe the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * A list of IDs specifying the security groups to apply to all network interfaces created for file system
         * access. This list isn't returned in later requests to describe the file system.
         * </p>
         * 
         * @param securityGroupIds
         *        A list of IDs specifying the security groups to apply to all network interfaces created for file
         *        system access. This list isn't returned in later requests to describe the file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears in the
         * console as the file system name.
         * </p>
         * 
         * @param tags
         *        The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears
         *        in the console as the file system name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears in the
         * console as the file system name.
         * </p>
         * 
         * @param tags
         *        The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears
         *        in the console as the file system name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags to apply to the file system being created. The key value of the <code>Name</code> tag appears in the
         * console as the file system name.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * Sets the value of the KmsKeyId property for this object.
         *
         * @param kmsKeyId
         *        The new value for the KmsKeyId property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Microsoft Windows configuration for the file system being created.
         * </p>
         * 
         * @param windowsConfiguration
         *        The Microsoft Windows configuration for the file system being created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder windowsConfiguration(CreateFileSystemWindowsConfiguration windowsConfiguration);

        /**
         * <p>
         * The Microsoft Windows configuration for the file system being created.
         * </p>
         * This is a convenience that creates an instance of the {@link CreateFileSystemWindowsConfiguration.Builder}
         * avoiding the need to create one manually via {@link CreateFileSystemWindowsConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link CreateFileSystemWindowsConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #windowsConfiguration(CreateFileSystemWindowsConfiguration)}.
         * 
         * @param windowsConfiguration
         *        a consumer that will call methods on {@link CreateFileSystemWindowsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #windowsConfiguration(CreateFileSystemWindowsConfiguration)
         */
        default Builder windowsConfiguration(Consumer<CreateFileSystemWindowsConfiguration.Builder> windowsConfiguration) {
            return windowsConfiguration(CreateFileSystemWindowsConfiguration.builder().applyMutation(windowsConfiguration)
                    .build());
        }

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * @param lustreConfiguration
         *        The new value for the LustreConfiguration property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lustreConfiguration(CreateFileSystemLustreConfiguration lustreConfiguration);

        /**
         * Sets the value of the LustreConfiguration property for this object.
         *
         * This is a convenience that creates an instance of the {@link CreateFileSystemLustreConfiguration.Builder}
         * avoiding the need to create one manually via {@link CreateFileSystemLustreConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link CreateFileSystemLustreConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #lustreConfiguration(CreateFileSystemLustreConfiguration)}.
         * 
         * @param lustreConfiguration
         *        a consumer that will call methods on {@link CreateFileSystemLustreConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lustreConfiguration(CreateFileSystemLustreConfiguration)
         */
        default Builder lustreConfiguration(Consumer<CreateFileSystemLustreConfiguration.Builder> lustreConfiguration) {
            return lustreConfiguration(CreateFileSystemLustreConfiguration.builder().applyMutation(lustreConfiguration).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends FSxRequest.BuilderImpl implements Builder {
        private String clientRequestToken;

        private String fileSystemType;

        private Integer storageCapacity;

        private String storageType;

        private List<String> subnetIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String kmsKeyId;

        private CreateFileSystemWindowsConfiguration windowsConfiguration;

        private CreateFileSystemLustreConfiguration lustreConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateFileSystemRequest model) {
            super(model);
            clientRequestToken(model.clientRequestToken);
            fileSystemType(model.fileSystemType);
            storageCapacity(model.storageCapacity);
            storageType(model.storageType);
            subnetIds(model.subnetIds);
            securityGroupIds(model.securityGroupIds);
            tags(model.tags);
            kmsKeyId(model.kmsKeyId);
            windowsConfiguration(model.windowsConfiguration);
            lustreConfiguration(model.lustreConfiguration);
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final String getFileSystemType() {
            return fileSystemType;
        }

        @Override
        public final Builder fileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
            return this;
        }

        @Override
        public final Builder fileSystemType(FileSystemType fileSystemType) {
            this.fileSystemType(fileSystemType == null ? null : fileSystemType.toString());
            return this;
        }

        public final void setFileSystemType(String fileSystemType) {
            this.fileSystemType = fileSystemType;
        }

        public final Integer getStorageCapacity() {
            return storageCapacity;
        }

        @Override
        public final Builder storageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
            return this;
        }

        public final void setStorageCapacity(Integer storageCapacity) {
            this.storageCapacity = storageCapacity;
        }

        public final String getStorageType() {
            return storageType;
        }

        @Override
        public final Builder storageType(String storageType) {
            this.storageType = storageType;
            return this;
        }

        @Override
        public final Builder storageType(StorageType storageType) {
            this.storageType(storageType == null ? null : storageType.toString());
            return this;
        }

        public final void setStorageType(String storageType) {
            this.storageType = storageType;
        }

        public final Collection<String> getSubnetIds() {
            return subnetIds;
        }

        @Override
        public final Builder subnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnetIds(String... subnetIds) {
            subnetIds(Arrays.asList(subnetIds));
            return this;
        }

        public final void setSubnetIds(Collection<String> subnetIds) {
            this.subnetIds = SubnetIdsCopier.copy(subnetIds);
        }

        public final Collection<String> getSecurityGroupIds() {
            return securityGroupIds;
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdsCopier.copy(securityGroupIds);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final CreateFileSystemWindowsConfiguration.Builder getWindowsConfiguration() {
            return windowsConfiguration != null ? windowsConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder windowsConfiguration(CreateFileSystemWindowsConfiguration windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration;
            return this;
        }

        public final void setWindowsConfiguration(CreateFileSystemWindowsConfiguration.BuilderImpl windowsConfiguration) {
            this.windowsConfiguration = windowsConfiguration != null ? windowsConfiguration.build() : null;
        }

        public final CreateFileSystemLustreConfiguration.Builder getLustreConfiguration() {
            return lustreConfiguration != null ? lustreConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder lustreConfiguration(CreateFileSystemLustreConfiguration lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration;
            return this;
        }

        public final void setLustreConfiguration(CreateFileSystemLustreConfiguration.BuilderImpl lustreConfiguration) {
            this.lustreConfiguration = lustreConfiguration != null ? lustreConfiguration.build() : null;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateFileSystemRequest build() {
            return new CreateFileSystemRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
