/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.frauddetector;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.frauddetector.model.BatchCreateVariableRequest;
import software.amazon.awssdk.services.frauddetector.model.BatchCreateVariableResponse;
import software.amazon.awssdk.services.frauddetector.model.BatchGetVariableRequest;
import software.amazon.awssdk.services.frauddetector.model.BatchGetVariableResponse;
import software.amazon.awssdk.services.frauddetector.model.ConflictException;
import software.amazon.awssdk.services.frauddetector.model.CreateDetectorVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.CreateDetectorVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.CreateModelVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.CreateModelVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.CreateRuleRequest;
import software.amazon.awssdk.services.frauddetector.model.CreateRuleResponse;
import software.amazon.awssdk.services.frauddetector.model.CreateVariableRequest;
import software.amazon.awssdk.services.frauddetector.model.CreateVariableResponse;
import software.amazon.awssdk.services.frauddetector.model.DeleteDetectorRequest;
import software.amazon.awssdk.services.frauddetector.model.DeleteDetectorResponse;
import software.amazon.awssdk.services.frauddetector.model.DeleteDetectorVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.DeleteDetectorVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.DeleteEventRequest;
import software.amazon.awssdk.services.frauddetector.model.DeleteEventResponse;
import software.amazon.awssdk.services.frauddetector.model.DeleteRuleVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.DeleteRuleVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.DescribeDetectorRequest;
import software.amazon.awssdk.services.frauddetector.model.DescribeDetectorResponse;
import software.amazon.awssdk.services.frauddetector.model.DescribeModelVersionsRequest;
import software.amazon.awssdk.services.frauddetector.model.DescribeModelVersionsResponse;
import software.amazon.awssdk.services.frauddetector.model.FraudDetectorException;
import software.amazon.awssdk.services.frauddetector.model.FraudDetectorRequest;
import software.amazon.awssdk.services.frauddetector.model.GetDetectorVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.GetDetectorVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.GetDetectorsRequest;
import software.amazon.awssdk.services.frauddetector.model.GetDetectorsResponse;
import software.amazon.awssdk.services.frauddetector.model.GetExternalModelsRequest;
import software.amazon.awssdk.services.frauddetector.model.GetExternalModelsResponse;
import software.amazon.awssdk.services.frauddetector.model.GetModelVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.GetModelVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.GetModelsRequest;
import software.amazon.awssdk.services.frauddetector.model.GetModelsResponse;
import software.amazon.awssdk.services.frauddetector.model.GetOutcomesRequest;
import software.amazon.awssdk.services.frauddetector.model.GetOutcomesResponse;
import software.amazon.awssdk.services.frauddetector.model.GetPredictionRequest;
import software.amazon.awssdk.services.frauddetector.model.GetPredictionResponse;
import software.amazon.awssdk.services.frauddetector.model.GetRulesRequest;
import software.amazon.awssdk.services.frauddetector.model.GetRulesResponse;
import software.amazon.awssdk.services.frauddetector.model.GetVariablesRequest;
import software.amazon.awssdk.services.frauddetector.model.GetVariablesResponse;
import software.amazon.awssdk.services.frauddetector.model.InternalServerException;
import software.amazon.awssdk.services.frauddetector.model.PutDetectorRequest;
import software.amazon.awssdk.services.frauddetector.model.PutDetectorResponse;
import software.amazon.awssdk.services.frauddetector.model.PutExternalModelRequest;
import software.amazon.awssdk.services.frauddetector.model.PutExternalModelResponse;
import software.amazon.awssdk.services.frauddetector.model.PutModelRequest;
import software.amazon.awssdk.services.frauddetector.model.PutModelResponse;
import software.amazon.awssdk.services.frauddetector.model.PutOutcomeRequest;
import software.amazon.awssdk.services.frauddetector.model.PutOutcomeResponse;
import software.amazon.awssdk.services.frauddetector.model.ResourceNotFoundException;
import software.amazon.awssdk.services.frauddetector.model.ThrottlingException;
import software.amazon.awssdk.services.frauddetector.model.UpdateDetectorVersionMetadataRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateDetectorVersionMetadataResponse;
import software.amazon.awssdk.services.frauddetector.model.UpdateDetectorVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateDetectorVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.UpdateDetectorVersionStatusRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateDetectorVersionStatusResponse;
import software.amazon.awssdk.services.frauddetector.model.UpdateModelVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateModelVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.UpdateRuleMetadataRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateRuleMetadataResponse;
import software.amazon.awssdk.services.frauddetector.model.UpdateRuleVersionRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateRuleVersionResponse;
import software.amazon.awssdk.services.frauddetector.model.UpdateVariableRequest;
import software.amazon.awssdk.services.frauddetector.model.UpdateVariableResponse;
import software.amazon.awssdk.services.frauddetector.model.ValidationException;
import software.amazon.awssdk.services.frauddetector.paginators.DescribeModelVersionsPublisher;
import software.amazon.awssdk.services.frauddetector.paginators.GetDetectorsPublisher;
import software.amazon.awssdk.services.frauddetector.paginators.GetExternalModelsPublisher;
import software.amazon.awssdk.services.frauddetector.paginators.GetModelsPublisher;
import software.amazon.awssdk.services.frauddetector.paginators.GetOutcomesPublisher;
import software.amazon.awssdk.services.frauddetector.paginators.GetRulesPublisher;
import software.amazon.awssdk.services.frauddetector.paginators.GetVariablesPublisher;
import software.amazon.awssdk.services.frauddetector.transform.BatchCreateVariableRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.BatchGetVariableRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.CreateDetectorVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.CreateModelVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.CreateRuleRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.CreateVariableRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.DeleteDetectorRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.DeleteDetectorVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.DeleteEventRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.DeleteRuleVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.DescribeDetectorRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.DescribeModelVersionsRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetDetectorVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetDetectorsRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetExternalModelsRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetModelVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetModelsRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetOutcomesRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetPredictionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetRulesRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.GetVariablesRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.PutDetectorRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.PutExternalModelRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.PutModelRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.PutOutcomeRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateDetectorVersionMetadataRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateDetectorVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateDetectorVersionStatusRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateModelVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateRuleMetadataRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateRuleVersionRequestMarshaller;
import software.amazon.awssdk.services.frauddetector.transform.UpdateVariableRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link FraudDetectorAsyncClient}.
 *
 * @see FraudDetectorAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFraudDetectorAsyncClient implements FraudDetectorAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultFraudDetectorAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFraudDetectorAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a batch of variables.
     * </p>
     *
     * @param batchCreateVariableRequest
     * @return A Java Future containing the result of the BatchCreateVariable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.BatchCreateVariable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/BatchCreateVariable"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchCreateVariableResponse> batchCreateVariable(
            BatchCreateVariableRequest batchCreateVariableRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchCreateVariableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchCreateVariableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchCreateVariableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchCreateVariableRequest, BatchCreateVariableResponse>()
                            .withOperationName("BatchCreateVariable")
                            .withMarshaller(new BatchCreateVariableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchCreateVariableRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a batch of variables.
     * </p>
     *
     * @param batchGetVariableRequest
     * @return A Java Future containing the result of the BatchGetVariable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.BatchGetVariable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/BatchGetVariable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetVariableResponse> batchGetVariable(BatchGetVariableRequest batchGetVariableRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetVariableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchGetVariableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<BatchGetVariableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetVariableRequest, BatchGetVariableResponse>()
                            .withOperationName("BatchGetVariable")
                            .withMarshaller(new BatchGetVariableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(batchGetVariableRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a detector version. The detector version starts in a <code>DRAFT</code> status.
     * </p>
     *
     * @param createDetectorVersionRequest
     * @return A Java Future containing the result of the CreateDetectorVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.CreateDetectorVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/CreateDetectorVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateDetectorVersionResponse> createDetectorVersion(
            CreateDetectorVersionRequest createDetectorVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateDetectorVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateDetectorVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateDetectorVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateDetectorVersionRequest, CreateDetectorVersionResponse>()
                            .withOperationName("CreateDetectorVersion")
                            .withMarshaller(new CreateDetectorVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createDetectorVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a version of the model using the specified model type.
     * </p>
     *
     * @param createModelVersionRequest
     * @return A Java Future containing the result of the CreateModelVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.CreateModelVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/CreateModelVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateModelVersionResponse> createModelVersion(CreateModelVersionRequest createModelVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateModelVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateModelVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateModelVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateModelVersionRequest, CreateModelVersionResponse>()
                            .withOperationName("CreateModelVersion")
                            .withMarshaller(new CreateModelVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createModelVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a rule for use with the specified detector.
     * </p>
     *
     * @param createRuleRequest
     * @return A Java Future containing the result of the CreateRule operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.CreateRule
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/CreateRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateRuleResponse> createRule(CreateRuleRequest createRuleRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateRuleResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateRuleResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateRuleResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateRuleRequest, CreateRuleResponse>().withOperationName("CreateRule")
                            .withMarshaller(new CreateRuleRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createRuleRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a variable.
     * </p>
     *
     * @param createVariableRequest
     * @return A Java Future containing the result of the CreateVariable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.CreateVariable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/CreateVariable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateVariableResponse> createVariable(CreateVariableRequest createVariableRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateVariableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateVariableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateVariableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateVariableRequest, CreateVariableResponse>()
                            .withOperationName("CreateVariable")
                            .withMarshaller(new CreateVariableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createVariableRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the detector. Before deleting a detector, you must first delete all detector versions and rule versions
     * associated with the detector.
     * </p>
     *
     * @param deleteDetectorRequest
     * @return A Java Future containing the result of the DeleteDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException An exception indicating there was a conflict during a delete operation. The
     *         following delete operations can cause a conflict exception:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         DeleteDetector: A conflict exception will occur if the detector has associated <code>Rules</code> or
     *         <code>DetectorVersions</code>. You can only delete a detector if it has no <code>Rules</code> or
     *         <code>DetectorVersions</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DeleteDetectorVersion: A conflict exception will occur if the <code>DetectorVersion</code> status is
     *         <code>ACTIVE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DeleteRuleVersion: A conflict exception will occur if the <code>RuleVersion</code> is in use by an
     *         associated <code>ACTIVE</code> or <code>INACTIVE DetectorVersion</code>.
     *         </p>
     *         </li></li>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DeleteDetector
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DeleteDetector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDetectorResponse> deleteDetector(DeleteDetectorRequest deleteDetectorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDetectorRequest, DeleteDetectorResponse>()
                            .withOperationName("DeleteDetector")
                            .withMarshaller(new DeleteDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDetectorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the detector version. You cannot delete detector versions that are in <code>ACTIVE</code> status.
     * </p>
     *
     * @param deleteDetectorVersionRequest
     * @return A Java Future containing the result of the DeleteDetectorVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>ConflictException An exception indicating there was a conflict during a delete operation. The
     *         following delete operations can cause a conflict exception:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         DeleteDetector: A conflict exception will occur if the detector has associated <code>Rules</code> or
     *         <code>DetectorVersions</code>. You can only delete a detector if it has no <code>Rules</code> or
     *         <code>DetectorVersions</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DeleteDetectorVersion: A conflict exception will occur if the <code>DetectorVersion</code> status is
     *         <code>ACTIVE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DeleteRuleVersion: A conflict exception will occur if the <code>RuleVersion</code> is in use by an
     *         associated <code>ACTIVE</code> or <code>INACTIVE DetectorVersion</code>.
     *         </p>
     *         </li></li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DeleteDetectorVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DeleteDetectorVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteDetectorVersionResponse> deleteDetectorVersion(
            DeleteDetectorVersionRequest deleteDetectorVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteDetectorVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteDetectorVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteDetectorVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteDetectorVersionRequest, DeleteDetectorVersionResponse>()
                            .withOperationName("DeleteDetectorVersion")
                            .withMarshaller(new DeleteDetectorVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteDetectorVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified event.
     * </p>
     *
     * @param deleteEventRequest
     * @return A Java Future containing the result of the DeleteEvent operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DeleteEvent
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DeleteEvent" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteEventResponse> deleteEvent(DeleteEventRequest deleteEventRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteEventResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteEventResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteEventResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteEventRequest, DeleteEventResponse>()
                            .withOperationName("DeleteEvent").withMarshaller(new DeleteEventRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteEventRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the rule version. You cannot delete a rule version if it is used by an <code>ACTIVE</code> or
     * <code>INACTIVE</code> detector version.
     * </p>
     *
     * @param deleteRuleVersionRequest
     * @return A Java Future containing the result of the DeleteRuleVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConflictException An exception indicating there was a conflict during a delete operation. The
     *         following delete operations can cause a conflict exception:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         DeleteDetector: A conflict exception will occur if the detector has associated <code>Rules</code> or
     *         <code>DetectorVersions</code>. You can only delete a detector if it has no <code>Rules</code> or
     *         <code>DetectorVersions</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DeleteDetectorVersion: A conflict exception will occur if the <code>DetectorVersion</code> status is
     *         <code>ACTIVE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DeleteRuleVersion: A conflict exception will occur if the <code>RuleVersion</code> is in use by an
     *         associated <code>ACTIVE</code> or <code>INACTIVE DetectorVersion</code>.
     *         </p>
     *         </li></li>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DeleteRuleVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DeleteRuleVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteRuleVersionResponse> deleteRuleVersion(DeleteRuleVersionRequest deleteRuleVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteRuleVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteRuleVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteRuleVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteRuleVersionRequest, DeleteRuleVersionResponse>()
                            .withOperationName("DeleteRuleVersion")
                            .withMarshaller(new DeleteRuleVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteRuleVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all versions for a specified detector.
     * </p>
     *
     * @param describeDetectorRequest
     * @return A Java Future containing the result of the DescribeDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DescribeDetector
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DescribeDetector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeDetectorResponse> describeDetector(DescribeDetectorRequest describeDetectorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeDetectorResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeDetectorRequest, DescribeDetectorResponse>()
                            .withOperationName("DescribeDetector")
                            .withMarshaller(new DescribeDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeDetectorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of the model versions for the specified model type or for the specified model type and model ID. You can
     * also get details for a single, specified model version.
     * </p>
     *
     * @param describeModelVersionsRequest
     * @return A Java Future containing the result of the DescribeModelVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DescribeModelVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DescribeModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeModelVersionsResponse> describeModelVersions(
            DescribeModelVersionsRequest describeModelVersionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeModelVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeModelVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeModelVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeModelVersionsRequest, DescribeModelVersionsResponse>()
                            .withOperationName("DescribeModelVersions")
                            .withMarshaller(new DescribeModelVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeModelVersionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of the model versions for the specified model type or for the specified model type and model ID. You can
     * also get details for a single, specified model version.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeModelVersions(software.amazon.awssdk.services.frauddetector.model.DescribeModelVersionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.DescribeModelVersionsPublisher publisher = client.describeModelVersionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.DescribeModelVersionsPublisher publisher = client.describeModelVersionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.DescribeModelVersionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.DescribeModelVersionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeModelVersions(software.amazon.awssdk.services.frauddetector.model.DescribeModelVersionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeModelVersionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.DescribeModelVersions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/DescribeModelVersions"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeModelVersionsPublisher describeModelVersionsPaginator(DescribeModelVersionsRequest describeModelVersionsRequest) {
        return new DescribeModelVersionsPublisher(this, applyPaginatorUserAgent(describeModelVersionsRequest));
    }

    /**
     * <p>
     * Gets a particular detector version.
     * </p>
     *
     * @param getDetectorVersionRequest
     * @return A Java Future containing the result of the GetDetectorVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetDetectorVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetDetectorVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetDetectorVersionResponse> getDetectorVersion(GetDetectorVersionRequest getDetectorVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDetectorVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDetectorVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDetectorVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDetectorVersionRequest, GetDetectorVersionResponse>()
                            .withOperationName("GetDetectorVersion")
                            .withMarshaller(new GetDetectorVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDetectorVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of detectors. This is a paginated API. If you provide a null <code>maxSizePerPage</code>, this actions
     * retrieves a maximum of 10 records per page. If you provide a <code>maxSizePerPage</code>, the value must be
     * between 5 and 10. To get the next page results, provide the pagination token from the
     * <code>GetEventTypesResponse</code> as part of your request. A null pagination token fetches the records from the
     * beginning.
     * </p>
     *
     * @param getDetectorsRequest
     * @return A Java Future containing the result of the GetDetectors operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetDetectors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetDetectors" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDetectorsResponse> getDetectors(GetDetectorsRequest getDetectorsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDetectorsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetDetectorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDetectorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDetectorsRequest, GetDetectorsResponse>()
                            .withOperationName("GetDetectors").withMarshaller(new GetDetectorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDetectorsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of detectors. This is a paginated API. If you provide a null <code>maxSizePerPage</code>, this actions
     * retrieves a maximum of 10 records per page. If you provide a <code>maxSizePerPage</code>, the value must be
     * between 5 and 10. To get the next page results, provide the pagination token from the
     * <code>GetEventTypesResponse</code> as part of your request. A null pagination token fetches the records from the
     * beginning.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getDetectors(software.amazon.awssdk.services.frauddetector.model.GetDetectorsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetDetectorsPublisher publisher = client.getDetectorsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetDetectorsPublisher publisher = client.getDetectorsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.GetDetectorsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.GetDetectorsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getDetectors(software.amazon.awssdk.services.frauddetector.model.GetDetectorsRequest)} operation.</b>
     * </p>
     *
     * @param getDetectorsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetDetectors
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetDetectors" target="_top">AWS API
     *      Documentation</a>
     */
    public GetDetectorsPublisher getDetectorsPaginator(GetDetectorsRequest getDetectorsRequest) {
        return new GetDetectorsPublisher(this, applyPaginatorUserAgent(getDetectorsRequest));
    }

    /**
     * <p>
     * Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a
     * paginated API. If you provide a null <code>maxSizePerPage</code>, this actions retrieves a maximum of 10 records
     * per page. If you provide a <code>maxSizePerPage</code>, the value must be between 5 and 10. To get the next page
     * results, provide the pagination token from the <code>GetExternalModelsResult</code> as part of your request. A
     * null pagination token fetches the records from the beginning.
     * </p>
     *
     * @param getExternalModelsRequest
     * @return A Java Future containing the result of the GetExternalModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetExternalModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetExternalModels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetExternalModelsResponse> getExternalModels(GetExternalModelsRequest getExternalModelsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetExternalModelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetExternalModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetExternalModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetExternalModelsRequest, GetExternalModelsResponse>()
                            .withOperationName("GetExternalModels")
                            .withMarshaller(new GetExternalModelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getExternalModelsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the details for one or more Amazon SageMaker models that have been imported into the service. This is a
     * paginated API. If you provide a null <code>maxSizePerPage</code>, this actions retrieves a maximum of 10 records
     * per page. If you provide a <code>maxSizePerPage</code>, the value must be between 5 and 10. To get the next page
     * results, provide the pagination token from the <code>GetExternalModelsResult</code> as part of your request. A
     * null pagination token fetches the records from the beginning.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getExternalModels(software.amazon.awssdk.services.frauddetector.model.GetExternalModelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetExternalModelsPublisher publisher = client.getExternalModelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetExternalModelsPublisher publisher = client.getExternalModelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.GetExternalModelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.GetExternalModelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getExternalModels(software.amazon.awssdk.services.frauddetector.model.GetExternalModelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getExternalModelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetExternalModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetExternalModels"
     *      target="_top">AWS API Documentation</a>
     */
    public GetExternalModelsPublisher getExternalModelsPaginator(GetExternalModelsRequest getExternalModelsRequest) {
        return new GetExternalModelsPublisher(this, applyPaginatorUserAgent(getExternalModelsRequest));
    }

    /**
     * <p>
     * Gets a model version.
     * </p>
     *
     * @param getModelVersionRequest
     * @return A Java Future containing the result of the GetModelVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetModelVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetModelVersion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetModelVersionResponse> getModelVersion(GetModelVersionRequest getModelVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetModelVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetModelVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetModelVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetModelVersionRequest, GetModelVersionResponse>()
                            .withOperationName("GetModelVersion")
                            .withMarshaller(new GetModelVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getModelVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of the models for the AWS account, or the specified model type, or gets a single model for the specified
     * model type, model ID combination.
     * </p>
     *
     * @param getModelsRequest
     * @return A Java Future containing the result of the GetModels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetModels" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetModelsResponse> getModels(GetModelsRequest getModelsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetModelsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetModelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetModelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetModelsRequest, GetModelsResponse>().withOperationName("GetModels")
                            .withMarshaller(new GetModelsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getModelsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of the models for the AWS account, or the specified model type, or gets a single model for the specified
     * model type, model ID combination.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getModels(software.amazon.awssdk.services.frauddetector.model.GetModelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetModelsPublisher publisher = client.getModelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetModelsPublisher publisher = client.getModelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.GetModelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.GetModelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getModels(software.amazon.awssdk.services.frauddetector.model.GetModelsRequest)} operation.</b>
     * </p>
     *
     * @param getModelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetModels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetModels" target="_top">AWS API
     *      Documentation</a>
     */
    public GetModelsPublisher getModelsPaginator(GetModelsRequest getModelsRequest) {
        return new GetModelsPublisher(this, applyPaginatorUserAgent(getModelsRequest));
    }

    /**
     * <p>
     * Gets one or more outcomes. This is a paginated API. If you provide a null <code>maxSizePerPage</code>, this
     * actions retrieves a maximum of 10 records per page. If you provide a <code>maxSizePerPage</code>, the value must
     * be between 50 and 100. To get the next page results, provide the pagination token from the
     * <code>GetOutcomesResult</code> as part of your request. A null pagination token fetches the records from the
     * beginning.
     * </p>
     *
     * @param getOutcomesRequest
     * @return A Java Future containing the result of the GetOutcomes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetOutcomes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetOutcomes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetOutcomesResponse> getOutcomes(GetOutcomesRequest getOutcomesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetOutcomesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetOutcomesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetOutcomesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetOutcomesRequest, GetOutcomesResponse>()
                            .withOperationName("GetOutcomes").withMarshaller(new GetOutcomesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getOutcomesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets one or more outcomes. This is a paginated API. If you provide a null <code>maxSizePerPage</code>, this
     * actions retrieves a maximum of 10 records per page. If you provide a <code>maxSizePerPage</code>, the value must
     * be between 50 and 100. To get the next page results, provide the pagination token from the
     * <code>GetOutcomesResult</code> as part of your request. A null pagination token fetches the records from the
     * beginning.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getOutcomes(software.amazon.awssdk.services.frauddetector.model.GetOutcomesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetOutcomesPublisher publisher = client.getOutcomesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetOutcomesPublisher publisher = client.getOutcomesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.GetOutcomesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.GetOutcomesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getOutcomes(software.amazon.awssdk.services.frauddetector.model.GetOutcomesRequest)} operation.</b>
     * </p>
     *
     * @param getOutcomesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetOutcomes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetOutcomes" target="_top">AWS API
     *      Documentation</a>
     */
    public GetOutcomesPublisher getOutcomesPaginator(GetOutcomesRequest getOutcomesRequest) {
        return new GetOutcomesPublisher(this, applyPaginatorUserAgent(getOutcomesRequest));
    }

    /**
     * <p>
     * Evaluates an event against a detector version. If a version ID is not provided, the detector’s (
     * <code>ACTIVE</code>) version is used.
     * </p>
     *
     * @param getPredictionRequest
     * @return A Java Future containing the result of the GetPrediction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetPrediction
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetPrediction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetPredictionResponse> getPrediction(GetPredictionRequest getPredictionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPredictionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPredictionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPredictionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPredictionRequest, GetPredictionResponse>()
                            .withOperationName("GetPrediction")
                            .withMarshaller(new GetPredictionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getPredictionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all rules available for the specified detector.
     * </p>
     *
     * @param getRulesRequest
     * @return A Java Future containing the result of the GetRules operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetRules" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetRulesResponse> getRules(GetRulesRequest getRulesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRulesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetRulesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRulesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRulesRequest, GetRulesResponse>().withOperationName("GetRules")
                            .withMarshaller(new GetRulesRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getRulesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all rules available for the specified detector.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #getRules(software.amazon.awssdk.services.frauddetector.model.GetRulesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetRulesPublisher publisher = client.getRulesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetRulesPublisher publisher = client.getRulesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.GetRulesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.GetRulesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getRules(software.amazon.awssdk.services.frauddetector.model.GetRulesRequest)} operation.</b>
     * </p>
     *
     * @param getRulesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetRules
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetRules" target="_top">AWS API
     *      Documentation</a>
     */
    public GetRulesPublisher getRulesPaginator(GetRulesRequest getRulesRequest) {
        return new GetRulesPublisher(this, applyPaginatorUserAgent(getRulesRequest));
    }

    /**
     * <p>
     * Gets all of the variables or the specific variable. This is a paginated API. Providing null
     * <code>maxSizePerPage</code> results in retrieving maximum of 100 records per page. If you provide
     * <code>maxSizePerPage</code> the value must be between 50 and 100. To get the next page result, a provide a
     * pagination token from <code>GetVariablesResult</code> as part of your request. Null pagination token fetches the
     * records from the beginning.
     * </p>
     *
     * @param getVariablesRequest
     * @return A Java Future containing the result of the GetVariables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetVariables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetVariables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetVariablesResponse> getVariables(GetVariablesRequest getVariablesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetVariablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetVariablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetVariablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetVariablesRequest, GetVariablesResponse>()
                            .withOperationName("GetVariables").withMarshaller(new GetVariablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getVariablesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets all of the variables or the specific variable. This is a paginated API. Providing null
     * <code>maxSizePerPage</code> results in retrieving maximum of 100 records per page. If you provide
     * <code>maxSizePerPage</code> the value must be between 50 and 100. To get the next page result, a provide a
     * pagination token from <code>GetVariablesResult</code> as part of your request. Null pagination token fetches the
     * records from the beginning.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getVariables(software.amazon.awssdk.services.frauddetector.model.GetVariablesRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetVariablesPublisher publisher = client.getVariablesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.frauddetector.paginators.GetVariablesPublisher publisher = client.getVariablesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.frauddetector.model.GetVariablesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.frauddetector.model.GetVariablesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getVariables(software.amazon.awssdk.services.frauddetector.model.GetVariablesRequest)} operation.</b>
     * </p>
     *
     * @param getVariablesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.GetVariables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/GetVariables" target="_top">AWS API
     *      Documentation</a>
     */
    public GetVariablesPublisher getVariablesPaginator(GetVariablesRequest getVariablesRequest) {
        return new GetVariablesPublisher(this, applyPaginatorUserAgent(getVariablesRequest));
    }

    /**
     * <p>
     * Creates or updates a detector.
     * </p>
     *
     * @param putDetectorRequest
     * @return A Java Future containing the result of the PutDetector operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.PutDetector
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/PutDetector" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutDetectorResponse> putDetector(PutDetectorRequest putDetectorRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutDetectorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutDetectorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutDetectorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutDetectorRequest, PutDetectorResponse>()
                            .withOperationName("PutDetector").withMarshaller(new PutDetectorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putDetectorRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates an Amazon SageMaker model endpoint. You can also use this action to update the configuration
     * of the model endpoint, including the IAM role and/or the mapped variables.
     * </p>
     *
     * @param putExternalModelRequest
     * @return A Java Future containing the result of the PutExternalModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.PutExternalModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/PutExternalModel" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<PutExternalModelResponse> putExternalModel(PutExternalModelRequest putExternalModelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutExternalModelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutExternalModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutExternalModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutExternalModelRequest, PutExternalModelResponse>()
                            .withOperationName("PutExternalModel")
                            .withMarshaller(new PutExternalModelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putExternalModelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates a model.
     * </p>
     *
     * @param putModelRequest
     * @return A Java Future containing the result of the PutModel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.PutModel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/PutModel" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutModelResponse> putModel(PutModelRequest putModelRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutModelResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutModelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutModelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutModelRequest, PutModelResponse>().withOperationName("PutModel")
                            .withMarshaller(new PutModelRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(putModelRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates or updates an outcome.
     * </p>
     *
     * @param putOutcomeRequest
     * @return A Java Future containing the result of the PutOutcome operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.PutOutcome
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/PutOutcome" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutOutcomeResponse> putOutcome(PutOutcomeRequest putOutcomeRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutOutcomeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutOutcomeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutOutcomeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutOutcomeRequest, PutOutcomeResponse>().withOperationName("PutOutcome")
                            .withMarshaller(new PutOutcomeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(putOutcomeRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a detector version. The detector version attributes that you can update include models, external model
     * endpoints, rules, and description. You can only update a <code>DRAFT</code> detector version.
     * </p>
     *
     * @param updateDetectorVersionRequest
     * @return A Java Future containing the result of the UpdateDetectorVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateDetectorVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateDetectorVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDetectorVersionResponse> updateDetectorVersion(
            UpdateDetectorVersionRequest updateDetectorVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDetectorVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDetectorVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDetectorVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDetectorVersionRequest, UpdateDetectorVersionResponse>()
                            .withOperationName("UpdateDetectorVersion")
                            .withMarshaller(new UpdateDetectorVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDetectorVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the detector version's description. You can update the metadata for any detector version (
     * <code>DRAFT, ACTIVE,</code> or <code>INACTIVE</code>).
     * </p>
     *
     * @param updateDetectorVersionMetadataRequest
     * @return A Java Future containing the result of the UpdateDetectorVersionMetadata operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateDetectorVersionMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateDetectorVersionMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDetectorVersionMetadataResponse> updateDetectorVersionMetadata(
            UpdateDetectorVersionMetadataRequest updateDetectorVersionMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDetectorVersionMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDetectorVersionMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDetectorVersionMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDetectorVersionMetadataRequest, UpdateDetectorVersionMetadataResponse>()
                            .withOperationName("UpdateDetectorVersionMetadata")
                            .withMarshaller(new UpdateDetectorVersionMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDetectorVersionMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the detector version’s status. You can perform the following promotions or demotions using
     * <code>UpdateDetectorVersionStatus</code>: <code>DRAFT</code> to <code>ACTIVE</code>, <code>ACTIVE</code> to
     * <code>INACTIVE</code>, and <code>INACTIVE</code> to <code>ACTIVE</code>.
     * </p>
     *
     * @param updateDetectorVersionStatusRequest
     * @return A Java Future containing the result of the UpdateDetectorVersionStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateDetectorVersionStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateDetectorVersionStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateDetectorVersionStatusResponse> updateDetectorVersionStatus(
            UpdateDetectorVersionStatusRequest updateDetectorVersionStatusRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateDetectorVersionStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateDetectorVersionStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateDetectorVersionStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateDetectorVersionStatusRequest, UpdateDetectorVersionStatusResponse>()
                            .withOperationName("UpdateDetectorVersionStatus")
                            .withMarshaller(new UpdateDetectorVersionStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateDetectorVersionStatusRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a model version. You can update the description and status attributes using this action. You can perform
     * the following status updates:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Change the <code>TRAINING_COMPLETE</code> status to <code>ACTIVE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Change <code>ACTIVE</code> back to <code>TRAINING_COMPLETE</code>
     * </p>
     * </li>
     * </ol>
     *
     * @param updateModelVersionRequest
     * @return A Java Future containing the result of the UpdateModelVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateModelVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateModelVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateModelVersionResponse> updateModelVersion(UpdateModelVersionRequest updateModelVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateModelVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateModelVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateModelVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateModelVersionRequest, UpdateModelVersionResponse>()
                            .withOperationName("UpdateModelVersion")
                            .withMarshaller(new UpdateModelVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateModelVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a rule's metadata.
     * </p>
     *
     * @param updateRuleMetadataRequest
     * @return A Java Future containing the result of the UpdateRuleMetadata operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateRuleMetadata
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateRuleMetadata"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRuleMetadataResponse> updateRuleMetadata(UpdateRuleMetadataRequest updateRuleMetadataRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRuleMetadataResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRuleMetadataResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRuleMetadataResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRuleMetadataRequest, UpdateRuleMetadataResponse>()
                            .withOperationName("UpdateRuleMetadata")
                            .withMarshaller(new UpdateRuleMetadataRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateRuleMetadataRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a rule version resulting in a new rule version.
     * </p>
     *
     * @param updateRuleVersionRequest
     * @return A Java Future containing the result of the UpdateRuleVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateRuleVersion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateRuleVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateRuleVersionResponse> updateRuleVersion(UpdateRuleVersionRequest updateRuleVersionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateRuleVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateRuleVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateRuleVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateRuleVersionRequest, UpdateRuleVersionResponse>()
                            .withOperationName("UpdateRuleVersion")
                            .withMarshaller(new UpdateRuleVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateRuleVersionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a variable.
     * </p>
     *
     * @param updateVariableRequest
     * @return A Java Future containing the result of the UpdateVariable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException An exception indicating a specified value is not allowed.</li>
     *         <li>ResourceNotFoundException An exception indicating the specified resource was not found.</li>
     *         <li>InternalServerException An exception indicating an internal server error.</li>
     *         <li>ThrottlingException An exception indicating a throttling error.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>FraudDetectorException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample FraudDetectorAsyncClient.UpdateVariable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/frauddetector-2019-11-15/UpdateVariable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateVariableResponse> updateVariable(UpdateVariableRequest updateVariableRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateVariableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateVariableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateVariableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateVariableRequest, UpdateVariableResponse>()
                            .withOperationName("UpdateVariable")
                            .withMarshaller(new UpdateVariableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateVariableRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(FraudDetectorException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).build());
    }

    private <T extends FraudDetectorRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
