/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An instance type configuration for each instance type in an instance fleet, which determines the EC2 instances Amazon
 * EMR attempts to provision to fulfill On-Demand and Spot target capacities. There can be a maximum of 5 instance type
 * configurations in a fleet.
 * </p>
 * <note>
 * <p>
 * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceTypeConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceTypeConfig.Builder, InstanceTypeConfig> {
    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTypeConfig::instanceType)).setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType").build()).build();

    private static final SdkField<Integer> WEIGHTED_CAPACITY_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(InstanceTypeConfig::weightedCapacity)).setter(setter(Builder::weightedCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WeightedCapacity").build()).build();

    private static final SdkField<String> BID_PRICE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceTypeConfig::bidPrice)).setter(setter(Builder::bidPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BidPrice").build()).build();

    private static final SdkField<Double> BID_PRICE_AS_PERCENTAGE_OF_ON_DEMAND_PRICE_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(InstanceTypeConfig::bidPriceAsPercentageOfOnDemandPrice))
            .setter(setter(Builder::bidPriceAsPercentageOfOnDemandPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("BidPriceAsPercentageOfOnDemandPrice").build()).build();

    private static final SdkField<EbsConfiguration> EBS_CONFIGURATION_FIELD = SdkField
            .<EbsConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(InstanceTypeConfig::ebsConfiguration))
            .setter(setter(Builder::ebsConfiguration)).constructor(EbsConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EbsConfiguration").build()).build();

    private static final SdkField<List<Configuration>> CONFIGURATIONS_FIELD = SdkField
            .<List<Configuration>> builder(MarshallingType.LIST)
            .getter(getter(InstanceTypeConfig::configurations))
            .setter(setter(Builder::configurations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Configurations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Configuration> builder(MarshallingType.SDK_POJO)
                                            .constructor(Configuration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_TYPE_FIELD,
            WEIGHTED_CAPACITY_FIELD, BID_PRICE_FIELD, BID_PRICE_AS_PERCENTAGE_OF_ON_DEMAND_PRICE_FIELD, EBS_CONFIGURATION_FIELD,
            CONFIGURATIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceType;

    private final Integer weightedCapacity;

    private final String bidPrice;

    private final Double bidPriceAsPercentageOfOnDemandPrice;

    private final EbsConfiguration ebsConfiguration;

    private final List<Configuration> configurations;

    private InstanceTypeConfig(BuilderImpl builder) {
        this.instanceType = builder.instanceType;
        this.weightedCapacity = builder.weightedCapacity;
        this.bidPrice = builder.bidPrice;
        this.bidPriceAsPercentageOfOnDemandPrice = builder.bidPriceAsPercentageOfOnDemandPrice;
        this.ebsConfiguration = builder.ebsConfiguration;
        this.configurations = builder.configurations;
    }

    /**
     * <p>
     * An EC2 instance type, such as <code>m3.xlarge</code>.
     * </p>
     * 
     * @return An EC2 instance type, such as <code>m3.xlarge</code>.
     */
    public String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The number of units that a provisioned instance of this type provides toward fulfilling the target capacities
     * defined in <a>InstanceFleetConfig</a>. This value is 1 for a master instance fleet, and must be 1 or greater for
     * core and task instance fleets. Defaults to 1 if not specified.
     * </p>
     * 
     * @return The number of units that a provisioned instance of this type provides toward fulfilling the target
     *         capacities defined in <a>InstanceFleetConfig</a>. This value is 1 for a master instance fleet, and must
     *         be 1 or greater for core and task instance fleets. Defaults to 1 if not specified.
     */
    public Integer weightedCapacity() {
        return weightedCapacity;
    }

    /**
     * <p>
     * The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD. If
     * neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
     * <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
     * </p>
     * 
     * @return The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD.
     *         If neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
     *         <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
     */
    public String bidPrice() {
        return bidPrice;
    }

    /**
     * <p>
     * The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
     * <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%). If neither
     * <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
     * <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
     * </p>
     * 
     * @return The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
     *         <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%). If neither
     *         <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
     *         <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
     */
    public Double bidPriceAsPercentageOfOnDemandPrice() {
        return bidPriceAsPercentageOfOnDemandPrice;
    }

    /**
     * <p>
     * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
     * <code>InstanceType</code>.
     * </p>
     * 
     * @return The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
     *         <code>InstanceType</code>.
     */
    public EbsConfiguration ebsConfiguration() {
        return ebsConfiguration;
    }

    /**
     * Returns true if the Configurations property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasConfigurations() {
        return configurations != null && !(configurations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A configuration classification that applies when provisioning cluster instances, which can include configurations
     * for applications and software that run on the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasConfigurations()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A configuration classification that applies when provisioning cluster instances, which can include
     *         configurations for applications and software that run on the cluster.
     */
    public List<Configuration> configurations() {
        return configurations;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(weightedCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(bidPrice());
        hashCode = 31 * hashCode + Objects.hashCode(bidPriceAsPercentageOfOnDemandPrice());
        hashCode = 31 * hashCode + Objects.hashCode(ebsConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(configurations());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceTypeConfig)) {
            return false;
        }
        InstanceTypeConfig other = (InstanceTypeConfig) obj;
        return Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(weightedCapacity(), other.weightedCapacity()) && Objects.equals(bidPrice(), other.bidPrice())
                && Objects.equals(bidPriceAsPercentageOfOnDemandPrice(), other.bidPriceAsPercentageOfOnDemandPrice())
                && Objects.equals(ebsConfiguration(), other.ebsConfiguration())
                && Objects.equals(configurations(), other.configurations());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceTypeConfig").add("InstanceType", instanceType())
                .add("WeightedCapacity", weightedCapacity()).add("BidPrice", bidPrice())
                .add("BidPriceAsPercentageOfOnDemandPrice", bidPriceAsPercentageOfOnDemandPrice())
                .add("EbsConfiguration", ebsConfiguration()).add("Configurations", configurations()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "WeightedCapacity":
            return Optional.ofNullable(clazz.cast(weightedCapacity()));
        case "BidPrice":
            return Optional.ofNullable(clazz.cast(bidPrice()));
        case "BidPriceAsPercentageOfOnDemandPrice":
            return Optional.ofNullable(clazz.cast(bidPriceAsPercentageOfOnDemandPrice()));
        case "EbsConfiguration":
            return Optional.ofNullable(clazz.cast(ebsConfiguration()));
        case "Configurations":
            return Optional.ofNullable(clazz.cast(configurations()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceTypeConfig, T> g) {
        return obj -> g.apply((InstanceTypeConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceTypeConfig> {
        /**
         * <p>
         * An EC2 instance type, such as <code>m3.xlarge</code>.
         * </p>
         * 
         * @param instanceType
         *        An EC2 instance type, such as <code>m3.xlarge</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The number of units that a provisioned instance of this type provides toward fulfilling the target capacities
         * defined in <a>InstanceFleetConfig</a>. This value is 1 for a master instance fleet, and must be 1 or greater
         * for core and task instance fleets. Defaults to 1 if not specified.
         * </p>
         * 
         * @param weightedCapacity
         *        The number of units that a provisioned instance of this type provides toward fulfilling the target
         *        capacities defined in <a>InstanceFleetConfig</a>. This value is 1 for a master instance fleet, and
         *        must be 1 or greater for core and task instance fleets. Defaults to 1 if not specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder weightedCapacity(Integer weightedCapacity);

        /**
         * <p>
         * The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in USD. If
         * neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
         * <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
         * </p>
         * 
         * @param bidPrice
         *        The bid price for each EC2 Spot instance type as defined by <code>InstanceType</code>. Expressed in
         *        USD. If neither <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is
         *        provided, <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPrice(String bidPrice);

        /**
         * <p>
         * The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
         * <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%). If neither
         * <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
         * <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
         * </p>
         * 
         * @param bidPriceAsPercentageOfOnDemandPrice
         *        The bid price, as a percentage of On-Demand price, for each EC2 Spot instance as defined by
         *        <code>InstanceType</code>. Expressed as a number (for example, 20 specifies 20%). If neither
         *        <code>BidPrice</code> nor <code>BidPriceAsPercentageOfOnDemandPrice</code> is provided,
         *        <code>BidPriceAsPercentageOfOnDemandPrice</code> defaults to 100%.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bidPriceAsPercentageOfOnDemandPrice(Double bidPriceAsPercentageOfOnDemandPrice);

        /**
         * <p>
         * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         * <code>InstanceType</code>.
         * </p>
         * 
         * @param ebsConfiguration
         *        The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         *        <code>InstanceType</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsConfiguration(EbsConfiguration ebsConfiguration);

        /**
         * <p>
         * The configuration of Amazon Elastic Block Storage (EBS) attached to each instance as defined by
         * <code>InstanceType</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link EbsConfiguration.Builder} avoiding the need to
         * create one manually via {@link EbsConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link EbsConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #ebsConfiguration(EbsConfiguration)}.
         * 
         * @param ebsConfiguration
         *        a consumer that will call methods on {@link EbsConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ebsConfiguration(EbsConfiguration)
         */
        default Builder ebsConfiguration(Consumer<EbsConfiguration.Builder> ebsConfiguration) {
            return ebsConfiguration(EbsConfiguration.builder().applyMutation(ebsConfiguration).build());
        }

        /**
         * <p>
         * A configuration classification that applies when provisioning cluster instances, which can include
         * configurations for applications and software that run on the cluster.
         * </p>
         * 
         * @param configurations
         *        A configuration classification that applies when provisioning cluster instances, which can include
         *        configurations for applications and software that run on the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Collection<Configuration> configurations);

        /**
         * <p>
         * A configuration classification that applies when provisioning cluster instances, which can include
         * configurations for applications and software that run on the cluster.
         * </p>
         * 
         * @param configurations
         *        A configuration classification that applies when provisioning cluster instances, which can include
         *        configurations for applications and software that run on the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configurations(Configuration... configurations);

        /**
         * <p>
         * A configuration classification that applies when provisioning cluster instances, which can include
         * configurations for applications and software that run on the cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Configuration>.Builder} avoiding the need
         * to create one manually via {@link List<Configuration>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Configuration>.Builder#build()} is called immediately and
         * its result is passed to {@link #configurations(List<Configuration>)}.
         * 
         * @param configurations
         *        a consumer that will call methods on {@link List<Configuration>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configurations(List<Configuration>)
         */
        Builder configurations(Consumer<Configuration.Builder>... configurations);
    }

    static final class BuilderImpl implements Builder {
        private String instanceType;

        private Integer weightedCapacity;

        private String bidPrice;

        private Double bidPriceAsPercentageOfOnDemandPrice;

        private EbsConfiguration ebsConfiguration;

        private List<Configuration> configurations = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceTypeConfig model) {
            instanceType(model.instanceType);
            weightedCapacity(model.weightedCapacity);
            bidPrice(model.bidPrice);
            bidPriceAsPercentageOfOnDemandPrice(model.bidPriceAsPercentageOfOnDemandPrice);
            ebsConfiguration(model.ebsConfiguration);
            configurations(model.configurations);
        }

        public final String getInstanceType() {
            return instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        public final Integer getWeightedCapacity() {
            return weightedCapacity;
        }

        @Override
        public final Builder weightedCapacity(Integer weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
            return this;
        }

        public final void setWeightedCapacity(Integer weightedCapacity) {
            this.weightedCapacity = weightedCapacity;
        }

        public final String getBidPrice() {
            return bidPrice;
        }

        @Override
        public final Builder bidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
            return this;
        }

        public final void setBidPrice(String bidPrice) {
            this.bidPrice = bidPrice;
        }

        public final Double getBidPriceAsPercentageOfOnDemandPrice() {
            return bidPriceAsPercentageOfOnDemandPrice;
        }

        @Override
        public final Builder bidPriceAsPercentageOfOnDemandPrice(Double bidPriceAsPercentageOfOnDemandPrice) {
            this.bidPriceAsPercentageOfOnDemandPrice = bidPriceAsPercentageOfOnDemandPrice;
            return this;
        }

        public final void setBidPriceAsPercentageOfOnDemandPrice(Double bidPriceAsPercentageOfOnDemandPrice) {
            this.bidPriceAsPercentageOfOnDemandPrice = bidPriceAsPercentageOfOnDemandPrice;
        }

        public final EbsConfiguration.Builder getEbsConfiguration() {
            return ebsConfiguration != null ? ebsConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder ebsConfiguration(EbsConfiguration ebsConfiguration) {
            this.ebsConfiguration = ebsConfiguration;
            return this;
        }

        public final void setEbsConfiguration(EbsConfiguration.BuilderImpl ebsConfiguration) {
            this.ebsConfiguration = ebsConfiguration != null ? ebsConfiguration.build() : null;
        }

        public final Collection<Configuration.Builder> getConfigurations() {
            return configurations != null ? configurations.stream().map(Configuration::toBuilder).collect(Collectors.toList())
                    : null;
        }

        @Override
        public final Builder configurations(Collection<Configuration> configurations) {
            this.configurations = ConfigurationListCopier.copy(configurations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Configuration... configurations) {
            configurations(Arrays.asList(configurations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder configurations(Consumer<Configuration.Builder>... configurations) {
            configurations(Stream.of(configurations).map(c -> Configuration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setConfigurations(Collection<Configuration.BuilderImpl> configurations) {
            this.configurations = ConfigurationListCopier.copyFromBuilder(configurations);
        }

        @Override
        public InstanceTypeConfig build() {
            return new InstanceTypeConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
