/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The definition of a CloudWatch metric alarm, which determines when an automatic scaling activity is triggered. When
 * the defined alarm conditions are satisfied, scaling activity begins.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CloudWatchAlarmDefinition implements SdkPojo, Serializable,
        ToCopyableBuilder<CloudWatchAlarmDefinition.Builder, CloudWatchAlarmDefinition> {
    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CloudWatchAlarmDefinition::comparisonOperatorAsString)).setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final SdkField<Integer> EVALUATION_PERIODS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CloudWatchAlarmDefinition::evaluationPeriods)).setter(setter(Builder::evaluationPeriods))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationPeriods").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CloudWatchAlarmDefinition::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CloudWatchAlarmDefinition::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CloudWatchAlarmDefinition::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Period").build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CloudWatchAlarmDefinition::statisticAsString)).setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistic").build()).build();

    private static final SdkField<Double> THRESHOLD_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(CloudWatchAlarmDefinition::threshold)).setter(setter(Builder::threshold))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Threshold").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CloudWatchAlarmDefinition::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final SdkField<List<MetricDimension>> DIMENSIONS_FIELD = SdkField
            .<List<MetricDimension>> builder(MarshallingType.LIST)
            .getter(getter(CloudWatchAlarmDefinition::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MetricDimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(MetricDimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMPARISON_OPERATOR_FIELD,
            EVALUATION_PERIODS_FIELD, METRIC_NAME_FIELD, NAMESPACE_FIELD, PERIOD_FIELD, STATISTIC_FIELD, THRESHOLD_FIELD,
            UNIT_FIELD, DIMENSIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String comparisonOperator;

    private final Integer evaluationPeriods;

    private final String metricName;

    private final String namespace;

    private final Integer period;

    private final String statistic;

    private final Double threshold;

    private final String unit;

    private final List<MetricDimension> dimensions;

    private CloudWatchAlarmDefinition(BuilderImpl builder) {
        this.comparisonOperator = builder.comparisonOperator;
        this.evaluationPeriods = builder.evaluationPeriods;
        this.metricName = builder.metricName;
        this.namespace = builder.namespace;
        this.period = builder.period;
        this.statistic = builder.statistic;
        this.threshold = builder.threshold;
        this.unit = builder.unit;
        this.dimensions = builder.dimensions;
    }

    /**
     * <p>
     * Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
     * <code>Threshold</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
     *         <code>Threshold</code>.
     * @see ComparisonOperator
     */
    public ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
     * <code>Threshold</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
     *         <code>Threshold</code>.
     * @see ComparisonOperator
     */
    public String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    /**
     * <p>
     * The number of periods, in five-minute increments, during which the alarm condition must exist before the alarm
     * triggers automatic scaling activity. The default value is <code>1</code>.
     * </p>
     * 
     * @return The number of periods, in five-minute increments, during which the alarm condition must exist before the
     *         alarm triggers automatic scaling activity. The default value is <code>1</code>.
     */
    public Integer evaluationPeriods() {
        return evaluationPeriods;
    }

    /**
     * <p>
     * The name of the CloudWatch metric that is watched to determine an alarm condition.
     * </p>
     * 
     * @return The name of the CloudWatch metric that is watched to determine an alarm condition.
     */
    public String metricName() {
        return metricName;
    }

    /**
     * <p>
     * The namespace for the CloudWatch metric. The default is <code>AWS/ElasticMapReduce</code>.
     * </p>
     * 
     * @return The namespace for the CloudWatch metric. The default is <code>AWS/ElasticMapReduce</code>.
     */
    public String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The period, in seconds, over which the statistic is applied. EMR CloudWatch metrics are emitted every five
     * minutes (300 seconds), so if an EMR CloudWatch metric is specified, specify <code>300</code>.
     * </p>
     * 
     * @return The period, in seconds, over which the statistic is applied. EMR CloudWatch metrics are emitted every
     *         five minutes (300 seconds), so if an EMR CloudWatch metric is specified, specify <code>300</code>.
     */
    public Integer period() {
        return period;
    }

    /**
     * <p>
     * The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link Statistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
     * @see Statistic
     */
    public Statistic statistic() {
        return Statistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link Statistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
     * @see Statistic
     */
    public String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The value against which the specified statistic is compared.
     * </p>
     * 
     * @return The value against which the specified statistic is compared.
     */
    public Double threshold() {
        return threshold;
    }

    /**
     * <p>
     * The unit of measure associated with the CloudWatch metric being watched. The value specified for
     * <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link Unit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of measure associated with the CloudWatch metric being watched. The value specified for
     *         <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
     * @see Unit
     */
    public Unit unit() {
        return Unit.fromValue(unit);
    }

    /**
     * <p>
     * The unit of measure associated with the CloudWatch metric being watched. The value specified for
     * <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link Unit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit of measure associated with the CloudWatch metric being watched. The value specified for
     *         <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
     * @see Unit
     */
    public String unitAsString() {
        return unit;
    }

    /**
     * Returns true if the Dimensions property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A CloudWatch metric dimension.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasDimensions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A CloudWatch metric dimension.
     */
    public List<MetricDimension> dimensions() {
        return dimensions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(evaluationPeriods());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(threshold());
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dimensions());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CloudWatchAlarmDefinition)) {
            return false;
        }
        CloudWatchAlarmDefinition other = (CloudWatchAlarmDefinition) obj;
        return Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString())
                && Objects.equals(evaluationPeriods(), other.evaluationPeriods())
                && Objects.equals(metricName(), other.metricName()) && Objects.equals(namespace(), other.namespace())
                && Objects.equals(period(), other.period()) && Objects.equals(statisticAsString(), other.statisticAsString())
                && Objects.equals(threshold(), other.threshold()) && Objects.equals(unitAsString(), other.unitAsString())
                && Objects.equals(dimensions(), other.dimensions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CloudWatchAlarmDefinition").add("ComparisonOperator", comparisonOperatorAsString())
                .add("EvaluationPeriods", evaluationPeriods()).add("MetricName", metricName()).add("Namespace", namespace())
                .add("Period", period()).add("Statistic", statisticAsString()).add("Threshold", threshold())
                .add("Unit", unitAsString()).add("Dimensions", dimensions()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        case "EvaluationPeriods":
            return Optional.ofNullable(clazz.cast(evaluationPeriods()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "Period":
            return Optional.ofNullable(clazz.cast(period()));
        case "Statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "Threshold":
            return Optional.ofNullable(clazz.cast(threshold()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CloudWatchAlarmDefinition, T> g) {
        return obj -> g.apply((CloudWatchAlarmDefinition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CloudWatchAlarmDefinition> {
        /**
         * <p>
         * Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
         * <code>Threshold</code>.
         * </p>
         * 
         * @param comparisonOperator
         *        Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
         *        <code>Threshold</code>.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
         * <code>Threshold</code>.
         * </p>
         * 
         * @param comparisonOperator
         *        Determines how the metric specified by <code>MetricName</code> is compared to the value specified by
         *        <code>Threshold</code>.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);

        /**
         * <p>
         * The number of periods, in five-minute increments, during which the alarm condition must exist before the
         * alarm triggers automatic scaling activity. The default value is <code>1</code>.
         * </p>
         * 
         * @param evaluationPeriods
         *        The number of periods, in five-minute increments, during which the alarm condition must exist before
         *        the alarm triggers automatic scaling activity. The default value is <code>1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationPeriods(Integer evaluationPeriods);

        /**
         * <p>
         * The name of the CloudWatch metric that is watched to determine an alarm condition.
         * </p>
         * 
         * @param metricName
         *        The name of the CloudWatch metric that is watched to determine an alarm condition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The namespace for the CloudWatch metric. The default is <code>AWS/ElasticMapReduce</code>.
         * </p>
         * 
         * @param namespace
         *        The namespace for the CloudWatch metric. The default is <code>AWS/ElasticMapReduce</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The period, in seconds, over which the statistic is applied. EMR CloudWatch metrics are emitted every five
         * minutes (300 seconds), so if an EMR CloudWatch metric is specified, specify <code>300</code>.
         * </p>
         * 
         * @param period
         *        The period, in seconds, over which the statistic is applied. EMR CloudWatch metrics are emitted every
         *        five minutes (300 seconds), so if an EMR CloudWatch metric is specified, specify <code>300</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
         * @see Statistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Statistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
         * </p>
         * 
         * @param statistic
         *        The statistic to apply to the metric associated with the alarm. The default is <code>AVERAGE</code>.
         * @see Statistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Statistic
         */
        Builder statistic(Statistic statistic);

        /**
         * <p>
         * The value against which the specified statistic is compared.
         * </p>
         * 
         * @param threshold
         *        The value against which the specified statistic is compared.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threshold(Double threshold);

        /**
         * <p>
         * The unit of measure associated with the CloudWatch metric being watched. The value specified for
         * <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
         * </p>
         * 
         * @param unit
         *        The unit of measure associated with the CloudWatch metric being watched. The value specified for
         *        <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
         * @see Unit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Unit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit of measure associated with the CloudWatch metric being watched. The value specified for
         * <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
         * </p>
         * 
         * @param unit
         *        The unit of measure associated with the CloudWatch metric being watched. The value specified for
         *        <code>Unit</code> must correspond to the units specified in the CloudWatch metric.
         * @see Unit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Unit
         */
        Builder unit(Unit unit);

        /**
         * <p>
         * A CloudWatch metric dimension.
         * </p>
         * 
         * @param dimensions
         *        A CloudWatch metric dimension.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<MetricDimension> dimensions);

        /**
         * <p>
         * A CloudWatch metric dimension.
         * </p>
         * 
         * @param dimensions
         *        A CloudWatch metric dimension.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(MetricDimension... dimensions);

        /**
         * <p>
         * A CloudWatch metric dimension.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MetricDimension>.Builder} avoiding the need
         * to create one manually via {@link List<MetricDimension>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MetricDimension>.Builder#build()} is called immediately and
         * its result is passed to {@link #dimensions(List<MetricDimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on {@link List<MetricDimension>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(List<MetricDimension>)
         */
        Builder dimensions(Consumer<MetricDimension.Builder>... dimensions);
    }

    static final class BuilderImpl implements Builder {
        private String comparisonOperator;

        private Integer evaluationPeriods;

        private String metricName;

        private String namespace;

        private Integer period;

        private String statistic;

        private Double threshold;

        private String unit;

        private List<MetricDimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CloudWatchAlarmDefinition model) {
            comparisonOperator(model.comparisonOperator);
            evaluationPeriods(model.evaluationPeriods);
            metricName(model.metricName);
            namespace(model.namespace);
            period(model.period);
            statistic(model.statistic);
            threshold(model.threshold);
            unit(model.unit);
            dimensions(model.dimensions);
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        @Override
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        public final Integer getEvaluationPeriods() {
            return evaluationPeriods;
        }

        @Override
        public final Builder evaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
            return this;
        }

        public final void setEvaluationPeriods(Integer evaluationPeriods) {
            this.evaluationPeriods = evaluationPeriods;
        }

        public final String getMetricName() {
            return metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        public final String getNamespace() {
            return namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        public final Integer getPeriod() {
            return period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        public final String getStatistic() {
            return statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(Statistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        public final Double getThreshold() {
            return threshold;
        }

        @Override
        public final Builder threshold(Double threshold) {
            this.threshold = threshold;
            return this;
        }

        public final void setThreshold(Double threshold) {
            this.threshold = threshold;
        }

        public final String getUnit() {
            return unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(Unit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        public final Collection<MetricDimension.Builder> getDimensions() {
            return dimensions != null ? dimensions.stream().map(MetricDimension::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder dimensions(Collection<MetricDimension> dimensions) {
            this.dimensions = MetricDimensionListCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(MetricDimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<MetricDimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> MetricDimension.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setDimensions(Collection<MetricDimension.BuilderImpl> dimensions) {
            this.dimensions = MetricDimensionListCopier.copyFromBuilder(dimensions);
        }

        @Override
        public CloudWatchAlarmDefinition build() {
            return new CloudWatchAlarmDefinition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
