/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An automatic scaling policy for a core instance group or task instance group in an Amazon EMR cluster. An automatic
 * scaling policy defines how an instance group dynamically adds and terminates EC2 instances in response to the value
 * of a CloudWatch metric. See <a>PutAutoScalingPolicy</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingPolicy.Builder, AutoScalingPolicy> {
    private static final SdkField<ScalingConstraints> CONSTRAINTS_FIELD = SdkField
            .<ScalingConstraints> builder(MarshallingType.SDK_POJO).getter(getter(AutoScalingPolicy::constraints))
            .setter(setter(Builder::constraints)).constructor(ScalingConstraints::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Constraints").build()).build();

    private static final SdkField<List<ScalingRule>> RULES_FIELD = SdkField
            .<List<ScalingRule>> builder(MarshallingType.LIST)
            .getter(getter(AutoScalingPolicy::rules))
            .setter(setter(Builder::rules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Rules").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ScalingRule> builder(MarshallingType.SDK_POJO)
                                            .constructor(ScalingRule::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(CONSTRAINTS_FIELD, RULES_FIELD));

    private static final long serialVersionUID = 1L;

    private final ScalingConstraints constraints;

    private final List<ScalingRule> rules;

    private AutoScalingPolicy(BuilderImpl builder) {
        this.constraints = builder.constraints;
        this.rules = builder.rules;
    }

    /**
     * <p>
     * The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity will not
     * cause an instance group to grow above or below these limits.
     * </p>
     * 
     * @return The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity will
     *         not cause an instance group to grow above or below these limits.
     */
    public ScalingConstraints constraints() {
        return constraints;
    }

    /**
     * Returns true if the Rules property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRules() {
        return rules != null && !(rules instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The scale-in and scale-out rules that comprise the automatic scaling policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRules()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The scale-in and scale-out rules that comprise the automatic scaling policy.
     */
    public List<ScalingRule> rules() {
        return rules;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(constraints());
        hashCode = 31 * hashCode + Objects.hashCode(rules());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingPolicy)) {
            return false;
        }
        AutoScalingPolicy other = (AutoScalingPolicy) obj;
        return Objects.equals(constraints(), other.constraints()) && Objects.equals(rules(), other.rules());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AutoScalingPolicy").add("Constraints", constraints()).add("Rules", rules()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Constraints":
            return Optional.ofNullable(clazz.cast(constraints()));
        case "Rules":
            return Optional.ofNullable(clazz.cast(rules()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingPolicy, T> g) {
        return obj -> g.apply((AutoScalingPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingPolicy> {
        /**
         * <p>
         * The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity will not
         * cause an instance group to grow above or below these limits.
         * </p>
         * 
         * @param constraints
         *        The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity
         *        will not cause an instance group to grow above or below these limits.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder constraints(ScalingConstraints constraints);

        /**
         * <p>
         * The upper and lower EC2 instance limits for an automatic scaling policy. Automatic scaling activity will not
         * cause an instance group to grow above or below these limits.
         * </p>
         * This is a convenience that creates an instance of the {@link ScalingConstraints.Builder} avoiding the need to
         * create one manually via {@link ScalingConstraints#builder()}.
         *
         * When the {@link Consumer} completes, {@link ScalingConstraints.Builder#build()} is called immediately and its
         * result is passed to {@link #constraints(ScalingConstraints)}.
         * 
         * @param constraints
         *        a consumer that will call methods on {@link ScalingConstraints.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #constraints(ScalingConstraints)
         */
        default Builder constraints(Consumer<ScalingConstraints.Builder> constraints) {
            return constraints(ScalingConstraints.builder().applyMutation(constraints).build());
        }

        /**
         * <p>
         * The scale-in and scale-out rules that comprise the automatic scaling policy.
         * </p>
         * 
         * @param rules
         *        The scale-in and scale-out rules that comprise the automatic scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(Collection<ScalingRule> rules);

        /**
         * <p>
         * The scale-in and scale-out rules that comprise the automatic scaling policy.
         * </p>
         * 
         * @param rules
         *        The scale-in and scale-out rules that comprise the automatic scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rules(ScalingRule... rules);

        /**
         * <p>
         * The scale-in and scale-out rules that comprise the automatic scaling policy.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ScalingRule>.Builder} avoiding the need to
         * create one manually via {@link List<ScalingRule>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ScalingRule>.Builder#build()} is called immediately and its
         * result is passed to {@link #rules(List<ScalingRule>)}.
         * 
         * @param rules
         *        a consumer that will call methods on {@link List<ScalingRule>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rules(List<ScalingRule>)
         */
        Builder rules(Consumer<ScalingRule.Builder>... rules);
    }

    static final class BuilderImpl implements Builder {
        private ScalingConstraints constraints;

        private List<ScalingRule> rules = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingPolicy model) {
            constraints(model.constraints);
            rules(model.rules);
        }

        public final ScalingConstraints.Builder getConstraints() {
            return constraints != null ? constraints.toBuilder() : null;
        }

        @Override
        public final Builder constraints(ScalingConstraints constraints) {
            this.constraints = constraints;
            return this;
        }

        public final void setConstraints(ScalingConstraints.BuilderImpl constraints) {
            this.constraints = constraints != null ? constraints.build() : null;
        }

        public final Collection<ScalingRule.Builder> getRules() {
            return rules != null ? rules.stream().map(ScalingRule::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder rules(Collection<ScalingRule> rules) {
            this.rules = ScalingRuleListCopier.copy(rules);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(ScalingRule... rules) {
            rules(Arrays.asList(rules));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rules(Consumer<ScalingRule.Builder>... rules) {
            rules(Stream.of(rules).map(c -> ScalingRule.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRules(Collection<ScalingRule.BuilderImpl> rules) {
            this.rules = ScalingRuleListCopier.copyFromBuilder(rules);
        }

        @Override
        public AutoScalingPolicy build() {
            return new AutoScalingPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
