/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * With Amazon EMR release version 4.0 and later, the only accepted parameter is the application name. To pass arguments
 * to applications, you use configuration classifications specified using configuration JSON objects. For more
 * information, see <a href="https://docs.aws.amazon.com/emr/latest/ReleaseGuide/emr-configure-apps.html">Configuring
 * Applications</a>.
 * </p>
 * <p>
 * With earlier Amazon EMR releases, the application is any Amazon or third-party software that you can add to the
 * cluster. This structure contains a list of strings that indicates the software to use with the cluster and accepts a
 * user argument list. Amazon EMR accepts and forwards the argument list to the corresponding installation script as
 * bootstrap action argument.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Application implements SdkPojo, Serializable, ToCopyableBuilder<Application.Builder, Application> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Application::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Application::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<List<String>> ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Application::args))
            .setter(setter(Builder::args))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Args").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, String>> ADDITIONAL_INFO_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Application::additionalInfo))
            .setter(setter(Builder::additionalInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalInfo").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, VERSION_FIELD,
            ARGS_FIELD, ADDITIONAL_INFO_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String version;

    private final List<String> args;

    private final Map<String, String> additionalInfo;

    private Application(BuilderImpl builder) {
        this.name = builder.name;
        this.version = builder.version;
        this.args = builder.args;
        this.additionalInfo = builder.additionalInfo;
    }

    /**
     * <p>
     * The name of the application.
     * </p>
     * 
     * @return The name of the application.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The version of the application.
     * </p>
     * 
     * @return The version of the application.
     */
    public String version() {
        return version;
    }

    /**
     * Returns true if the Args property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasArgs() {
        return args != null && !(args instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Arguments for Amazon EMR to pass to the application.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasArgs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Arguments for Amazon EMR to pass to the application.
     */
    public List<String> args() {
        return args;
    }

    /**
     * Returns true if the AdditionalInfo property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAdditionalInfo() {
        return additionalInfo != null && !(additionalInfo instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * This option is for advanced users only. This is meta information about third-party applications that third-party
     * vendors use for testing purposes.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAdditionalInfo()} to see if a value was sent in this field.
     * </p>
     * 
     * @return This option is for advanced users only. This is meta information about third-party applications that
     *         third-party vendors use for testing purposes.
     */
    public Map<String, String> additionalInfo() {
        return additionalInfo;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(args());
        hashCode = 31 * hashCode + Objects.hashCode(additionalInfo());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Application)) {
            return false;
        }
        Application other = (Application) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(version(), other.version())
                && Objects.equals(args(), other.args()) && Objects.equals(additionalInfo(), other.additionalInfo());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Application").add("Name", name()).add("Version", version()).add("Args", args())
                .add("AdditionalInfo", additionalInfo()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "Args":
            return Optional.ofNullable(clazz.cast(args()));
        case "AdditionalInfo":
            return Optional.ofNullable(clazz.cast(additionalInfo()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Application, T> g) {
        return obj -> g.apply((Application) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Application> {
        /**
         * <p>
         * The name of the application.
         * </p>
         * 
         * @param name
         *        The name of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The version of the application.
         * </p>
         * 
         * @param version
         *        The version of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * Arguments for Amazon EMR to pass to the application.
         * </p>
         * 
         * @param args
         *        Arguments for Amazon EMR to pass to the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(Collection<String> args);

        /**
         * <p>
         * Arguments for Amazon EMR to pass to the application.
         * </p>
         * 
         * @param args
         *        Arguments for Amazon EMR to pass to the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(String... args);

        /**
         * <p>
         * This option is for advanced users only. This is meta information about third-party applications that
         * third-party vendors use for testing purposes.
         * </p>
         * 
         * @param additionalInfo
         *        This option is for advanced users only. This is meta information about third-party applications that
         *        third-party vendors use for testing purposes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInfo(Map<String, String> additionalInfo);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String version;

        private List<String> args = DefaultSdkAutoConstructList.getInstance();

        private Map<String, String> additionalInfo = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Application model) {
            name(model.name);
            version(model.version);
            args(model.args);
            additionalInfo(model.additionalInfo);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final Collection<String> getArgs() {
            return args;
        }

        @Override
        public final Builder args(Collection<String> args) {
            this.args = StringListCopier.copy(args);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder args(String... args) {
            args(Arrays.asList(args));
            return this;
        }

        public final void setArgs(Collection<String> args) {
            this.args = StringListCopier.copy(args);
        }

        public final Map<String, String> getAdditionalInfo() {
            return additionalInfo;
        }

        @Override
        public final Builder additionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
            return this;
        }

        public final void setAdditionalInfo(Map<String, String> additionalInfo) {
            this.additionalInfo = StringMapCopier.copy(additionalInfo);
        }

        @Override
        public Application build() {
            return new Application(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
