/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The launch specification for Spot instances in the instance fleet, which determines the defined duration and
 * provisioning timeout behavior.
 * </p>
 * <note>
 * <p>
 * The instance fleet configuration is available only in Amazon EMR versions 4.8.0 and later, excluding 5.0.x versions.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpotProvisioningSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<SpotProvisioningSpecification.Builder, SpotProvisioningSpecification> {
    private static final SdkField<Integer> TIMEOUT_DURATION_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(SpotProvisioningSpecification::timeoutDurationMinutes))
            .setter(setter(Builder::timeoutDurationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutDurationMinutes").build())
            .build();

    private static final SdkField<String> TIMEOUT_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(SpotProvisioningSpecification::timeoutActionAsString)).setter(setter(Builder::timeoutAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeoutAction").build()).build();

    private static final SdkField<Integer> BLOCK_DURATION_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(SpotProvisioningSpecification::blockDurationMinutes)).setter(setter(Builder::blockDurationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockDurationMinutes").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            TIMEOUT_DURATION_MINUTES_FIELD, TIMEOUT_ACTION_FIELD, BLOCK_DURATION_MINUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer timeoutDurationMinutes;

    private final String timeoutAction;

    private final Integer blockDurationMinutes;

    private SpotProvisioningSpecification(BuilderImpl builder) {
        this.timeoutDurationMinutes = builder.timeoutDurationMinutes;
        this.timeoutAction = builder.timeoutAction;
        this.blockDurationMinutes = builder.blockDurationMinutes;
    }

    /**
     * <p>
     * The spot provisioning timeout period in minutes. If Spot instances are not provisioned within this time period,
     * the <code>TimeOutAction</code> is taken. Minimum value is 5 and maximum value is 1440. The timeout applies only
     * during initial provisioning, when the cluster is first created.
     * </p>
     * 
     * @return The spot provisioning timeout period in minutes. If Spot instances are not provisioned within this time
     *         period, the <code>TimeOutAction</code> is taken. Minimum value is 5 and maximum value is 1440. The
     *         timeout applies only during initial provisioning, when the cluster is first created.
     */
    public Integer timeoutDurationMinutes() {
        return timeoutDurationMinutes;
    }

    /**
     * <p>
     * The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
     * <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be provisioned within
     * the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
     * <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are available,
     * On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeoutAction}
     * will return {@link SpotProvisioningTimeoutAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #timeoutActionAsString}.
     * </p>
     * 
     * @return The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
     *         <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be
     *         provisioned within the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
     *         <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are available,
     *         On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
     * @see SpotProvisioningTimeoutAction
     */
    public SpotProvisioningTimeoutAction timeoutAction() {
        return SpotProvisioningTimeoutAction.fromValue(timeoutAction);
    }

    /**
     * <p>
     * The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
     * <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be provisioned within
     * the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
     * <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are available,
     * On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #timeoutAction}
     * will return {@link SpotProvisioningTimeoutAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #timeoutActionAsString}.
     * </p>
     * 
     * @return The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
     *         <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be
     *         provisioned within the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
     *         <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are available,
     *         On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
     * @see SpotProvisioningTimeoutAction
     */
    public String timeoutActionAsString() {
        return timeoutAction;
    }

    /**
     * <p>
     * The defined duration for Spot instances (also known as Spot blocks) in minutes. When specified, the Spot instance
     * does not terminate before the defined duration expires, and defined duration pricing for Spot instances applies.
     * Valid values are 60, 120, 180, 240, 300, or 360. The duration period starts as soon as a Spot instance receives
     * its instance ID. At the end of the duration, Amazon EC2 marks the Spot instance for termination and provides a
     * Spot instance termination notice, which gives the instance a two-minute warning before it terminates.
     * </p>
     * 
     * @return The defined duration for Spot instances (also known as Spot blocks) in minutes. When specified, the Spot
     *         instance does not terminate before the defined duration expires, and defined duration pricing for Spot
     *         instances applies. Valid values are 60, 120, 180, 240, 300, or 360. The duration period starts as soon as
     *         a Spot instance receives its instance ID. At the end of the duration, Amazon EC2 marks the Spot instance
     *         for termination and provides a Spot instance termination notice, which gives the instance a two-minute
     *         warning before it terminates.
     */
    public Integer blockDurationMinutes() {
        return blockDurationMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(timeoutDurationMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(timeoutActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(blockDurationMinutes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpotProvisioningSpecification)) {
            return false;
        }
        SpotProvisioningSpecification other = (SpotProvisioningSpecification) obj;
        return Objects.equals(timeoutDurationMinutes(), other.timeoutDurationMinutes())
                && Objects.equals(timeoutActionAsString(), other.timeoutActionAsString())
                && Objects.equals(blockDurationMinutes(), other.blockDurationMinutes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("SpotProvisioningSpecification").add("TimeoutDurationMinutes", timeoutDurationMinutes())
                .add("TimeoutAction", timeoutActionAsString()).add("BlockDurationMinutes", blockDurationMinutes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TimeoutDurationMinutes":
            return Optional.ofNullable(clazz.cast(timeoutDurationMinutes()));
        case "TimeoutAction":
            return Optional.ofNullable(clazz.cast(timeoutActionAsString()));
        case "BlockDurationMinutes":
            return Optional.ofNullable(clazz.cast(blockDurationMinutes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpotProvisioningSpecification, T> g) {
        return obj -> g.apply((SpotProvisioningSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpotProvisioningSpecification> {
        /**
         * <p>
         * The spot provisioning timeout period in minutes. If Spot instances are not provisioned within this time
         * period, the <code>TimeOutAction</code> is taken. Minimum value is 5 and maximum value is 1440. The timeout
         * applies only during initial provisioning, when the cluster is first created.
         * </p>
         * 
         * @param timeoutDurationMinutes
         *        The spot provisioning timeout period in minutes. If Spot instances are not provisioned within this
         *        time period, the <code>TimeOutAction</code> is taken. Minimum value is 5 and maximum value is 1440.
         *        The timeout applies only during initial provisioning, when the cluster is first created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeoutDurationMinutes(Integer timeoutDurationMinutes);

        /**
         * <p>
         * The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
         * <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be provisioned
         * within the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
         * <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are available,
         * On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
         * </p>
         * 
         * @param timeoutAction
         *        The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
         *        <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be
         *        provisioned within the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
         *        <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are
         *        available, On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
         * @see SpotProvisioningTimeoutAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotProvisioningTimeoutAction
         */
        Builder timeoutAction(String timeoutAction);

        /**
         * <p>
         * The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
         * <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be provisioned
         * within the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
         * <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are available,
         * On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
         * </p>
         * 
         * @param timeoutAction
         *        The action to take when <code>TargetSpotCapacity</code> has not been fulfilled when the
         *        <code>TimeoutDurationMinutes</code> has expired; that is, when all Spot instances could not be
         *        provisioned within the Spot provisioning timeout. Valid values are <code>TERMINATE_CLUSTER</code> and
         *        <code>SWITCH_TO_ON_DEMAND</code>. SWITCH_TO_ON_DEMAND specifies that if no Spot instances are
         *        available, On-Demand Instances should be provisioned to fulfill any remaining Spot capacity.
         * @see SpotProvisioningTimeoutAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotProvisioningTimeoutAction
         */
        Builder timeoutAction(SpotProvisioningTimeoutAction timeoutAction);

        /**
         * <p>
         * The defined duration for Spot instances (also known as Spot blocks) in minutes. When specified, the Spot
         * instance does not terminate before the defined duration expires, and defined duration pricing for Spot
         * instances applies. Valid values are 60, 120, 180, 240, 300, or 360. The duration period starts as soon as a
         * Spot instance receives its instance ID. At the end of the duration, Amazon EC2 marks the Spot instance for
         * termination and provides a Spot instance termination notice, which gives the instance a two-minute warning
         * before it terminates.
         * </p>
         * 
         * @param blockDurationMinutes
         *        The defined duration for Spot instances (also known as Spot blocks) in minutes. When specified, the
         *        Spot instance does not terminate before the defined duration expires, and defined duration pricing for
         *        Spot instances applies. Valid values are 60, 120, 180, 240, 300, or 360. The duration period starts as
         *        soon as a Spot instance receives its instance ID. At the end of the duration, Amazon EC2 marks the
         *        Spot instance for termination and provides a Spot instance termination notice, which gives the
         *        instance a two-minute warning before it terminates.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDurationMinutes(Integer blockDurationMinutes);
    }

    static final class BuilderImpl implements Builder {
        private Integer timeoutDurationMinutes;

        private String timeoutAction;

        private Integer blockDurationMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(SpotProvisioningSpecification model) {
            timeoutDurationMinutes(model.timeoutDurationMinutes);
            timeoutAction(model.timeoutAction);
            blockDurationMinutes(model.blockDurationMinutes);
        }

        public final Integer getTimeoutDurationMinutes() {
            return timeoutDurationMinutes;
        }

        @Override
        public final Builder timeoutDurationMinutes(Integer timeoutDurationMinutes) {
            this.timeoutDurationMinutes = timeoutDurationMinutes;
            return this;
        }

        public final void setTimeoutDurationMinutes(Integer timeoutDurationMinutes) {
            this.timeoutDurationMinutes = timeoutDurationMinutes;
        }

        public final String getTimeoutActionAsString() {
            return timeoutAction;
        }

        @Override
        public final Builder timeoutAction(String timeoutAction) {
            this.timeoutAction = timeoutAction;
            return this;
        }

        @Override
        public final Builder timeoutAction(SpotProvisioningTimeoutAction timeoutAction) {
            this.timeoutAction(timeoutAction == null ? null : timeoutAction.toString());
            return this;
        }

        public final void setTimeoutAction(String timeoutAction) {
            this.timeoutAction = timeoutAction;
        }

        public final Integer getBlockDurationMinutes() {
            return blockDurationMinutes;
        }

        @Override
        public final Builder blockDurationMinutes(Integer blockDurationMinutes) {
            this.blockDurationMinutes = blockDurationMinutes;
            return this;
        }

        public final void setBlockDurationMinutes(Integer blockDurationMinutes) {
            this.blockDurationMinutes = blockDurationMinutes;
        }

        @Override
        public SpotProvisioningSpecification build() {
            return new SpotProvisioningSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
