/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A configuration for Amazon EMR block public access. When <code>BlockPublicSecurityGroupRules</code> is set to
 * <code>true</code>, Amazon EMR prevents cluster creation if one of the cluster's security groups has a rule that
 * allows inbound traffic from 0.0.0.0/0 or ::/0 on a port, unless the port is specified as an exception using
 * <code>PermittedPublicSecurityGroupRuleRanges</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BlockPublicAccessConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<BlockPublicAccessConfiguration.Builder, BlockPublicAccessConfiguration> {
    private static final SdkField<Boolean> BLOCK_PUBLIC_SECURITY_GROUP_RULES_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(BlockPublicAccessConfiguration::blockPublicSecurityGroupRules))
            .setter(setter(Builder::blockPublicSecurityGroupRules))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockPublicSecurityGroupRules")
                    .build()).build();

    private static final SdkField<List<PortRange>> PERMITTED_PUBLIC_SECURITY_GROUP_RULE_RANGES_FIELD = SdkField
            .<List<PortRange>> builder(MarshallingType.LIST)
            .getter(getter(BlockPublicAccessConfiguration::permittedPublicSecurityGroupRuleRanges))
            .setter(setter(Builder::permittedPublicSecurityGroupRuleRanges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("PermittedPublicSecurityGroupRuleRanges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PortRange> builder(MarshallingType.SDK_POJO)
                                            .constructor(PortRange::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            BLOCK_PUBLIC_SECURITY_GROUP_RULES_FIELD, PERMITTED_PUBLIC_SECURITY_GROUP_RULE_RANGES_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean blockPublicSecurityGroupRules;

    private final List<PortRange> permittedPublicSecurityGroupRuleRanges;

    private BlockPublicAccessConfiguration(BuilderImpl builder) {
        this.blockPublicSecurityGroupRules = builder.blockPublicSecurityGroupRules;
        this.permittedPublicSecurityGroupRuleRanges = builder.permittedPublicSecurityGroupRuleRanges;
    }

    /**
     * <p>
     * Indicates whether EMR block public access is enabled (<code>true</code>) or disabled (<code>false</code>). By
     * default, the value is <code>false</code> for accounts that have created EMR clusters before July 2019. For
     * accounts created after this, the default is <code>true</code>.
     * </p>
     * 
     * @return Indicates whether EMR block public access is enabled (<code>true</code>) or disabled (<code>false</code>
     *         ). By default, the value is <code>false</code> for accounts that have created EMR clusters before July
     *         2019. For accounts created after this, the default is <code>true</code>.
     */
    public Boolean blockPublicSecurityGroupRules() {
        return blockPublicSecurityGroupRules;
    }

    /**
     * Returns true if the PermittedPublicSecurityGroupRuleRanges property was specified by the sender (it may be
     * empty), or false if the sender did not specify the value (it will be empty). For responses returned by the SDK,
     * the sender is the AWS service.
     */
    public boolean hasPermittedPublicSecurityGroupRuleRanges() {
        return permittedPublicSecurityGroupRuleRanges != null
                && !(permittedPublicSecurityGroupRuleRanges instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Specifies ports and port ranges that are permitted to have security group rules that allow inbound traffic from
     * all public sources. For example, if Port 23 (Telnet) is specified for
     * <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security group
     * associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4 0.0.0.0/0 or IPv6 port
     * ::/0 as the source.
     * </p>
     * <p>
     * By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
     * <code>PermittedPublicSecurityGroupRuleRanges</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPermittedPublicSecurityGroupRuleRanges()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Specifies ports and port ranges that are permitted to have security group rules that allow inbound
     *         traffic from all public sources. For example, if Port 23 (Telnet) is specified for
     *         <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security
     *         group associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4 0.0.0.0/0
     *         or IPv6 port ::/0 as the source.</p>
     *         <p>
     *         By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
     *         <code>PermittedPublicSecurityGroupRuleRanges</code>.
     */
    public List<PortRange> permittedPublicSecurityGroupRuleRanges() {
        return permittedPublicSecurityGroupRuleRanges;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(blockPublicSecurityGroupRules());
        hashCode = 31 * hashCode + Objects.hashCode(permittedPublicSecurityGroupRuleRanges());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BlockPublicAccessConfiguration)) {
            return false;
        }
        BlockPublicAccessConfiguration other = (BlockPublicAccessConfiguration) obj;
        return Objects.equals(blockPublicSecurityGroupRules(), other.blockPublicSecurityGroupRules())
                && Objects.equals(permittedPublicSecurityGroupRuleRanges(), other.permittedPublicSecurityGroupRuleRanges());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("BlockPublicAccessConfiguration")
                .add("BlockPublicSecurityGroupRules", blockPublicSecurityGroupRules())
                .add("PermittedPublicSecurityGroupRuleRanges", permittedPublicSecurityGroupRuleRanges()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "BlockPublicSecurityGroupRules":
            return Optional.ofNullable(clazz.cast(blockPublicSecurityGroupRules()));
        case "PermittedPublicSecurityGroupRuleRanges":
            return Optional.ofNullable(clazz.cast(permittedPublicSecurityGroupRuleRanges()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BlockPublicAccessConfiguration, T> g) {
        return obj -> g.apply((BlockPublicAccessConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BlockPublicAccessConfiguration> {
        /**
         * <p>
         * Indicates whether EMR block public access is enabled (<code>true</code>) or disabled (<code>false</code>). By
         * default, the value is <code>false</code> for accounts that have created EMR clusters before July 2019. For
         * accounts created after this, the default is <code>true</code>.
         * </p>
         * 
         * @param blockPublicSecurityGroupRules
         *        Indicates whether EMR block public access is enabled (<code>true</code>) or disabled (
         *        <code>false</code>). By default, the value is <code>false</code> for accounts that have created EMR
         *        clusters before July 2019. For accounts created after this, the default is <code>true</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockPublicSecurityGroupRules(Boolean blockPublicSecurityGroupRules);

        /**
         * <p>
         * Specifies ports and port ranges that are permitted to have security group rules that allow inbound traffic
         * from all public sources. For example, if Port 23 (Telnet) is specified for
         * <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security group
         * associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4 0.0.0.0/0 or IPv6
         * port ::/0 as the source.
         * </p>
         * <p>
         * By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
         * <code>PermittedPublicSecurityGroupRuleRanges</code>.
         * </p>
         * 
         * @param permittedPublicSecurityGroupRuleRanges
         *        Specifies ports and port ranges that are permitted to have security group rules that allow inbound
         *        traffic from all public sources. For example, if Port 23 (Telnet) is specified for
         *        <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security
         *        group associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4
         *        0.0.0.0/0 or IPv6 port ::/0 as the source.</p>
         *        <p>
         *        By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
         *        <code>PermittedPublicSecurityGroupRuleRanges</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permittedPublicSecurityGroupRuleRanges(Collection<PortRange> permittedPublicSecurityGroupRuleRanges);

        /**
         * <p>
         * Specifies ports and port ranges that are permitted to have security group rules that allow inbound traffic
         * from all public sources. For example, if Port 23 (Telnet) is specified for
         * <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security group
         * associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4 0.0.0.0/0 or IPv6
         * port ::/0 as the source.
         * </p>
         * <p>
         * By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
         * <code>PermittedPublicSecurityGroupRuleRanges</code>.
         * </p>
         * 
         * @param permittedPublicSecurityGroupRuleRanges
         *        Specifies ports and port ranges that are permitted to have security group rules that allow inbound
         *        traffic from all public sources. For example, if Port 23 (Telnet) is specified for
         *        <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security
         *        group associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4
         *        0.0.0.0/0 or IPv6 port ::/0 as the source.</p>
         *        <p>
         *        By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
         *        <code>PermittedPublicSecurityGroupRuleRanges</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder permittedPublicSecurityGroupRuleRanges(PortRange... permittedPublicSecurityGroupRuleRanges);

        /**
         * <p>
         * Specifies ports and port ranges that are permitted to have security group rules that allow inbound traffic
         * from all public sources. For example, if Port 23 (Telnet) is specified for
         * <code>PermittedPublicSecurityGroupRuleRanges</code>, Amazon EMR allows cluster creation if a security group
         * associated with the cluster has a rule that allows inbound traffic on Port 23 from IPv4 0.0.0.0/0 or IPv6
         * port ::/0 as the source.
         * </p>
         * <p>
         * By default, Port 22, which is used for SSH access to the cluster EC2 instances, is in the list of
         * <code>PermittedPublicSecurityGroupRuleRanges</code>.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PortRange>.Builder} avoiding the need to
         * create one manually via {@link List<PortRange>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PortRange>.Builder#build()} is called immediately and its
         * result is passed to {@link #permittedPublicSecurityGroupRuleRanges(List<PortRange>)}.
         * 
         * @param permittedPublicSecurityGroupRuleRanges
         *        a consumer that will call methods on {@link List<PortRange>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #permittedPublicSecurityGroupRuleRanges(List<PortRange>)
         */
        Builder permittedPublicSecurityGroupRuleRanges(Consumer<PortRange.Builder>... permittedPublicSecurityGroupRuleRanges);
    }

    static final class BuilderImpl implements Builder {
        private Boolean blockPublicSecurityGroupRules;

        private List<PortRange> permittedPublicSecurityGroupRuleRanges = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(BlockPublicAccessConfiguration model) {
            blockPublicSecurityGroupRules(model.blockPublicSecurityGroupRules);
            permittedPublicSecurityGroupRuleRanges(model.permittedPublicSecurityGroupRuleRanges);
        }

        public final Boolean getBlockPublicSecurityGroupRules() {
            return blockPublicSecurityGroupRules;
        }

        @Override
        public final Builder blockPublicSecurityGroupRules(Boolean blockPublicSecurityGroupRules) {
            this.blockPublicSecurityGroupRules = blockPublicSecurityGroupRules;
            return this;
        }

        public final void setBlockPublicSecurityGroupRules(Boolean blockPublicSecurityGroupRules) {
            this.blockPublicSecurityGroupRules = blockPublicSecurityGroupRules;
        }

        public final Collection<PortRange.Builder> getPermittedPublicSecurityGroupRuleRanges() {
            return permittedPublicSecurityGroupRuleRanges != null ? permittedPublicSecurityGroupRuleRanges.stream()
                    .map(PortRange::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder permittedPublicSecurityGroupRuleRanges(Collection<PortRange> permittedPublicSecurityGroupRuleRanges) {
            this.permittedPublicSecurityGroupRuleRanges = PortRangesCopier.copy(permittedPublicSecurityGroupRuleRanges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permittedPublicSecurityGroupRuleRanges(PortRange... permittedPublicSecurityGroupRuleRanges) {
            permittedPublicSecurityGroupRuleRanges(Arrays.asList(permittedPublicSecurityGroupRuleRanges));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder permittedPublicSecurityGroupRuleRanges(
                Consumer<PortRange.Builder>... permittedPublicSecurityGroupRuleRanges) {
            permittedPublicSecurityGroupRuleRanges(Stream.of(permittedPublicSecurityGroupRuleRanges)
                    .map(c -> PortRange.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPermittedPublicSecurityGroupRuleRanges(
                Collection<PortRange.BuilderImpl> permittedPublicSecurityGroupRuleRanges) {
            this.permittedPublicSecurityGroupRuleRanges = PortRangesCopier
                    .copyFromBuilder(permittedPublicSecurityGroupRuleRanges);
        }

        @Override
        public BlockPublicAccessConfiguration build() {
            return new BlockPublicAccessConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
