/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticsearch.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * History of the last 10 Upgrades and Upgrade Eligibility Checks.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpgradeHistory implements SdkPojo, Serializable, ToCopyableBuilder<UpgradeHistory.Builder, UpgradeHistory> {
    private static final SdkField<String> UPGRADE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpgradeName").getter(getter(UpgradeHistory::upgradeName)).setter(setter(Builder::upgradeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeName").build()).build();

    private static final SdkField<Instant> START_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTimestamp").getter(getter(UpgradeHistory::startTimestamp)).setter(setter(Builder::startTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTimestamp").build()).build();

    private static final SdkField<String> UPGRADE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("UpgradeStatus").getter(getter(UpgradeHistory::upgradeStatusAsString))
            .setter(setter(Builder::upgradeStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UpgradeStatus").build()).build();

    private static final SdkField<List<UpgradeStepItem>> STEPS_LIST_FIELD = SdkField
            .<List<UpgradeStepItem>> builder(MarshallingType.LIST)
            .memberName("StepsList")
            .getter(getter(UpgradeHistory::stepsList))
            .setter(setter(Builder::stepsList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StepsList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UpgradeStepItem> builder(MarshallingType.SDK_POJO)
                                            .constructor(UpgradeStepItem::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(UPGRADE_NAME_FIELD,
            START_TIMESTAMP_FIELD, UPGRADE_STATUS_FIELD, STEPS_LIST_FIELD));

    private static final long serialVersionUID = 1L;

    private final String upgradeName;

    private final Instant startTimestamp;

    private final String upgradeStatus;

    private final List<UpgradeStepItem> stepsList;

    private UpgradeHistory(BuilderImpl builder) {
        this.upgradeName = builder.upgradeName;
        this.startTimestamp = builder.startTimestamp;
        this.upgradeStatus = builder.upgradeStatus;
        this.stepsList = builder.stepsList;
    }

    /**
     * <p>
     * A string that describes the update briefly
     * </p>
     * 
     * @return A string that describes the update briefly
     */
    public final String upgradeName() {
        return upgradeName;
    }

    /**
     * <p>
     * UTC Timestamp at which the Upgrade API call was made in "yyyy-MM-ddTHH:mm:ssZ" format.
     * </p>
     * 
     * @return UTC Timestamp at which the Upgrade API call was made in "yyyy-MM-ddTHH:mm:ssZ" format.
     */
    public final Instant startTimestamp() {
        return startTimestamp;
    }

    /**
     * <p>
     * The overall status of the update. The status can take one of the following values:
     * <ul>
     * <li>In Progress</li>
     * <li>Succeeded</li>
     * <li>Succeeded with Issues</li>
     * <li>Failed</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #upgradeStatus}
     * will return {@link UpgradeStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #upgradeStatusAsString}.
     * </p>
     * 
     * @return The overall status of the update. The status can take one of the following values:
     *         <ul>
     *         <li>In Progress</li>
     *         <li>Succeeded</li>
     *         <li>Succeeded with Issues</li>
     *         <li>Failed</li>
     *         </ul>
     * @see UpgradeStatus
     */
    public final UpgradeStatus upgradeStatus() {
        return UpgradeStatus.fromValue(upgradeStatus);
    }

    /**
     * <p>
     * The overall status of the update. The status can take one of the following values:
     * <ul>
     * <li>In Progress</li>
     * <li>Succeeded</li>
     * <li>Succeeded with Issues</li>
     * <li>Failed</li>
     * </ul>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #upgradeStatus}
     * will return {@link UpgradeStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #upgradeStatusAsString}.
     * </p>
     * 
     * @return The overall status of the update. The status can take one of the following values:
     *         <ul>
     *         <li>In Progress</li>
     *         <li>Succeeded</li>
     *         <li>Succeeded with Issues</li>
     *         <li>Failed</li>
     *         </ul>
     * @see UpgradeStatus
     */
    public final String upgradeStatusAsString() {
        return upgradeStatus;
    }

    /**
     * Returns true if the StepsList property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasStepsList() {
        return stepsList != null && !(stepsList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed as pard of a
     * specific Upgrade or Upgrade Eligibility Check.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasStepsList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed as
     *         pard of a specific Upgrade or Upgrade Eligibility Check.
     */
    public final List<UpgradeStepItem> stepsList() {
        return stepsList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(upgradeName());
        hashCode = 31 * hashCode + Objects.hashCode(startTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(upgradeStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasStepsList() ? stepsList() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpgradeHistory)) {
            return false;
        }
        UpgradeHistory other = (UpgradeHistory) obj;
        return Objects.equals(upgradeName(), other.upgradeName()) && Objects.equals(startTimestamp(), other.startTimestamp())
                && Objects.equals(upgradeStatusAsString(), other.upgradeStatusAsString())
                && hasStepsList() == other.hasStepsList() && Objects.equals(stepsList(), other.stepsList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpgradeHistory").add("UpgradeName", upgradeName()).add("StartTimestamp", startTimestamp())
                .add("UpgradeStatus", upgradeStatusAsString()).add("StepsList", hasStepsList() ? stepsList() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "UpgradeName":
            return Optional.ofNullable(clazz.cast(upgradeName()));
        case "StartTimestamp":
            return Optional.ofNullable(clazz.cast(startTimestamp()));
        case "UpgradeStatus":
            return Optional.ofNullable(clazz.cast(upgradeStatusAsString()));
        case "StepsList":
            return Optional.ofNullable(clazz.cast(stepsList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpgradeHistory, T> g) {
        return obj -> g.apply((UpgradeHistory) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UpgradeHistory> {
        /**
         * <p>
         * A string that describes the update briefly
         * </p>
         * 
         * @param upgradeName
         *        A string that describes the update briefly
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder upgradeName(String upgradeName);

        /**
         * <p>
         * UTC Timestamp at which the Upgrade API call was made in "yyyy-MM-ddTHH:mm:ssZ" format.
         * </p>
         * 
         * @param startTimestamp
         *        UTC Timestamp at which the Upgrade API call was made in "yyyy-MM-ddTHH:mm:ssZ" format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTimestamp(Instant startTimestamp);

        /**
         * <p>
         * The overall status of the update. The status can take one of the following values:
         * <ul>
         * <li>In Progress</li>
         * <li>Succeeded</li>
         * <li>Succeeded with Issues</li>
         * <li>Failed</li>
         * </ul>
         * </p>
         * 
         * @param upgradeStatus
         *        The overall status of the update. The status can take one of the following values:
         *        <ul>
         *        <li>In Progress</li>
         *        <li>Succeeded</li>
         *        <li>Succeeded with Issues</li>
         *        <li>Failed</li>
         *        </ul>
         * @see UpgradeStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpgradeStatus
         */
        Builder upgradeStatus(String upgradeStatus);

        /**
         * <p>
         * The overall status of the update. The status can take one of the following values:
         * <ul>
         * <li>In Progress</li>
         * <li>Succeeded</li>
         * <li>Succeeded with Issues</li>
         * <li>Failed</li>
         * </ul>
         * </p>
         * 
         * @param upgradeStatus
         *        The overall status of the update. The status can take one of the following values:
         *        <ul>
         *        <li>In Progress</li>
         *        <li>Succeeded</li>
         *        <li>Succeeded with Issues</li>
         *        <li>Failed</li>
         *        </ul>
         * @see UpgradeStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UpgradeStatus
         */
        Builder upgradeStatus(UpgradeStatus upgradeStatus);

        /**
         * <p>
         * A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed as pard
         * of a specific Upgrade or Upgrade Eligibility Check.
         * </p>
         * 
         * @param stepsList
         *        A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed
         *        as pard of a specific Upgrade or Upgrade Eligibility Check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepsList(Collection<UpgradeStepItem> stepsList);

        /**
         * <p>
         * A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed as pard
         * of a specific Upgrade or Upgrade Eligibility Check.
         * </p>
         * 
         * @param stepsList
         *        A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed
         *        as pard of a specific Upgrade or Upgrade Eligibility Check.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stepsList(UpgradeStepItem... stepsList);

        /**
         * <p>
         * A list of <code> <a>UpgradeStepItem</a> </code> s representing information about each step performed as pard
         * of a specific Upgrade or Upgrade Eligibility Check.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UpgradeStepItem>.Builder} avoiding the need
         * to create one manually via {@link List<UpgradeStepItem>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UpgradeStepItem>.Builder#build()} is called immediately and
         * its result is passed to {@link #stepsList(List<UpgradeStepItem>)}.
         * 
         * @param stepsList
         *        a consumer that will call methods on {@link List<UpgradeStepItem>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stepsList(List<UpgradeStepItem>)
         */
        Builder stepsList(Consumer<UpgradeStepItem.Builder>... stepsList);
    }

    static final class BuilderImpl implements Builder {
        private String upgradeName;

        private Instant startTimestamp;

        private String upgradeStatus;

        private List<UpgradeStepItem> stepsList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(UpgradeHistory model) {
            upgradeName(model.upgradeName);
            startTimestamp(model.startTimestamp);
            upgradeStatus(model.upgradeStatus);
            stepsList(model.stepsList);
        }

        public final String getUpgradeName() {
            return upgradeName;
        }

        @Override
        public final Builder upgradeName(String upgradeName) {
            this.upgradeName = upgradeName;
            return this;
        }

        public final void setUpgradeName(String upgradeName) {
            this.upgradeName = upgradeName;
        }

        public final Instant getStartTimestamp() {
            return startTimestamp;
        }

        @Override
        public final Builder startTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
            return this;
        }

        public final void setStartTimestamp(Instant startTimestamp) {
            this.startTimestamp = startTimestamp;
        }

        public final String getUpgradeStatus() {
            return upgradeStatus;
        }

        @Override
        public final Builder upgradeStatus(String upgradeStatus) {
            this.upgradeStatus = upgradeStatus;
            return this;
        }

        @Override
        public final Builder upgradeStatus(UpgradeStatus upgradeStatus) {
            this.upgradeStatus(upgradeStatus == null ? null : upgradeStatus.toString());
            return this;
        }

        public final void setUpgradeStatus(String upgradeStatus) {
            this.upgradeStatus = upgradeStatus;
        }

        public final Collection<UpgradeStepItem.Builder> getStepsList() {
            if (stepsList instanceof SdkAutoConstructList) {
                return null;
            }
            return stepsList != null ? stepsList.stream().map(UpgradeStepItem::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder stepsList(Collection<UpgradeStepItem> stepsList) {
            this.stepsList = UpgradeStepsListCopier.copy(stepsList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepsList(UpgradeStepItem... stepsList) {
            stepsList(Arrays.asList(stepsList));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder stepsList(Consumer<UpgradeStepItem.Builder>... stepsList) {
            stepsList(Stream.of(stepsList).map(c -> UpgradeStepItem.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setStepsList(Collection<UpgradeStepItem.BuilderImpl> stepsList) {
            this.stepsList = UpgradeStepsListCopier.copyFromBuilder(stepsList);
        }

        @Override
        public UpgradeHistory build() {
            return new UpgradeHistory(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
