/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticsearch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Options to enable, disable, and specify the properties of EBS storage volumes. For more information, see <a href=
 * "http://docs.aws.amazon.com/elasticsearch-service/latest/developerguide/es-createupdatedomains.html#es-createdomain-configure-ebs"
 * target="_blank"> Configuring EBS-based Storage</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EBSOptions implements SdkPojo, Serializable, ToCopyableBuilder<EBSOptions.Builder, EBSOptions> {
    private static final SdkField<Boolean> EBS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(EBSOptions::ebsEnabled)).setter(setter(Builder::ebsEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EBSEnabled").build()).build();

    private static final SdkField<String> VOLUME_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(EBSOptions::volumeTypeAsString)).setter(setter(Builder::volumeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeType").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EBSOptions::volumeSize)).setter(setter(Builder::volumeSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSize").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EBSOptions::iops)).setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EBS_ENABLED_FIELD,
            VOLUME_TYPE_FIELD, VOLUME_SIZE_FIELD, IOPS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean ebsEnabled;

    private final String volumeType;

    private final Integer volumeSize;

    private final Integer iops;

    private EBSOptions(BuilderImpl builder) {
        this.ebsEnabled = builder.ebsEnabled;
        this.volumeType = builder.volumeType;
        this.volumeSize = builder.volumeSize;
        this.iops = builder.iops;
    }

    /**
     * <p>
     * Specifies whether EBS-based storage is enabled.
     * </p>
     * 
     * @return Specifies whether EBS-based storage is enabled.
     */
    public Boolean ebsEnabled() {
        return ebsEnabled;
    }

    /**
     * <p>
     * Specifies the volume type for EBS-based storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return Specifies the volume type for EBS-based storage.
     * @see VolumeType
     */
    public VolumeType volumeType() {
        return VolumeType.fromValue(volumeType);
    }

    /**
     * <p>
     * Specifies the volume type for EBS-based storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return Specifies the volume type for EBS-based storage.
     * @see VolumeType
     */
    public String volumeTypeAsString() {
        return volumeType;
    }

    /**
     * <p>
     * Integer to specify the size of an EBS volume.
     * </p>
     * 
     * @return Integer to specify the size of an EBS volume.
     */
    public Integer volumeSize() {
        return volumeSize;
    }

    /**
     * <p>
     * Specifies the IOPD for a Provisioned IOPS EBS volume (SSD).
     * </p>
     * 
     * @return Specifies the IOPD for a Provisioned IOPS EBS volume (SSD).
     */
    public Integer iops() {
        return iops;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ebsEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(volumeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSize());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EBSOptions)) {
            return false;
        }
        EBSOptions other = (EBSOptions) obj;
        return Objects.equals(ebsEnabled(), other.ebsEnabled())
                && Objects.equals(volumeTypeAsString(), other.volumeTypeAsString())
                && Objects.equals(volumeSize(), other.volumeSize()) && Objects.equals(iops(), other.iops());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("EBSOptions").add("EBSEnabled", ebsEnabled()).add("VolumeType", volumeTypeAsString())
                .add("VolumeSize", volumeSize()).add("Iops", iops()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EBSEnabled":
            return Optional.ofNullable(clazz.cast(ebsEnabled()));
        case "VolumeType":
            return Optional.ofNullable(clazz.cast(volumeTypeAsString()));
        case "VolumeSize":
            return Optional.ofNullable(clazz.cast(volumeSize()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EBSOptions, T> g) {
        return obj -> g.apply((EBSOptions) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EBSOptions> {
        /**
         * <p>
         * Specifies whether EBS-based storage is enabled.
         * </p>
         * 
         * @param ebsEnabled
         *        Specifies whether EBS-based storage is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ebsEnabled(Boolean ebsEnabled);

        /**
         * <p>
         * Specifies the volume type for EBS-based storage.
         * </p>
         * 
         * @param volumeType
         *        Specifies the volume type for EBS-based storage.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * Specifies the volume type for EBS-based storage.
         * </p>
         * 
         * @param volumeType
         *        Specifies the volume type for EBS-based storage.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(VolumeType volumeType);

        /**
         * <p>
         * Integer to specify the size of an EBS volume.
         * </p>
         * 
         * @param volumeSize
         *        Integer to specify the size of an EBS volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSize(Integer volumeSize);

        /**
         * <p>
         * Specifies the IOPD for a Provisioned IOPS EBS volume (SSD).
         * </p>
         * 
         * @param iops
         *        Specifies the IOPD for a Provisioned IOPS EBS volume (SSD).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);
    }

    static final class BuilderImpl implements Builder {
        private Boolean ebsEnabled;

        private String volumeType;

        private Integer volumeSize;

        private Integer iops;

        private BuilderImpl() {
        }

        private BuilderImpl(EBSOptions model) {
            ebsEnabled(model.ebsEnabled);
            volumeType(model.volumeType);
            volumeSize(model.volumeSize);
            iops(model.iops);
        }

        public final Boolean getEbsEnabled() {
            return ebsEnabled;
        }

        @Override
        public final Builder ebsEnabled(Boolean ebsEnabled) {
            this.ebsEnabled = ebsEnabled;
            return this;
        }

        public final void setEbsEnabled(Boolean ebsEnabled) {
            this.ebsEnabled = ebsEnabled;
        }

        public final String getVolumeTypeAsString() {
            return volumeType;
        }

        @Override
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        @Override
        public final Builder volumeType(VolumeType volumeType) {
            this.volumeType(volumeType == null ? null : volumeType.toString());
            return this;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        public final Integer getVolumeSize() {
            return volumeSize;
        }

        @Override
        public final Builder volumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
            return this;
        }

        public final void setVolumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        public EBSOptions build() {
            return new EBSOptions(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
