/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A list of <code>PreferredAvailabilityZones</code> objects that specifies the configuration of a node group in the
 * resharded cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReshardingConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ReshardingConfiguration.Builder, ReshardingConfiguration> {
    private static final SdkField<String> NODE_GROUP_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReshardingConfiguration::nodeGroupId)).setter(setter(Builder::nodeGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NodeGroupId").build()).build();

    private static final SdkField<List<String>> PREFERRED_AVAILABILITY_ZONES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(ReshardingConfiguration::preferredAvailabilityZones))
            .setter(setter(Builder::preferredAvailabilityZones))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PreferredAvailabilityZones").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("AvailabilityZone")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("AvailabilityZone").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NODE_GROUP_ID_FIELD,
            PREFERRED_AVAILABILITY_ZONES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nodeGroupId;

    private final List<String> preferredAvailabilityZones;

    private ReshardingConfiguration(BuilderImpl builder) {
        this.nodeGroupId = builder.nodeGroupId;
        this.preferredAvailabilityZones = builder.preferredAvailabilityZones;
    }

    /**
     * <p>
     * Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these configuration
     * values apply to.
     * </p>
     * 
     * @return Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these
     *         configuration values apply to.
     */
    public String nodeGroupId() {
        return nodeGroupId;
    }

    /**
     * Returns true if the PreferredAvailabilityZones property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasPreferredAvailabilityZones() {
        return preferredAvailabilityZones != null && !(preferredAvailabilityZones instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of preferred availability zones for the nodes in this cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPreferredAvailabilityZones()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of preferred availability zones for the nodes in this cluster.
     */
    public List<String> preferredAvailabilityZones() {
        return preferredAvailabilityZones;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nodeGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(preferredAvailabilityZones());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReshardingConfiguration)) {
            return false;
        }
        ReshardingConfiguration other = (ReshardingConfiguration) obj;
        return Objects.equals(nodeGroupId(), other.nodeGroupId())
                && Objects.equals(preferredAvailabilityZones(), other.preferredAvailabilityZones());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ReshardingConfiguration").add("NodeGroupId", nodeGroupId())
                .add("PreferredAvailabilityZones", preferredAvailabilityZones()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "NodeGroupId":
            return Optional.ofNullable(clazz.cast(nodeGroupId()));
        case "PreferredAvailabilityZones":
            return Optional.ofNullable(clazz.cast(preferredAvailabilityZones()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReshardingConfiguration, T> g) {
        return obj -> g.apply((ReshardingConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReshardingConfiguration> {
        /**
         * <p>
         * Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these
         * configuration values apply to.
         * </p>
         * 
         * @param nodeGroupId
         *        Either the ElastiCache for Redis supplied 4-digit id or a user supplied id for the node group these
         *        configuration values apply to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeGroupId(String nodeGroupId);

        /**
         * <p>
         * A list of preferred availability zones for the nodes in this cluster.
         * </p>
         * 
         * @param preferredAvailabilityZones
         *        A list of preferred availability zones for the nodes in this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZones(Collection<String> preferredAvailabilityZones);

        /**
         * <p>
         * A list of preferred availability zones for the nodes in this cluster.
         * </p>
         * 
         * @param preferredAvailabilityZones
         *        A list of preferred availability zones for the nodes in this cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder preferredAvailabilityZones(String... preferredAvailabilityZones);
    }

    static final class BuilderImpl implements Builder {
        private String nodeGroupId;

        private List<String> preferredAvailabilityZones = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReshardingConfiguration model) {
            nodeGroupId(model.nodeGroupId);
            preferredAvailabilityZones(model.preferredAvailabilityZones);
        }

        public final String getNodeGroupId() {
            return nodeGroupId;
        }

        @Override
        public final Builder nodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
            return this;
        }

        public final void setNodeGroupId(String nodeGroupId) {
            this.nodeGroupId = nodeGroupId;
        }

        public final Collection<String> getPreferredAvailabilityZones() {
            return preferredAvailabilityZones;
        }

        @Override
        public final Builder preferredAvailabilityZones(Collection<String> preferredAvailabilityZones) {
            this.preferredAvailabilityZones = AvailabilityZonesListCopier.copy(preferredAvailabilityZones);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder preferredAvailabilityZones(String... preferredAvailabilityZones) {
            preferredAvailabilityZones(Arrays.asList(preferredAvailabilityZones));
            return this;
        }

        public final void setPreferredAvailabilityZones(Collection<String> preferredAvailabilityZones) {
            this.preferredAvailabilityZones = AvailabilityZonesListCopier.copy(preferredAvailabilityZones);
        }

        @Override
        public ReshardingConfiguration build() {
            return new ReshardingConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
