/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes all of the attributes of a reserved cache node offering.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReservedCacheNodesOffering implements SdkPojo, Serializable,
        ToCopyableBuilder<ReservedCacheNodesOffering.Builder, ReservedCacheNodesOffering> {
    private static final SdkField<String> RESERVED_CACHE_NODES_OFFERING_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNodesOffering::reservedCacheNodesOfferingId))
            .setter(setter(Builder::reservedCacheNodesOfferingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedCacheNodesOfferingId")
                    .build()).build();

    private static final SdkField<String> CACHE_NODE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNodesOffering::cacheNodeType)).setter(setter(Builder::cacheNodeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeType").build()).build();

    private static final SdkField<Integer> DURATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ReservedCacheNodesOffering::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<Double> FIXED_PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(ReservedCacheNodesOffering::fixedPrice)).setter(setter(Builder::fixedPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FixedPrice").build()).build();

    private static final SdkField<Double> USAGE_PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(ReservedCacheNodesOffering::usagePrice)).setter(setter(Builder::usagePrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsagePrice").build()).build();

    private static final SdkField<String> PRODUCT_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNodesOffering::productDescription)).setter(setter(Builder::productDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductDescription").build())
            .build();

    private static final SdkField<String> OFFERING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNodesOffering::offeringType)).setter(setter(Builder::offeringType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OfferingType").build()).build();

    private static final SdkField<List<RecurringCharge>> RECURRING_CHARGES_FIELD = SdkField
            .<List<RecurringCharge>> builder(MarshallingType.LIST)
            .getter(getter(ReservedCacheNodesOffering::recurringCharges))
            .setter(setter(Builder::recurringCharges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecurringCharges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("RecurringCharge")
                            .memberFieldInfo(
                                    SdkField.<RecurringCharge> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecurringCharge::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("RecurringCharge").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            RESERVED_CACHE_NODES_OFFERING_ID_FIELD, CACHE_NODE_TYPE_FIELD, DURATION_FIELD, FIXED_PRICE_FIELD, USAGE_PRICE_FIELD,
            PRODUCT_DESCRIPTION_FIELD, OFFERING_TYPE_FIELD, RECURRING_CHARGES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String reservedCacheNodesOfferingId;

    private final String cacheNodeType;

    private final Integer duration;

    private final Double fixedPrice;

    private final Double usagePrice;

    private final String productDescription;

    private final String offeringType;

    private final List<RecurringCharge> recurringCharges;

    private ReservedCacheNodesOffering(BuilderImpl builder) {
        this.reservedCacheNodesOfferingId = builder.reservedCacheNodesOfferingId;
        this.cacheNodeType = builder.cacheNodeType;
        this.duration = builder.duration;
        this.fixedPrice = builder.fixedPrice;
        this.usagePrice = builder.usagePrice;
        this.productDescription = builder.productDescription;
        this.offeringType = builder.offeringType;
        this.recurringCharges = builder.recurringCharges;
    }

    /**
     * <p>
     * A unique identifier for the reserved cache node offering.
     * </p>
     * 
     * @return A unique identifier for the reserved cache node offering.
     */
    public String reservedCacheNodesOfferingId() {
        return reservedCacheNodesOfferingId;
    }

    /**
     * <p>
     * The cache node type for the reserved cache node.
     * </p>
     * <p>
     * The following node types are supported by ElastiCache. Generally speaking, the current generation types provide
     * more memory and computational power at lower cost when compared to their equivalent previous generation
     * counterparts.
     * </p>
     * <ul>
     * <li>
     * <p>
     * General purpose:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Current generation:
     * </p>
     * <p>
     * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>, <code>cache.m5.2xlarge</code>,
     * <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>, <code>cache.m5.24xlarge</code>
     * </p>
     * <p>
     * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>, <code>cache.m4.2xlarge</code>,
     * <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
     * </p>
     * <p>
     * <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>, <code>cache.t3.medium</code>
     * </p>
     * <p>
     * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Previous generation: (not recommended)
     * </p>
     * <p>
     * <b>T1 node types:</b> <code>cache.t1.micro</code>
     * </p>
     * <p>
     * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
     * <code>cache.m1.xlarge</code>
     * </p>
     * <p>
     * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>,
     * <code>cache.m3.2xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Compute optimized:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Previous generation: (not recommended)
     * </p>
     * <p>
     * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Memory optimized:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Current generation:
     * </p>
     * <p>
     * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>, <code>cache.r5.2xlarge</code>,
     * <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>, <code>cache.r5.24xlarge</code>
     * </p>
     * <p>
     * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>, <code>cache.r4.2xlarge</code>,
     * <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>, <code>cache.r4.16xlarge</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Previous generation: (not recommended)
     * </p>
     * <p>
     * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>, <code>cache.m2.4xlarge</code>
     * </p>
     * <p>
     * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>, <code>cache.r3.2xlarge</code>,
     * <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * <b>Additional node type info</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * All current generation instance types are created in Amazon VPC by default.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redis append-only files (AOF) are not supported for T1 or T2 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redis Multi-AZ with automatic failover is not supported on T1 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Redis
     * version 2.8.22 and later.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The cache node type for the reserved cache node.</p>
     *         <p>
     *         The following node types are supported by ElastiCache. Generally speaking, the current generation types
     *         provide more memory and computational power at lower cost when compared to their equivalent previous
     *         generation counterparts.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         General purpose:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Current generation:
     *         </p>
     *         <p>
     *         <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
     *         <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
     *         <code>cache.m5.24xlarge</code>
     *         </p>
     *         <p>
     *         <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
     *         <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
     *         </p>
     *         <p>
     *         <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
     *         <code>cache.t3.medium</code>
     *         </p>
     *         <p>
     *         <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
     *         <code>cache.t2.medium</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended)
     *         </p>
     *         <p>
     *         <b>T1 node types:</b> <code>cache.t1.micro</code>
     *         </p>
     *         <p>
     *         <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
     *         <code>cache.m1.large</code>, <code>cache.m1.xlarge</code>
     *         </p>
     *         <p>
     *         <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
     *         <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Compute optimized:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended)
     *         </p>
     *         <p>
     *         <b>C1 node types:</b> <code>cache.c1.xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Memory optimized:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Current generation:
     *         </p>
     *         <p>
     *         <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
     *         <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
     *         <code>cache.r5.24xlarge</code>
     *         </p>
     *         <p>
     *         <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
     *         <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
     *         <code>cache.r4.16xlarge</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended)
     *         </p>
     *         <p>
     *         <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
     *         <code>cache.m2.4xlarge</code>
     *         </p>
     *         <p>
     *         <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
     *         <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Additional node type info</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         All current generation instance types are created in Amazon VPC by default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Redis append-only files (AOF) are not supported for T1 or T2 instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Redis Multi-AZ with automatic failover is not supported on T1 instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on
     *         Redis version 2.8.22 and later.
     *         </p>
     *         </li>
     */
    public String cacheNodeType() {
        return cacheNodeType;
    }

    /**
     * <p>
     * The duration of the offering. in seconds.
     * </p>
     * 
     * @return The duration of the offering. in seconds.
     */
    public Integer duration() {
        return duration;
    }

    /**
     * <p>
     * The fixed price charged for this offering.
     * </p>
     * 
     * @return The fixed price charged for this offering.
     */
    public Double fixedPrice() {
        return fixedPrice;
    }

    /**
     * <p>
     * The hourly price charged for this offering.
     * </p>
     * 
     * @return The hourly price charged for this offering.
     */
    public Double usagePrice() {
        return usagePrice;
    }

    /**
     * <p>
     * The cache engine used by the offering.
     * </p>
     * 
     * @return The cache engine used by the offering.
     */
    public String productDescription() {
        return productDescription;
    }

    /**
     * <p>
     * The offering type.
     * </p>
     * 
     * @return The offering type.
     */
    public String offeringType() {
        return offeringType;
    }

    /**
     * Returns true if the RecurringCharges property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRecurringCharges() {
        return recurringCharges != null && !(recurringCharges instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recurring price charged to run this reserved cache node.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecurringCharges()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The recurring price charged to run this reserved cache node.
     */
    public List<RecurringCharge> recurringCharges() {
        return recurringCharges;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reservedCacheNodesOfferingId());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeType());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(fixedPrice());
        hashCode = 31 * hashCode + Objects.hashCode(usagePrice());
        hashCode = 31 * hashCode + Objects.hashCode(productDescription());
        hashCode = 31 * hashCode + Objects.hashCode(offeringType());
        hashCode = 31 * hashCode + Objects.hashCode(recurringCharges());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReservedCacheNodesOffering)) {
            return false;
        }
        ReservedCacheNodesOffering other = (ReservedCacheNodesOffering) obj;
        return Objects.equals(reservedCacheNodesOfferingId(), other.reservedCacheNodesOfferingId())
                && Objects.equals(cacheNodeType(), other.cacheNodeType()) && Objects.equals(duration(), other.duration())
                && Objects.equals(fixedPrice(), other.fixedPrice()) && Objects.equals(usagePrice(), other.usagePrice())
                && Objects.equals(productDescription(), other.productDescription())
                && Objects.equals(offeringType(), other.offeringType())
                && Objects.equals(recurringCharges(), other.recurringCharges());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ReservedCacheNodesOffering").add("ReservedCacheNodesOfferingId", reservedCacheNodesOfferingId())
                .add("CacheNodeType", cacheNodeType()).add("Duration", duration()).add("FixedPrice", fixedPrice())
                .add("UsagePrice", usagePrice()).add("ProductDescription", productDescription())
                .add("OfferingType", offeringType()).add("RecurringCharges", recurringCharges()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReservedCacheNodesOfferingId":
            return Optional.ofNullable(clazz.cast(reservedCacheNodesOfferingId()));
        case "CacheNodeType":
            return Optional.ofNullable(clazz.cast(cacheNodeType()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "FixedPrice":
            return Optional.ofNullable(clazz.cast(fixedPrice()));
        case "UsagePrice":
            return Optional.ofNullable(clazz.cast(usagePrice()));
        case "ProductDescription":
            return Optional.ofNullable(clazz.cast(productDescription()));
        case "OfferingType":
            return Optional.ofNullable(clazz.cast(offeringType()));
        case "RecurringCharges":
            return Optional.ofNullable(clazz.cast(recurringCharges()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReservedCacheNodesOffering, T> g) {
        return obj -> g.apply((ReservedCacheNodesOffering) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReservedCacheNodesOffering> {
        /**
         * <p>
         * A unique identifier for the reserved cache node offering.
         * </p>
         * 
         * @param reservedCacheNodesOfferingId
         *        A unique identifier for the reserved cache node offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedCacheNodesOfferingId(String reservedCacheNodesOfferingId);

        /**
         * <p>
         * The cache node type for the reserved cache node.
         * </p>
         * <p>
         * The following node types are supported by ElastiCache. Generally speaking, the current generation types
         * provide more memory and computational power at lower cost when compared to their equivalent previous
         * generation counterparts.
         * </p>
         * <ul>
         * <li>
         * <p>
         * General purpose:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Current generation:
         * </p>
         * <p>
         * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
         * <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
         * <code>cache.m5.24xlarge</code>
         * </p>
         * <p>
         * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
         * <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
         * </p>
         * <p>
         * <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>, <code>cache.t3.medium</code>
         * </p>
         * <p>
         * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Previous generation: (not recommended)
         * </p>
         * <p>
         * <b>T1 node types:</b> <code>cache.t1.micro</code>
         * </p>
         * <p>
         * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
         * <code>cache.m1.xlarge</code>
         * </p>
         * <p>
         * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Compute optimized:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Previous generation: (not recommended)
         * </p>
         * <p>
         * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Memory optimized:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Current generation:
         * </p>
         * <p>
         * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
         * <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
         * <code>cache.r5.24xlarge</code>
         * </p>
         * <p>
         * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
         * <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
         * <code>cache.r4.16xlarge</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Previous generation: (not recommended)
         * </p>
         * <p>
         * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
         * <code>cache.m2.4xlarge</code>
         * </p>
         * <p>
         * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
         * <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * <b>Additional node type info</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * All current generation instance types are created in Amazon VPC by default.
         * </p>
         * </li>
         * <li>
         * <p>
         * Redis append-only files (AOF) are not supported for T1 or T2 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Redis Multi-AZ with automatic failover is not supported on T1 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Redis
         * version 2.8.22 and later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cacheNodeType
         *        The cache node type for the reserved cache node.</p>
         *        <p>
         *        The following node types are supported by ElastiCache. Generally speaking, the current generation
         *        types provide more memory and computational power at lower cost when compared to their equivalent
         *        previous generation counterparts.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        General purpose:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Current generation:
         *        </p>
         *        <p>
         *        <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
         *        <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
         *        <code>cache.m5.24xlarge</code>
         *        </p>
         *        <p>
         *        <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
         *        <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
         *        </p>
         *        <p>
         *        <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
         *        <code>cache.t3.medium</code>
         *        </p>
         *        <p>
         *        <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
         *        <code>cache.t2.medium</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended)
         *        </p>
         *        <p>
         *        <b>T1 node types:</b> <code>cache.t1.micro</code>
         *        </p>
         *        <p>
         *        <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
         *        <code>cache.m1.large</code>, <code>cache.m1.xlarge</code>
         *        </p>
         *        <p>
         *        <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
         *        <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Compute optimized:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended)
         *        </p>
         *        <p>
         *        <b>C1 node types:</b> <code>cache.c1.xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Memory optimized:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Current generation:
         *        </p>
         *        <p>
         *        <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
         *        <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
         *        <code>cache.r5.24xlarge</code>
         *        </p>
         *        <p>
         *        <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
         *        <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
         *        <code>cache.r4.16xlarge</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended)
         *        </p>
         *        <p>
         *        <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
         *        <code>cache.m2.4xlarge</code>
         *        </p>
         *        <p>
         *        <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
         *        <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Additional node type info</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        All current generation instance types are created in Amazon VPC by default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Redis append-only files (AOF) are not supported for T1 or T2 instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Redis Multi-AZ with automatic failover is not supported on T1 instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported
         *        on Redis version 2.8.22 and later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeType(String cacheNodeType);

        /**
         * <p>
         * The duration of the offering. in seconds.
         * </p>
         * 
         * @param duration
         *        The duration of the offering. in seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Integer duration);

        /**
         * <p>
         * The fixed price charged for this offering.
         * </p>
         * 
         * @param fixedPrice
         *        The fixed price charged for this offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fixedPrice(Double fixedPrice);

        /**
         * <p>
         * The hourly price charged for this offering.
         * </p>
         * 
         * @param usagePrice
         *        The hourly price charged for this offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usagePrice(Double usagePrice);

        /**
         * <p>
         * The cache engine used by the offering.
         * </p>
         * 
         * @param productDescription
         *        The cache engine used by the offering.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productDescription(String productDescription);

        /**
         * <p>
         * The offering type.
         * </p>
         * 
         * @param offeringType
         *        The offering type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder offeringType(String offeringType);

        /**
         * <p>
         * The recurring price charged to run this reserved cache node.
         * </p>
         * 
         * @param recurringCharges
         *        The recurring price charged to run this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCharges(Collection<RecurringCharge> recurringCharges);

        /**
         * <p>
         * The recurring price charged to run this reserved cache node.
         * </p>
         * 
         * @param recurringCharges
         *        The recurring price charged to run this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCharges(RecurringCharge... recurringCharges);

        /**
         * <p>
         * The recurring price charged to run this reserved cache node.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RecurringCharge>.Builder} avoiding the need
         * to create one manually via {@link List<RecurringCharge>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RecurringCharge>.Builder#build()} is called immediately and
         * its result is passed to {@link #recurringCharges(List<RecurringCharge>)}.
         * 
         * @param recurringCharges
         *        a consumer that will call methods on {@link List<RecurringCharge>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recurringCharges(List<RecurringCharge>)
         */
        Builder recurringCharges(Consumer<RecurringCharge.Builder>... recurringCharges);
    }

    static final class BuilderImpl implements Builder {
        private String reservedCacheNodesOfferingId;

        private String cacheNodeType;

        private Integer duration;

        private Double fixedPrice;

        private Double usagePrice;

        private String productDescription;

        private String offeringType;

        private List<RecurringCharge> recurringCharges = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReservedCacheNodesOffering model) {
            reservedCacheNodesOfferingId(model.reservedCacheNodesOfferingId);
            cacheNodeType(model.cacheNodeType);
            duration(model.duration);
            fixedPrice(model.fixedPrice);
            usagePrice(model.usagePrice);
            productDescription(model.productDescription);
            offeringType(model.offeringType);
            recurringCharges(model.recurringCharges);
        }

        public final String getReservedCacheNodesOfferingId() {
            return reservedCacheNodesOfferingId;
        }

        @Override
        public final Builder reservedCacheNodesOfferingId(String reservedCacheNodesOfferingId) {
            this.reservedCacheNodesOfferingId = reservedCacheNodesOfferingId;
            return this;
        }

        public final void setReservedCacheNodesOfferingId(String reservedCacheNodesOfferingId) {
            this.reservedCacheNodesOfferingId = reservedCacheNodesOfferingId;
        }

        public final String getCacheNodeType() {
            return cacheNodeType;
        }

        @Override
        public final Builder cacheNodeType(String cacheNodeType) {
            this.cacheNodeType = cacheNodeType;
            return this;
        }

        public final void setCacheNodeType(String cacheNodeType) {
            this.cacheNodeType = cacheNodeType;
        }

        public final Integer getDuration() {
            return duration;
        }

        @Override
        public final Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public final void setDuration(Integer duration) {
            this.duration = duration;
        }

        public final Double getFixedPrice() {
            return fixedPrice;
        }

        @Override
        public final Builder fixedPrice(Double fixedPrice) {
            this.fixedPrice = fixedPrice;
            return this;
        }

        public final void setFixedPrice(Double fixedPrice) {
            this.fixedPrice = fixedPrice;
        }

        public final Double getUsagePrice() {
            return usagePrice;
        }

        @Override
        public final Builder usagePrice(Double usagePrice) {
            this.usagePrice = usagePrice;
            return this;
        }

        public final void setUsagePrice(Double usagePrice) {
            this.usagePrice = usagePrice;
        }

        public final String getProductDescription() {
            return productDescription;
        }

        @Override
        public final Builder productDescription(String productDescription) {
            this.productDescription = productDescription;
            return this;
        }

        public final void setProductDescription(String productDescription) {
            this.productDescription = productDescription;
        }

        public final String getOfferingType() {
            return offeringType;
        }

        @Override
        public final Builder offeringType(String offeringType) {
            this.offeringType = offeringType;
            return this;
        }

        public final void setOfferingType(String offeringType) {
            this.offeringType = offeringType;
        }

        public final Collection<RecurringCharge.Builder> getRecurringCharges() {
            return recurringCharges != null ? recurringCharges.stream().map(RecurringCharge::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder recurringCharges(Collection<RecurringCharge> recurringCharges) {
            this.recurringCharges = RecurringChargeListCopier.copy(recurringCharges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recurringCharges(RecurringCharge... recurringCharges) {
            recurringCharges(Arrays.asList(recurringCharges));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recurringCharges(Consumer<RecurringCharge.Builder>... recurringCharges) {
            recurringCharges(Stream.of(recurringCharges).map(c -> RecurringCharge.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRecurringCharges(Collection<RecurringCharge.BuilderImpl> recurringCharges) {
            this.recurringCharges = RecurringChargeListCopier.copyFromBuilder(recurringCharges);
        }

        @Override
        public ReservedCacheNodesOffering build() {
            return new ReservedCacheNodesOffering(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
