/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of a <code>PurchaseReservedCacheNodesOffering</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReservedCacheNode implements SdkPojo, Serializable,
        ToCopyableBuilder<ReservedCacheNode.Builder, ReservedCacheNode> {
    private static final SdkField<String> RESERVED_CACHE_NODE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::reservedCacheNodeId)).setter(setter(Builder::reservedCacheNodeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedCacheNodeId").build())
            .build();

    private static final SdkField<String> RESERVED_CACHE_NODES_OFFERING_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::reservedCacheNodesOfferingId))
            .setter(setter(Builder::reservedCacheNodesOfferingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedCacheNodesOfferingId")
                    .build()).build();

    private static final SdkField<String> CACHE_NODE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::cacheNodeType)).setter(setter(Builder::cacheNodeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeType").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(ReservedCacheNode::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Integer> DURATION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ReservedCacheNode::duration)).setter(setter(Builder::duration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Duration").build()).build();

    private static final SdkField<Double> FIXED_PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(ReservedCacheNode::fixedPrice)).setter(setter(Builder::fixedPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FixedPrice").build()).build();

    private static final SdkField<Double> USAGE_PRICE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(ReservedCacheNode::usagePrice)).setter(setter(Builder::usagePrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UsagePrice").build()).build();

    private static final SdkField<Integer> CACHE_NODE_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(ReservedCacheNode::cacheNodeCount)).setter(setter(Builder::cacheNodeCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CacheNodeCount").build()).build();

    private static final SdkField<String> PRODUCT_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::productDescription)).setter(setter(Builder::productDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductDescription").build())
            .build();

    private static final SdkField<String> OFFERING_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::offeringType)).setter(setter(Builder::offeringType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OfferingType").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::state)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<List<RecurringCharge>> RECURRING_CHARGES_FIELD = SdkField
            .<List<RecurringCharge>> builder(MarshallingType.LIST)
            .getter(getter(ReservedCacheNode::recurringCharges))
            .setter(setter(Builder::recurringCharges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecurringCharges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("RecurringCharge")
                            .memberFieldInfo(
                                    SdkField.<RecurringCharge> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecurringCharge::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("RecurringCharge").build()).build()).build()).build();

    private static final SdkField<String> RESERVATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ReservedCacheNode::reservationARN)).setter(setter(Builder::reservationARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationARN").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESERVED_CACHE_NODE_ID_FIELD,
            RESERVED_CACHE_NODES_OFFERING_ID_FIELD, CACHE_NODE_TYPE_FIELD, START_TIME_FIELD, DURATION_FIELD, FIXED_PRICE_FIELD,
            USAGE_PRICE_FIELD, CACHE_NODE_COUNT_FIELD, PRODUCT_DESCRIPTION_FIELD, OFFERING_TYPE_FIELD, STATE_FIELD,
            RECURRING_CHARGES_FIELD, RESERVATION_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String reservedCacheNodeId;

    private final String reservedCacheNodesOfferingId;

    private final String cacheNodeType;

    private final Instant startTime;

    private final Integer duration;

    private final Double fixedPrice;

    private final Double usagePrice;

    private final Integer cacheNodeCount;

    private final String productDescription;

    private final String offeringType;

    private final String state;

    private final List<RecurringCharge> recurringCharges;

    private final String reservationARN;

    private ReservedCacheNode(BuilderImpl builder) {
        this.reservedCacheNodeId = builder.reservedCacheNodeId;
        this.reservedCacheNodesOfferingId = builder.reservedCacheNodesOfferingId;
        this.cacheNodeType = builder.cacheNodeType;
        this.startTime = builder.startTime;
        this.duration = builder.duration;
        this.fixedPrice = builder.fixedPrice;
        this.usagePrice = builder.usagePrice;
        this.cacheNodeCount = builder.cacheNodeCount;
        this.productDescription = builder.productDescription;
        this.offeringType = builder.offeringType;
        this.state = builder.state;
        this.recurringCharges = builder.recurringCharges;
        this.reservationARN = builder.reservationARN;
    }

    /**
     * <p>
     * The unique identifier for the reservation.
     * </p>
     * 
     * @return The unique identifier for the reservation.
     */
    public String reservedCacheNodeId() {
        return reservedCacheNodeId;
    }

    /**
     * <p>
     * The offering identifier.
     * </p>
     * 
     * @return The offering identifier.
     */
    public String reservedCacheNodesOfferingId() {
        return reservedCacheNodesOfferingId;
    }

    /**
     * <p>
     * The cache node type for the reserved cache nodes.
     * </p>
     * <p>
     * The following node types are supported by ElastiCache. Generally speaking, the current generation types provide
     * more memory and computational power at lower cost when compared to their equivalent previous generation
     * counterparts.
     * </p>
     * <ul>
     * <li>
     * <p>
     * General purpose:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Current generation:
     * </p>
     * <p>
     * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>, <code>cache.m5.2xlarge</code>,
     * <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>, <code>cache.m5.24xlarge</code>
     * </p>
     * <p>
     * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>, <code>cache.m4.2xlarge</code>,
     * <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
     * </p>
     * <p>
     * <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>, <code>cache.t3.medium</code>
     * </p>
     * <p>
     * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Previous generation: (not recommended)
     * </p>
     * <p>
     * <b>T1 node types:</b> <code>cache.t1.micro</code>
     * </p>
     * <p>
     * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
     * <code>cache.m1.xlarge</code>
     * </p>
     * <p>
     * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>,
     * <code>cache.m3.2xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Compute optimized:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Previous generation: (not recommended)
     * </p>
     * <p>
     * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Memory optimized:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Current generation:
     * </p>
     * <p>
     * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>, <code>cache.r5.2xlarge</code>,
     * <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>, <code>cache.r5.24xlarge</code>
     * </p>
     * <p>
     * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>, <code>cache.r4.2xlarge</code>,
     * <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>, <code>cache.r4.16xlarge</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * Previous generation: (not recommended)
     * </p>
     * <p>
     * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>, <code>cache.m2.4xlarge</code>
     * </p>
     * <p>
     * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>, <code>cache.r3.2xlarge</code>,
     * <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
     * </p>
     * </li>
     * </ul>
     * </li>
     * </ul>
     * <p>
     * <b>Additional node type info</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * All current generation instance types are created in Amazon VPC by default.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redis append-only files (AOF) are not supported for T1 or T2 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redis Multi-AZ with automatic failover is not supported on T1 instances.
     * </p>
     * </li>
     * <li>
     * <p>
     * Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Redis
     * version 2.8.22 and later.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The cache node type for the reserved cache nodes.</p>
     *         <p>
     *         The following node types are supported by ElastiCache. Generally speaking, the current generation types
     *         provide more memory and computational power at lower cost when compared to their equivalent previous
     *         generation counterparts.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         General purpose:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Current generation:
     *         </p>
     *         <p>
     *         <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
     *         <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
     *         <code>cache.m5.24xlarge</code>
     *         </p>
     *         <p>
     *         <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
     *         <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
     *         </p>
     *         <p>
     *         <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
     *         <code>cache.t3.medium</code>
     *         </p>
     *         <p>
     *         <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
     *         <code>cache.t2.medium</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended)
     *         </p>
     *         <p>
     *         <b>T1 node types:</b> <code>cache.t1.micro</code>
     *         </p>
     *         <p>
     *         <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
     *         <code>cache.m1.large</code>, <code>cache.m1.xlarge</code>
     *         </p>
     *         <p>
     *         <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
     *         <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Compute optimized:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended)
     *         </p>
     *         <p>
     *         <b>C1 node types:</b> <code>cache.c1.xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Memory optimized:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Current generation:
     *         </p>
     *         <p>
     *         <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
     *         <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
     *         <code>cache.r5.24xlarge</code>
     *         </p>
     *         <p>
     *         <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
     *         <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
     *         <code>cache.r4.16xlarge</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Previous generation: (not recommended)
     *         </p>
     *         <p>
     *         <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
     *         <code>cache.m2.4xlarge</code>
     *         </p>
     *         <p>
     *         <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
     *         <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>Additional node type info</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         All current generation instance types are created in Amazon VPC by default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Redis append-only files (AOF) are not supported for T1 or T2 instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Redis Multi-AZ with automatic failover is not supported on T1 instances.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on
     *         Redis version 2.8.22 and later.
     *         </p>
     *         </li>
     */
    public String cacheNodeType() {
        return cacheNodeType;
    }

    /**
     * <p>
     * The time the reservation started.
     * </p>
     * 
     * @return The time the reservation started.
     */
    public Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The duration of the reservation in seconds.
     * </p>
     * 
     * @return The duration of the reservation in seconds.
     */
    public Integer duration() {
        return duration;
    }

    /**
     * <p>
     * The fixed price charged for this reserved cache node.
     * </p>
     * 
     * @return The fixed price charged for this reserved cache node.
     */
    public Double fixedPrice() {
        return fixedPrice;
    }

    /**
     * <p>
     * The hourly price charged for this reserved cache node.
     * </p>
     * 
     * @return The hourly price charged for this reserved cache node.
     */
    public Double usagePrice() {
        return usagePrice;
    }

    /**
     * <p>
     * The number of cache nodes that have been reserved.
     * </p>
     * 
     * @return The number of cache nodes that have been reserved.
     */
    public Integer cacheNodeCount() {
        return cacheNodeCount;
    }

    /**
     * <p>
     * The description of the reserved cache node.
     * </p>
     * 
     * @return The description of the reserved cache node.
     */
    public String productDescription() {
        return productDescription;
    }

    /**
     * <p>
     * The offering type of this reserved cache node.
     * </p>
     * 
     * @return The offering type of this reserved cache node.
     */
    public String offeringType() {
        return offeringType;
    }

    /**
     * <p>
     * The state of the reserved cache node.
     * </p>
     * 
     * @return The state of the reserved cache node.
     */
    public String state() {
        return state;
    }

    /**
     * Returns true if the RecurringCharges property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRecurringCharges() {
        return recurringCharges != null && !(recurringCharges instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The recurring price charged to run this reserved cache node.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecurringCharges()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The recurring price charged to run this reserved cache node.
     */
    public List<RecurringCharge> recurringCharges() {
        return recurringCharges;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the reserved cache node.
     * </p>
     * <p>
     * Example: <code>arn:aws:elasticache:us-east-1:123456789012:reserved-instance:ri-2017-03-27-08-33-25-582</code>
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the reserved cache node.</p>
     *         <p>
     *         Example:
     *         <code>arn:aws:elasticache:us-east-1:123456789012:reserved-instance:ri-2017-03-27-08-33-25-582</code>
     */
    public String reservationARN() {
        return reservationARN;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reservedCacheNodeId());
        hashCode = 31 * hashCode + Objects.hashCode(reservedCacheNodesOfferingId());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeType());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(duration());
        hashCode = 31 * hashCode + Objects.hashCode(fixedPrice());
        hashCode = 31 * hashCode + Objects.hashCode(usagePrice());
        hashCode = 31 * hashCode + Objects.hashCode(cacheNodeCount());
        hashCode = 31 * hashCode + Objects.hashCode(productDescription());
        hashCode = 31 * hashCode + Objects.hashCode(offeringType());
        hashCode = 31 * hashCode + Objects.hashCode(state());
        hashCode = 31 * hashCode + Objects.hashCode(recurringCharges());
        hashCode = 31 * hashCode + Objects.hashCode(reservationARN());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReservedCacheNode)) {
            return false;
        }
        ReservedCacheNode other = (ReservedCacheNode) obj;
        return Objects.equals(reservedCacheNodeId(), other.reservedCacheNodeId())
                && Objects.equals(reservedCacheNodesOfferingId(), other.reservedCacheNodesOfferingId())
                && Objects.equals(cacheNodeType(), other.cacheNodeType()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(duration(), other.duration()) && Objects.equals(fixedPrice(), other.fixedPrice())
                && Objects.equals(usagePrice(), other.usagePrice()) && Objects.equals(cacheNodeCount(), other.cacheNodeCount())
                && Objects.equals(productDescription(), other.productDescription())
                && Objects.equals(offeringType(), other.offeringType()) && Objects.equals(state(), other.state())
                && Objects.equals(recurringCharges(), other.recurringCharges())
                && Objects.equals(reservationARN(), other.reservationARN());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ReservedCacheNode").add("ReservedCacheNodeId", reservedCacheNodeId())
                .add("ReservedCacheNodesOfferingId", reservedCacheNodesOfferingId()).add("CacheNodeType", cacheNodeType())
                .add("StartTime", startTime()).add("Duration", duration()).add("FixedPrice", fixedPrice())
                .add("UsagePrice", usagePrice()).add("CacheNodeCount", cacheNodeCount())
                .add("ProductDescription", productDescription()).add("OfferingType", offeringType()).add("State", state())
                .add("RecurringCharges", recurringCharges()).add("ReservationARN", reservationARN()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReservedCacheNodeId":
            return Optional.ofNullable(clazz.cast(reservedCacheNodeId()));
        case "ReservedCacheNodesOfferingId":
            return Optional.ofNullable(clazz.cast(reservedCacheNodesOfferingId()));
        case "CacheNodeType":
            return Optional.ofNullable(clazz.cast(cacheNodeType()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "Duration":
            return Optional.ofNullable(clazz.cast(duration()));
        case "FixedPrice":
            return Optional.ofNullable(clazz.cast(fixedPrice()));
        case "UsagePrice":
            return Optional.ofNullable(clazz.cast(usagePrice()));
        case "CacheNodeCount":
            return Optional.ofNullable(clazz.cast(cacheNodeCount()));
        case "ProductDescription":
            return Optional.ofNullable(clazz.cast(productDescription()));
        case "OfferingType":
            return Optional.ofNullable(clazz.cast(offeringType()));
        case "State":
            return Optional.ofNullable(clazz.cast(state()));
        case "RecurringCharges":
            return Optional.ofNullable(clazz.cast(recurringCharges()));
        case "ReservationARN":
            return Optional.ofNullable(clazz.cast(reservationARN()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReservedCacheNode, T> g) {
        return obj -> g.apply((ReservedCacheNode) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReservedCacheNode> {
        /**
         * <p>
         * The unique identifier for the reservation.
         * </p>
         * 
         * @param reservedCacheNodeId
         *        The unique identifier for the reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedCacheNodeId(String reservedCacheNodeId);

        /**
         * <p>
         * The offering identifier.
         * </p>
         * 
         * @param reservedCacheNodesOfferingId
         *        The offering identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedCacheNodesOfferingId(String reservedCacheNodesOfferingId);

        /**
         * <p>
         * The cache node type for the reserved cache nodes.
         * </p>
         * <p>
         * The following node types are supported by ElastiCache. Generally speaking, the current generation types
         * provide more memory and computational power at lower cost when compared to their equivalent previous
         * generation counterparts.
         * </p>
         * <ul>
         * <li>
         * <p>
         * General purpose:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Current generation:
         * </p>
         * <p>
         * <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
         * <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
         * <code>cache.m5.24xlarge</code>
         * </p>
         * <p>
         * <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
         * <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
         * </p>
         * <p>
         * <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>, <code>cache.t3.medium</code>
         * </p>
         * <p>
         * <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>, <code>cache.t2.medium</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Previous generation: (not recommended)
         * </p>
         * <p>
         * <b>T1 node types:</b> <code>cache.t1.micro</code>
         * </p>
         * <p>
         * <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>, <code>cache.m1.large</code>,
         * <code>cache.m1.xlarge</code>
         * </p>
         * <p>
         * <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>, <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Compute optimized:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Previous generation: (not recommended)
         * </p>
         * <p>
         * <b>C1 node types:</b> <code>cache.c1.xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * Memory optimized:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Current generation:
         * </p>
         * <p>
         * <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
         * <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
         * <code>cache.r5.24xlarge</code>
         * </p>
         * <p>
         * <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
         * <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
         * <code>cache.r4.16xlarge</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * Previous generation: (not recommended)
         * </p>
         * <p>
         * <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
         * <code>cache.m2.4xlarge</code>
         * </p>
         * <p>
         * <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
         * <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
         * </p>
         * </li>
         * </ul>
         * </li>
         * </ul>
         * <p>
         * <b>Additional node type info</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * All current generation instance types are created in Amazon VPC by default.
         * </p>
         * </li>
         * <li>
         * <p>
         * Redis append-only files (AOF) are not supported for T1 or T2 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Redis Multi-AZ with automatic failover is not supported on T1 instances.
         * </p>
         * </li>
         * <li>
         * <p>
         * Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported on Redis
         * version 2.8.22 and later.
         * </p>
         * </li>
         * </ul>
         * 
         * @param cacheNodeType
         *        The cache node type for the reserved cache nodes.</p>
         *        <p>
         *        The following node types are supported by ElastiCache. Generally speaking, the current generation
         *        types provide more memory and computational power at lower cost when compared to their equivalent
         *        previous generation counterparts.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        General purpose:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Current generation:
         *        </p>
         *        <p>
         *        <b>M5 node types:</b> <code>cache.m5.large</code>, <code>cache.m5.xlarge</code>,
         *        <code>cache.m5.2xlarge</code>, <code>cache.m5.4xlarge</code>, <code>cache.m5.12xlarge</code>,
         *        <code>cache.m5.24xlarge</code>
         *        </p>
         *        <p>
         *        <b>M4 node types:</b> <code>cache.m4.large</code>, <code>cache.m4.xlarge</code>,
         *        <code>cache.m4.2xlarge</code>, <code>cache.m4.4xlarge</code>, <code>cache.m4.10xlarge</code>
         *        </p>
         *        <p>
         *        <b>T3 node types:</b> <code>cache.t3.micro</code>, <code>cache.t3.small</code>,
         *        <code>cache.t3.medium</code>
         *        </p>
         *        <p>
         *        <b>T2 node types:</b> <code>cache.t2.micro</code>, <code>cache.t2.small</code>,
         *        <code>cache.t2.medium</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended)
         *        </p>
         *        <p>
         *        <b>T1 node types:</b> <code>cache.t1.micro</code>
         *        </p>
         *        <p>
         *        <b>M1 node types:</b> <code>cache.m1.small</code>, <code>cache.m1.medium</code>,
         *        <code>cache.m1.large</code>, <code>cache.m1.xlarge</code>
         *        </p>
         *        <p>
         *        <b>M3 node types:</b> <code>cache.m3.medium</code>, <code>cache.m3.large</code>,
         *        <code>cache.m3.xlarge</code>, <code>cache.m3.2xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Compute optimized:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended)
         *        </p>
         *        <p>
         *        <b>C1 node types:</b> <code>cache.c1.xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        Memory optimized:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Current generation:
         *        </p>
         *        <p>
         *        <b>R5 node types:</b> <code>cache.r5.large</code>, <code>cache.r5.xlarge</code>,
         *        <code>cache.r5.2xlarge</code>, <code>cache.r5.4xlarge</code>, <code>cache.r5.12xlarge</code>,
         *        <code>cache.r5.24xlarge</code>
         *        </p>
         *        <p>
         *        <b>R4 node types:</b> <code>cache.r4.large</code>, <code>cache.r4.xlarge</code>,
         *        <code>cache.r4.2xlarge</code>, <code>cache.r4.4xlarge</code>, <code>cache.r4.8xlarge</code>,
         *        <code>cache.r4.16xlarge</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Previous generation: (not recommended)
         *        </p>
         *        <p>
         *        <b>M2 node types:</b> <code>cache.m2.xlarge</code>, <code>cache.m2.2xlarge</code>,
         *        <code>cache.m2.4xlarge</code>
         *        </p>
         *        <p>
         *        <b>R3 node types:</b> <code>cache.r3.large</code>, <code>cache.r3.xlarge</code>,
         *        <code>cache.r3.2xlarge</code>, <code>cache.r3.4xlarge</code>, <code>cache.r3.8xlarge</code>
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>Additional node type info</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        All current generation instance types are created in Amazon VPC by default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Redis append-only files (AOF) are not supported for T1 or T2 instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Redis Multi-AZ with automatic failover is not supported on T1 instances.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Redis configuration variables <code>appendonly</code> and <code>appendfsync</code> are not supported
         *        on Redis version 2.8.22 and later.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeType(String cacheNodeType);

        /**
         * <p>
         * The time the reservation started.
         * </p>
         * 
         * @param startTime
         *        The time the reservation started.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The duration of the reservation in seconds.
         * </p>
         * 
         * @param duration
         *        The duration of the reservation in seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder duration(Integer duration);

        /**
         * <p>
         * The fixed price charged for this reserved cache node.
         * </p>
         * 
         * @param fixedPrice
         *        The fixed price charged for this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder fixedPrice(Double fixedPrice);

        /**
         * <p>
         * The hourly price charged for this reserved cache node.
         * </p>
         * 
         * @param usagePrice
         *        The hourly price charged for this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder usagePrice(Double usagePrice);

        /**
         * <p>
         * The number of cache nodes that have been reserved.
         * </p>
         * 
         * @param cacheNodeCount
         *        The number of cache nodes that have been reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cacheNodeCount(Integer cacheNodeCount);

        /**
         * <p>
         * The description of the reserved cache node.
         * </p>
         * 
         * @param productDescription
         *        The description of the reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder productDescription(String productDescription);

        /**
         * <p>
         * The offering type of this reserved cache node.
         * </p>
         * 
         * @param offeringType
         *        The offering type of this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder offeringType(String offeringType);

        /**
         * <p>
         * The state of the reserved cache node.
         * </p>
         * 
         * @param state
         *        The state of the reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder state(String state);

        /**
         * <p>
         * The recurring price charged to run this reserved cache node.
         * </p>
         * 
         * @param recurringCharges
         *        The recurring price charged to run this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCharges(Collection<RecurringCharge> recurringCharges);

        /**
         * <p>
         * The recurring price charged to run this reserved cache node.
         * </p>
         * 
         * @param recurringCharges
         *        The recurring price charged to run this reserved cache node.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCharges(RecurringCharge... recurringCharges);

        /**
         * <p>
         * The recurring price charged to run this reserved cache node.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RecurringCharge>.Builder} avoiding the need
         * to create one manually via {@link List<RecurringCharge>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RecurringCharge>.Builder#build()} is called immediately and
         * its result is passed to {@link #recurringCharges(List<RecurringCharge>)}.
         * 
         * @param recurringCharges
         *        a consumer that will call methods on {@link List<RecurringCharge>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recurringCharges(List<RecurringCharge>)
         */
        Builder recurringCharges(Consumer<RecurringCharge.Builder>... recurringCharges);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the reserved cache node.
         * </p>
         * <p>
         * Example: <code>arn:aws:elasticache:us-east-1:123456789012:reserved-instance:ri-2017-03-27-08-33-25-582</code>
         * </p>
         * 
         * @param reservationARN
         *        The Amazon Resource Name (ARN) of the reserved cache node.</p>
         *        <p>
         *        Example:
         *        <code>arn:aws:elasticache:us-east-1:123456789012:reserved-instance:ri-2017-03-27-08-33-25-582</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationARN(String reservationARN);
    }

    static final class BuilderImpl implements Builder {
        private String reservedCacheNodeId;

        private String reservedCacheNodesOfferingId;

        private String cacheNodeType;

        private Instant startTime;

        private Integer duration;

        private Double fixedPrice;

        private Double usagePrice;

        private Integer cacheNodeCount;

        private String productDescription;

        private String offeringType;

        private String state;

        private List<RecurringCharge> recurringCharges = DefaultSdkAutoConstructList.getInstance();

        private String reservationARN;

        private BuilderImpl() {
        }

        private BuilderImpl(ReservedCacheNode model) {
            reservedCacheNodeId(model.reservedCacheNodeId);
            reservedCacheNodesOfferingId(model.reservedCacheNodesOfferingId);
            cacheNodeType(model.cacheNodeType);
            startTime(model.startTime);
            duration(model.duration);
            fixedPrice(model.fixedPrice);
            usagePrice(model.usagePrice);
            cacheNodeCount(model.cacheNodeCount);
            productDescription(model.productDescription);
            offeringType(model.offeringType);
            state(model.state);
            recurringCharges(model.recurringCharges);
            reservationARN(model.reservationARN);
        }

        public final String getReservedCacheNodeId() {
            return reservedCacheNodeId;
        }

        @Override
        public final Builder reservedCacheNodeId(String reservedCacheNodeId) {
            this.reservedCacheNodeId = reservedCacheNodeId;
            return this;
        }

        public final void setReservedCacheNodeId(String reservedCacheNodeId) {
            this.reservedCacheNodeId = reservedCacheNodeId;
        }

        public final String getReservedCacheNodesOfferingId() {
            return reservedCacheNodesOfferingId;
        }

        @Override
        public final Builder reservedCacheNodesOfferingId(String reservedCacheNodesOfferingId) {
            this.reservedCacheNodesOfferingId = reservedCacheNodesOfferingId;
            return this;
        }

        public final void setReservedCacheNodesOfferingId(String reservedCacheNodesOfferingId) {
            this.reservedCacheNodesOfferingId = reservedCacheNodesOfferingId;
        }

        public final String getCacheNodeType() {
            return cacheNodeType;
        }

        @Override
        public final Builder cacheNodeType(String cacheNodeType) {
            this.cacheNodeType = cacheNodeType;
            return this;
        }

        public final void setCacheNodeType(String cacheNodeType) {
            this.cacheNodeType = cacheNodeType;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        public final Integer getDuration() {
            return duration;
        }

        @Override
        public final Builder duration(Integer duration) {
            this.duration = duration;
            return this;
        }

        public final void setDuration(Integer duration) {
            this.duration = duration;
        }

        public final Double getFixedPrice() {
            return fixedPrice;
        }

        @Override
        public final Builder fixedPrice(Double fixedPrice) {
            this.fixedPrice = fixedPrice;
            return this;
        }

        public final void setFixedPrice(Double fixedPrice) {
            this.fixedPrice = fixedPrice;
        }

        public final Double getUsagePrice() {
            return usagePrice;
        }

        @Override
        public final Builder usagePrice(Double usagePrice) {
            this.usagePrice = usagePrice;
            return this;
        }

        public final void setUsagePrice(Double usagePrice) {
            this.usagePrice = usagePrice;
        }

        public final Integer getCacheNodeCount() {
            return cacheNodeCount;
        }

        @Override
        public final Builder cacheNodeCount(Integer cacheNodeCount) {
            this.cacheNodeCount = cacheNodeCount;
            return this;
        }

        public final void setCacheNodeCount(Integer cacheNodeCount) {
            this.cacheNodeCount = cacheNodeCount;
        }

        public final String getProductDescription() {
            return productDescription;
        }

        @Override
        public final Builder productDescription(String productDescription) {
            this.productDescription = productDescription;
            return this;
        }

        public final void setProductDescription(String productDescription) {
            this.productDescription = productDescription;
        }

        public final String getOfferingType() {
            return offeringType;
        }

        @Override
        public final Builder offeringType(String offeringType) {
            this.offeringType = offeringType;
            return this;
        }

        public final void setOfferingType(String offeringType) {
            this.offeringType = offeringType;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final Collection<RecurringCharge.Builder> getRecurringCharges() {
            return recurringCharges != null ? recurringCharges.stream().map(RecurringCharge::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder recurringCharges(Collection<RecurringCharge> recurringCharges) {
            this.recurringCharges = RecurringChargeListCopier.copy(recurringCharges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recurringCharges(RecurringCharge... recurringCharges) {
            recurringCharges(Arrays.asList(recurringCharges));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recurringCharges(Consumer<RecurringCharge.Builder>... recurringCharges) {
            recurringCharges(Stream.of(recurringCharges).map(c -> RecurringCharge.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setRecurringCharges(Collection<RecurringCharge.BuilderImpl> recurringCharges) {
            this.recurringCharges = RecurringChargeListCopier.copyFromBuilder(recurringCharges);
        }

        public final String getReservationARN() {
            return reservationARN;
        }

        @Override
        public final Builder reservationARN(String reservationARN) {
            this.reservationARN = reservationARN;
            return this;
        }

        public final void setReservationARN(String reservationARN) {
            this.reservationARN = reservationARN;
        }

        @Override
        public ReservedCacheNode build() {
            return new ReservedCacheNode(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
