/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an Amazon EKS managed node group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Nodegroup implements SdkPojo, Serializable, ToCopyableBuilder<Nodegroup.Builder, Nodegroup> {
    private static final SdkField<String> NODEGROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::nodegroupName)).setter(setter(Builder::nodegroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodegroupName").build()).build();

    private static final SdkField<String> NODEGROUP_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::nodegroupArn)).setter(setter(Builder::nodegroupArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodegroupArn").build()).build();

    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clusterName").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> RELEASE_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::releaseVersion)).setter(setter(Builder::releaseVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("releaseVersion").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Nodegroup::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<Instant> MODIFIED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Nodegroup::modifiedAt)).setter(setter(Builder::modifiedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modifiedAt").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<NodegroupScalingConfig> SCALING_CONFIG_FIELD = SdkField
            .<NodegroupScalingConfig> builder(MarshallingType.SDK_POJO).getter(getter(Nodegroup::scalingConfig))
            .setter(setter(Builder::scalingConfig)).constructor(NodegroupScalingConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scalingConfig").build()).build();

    private static final SdkField<List<String>> INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Nodegroup::instanceTypes))
            .setter(setter(Builder::instanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> SUBNETS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(Nodegroup::subnets))
            .setter(setter(Builder::subnets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("subnets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RemoteAccessConfig> REMOTE_ACCESS_FIELD = SdkField
            .<RemoteAccessConfig> builder(MarshallingType.SDK_POJO).getter(getter(Nodegroup::remoteAccess))
            .setter(setter(Builder::remoteAccess)).constructor(RemoteAccessConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("remoteAccess").build()).build();

    private static final SdkField<String> AMI_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::amiTypeAsString)).setter(setter(Builder::amiType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("amiType").build()).build();

    private static final SdkField<String> NODE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Nodegroup::nodeRole)).setter(setter(Builder::nodeRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("nodeRole").build()).build();

    private static final SdkField<Map<String, String>> LABELS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Nodegroup::labels))
            .setter(setter(Builder::labels))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("labels").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<NodegroupResources> RESOURCES_FIELD = SdkField
            .<NodegroupResources> builder(MarshallingType.SDK_POJO).getter(getter(Nodegroup::resources))
            .setter(setter(Builder::resources)).constructor(NodegroupResources::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resources").build()).build();

    private static final SdkField<Integer> DISK_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Nodegroup::diskSize)).setter(setter(Builder::diskSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("diskSize").build()).build();

    private static final SdkField<NodegroupHealth> HEALTH_FIELD = SdkField.<NodegroupHealth> builder(MarshallingType.SDK_POJO)
            .getter(getter(Nodegroup::health)).setter(setter(Builder::health)).constructor(NodegroupHealth::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("health").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Nodegroup::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NODEGROUP_NAME_FIELD,
            NODEGROUP_ARN_FIELD, CLUSTER_NAME_FIELD, VERSION_FIELD, RELEASE_VERSION_FIELD, CREATED_AT_FIELD, MODIFIED_AT_FIELD,
            STATUS_FIELD, SCALING_CONFIG_FIELD, INSTANCE_TYPES_FIELD, SUBNETS_FIELD, REMOTE_ACCESS_FIELD, AMI_TYPE_FIELD,
            NODE_ROLE_FIELD, LABELS_FIELD, RESOURCES_FIELD, DISK_SIZE_FIELD, HEALTH_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String nodegroupName;

    private final String nodegroupArn;

    private final String clusterName;

    private final String version;

    private final String releaseVersion;

    private final Instant createdAt;

    private final Instant modifiedAt;

    private final String status;

    private final NodegroupScalingConfig scalingConfig;

    private final List<String> instanceTypes;

    private final List<String> subnets;

    private final RemoteAccessConfig remoteAccess;

    private final String amiType;

    private final String nodeRole;

    private final Map<String, String> labels;

    private final NodegroupResources resources;

    private final Integer diskSize;

    private final NodegroupHealth health;

    private final Map<String, String> tags;

    private Nodegroup(BuilderImpl builder) {
        this.nodegroupName = builder.nodegroupName;
        this.nodegroupArn = builder.nodegroupArn;
        this.clusterName = builder.clusterName;
        this.version = builder.version;
        this.releaseVersion = builder.releaseVersion;
        this.createdAt = builder.createdAt;
        this.modifiedAt = builder.modifiedAt;
        this.status = builder.status;
        this.scalingConfig = builder.scalingConfig;
        this.instanceTypes = builder.instanceTypes;
        this.subnets = builder.subnets;
        this.remoteAccess = builder.remoteAccess;
        this.amiType = builder.amiType;
        this.nodeRole = builder.nodeRole;
        this.labels = builder.labels;
        this.resources = builder.resources;
        this.diskSize = builder.diskSize;
        this.health = builder.health;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name associated with an Amazon EKS managed node group.
     * </p>
     * 
     * @return The name associated with an Amazon EKS managed node group.
     */
    public String nodegroupName() {
        return nodegroupName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) associated with the managed node group.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) associated with the managed node group.
     */
    public String nodegroupArn() {
        return nodegroupArn;
    }

    /**
     * <p>
     * The name of the cluster that the managed node group resides in.
     * </p>
     * 
     * @return The name of the cluster that the managed node group resides in.
     */
    public String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * The Kubernetes version of the managed node group.
     * </p>
     * 
     * @return The Kubernetes version of the managed node group.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * The AMI version of the managed node group. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html">Amazon EKS-Optimized Linux
     * AMI Versions </a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * 
     * @return The AMI version of the managed node group. For more information, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html">Amazon EKS-Optimized
     *         Linux AMI Versions </a> in the <i>Amazon EKS User Guide</i>.
     */
    public String releaseVersion() {
        return releaseVersion;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the managed node group was created.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the managed node group was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the managed node group was last modified.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the managed node group was last modified.
     */
    public Instant modifiedAt() {
        return modifiedAt;
    }

    /**
     * <p>
     * The current status of the managed node group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NodegroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the managed node group.
     * @see NodegroupStatus
     */
    public NodegroupStatus status() {
        return NodegroupStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the managed node group.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link NodegroupStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the managed node group.
     * @see NodegroupStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The scaling configuration details for the Auto Scaling group that is associated with your node group.
     * </p>
     * 
     * @return The scaling configuration details for the Auto Scaling group that is associated with your node group.
     */
    public NodegroupScalingConfig scalingConfig() {
        return scalingConfig;
    }

    /**
     * Returns true if the InstanceTypes property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasInstanceTypes() {
        return instanceTypes != null && !(instanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance types associated with your node group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasInstanceTypes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The instance types associated with your node group.
     */
    public List<String> instanceTypes() {
        return instanceTypes;
    }

    /**
     * Returns true if the Subnets property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasSubnets() {
        return subnets != null && !(subnets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The subnets allowed for the Auto Scaling group that is associated with your node group. These subnets must have
     * the following tag: <code>kubernetes.io/cluster/CLUSTER_NAME</code>, where <code>CLUSTER_NAME</code> is replaced
     * with the name of your cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSubnets()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The subnets allowed for the Auto Scaling group that is associated with your node group. These subnets
     *         must have the following tag: <code>kubernetes.io/cluster/CLUSTER_NAME</code>, where
     *         <code>CLUSTER_NAME</code> is replaced with the name of your cluster.
     */
    public List<String> subnets() {
        return subnets;
    }

    /**
     * <p>
     * The remote access (SSH) configuration that is associated with the node group.
     * </p>
     * 
     * @return The remote access (SSH) configuration that is associated with the node group.
     */
    public RemoteAccessConfig remoteAccess() {
        return remoteAccess;
    }

    /**
     * <p>
     * The AMI type associated with your node group. GPU instance types should use the <code>AL2_x86_64_GPU</code> AMI
     * type, which uses the Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the
     * <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized Linux AMI.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #amiType} will
     * return {@link AMITypes#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #amiTypeAsString}.
     * </p>
     * 
     * @return The AMI type associated with your node group. GPU instance types should use the
     *         <code>AL2_x86_64_GPU</code> AMI type, which uses the Amazon EKS-optimized Linux AMI with GPU support.
     *         Non-GPU instances should use the <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized
     *         Linux AMI.
     * @see AMITypes
     */
    public AMITypes amiType() {
        return AMITypes.fromValue(amiType);
    }

    /**
     * <p>
     * The AMI type associated with your node group. GPU instance types should use the <code>AL2_x86_64_GPU</code> AMI
     * type, which uses the Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the
     * <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized Linux AMI.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #amiType} will
     * return {@link AMITypes#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #amiTypeAsString}.
     * </p>
     * 
     * @return The AMI type associated with your node group. GPU instance types should use the
     *         <code>AL2_x86_64_GPU</code> AMI type, which uses the Amazon EKS-optimized Linux AMI with GPU support.
     *         Non-GPU instances should use the <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized
     *         Linux AMI.
     * @see AMITypes
     */
    public String amiTypeAsString() {
        return amiType;
    }

    /**
     * <p>
     * The IAM role associated with your node group. The Amazon EKS worker node <code>kubelet</code> daemon makes calls
     * to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through an IAM instance profile
     * and associated policies. Before you can launch worker nodes and register them into a cluster, you must create an
     * IAM role for those worker nodes to use when they are launched. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">Amazon EKS Worker Node IAM
     * Role</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * 
     * @return The IAM role associated with your node group. The Amazon EKS worker node <code>kubelet</code> daemon
     *         makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through an
     *         IAM instance profile and associated policies. Before you can launch worker nodes and register them into a
     *         cluster, you must create an IAM role for those worker nodes to use when they are launched. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">Amazon EKS Worker Node
     *         IAM Role</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
     */
    public String nodeRole() {
        return nodeRole;
    }

    /**
     * Returns true if the Labels property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasLabels() {
        return labels != null && !(labels instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The Kubernetes labels applied to the nodes in the node group.
     * </p>
     * <note>
     * <p>
     * Only labels that are applied with the Amazon EKS API are shown here. There may be other Kubernetes labels applied
     * to the nodes in this group.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLabels()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The Kubernetes labels applied to the nodes in the node group.</p> <note>
     *         <p>
     *         Only labels that are applied with the Amazon EKS API are shown here. There may be other Kubernetes labels
     *         applied to the nodes in this group.
     *         </p>
     */
    public Map<String, String> labels() {
        return labels;
    }

    /**
     * <p>
     * The resources associated with the node group, such as Auto Scaling groups and security groups for remote access.
     * </p>
     * 
     * @return The resources associated with the node group, such as Auto Scaling groups and security groups for remote
     *         access.
     */
    public NodegroupResources resources() {
        return resources;
    }

    /**
     * <p>
     * The root device disk size (in GiB) for your node group instances. The default disk size is 20 GiB.
     * </p>
     * 
     * @return The root device disk size (in GiB) for your node group instances. The default disk size is 20 GiB.
     */
    public Integer diskSize() {
        return diskSize;
    }

    /**
     * <p>
     * The health status of the node group. If there are issues with your node group's health, they are listed here.
     * </p>
     * 
     * @return The health status of the node group. If there are issues with your node group's health, they are listed
     *         here.
     */
    public NodegroupHealth health() {
        return health;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The metadata applied to the node group to assist with categorization and organization. Each tag consists of a key
     * and an optional value, both of which you define. Node group tags do not propagate to any other resources
     * associated with the node group, such as the Amazon EC2 instances or subnets.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The metadata applied to the node group to assist with categorization and organization. Each tag consists
     *         of a key and an optional value, both of which you define. Node group tags do not propagate to any other
     *         resources associated with the node group, such as the Amazon EC2 instances or subnets.
     */
    public Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nodegroupName());
        hashCode = 31 * hashCode + Objects.hashCode(nodegroupArn());
        hashCode = 31 * hashCode + Objects.hashCode(clusterName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(releaseVersion());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(modifiedAt());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scalingConfig());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypes());
        hashCode = 31 * hashCode + Objects.hashCode(subnets());
        hashCode = 31 * hashCode + Objects.hashCode(remoteAccess());
        hashCode = 31 * hashCode + Objects.hashCode(amiTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(nodeRole());
        hashCode = 31 * hashCode + Objects.hashCode(labels());
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(diskSize());
        hashCode = 31 * hashCode + Objects.hashCode(health());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Nodegroup)) {
            return false;
        }
        Nodegroup other = (Nodegroup) obj;
        return Objects.equals(nodegroupName(), other.nodegroupName()) && Objects.equals(nodegroupArn(), other.nodegroupArn())
                && Objects.equals(clusterName(), other.clusterName()) && Objects.equals(version(), other.version())
                && Objects.equals(releaseVersion(), other.releaseVersion()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(modifiedAt(), other.modifiedAt()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(scalingConfig(), other.scalingConfig())
                && Objects.equals(instanceTypes(), other.instanceTypes()) && Objects.equals(subnets(), other.subnets())
                && Objects.equals(remoteAccess(), other.remoteAccess())
                && Objects.equals(amiTypeAsString(), other.amiTypeAsString()) && Objects.equals(nodeRole(), other.nodeRole())
                && Objects.equals(labels(), other.labels()) && Objects.equals(resources(), other.resources())
                && Objects.equals(diskSize(), other.diskSize()) && Objects.equals(health(), other.health())
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Nodegroup").add("NodegroupName", nodegroupName()).add("NodegroupArn", nodegroupArn())
                .add("ClusterName", clusterName()).add("Version", version()).add("ReleaseVersion", releaseVersion())
                .add("CreatedAt", createdAt()).add("ModifiedAt", modifiedAt()).add("Status", statusAsString())
                .add("ScalingConfig", scalingConfig()).add("InstanceTypes", instanceTypes()).add("Subnets", subnets())
                .add("RemoteAccess", remoteAccess()).add("AmiType", amiTypeAsString()).add("NodeRole", nodeRole())
                .add("Labels", labels()).add("Resources", resources()).add("DiskSize", diskSize()).add("Health", health())
                .add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "nodegroupName":
            return Optional.ofNullable(clazz.cast(nodegroupName()));
        case "nodegroupArn":
            return Optional.ofNullable(clazz.cast(nodegroupArn()));
        case "clusterName":
            return Optional.ofNullable(clazz.cast(clusterName()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "releaseVersion":
            return Optional.ofNullable(clazz.cast(releaseVersion()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "modifiedAt":
            return Optional.ofNullable(clazz.cast(modifiedAt()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "scalingConfig":
            return Optional.ofNullable(clazz.cast(scalingConfig()));
        case "instanceTypes":
            return Optional.ofNullable(clazz.cast(instanceTypes()));
        case "subnets":
            return Optional.ofNullable(clazz.cast(subnets()));
        case "remoteAccess":
            return Optional.ofNullable(clazz.cast(remoteAccess()));
        case "amiType":
            return Optional.ofNullable(clazz.cast(amiTypeAsString()));
        case "nodeRole":
            return Optional.ofNullable(clazz.cast(nodeRole()));
        case "labels":
            return Optional.ofNullable(clazz.cast(labels()));
        case "resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "diskSize":
            return Optional.ofNullable(clazz.cast(diskSize()));
        case "health":
            return Optional.ofNullable(clazz.cast(health()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Nodegroup, T> g) {
        return obj -> g.apply((Nodegroup) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Nodegroup> {
        /**
         * <p>
         * The name associated with an Amazon EKS managed node group.
         * </p>
         * 
         * @param nodegroupName
         *        The name associated with an Amazon EKS managed node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodegroupName(String nodegroupName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) associated with the managed node group.
         * </p>
         * 
         * @param nodegroupArn
         *        The Amazon Resource Name (ARN) associated with the managed node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodegroupArn(String nodegroupArn);

        /**
         * <p>
         * The name of the cluster that the managed node group resides in.
         * </p>
         * 
         * @param clusterName
         *        The name of the cluster that the managed node group resides in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * The Kubernetes version of the managed node group.
         * </p>
         * 
         * @param version
         *        The Kubernetes version of the managed node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The AMI version of the managed node group. For more information, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html">Amazon EKS-Optimized
         * Linux AMI Versions </a> in the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param releaseVersion
         *        The AMI version of the managed node group. For more information, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/eks-linux-ami-versions.html">Amazon
         *        EKS-Optimized Linux AMI Versions </a> in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseVersion(String releaseVersion);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the managed node group was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix epoch timestamp in seconds for when the managed node group was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the managed node group was last modified.
         * </p>
         * 
         * @param modifiedAt
         *        The Unix epoch timestamp in seconds for when the managed node group was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modifiedAt(Instant modifiedAt);

        /**
         * <p>
         * The current status of the managed node group.
         * </p>
         * 
         * @param status
         *        The current status of the managed node group.
         * @see NodegroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NodegroupStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the managed node group.
         * </p>
         * 
         * @param status
         *        The current status of the managed node group.
         * @see NodegroupStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NodegroupStatus
         */
        Builder status(NodegroupStatus status);

        /**
         * <p>
         * The scaling configuration details for the Auto Scaling group that is associated with your node group.
         * </p>
         * 
         * @param scalingConfig
         *        The scaling configuration details for the Auto Scaling group that is associated with your node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingConfig(NodegroupScalingConfig scalingConfig);

        /**
         * <p>
         * The scaling configuration details for the Auto Scaling group that is associated with your node group.
         * </p>
         * This is a convenience that creates an instance of the {@link NodegroupScalingConfig.Builder} avoiding the
         * need to create one manually via {@link NodegroupScalingConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link NodegroupScalingConfig.Builder#build()} is called immediately and
         * its result is passed to {@link #scalingConfig(NodegroupScalingConfig)}.
         * 
         * @param scalingConfig
         *        a consumer that will call methods on {@link NodegroupScalingConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scalingConfig(NodegroupScalingConfig)
         */
        default Builder scalingConfig(Consumer<NodegroupScalingConfig.Builder> scalingConfig) {
            return scalingConfig(NodegroupScalingConfig.builder().applyMutation(scalingConfig).build());
        }

        /**
         * <p>
         * The instance types associated with your node group.
         * </p>
         * 
         * @param instanceTypes
         *        The instance types associated with your node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(Collection<String> instanceTypes);

        /**
         * <p>
         * The instance types associated with your node group.
         * </p>
         * 
         * @param instanceTypes
         *        The instance types associated with your node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypes(String... instanceTypes);

        /**
         * <p>
         * The subnets allowed for the Auto Scaling group that is associated with your node group. These subnets must
         * have the following tag: <code>kubernetes.io/cluster/CLUSTER_NAME</code>, where <code>CLUSTER_NAME</code> is
         * replaced with the name of your cluster.
         * </p>
         * 
         * @param subnets
         *        The subnets allowed for the Auto Scaling group that is associated with your node group. These subnets
         *        must have the following tag: <code>kubernetes.io/cluster/CLUSTER_NAME</code>, where
         *        <code>CLUSTER_NAME</code> is replaced with the name of your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(Collection<String> subnets);

        /**
         * <p>
         * The subnets allowed for the Auto Scaling group that is associated with your node group. These subnets must
         * have the following tag: <code>kubernetes.io/cluster/CLUSTER_NAME</code>, where <code>CLUSTER_NAME</code> is
         * replaced with the name of your cluster.
         * </p>
         * 
         * @param subnets
         *        The subnets allowed for the Auto Scaling group that is associated with your node group. These subnets
         *        must have the following tag: <code>kubernetes.io/cluster/CLUSTER_NAME</code>, where
         *        <code>CLUSTER_NAME</code> is replaced with the name of your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnets(String... subnets);

        /**
         * <p>
         * The remote access (SSH) configuration that is associated with the node group.
         * </p>
         * 
         * @param remoteAccess
         *        The remote access (SSH) configuration that is associated with the node group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder remoteAccess(RemoteAccessConfig remoteAccess);

        /**
         * <p>
         * The remote access (SSH) configuration that is associated with the node group.
         * </p>
         * This is a convenience that creates an instance of the {@link RemoteAccessConfig.Builder} avoiding the need to
         * create one manually via {@link RemoteAccessConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link RemoteAccessConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #remoteAccess(RemoteAccessConfig)}.
         * 
         * @param remoteAccess
         *        a consumer that will call methods on {@link RemoteAccessConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #remoteAccess(RemoteAccessConfig)
         */
        default Builder remoteAccess(Consumer<RemoteAccessConfig.Builder> remoteAccess) {
            return remoteAccess(RemoteAccessConfig.builder().applyMutation(remoteAccess).build());
        }

        /**
         * <p>
         * The AMI type associated with your node group. GPU instance types should use the <code>AL2_x86_64_GPU</code>
         * AMI type, which uses the Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the
         * <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized Linux AMI.
         * </p>
         * 
         * @param amiType
         *        The AMI type associated with your node group. GPU instance types should use the
         *        <code>AL2_x86_64_GPU</code> AMI type, which uses the Amazon EKS-optimized Linux AMI with GPU support.
         *        Non-GPU instances should use the <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized
         *        Linux AMI.
         * @see AMITypes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AMITypes
         */
        Builder amiType(String amiType);

        /**
         * <p>
         * The AMI type associated with your node group. GPU instance types should use the <code>AL2_x86_64_GPU</code>
         * AMI type, which uses the Amazon EKS-optimized Linux AMI with GPU support. Non-GPU instances should use the
         * <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized Linux AMI.
         * </p>
         * 
         * @param amiType
         *        The AMI type associated with your node group. GPU instance types should use the
         *        <code>AL2_x86_64_GPU</code> AMI type, which uses the Amazon EKS-optimized Linux AMI with GPU support.
         *        Non-GPU instances should use the <code>AL2_x86_64</code> AMI type, which uses the Amazon EKS-optimized
         *        Linux AMI.
         * @see AMITypes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AMITypes
         */
        Builder amiType(AMITypes amiType);

        /**
         * <p>
         * The IAM role associated with your node group. The Amazon EKS worker node <code>kubelet</code> daemon makes
         * calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through an IAM
         * instance profile and associated policies. Before you can launch worker nodes and register them into a
         * cluster, you must create an IAM role for those worker nodes to use when they are launched. For more
         * information, see <a href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">Amazon
         * EKS Worker Node IAM Role</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
         * </p>
         * 
         * @param nodeRole
         *        The IAM role associated with your node group. The Amazon EKS worker node <code>kubelet</code> daemon
         *        makes calls to AWS APIs on your behalf. Worker nodes receive permissions for these API calls through
         *        an IAM instance profile and associated policies. Before you can launch worker nodes and register them
         *        into a cluster, you must create an IAM role for those worker nodes to use when they are launched. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/worker_node_IAM_role.html">Amazon EKS Worker
         *        Node IAM Role</a> in the <i> <i>Amazon EKS User Guide</i> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nodeRole(String nodeRole);

        /**
         * <p>
         * The Kubernetes labels applied to the nodes in the node group.
         * </p>
         * <note>
         * <p>
         * Only labels that are applied with the Amazon EKS API are shown here. There may be other Kubernetes labels
         * applied to the nodes in this group.
         * </p>
         * </note>
         * 
         * @param labels
         *        The Kubernetes labels applied to the nodes in the node group.</p> <note>
         *        <p>
         *        Only labels that are applied with the Amazon EKS API are shown here. There may be other Kubernetes
         *        labels applied to the nodes in this group.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder labels(Map<String, String> labels);

        /**
         * <p>
         * The resources associated with the node group, such as Auto Scaling groups and security groups for remote
         * access.
         * </p>
         * 
         * @param resources
         *        The resources associated with the node group, such as Auto Scaling groups and security groups for
         *        remote access.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(NodegroupResources resources);

        /**
         * <p>
         * The resources associated with the node group, such as Auto Scaling groups and security groups for remote
         * access.
         * </p>
         * This is a convenience that creates an instance of the {@link NodegroupResources.Builder} avoiding the need to
         * create one manually via {@link NodegroupResources#builder()}.
         *
         * When the {@link Consumer} completes, {@link NodegroupResources.Builder#build()} is called immediately and its
         * result is passed to {@link #resources(NodegroupResources)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link NodegroupResources.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(NodegroupResources)
         */
        default Builder resources(Consumer<NodegroupResources.Builder> resources) {
            return resources(NodegroupResources.builder().applyMutation(resources).build());
        }

        /**
         * <p>
         * The root device disk size (in GiB) for your node group instances. The default disk size is 20 GiB.
         * </p>
         * 
         * @param diskSize
         *        The root device disk size (in GiB) for your node group instances. The default disk size is 20 GiB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskSize(Integer diskSize);

        /**
         * <p>
         * The health status of the node group. If there are issues with your node group's health, they are listed here.
         * </p>
         * 
         * @param health
         *        The health status of the node group. If there are issues with your node group's health, they are
         *        listed here.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder health(NodegroupHealth health);

        /**
         * <p>
         * The health status of the node group. If there are issues with your node group's health, they are listed here.
         * </p>
         * This is a convenience that creates an instance of the {@link NodegroupHealth.Builder} avoiding the need to
         * create one manually via {@link NodegroupHealth#builder()}.
         *
         * When the {@link Consumer} completes, {@link NodegroupHealth.Builder#build()} is called immediately and its
         * result is passed to {@link #health(NodegroupHealth)}.
         * 
         * @param health
         *        a consumer that will call methods on {@link NodegroupHealth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #health(NodegroupHealth)
         */
        default Builder health(Consumer<NodegroupHealth.Builder> health) {
            return health(NodegroupHealth.builder().applyMutation(health).build());
        }

        /**
         * <p>
         * The metadata applied to the node group to assist with categorization and organization. Each tag consists of a
         * key and an optional value, both of which you define. Node group tags do not propagate to any other resources
         * associated with the node group, such as the Amazon EC2 instances or subnets.
         * </p>
         * 
         * @param tags
         *        The metadata applied to the node group to assist with categorization and organization. Each tag
         *        consists of a key and an optional value, both of which you define. Node group tags do not propagate to
         *        any other resources associated with the node group, such as the Amazon EC2 instances or subnets.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String nodegroupName;

        private String nodegroupArn;

        private String clusterName;

        private String version;

        private String releaseVersion;

        private Instant createdAt;

        private Instant modifiedAt;

        private String status;

        private NodegroupScalingConfig scalingConfig;

        private List<String> instanceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> subnets = DefaultSdkAutoConstructList.getInstance();

        private RemoteAccessConfig remoteAccess;

        private String amiType;

        private String nodeRole;

        private Map<String, String> labels = DefaultSdkAutoConstructMap.getInstance();

        private NodegroupResources resources;

        private Integer diskSize;

        private NodegroupHealth health;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Nodegroup model) {
            nodegroupName(model.nodegroupName);
            nodegroupArn(model.nodegroupArn);
            clusterName(model.clusterName);
            version(model.version);
            releaseVersion(model.releaseVersion);
            createdAt(model.createdAt);
            modifiedAt(model.modifiedAt);
            status(model.status);
            scalingConfig(model.scalingConfig);
            instanceTypes(model.instanceTypes);
            subnets(model.subnets);
            remoteAccess(model.remoteAccess);
            amiType(model.amiType);
            nodeRole(model.nodeRole);
            labels(model.labels);
            resources(model.resources);
            diskSize(model.diskSize);
            health(model.health);
            tags(model.tags);
        }

        public final String getNodegroupName() {
            return nodegroupName;
        }

        @Override
        public final Builder nodegroupName(String nodegroupName) {
            this.nodegroupName = nodegroupName;
            return this;
        }

        public final void setNodegroupName(String nodegroupName) {
            this.nodegroupName = nodegroupName;
        }

        public final String getNodegroupArn() {
            return nodegroupArn;
        }

        @Override
        public final Builder nodegroupArn(String nodegroupArn) {
            this.nodegroupArn = nodegroupArn;
            return this;
        }

        public final void setNodegroupArn(String nodegroupArn) {
            this.nodegroupArn = nodegroupArn;
        }

        public final String getClusterName() {
            return clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final String getReleaseVersion() {
            return releaseVersion;
        }

        @Override
        public final Builder releaseVersion(String releaseVersion) {
            this.releaseVersion = releaseVersion;
            return this;
        }

        public final void setReleaseVersion(String releaseVersion) {
            this.releaseVersion = releaseVersion;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final Instant getModifiedAt() {
            return modifiedAt;
        }

        @Override
        public final Builder modifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
            return this;
        }

        public final void setModifiedAt(Instant modifiedAt) {
            this.modifiedAt = modifiedAt;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(NodegroupStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final NodegroupScalingConfig.Builder getScalingConfig() {
            return scalingConfig != null ? scalingConfig.toBuilder() : null;
        }

        @Override
        public final Builder scalingConfig(NodegroupScalingConfig scalingConfig) {
            this.scalingConfig = scalingConfig;
            return this;
        }

        public final void setScalingConfig(NodegroupScalingConfig.BuilderImpl scalingConfig) {
            this.scalingConfig = scalingConfig != null ? scalingConfig.build() : null;
        }

        public final Collection<String> getInstanceTypes() {
            return instanceTypes;
        }

        @Override
        public final Builder instanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceTypes(String... instanceTypes) {
            instanceTypes(Arrays.asList(instanceTypes));
            return this;
        }

        public final void setInstanceTypes(Collection<String> instanceTypes) {
            this.instanceTypes = StringListCopier.copy(instanceTypes);
        }

        public final Collection<String> getSubnets() {
            return subnets;
        }

        @Override
        public final Builder subnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder subnets(String... subnets) {
            subnets(Arrays.asList(subnets));
            return this;
        }

        public final void setSubnets(Collection<String> subnets) {
            this.subnets = StringListCopier.copy(subnets);
        }

        public final RemoteAccessConfig.Builder getRemoteAccess() {
            return remoteAccess != null ? remoteAccess.toBuilder() : null;
        }

        @Override
        public final Builder remoteAccess(RemoteAccessConfig remoteAccess) {
            this.remoteAccess = remoteAccess;
            return this;
        }

        public final void setRemoteAccess(RemoteAccessConfig.BuilderImpl remoteAccess) {
            this.remoteAccess = remoteAccess != null ? remoteAccess.build() : null;
        }

        public final String getAmiType() {
            return amiType;
        }

        @Override
        public final Builder amiType(String amiType) {
            this.amiType = amiType;
            return this;
        }

        @Override
        public final Builder amiType(AMITypes amiType) {
            this.amiType(amiType == null ? null : amiType.toString());
            return this;
        }

        public final void setAmiType(String amiType) {
            this.amiType = amiType;
        }

        public final String getNodeRole() {
            return nodeRole;
        }

        @Override
        public final Builder nodeRole(String nodeRole) {
            this.nodeRole = nodeRole;
            return this;
        }

        public final void setNodeRole(String nodeRole) {
            this.nodeRole = nodeRole;
        }

        public final Map<String, String> getLabels() {
            return labels;
        }

        @Override
        public final Builder labels(Map<String, String> labels) {
            this.labels = _labelsMapCopier.copy(labels);
            return this;
        }

        public final void setLabels(Map<String, String> labels) {
            this.labels = _labelsMapCopier.copy(labels);
        }

        public final NodegroupResources.Builder getResources() {
            return resources != null ? resources.toBuilder() : null;
        }

        @Override
        public final Builder resources(NodegroupResources resources) {
            this.resources = resources;
            return this;
        }

        public final void setResources(NodegroupResources.BuilderImpl resources) {
            this.resources = resources != null ? resources.build() : null;
        }

        public final Integer getDiskSize() {
            return diskSize;
        }

        @Override
        public final Builder diskSize(Integer diskSize) {
            this.diskSize = diskSize;
            return this;
        }

        public final void setDiskSize(Integer diskSize) {
            this.diskSize = diskSize;
        }

        public final NodegroupHealth.Builder getHealth() {
            return health != null ? health.toBuilder() : null;
        }

        @Override
        public final Builder health(NodegroupHealth health) {
            this.health = health;
            return this;
        }

        public final void setHealth(NodegroupHealth.BuilderImpl health) {
            this.health = health != null ? health.build() : null;
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public Nodegroup build() {
            return new Nodegroup(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
