/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object representing an Amazon EKS cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Cluster implements SdkPojo, Serializable, ToCopyableBuilder<Cluster.Builder, Cluster> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(Cluster::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<String> ENDPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::endpoint)).setter(setter(Builder::endpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("endpoint").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<VpcConfigResponse> RESOURCES_VPC_CONFIG_FIELD = SdkField
            .<VpcConfigResponse> builder(MarshallingType.SDK_POJO).getter(getter(Cluster::resourcesVpcConfig))
            .setter(setter(Builder::resourcesVpcConfig)).constructor(VpcConfigResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resourcesVpcConfig").build())
            .build();

    private static final SdkField<Logging> LOGGING_FIELD = SdkField.<Logging> builder(MarshallingType.SDK_POJO)
            .getter(getter(Cluster::logging)).setter(setter(Builder::logging)).constructor(Logging::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("logging").build()).build();

    private static final SdkField<Identity> IDENTITY_FIELD = SdkField.<Identity> builder(MarshallingType.SDK_POJO)
            .getter(getter(Cluster::identity)).setter(setter(Builder::identity)).constructor(Identity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identity").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Certificate> CERTIFICATE_AUTHORITY_FIELD = SdkField
            .<Certificate> builder(MarshallingType.SDK_POJO).getter(getter(Cluster::certificateAuthority))
            .setter(setter(Builder::certificateAuthority)).constructor(Certificate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("certificateAuthority").build())
            .build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::clientRequestToken)).setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientRequestToken").build())
            .build();

    private static final SdkField<String> PLATFORM_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Cluster::platformVersion)).setter(setter(Builder::platformVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("platformVersion").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .getter(getter(Cluster::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<EncryptionConfig>> ENCRYPTION_CONFIG_FIELD = SdkField
            .<List<EncryptionConfig>> builder(MarshallingType.LIST)
            .getter(getter(Cluster::encryptionConfig))
            .setter(setter(Builder::encryptionConfig))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionConfig").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EncryptionConfig> builder(MarshallingType.SDK_POJO)
                                            .constructor(EncryptionConfig::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            CREATED_AT_FIELD, VERSION_FIELD, ENDPOINT_FIELD, ROLE_ARN_FIELD, RESOURCES_VPC_CONFIG_FIELD, LOGGING_FIELD,
            IDENTITY_FIELD, STATUS_FIELD, CERTIFICATE_AUTHORITY_FIELD, CLIENT_REQUEST_TOKEN_FIELD, PLATFORM_VERSION_FIELD,
            TAGS_FIELD, ENCRYPTION_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final Instant createdAt;

    private final String version;

    private final String endpoint;

    private final String roleArn;

    private final VpcConfigResponse resourcesVpcConfig;

    private final Logging logging;

    private final Identity identity;

    private final String status;

    private final Certificate certificateAuthority;

    private final String clientRequestToken;

    private final String platformVersion;

    private final Map<String, String> tags;

    private final List<EncryptionConfig> encryptionConfig;

    private Cluster(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.version = builder.version;
        this.endpoint = builder.endpoint;
        this.roleArn = builder.roleArn;
        this.resourcesVpcConfig = builder.resourcesVpcConfig;
        this.logging = builder.logging;
        this.identity = builder.identity;
        this.status = builder.status;
        this.certificateAuthority = builder.certificateAuthority;
        this.clientRequestToken = builder.clientRequestToken;
        this.platformVersion = builder.platformVersion;
        this.tags = builder.tags;
        this.encryptionConfig = builder.encryptionConfig;
    }

    /**
     * <p>
     * The name of the cluster.
     * </p>
     * 
     * @return The name of the cluster.
     */
    public String name() {
        return name;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the cluster.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the cluster.
     */
    public String arn() {
        return arn;
    }

    /**
     * <p>
     * The Unix epoch timestamp in seconds for when the cluster was created.
     * </p>
     * 
     * @return The Unix epoch timestamp in seconds for when the cluster was created.
     */
    public Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The Kubernetes server version for the cluster.
     * </p>
     * 
     * @return The Kubernetes server version for the cluster.
     */
    public String version() {
        return version;
    }

    /**
     * <p>
     * The endpoint for your Kubernetes API server.
     * </p>
     * 
     * @return The endpoint for your Kubernetes API server.
     */
    public String endpoint() {
        return endpoint;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to make
     * calls to AWS API operations on your behalf.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane
     *         to make calls to AWS API operations on your behalf.
     */
    public String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific requirements to
     * work properly with Kubernetes. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC Considerations</a> and <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster Security Group
     * Considerations</a> in the <i>Amazon EKS User Guide</i>.
     * </p>
     * 
     * @return The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific
     *         requirements to work properly with Kubernetes. For more information, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC Considerations</a>
     *         and <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster Security Group
     *         Considerations</a> in the <i>Amazon EKS User Guide</i>.
     */
    public VpcConfigResponse resourcesVpcConfig() {
        return resourcesVpcConfig;
    }

    /**
     * <p>
     * The logging configuration for your cluster.
     * </p>
     * 
     * @return The logging configuration for your cluster.
     */
    public Logging logging() {
        return logging;
    }

    /**
     * <p>
     * The identity provider information for the cluster.
     * </p>
     * 
     * @return The identity provider information for the cluster.
     */
    public Identity identity() {
        return identity;
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterStatus
     */
    public ClusterStatus status() {
        return ClusterStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the cluster.
     * @see ClusterStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The <code>certificate-authority-data</code> for your cluster.
     * </p>
     * 
     * @return The <code>certificate-authority-data</code> for your cluster.
     */
    public Certificate certificateAuthority() {
        return certificateAuthority;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
     * </p>
     * 
     * @return Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
     */
    public String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The platform version of your Amazon EKS cluster. For more information, see <a
     * href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform Versions</a> in the <i>
     * <i>Amazon EKS User Guide</i> </i>.
     * </p>
     * 
     * @return The platform version of your Amazon EKS cluster. For more information, see <a
     *         href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform Versions</a> in
     *         the <i> <i>Amazon EKS User Guide</i> </i>.
     */
    public String platformVersion() {
        return platformVersion;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The metadata that you apply to the cluster to assist with categorization and organization. Each tag consists of a
     * key and an optional value, both of which you define. Cluster tags do not propagate to any other resources
     * associated with the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The metadata that you apply to the cluster to assist with categorization and organization. Each tag
     *         consists of a key and an optional value, both of which you define. Cluster tags do not propagate to any
     *         other resources associated with the cluster.
     */
    public Map<String, String> tags() {
        return tags;
    }

    /**
     * Returns true if the EncryptionConfig property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasEncryptionConfig() {
        return encryptionConfig != null && !(encryptionConfig instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The encryption configuration for the cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasEncryptionConfig()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The encryption configuration for the cluster.
     */
    public List<EncryptionConfig> encryptionConfig() {
        return encryptionConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(endpoint());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourcesVpcConfig());
        hashCode = 31 * hashCode + Objects.hashCode(logging());
        hashCode = 31 * hashCode + Objects.hashCode(identity());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(certificateAuthority());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(platformVersion());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionConfig());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Cluster)) {
            return false;
        }
        Cluster other = (Cluster) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(createdAt(), other.createdAt()) && Objects.equals(version(), other.version())
                && Objects.equals(endpoint(), other.endpoint()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(resourcesVpcConfig(), other.resourcesVpcConfig()) && Objects.equals(logging(), other.logging())
                && Objects.equals(identity(), other.identity()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(certificateAuthority(), other.certificateAuthority())
                && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(platformVersion(), other.platformVersion()) && Objects.equals(tags(), other.tags())
                && Objects.equals(encryptionConfig(), other.encryptionConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Cluster").add("Name", name()).add("Arn", arn()).add("CreatedAt", createdAt())
                .add("Version", version()).add("Endpoint", endpoint()).add("RoleArn", roleArn())
                .add("ResourcesVpcConfig", resourcesVpcConfig()).add("Logging", logging()).add("Identity", identity())
                .add("Status", statusAsString()).add("CertificateAuthority", certificateAuthority())
                .add("ClientRequestToken", clientRequestToken()).add("PlatformVersion", platformVersion()).add("Tags", tags())
                .add("EncryptionConfig", encryptionConfig()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "endpoint":
            return Optional.ofNullable(clazz.cast(endpoint()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "resourcesVpcConfig":
            return Optional.ofNullable(clazz.cast(resourcesVpcConfig()));
        case "logging":
            return Optional.ofNullable(clazz.cast(logging()));
        case "identity":
            return Optional.ofNullable(clazz.cast(identity()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "certificateAuthority":
            return Optional.ofNullable(clazz.cast(certificateAuthority()));
        case "clientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "platformVersion":
            return Optional.ofNullable(clazz.cast(platformVersion()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "encryptionConfig":
            return Optional.ofNullable(clazz.cast(encryptionConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Cluster, T> g) {
        return obj -> g.apply((Cluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Cluster> {
        /**
         * <p>
         * The name of the cluster.
         * </p>
         * 
         * @param name
         *        The name of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the cluster.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The Unix epoch timestamp in seconds for when the cluster was created.
         * </p>
         * 
         * @param createdAt
         *        The Unix epoch timestamp in seconds for when the cluster was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The Kubernetes server version for the cluster.
         * </p>
         * 
         * @param version
         *        The Kubernetes server version for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * The endpoint for your Kubernetes API server.
         * </p>
         * 
         * @param endpoint
         *        The endpoint for your Kubernetes API server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpoint(String endpoint);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control plane to
         * make calls to AWS API operations on your behalf.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role that provides permissions for the Kubernetes control
         *        plane to make calls to AWS API operations on your behalf.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific requirements
         * to work properly with Kubernetes. For more information, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC Considerations</a> and
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster Security Group
         * Considerations</a> in the <i>Amazon EKS User Guide</i>.
         * </p>
         * 
         * @param resourcesVpcConfig
         *        The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific
         *        requirements to work properly with Kubernetes. For more information, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC
         *        Considerations</a> and <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster Security Group
         *        Considerations</a> in the <i>Amazon EKS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourcesVpcConfig(VpcConfigResponse resourcesVpcConfig);

        /**
         * <p>
         * The VPC configuration used by the cluster control plane. Amazon EKS VPC resources have specific requirements
         * to work properly with Kubernetes. For more information, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html">Cluster VPC Considerations</a> and
         * <a href="https://docs.aws.amazon.com/eks/latest/userguide/sec-group-reqs.html">Cluster Security Group
         * Considerations</a> in the <i>Amazon EKS User Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link VpcConfigResponse.Builder} avoiding the need to
         * create one manually via {@link VpcConfigResponse#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpcConfigResponse.Builder#build()} is called immediately and its
         * result is passed to {@link #resourcesVpcConfig(VpcConfigResponse)}.
         * 
         * @param resourcesVpcConfig
         *        a consumer that will call methods on {@link VpcConfigResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourcesVpcConfig(VpcConfigResponse)
         */
        default Builder resourcesVpcConfig(Consumer<VpcConfigResponse.Builder> resourcesVpcConfig) {
            return resourcesVpcConfig(VpcConfigResponse.builder().applyMutation(resourcesVpcConfig).build());
        }

        /**
         * <p>
         * The logging configuration for your cluster.
         * </p>
         * 
         * @param logging
         *        The logging configuration for your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logging(Logging logging);

        /**
         * <p>
         * The logging configuration for your cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link Logging.Builder} avoiding the need to create one
         * manually via {@link Logging#builder()}.
         *
         * When the {@link Consumer} completes, {@link Logging.Builder#build()} is called immediately and its result is
         * passed to {@link #logging(Logging)}.
         * 
         * @param logging
         *        a consumer that will call methods on {@link Logging.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #logging(Logging)
         */
        default Builder logging(Consumer<Logging.Builder> logging) {
            return logging(Logging.builder().applyMutation(logging).build());
        }

        /**
         * <p>
         * The identity provider information for the cluster.
         * </p>
         * 
         * @param identity
         *        The identity provider information for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identity(Identity identity);

        /**
         * <p>
         * The identity provider information for the cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link Identity.Builder} avoiding the need to create
         * one manually via {@link Identity#builder()}.
         *
         * When the {@link Consumer} completes, {@link Identity.Builder#build()} is called immediately and its result is
         * passed to {@link #identity(Identity)}.
         * 
         * @param identity
         *        a consumer that will call methods on {@link Identity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #identity(Identity)
         */
        default Builder identity(Consumer<Identity.Builder> identity) {
            return identity(Identity.builder().applyMutation(identity).build());
        }

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param status
         *        The current status of the cluster.
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the cluster.
         * </p>
         * 
         * @param status
         *        The current status of the cluster.
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(ClusterStatus status);

        /**
         * <p>
         * The <code>certificate-authority-data</code> for your cluster.
         * </p>
         * 
         * @param certificateAuthority
         *        The <code>certificate-authority-data</code> for your cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateAuthority(Certificate certificateAuthority);

        /**
         * <p>
         * The <code>certificate-authority-data</code> for your cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link Certificate.Builder} avoiding the need to create
         * one manually via {@link Certificate#builder()}.
         *
         * When the {@link Consumer} completes, {@link Certificate.Builder#build()} is called immediately and its result
         * is passed to {@link #certificateAuthority(Certificate)}.
         * 
         * @param certificateAuthority
         *        a consumer that will call methods on {@link Certificate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateAuthority(Certificate)
         */
        default Builder certificateAuthority(Consumer<Certificate.Builder> certificateAuthority) {
            return certificateAuthority(Certificate.builder().applyMutation(certificateAuthority).build());
        }

        /**
         * <p>
         * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
         * </p>
         * 
         * @param clientRequestToken
         *        Unique, case-sensitive identifier that you provide to ensure the idempotency of the request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The platform version of your Amazon EKS cluster. For more information, see <a
         * href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform Versions</a> in the
         * <i> <i>Amazon EKS User Guide</i> </i>.
         * </p>
         * 
         * @param platformVersion
         *        The platform version of your Amazon EKS cluster. For more information, see <a
         *        href="https://docs.aws.amazon.com/eks/latest/userguide/platform-versions.html">Platform Versions</a>
         *        in the <i> <i>Amazon EKS User Guide</i> </i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder platformVersion(String platformVersion);

        /**
         * <p>
         * The metadata that you apply to the cluster to assist with categorization and organization. Each tag consists
         * of a key and an optional value, both of which you define. Cluster tags do not propagate to any other
         * resources associated with the cluster.
         * </p>
         * 
         * @param tags
         *        The metadata that you apply to the cluster to assist with categorization and organization. Each tag
         *        consists of a key and an optional value, both of which you define. Cluster tags do not propagate to
         *        any other resources associated with the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * The encryption configuration for the cluster.
         * </p>
         * 
         * @param encryptionConfig
         *        The encryption configuration for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfig(Collection<EncryptionConfig> encryptionConfig);

        /**
         * <p>
         * The encryption configuration for the cluster.
         * </p>
         * 
         * @param encryptionConfig
         *        The encryption configuration for the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionConfig(EncryptionConfig... encryptionConfig);

        /**
         * <p>
         * The encryption configuration for the cluster.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EncryptionConfig>.Builder} avoiding the
         * need to create one manually via {@link List<EncryptionConfig>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EncryptionConfig>.Builder#build()} is called immediately and
         * its result is passed to {@link #encryptionConfig(List<EncryptionConfig>)}.
         * 
         * @param encryptionConfig
         *        a consumer that will call methods on {@link List<EncryptionConfig>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #encryptionConfig(List<EncryptionConfig>)
         */
        Builder encryptionConfig(Consumer<EncryptionConfig.Builder>... encryptionConfig);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private Instant createdAt;

        private String version;

        private String endpoint;

        private String roleArn;

        private VpcConfigResponse resourcesVpcConfig;

        private Logging logging;

        private Identity identity;

        private String status;

        private Certificate certificateAuthority;

        private String clientRequestToken;

        private String platformVersion;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private List<EncryptionConfig> encryptionConfig = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Cluster model) {
            name(model.name);
            arn(model.arn);
            createdAt(model.createdAt);
            version(model.version);
            endpoint(model.endpoint);
            roleArn(model.roleArn);
            resourcesVpcConfig(model.resourcesVpcConfig);
            logging(model.logging);
            identity(model.identity);
            status(model.status);
            certificateAuthority(model.certificateAuthority);
            clientRequestToken(model.clientRequestToken);
            platformVersion(model.platformVersion);
            tags(model.tags);
            encryptionConfig(model.encryptionConfig);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getArn() {
            return arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        public final String getVersion() {
            return version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        public final String getEndpoint() {
            return endpoint;
        }

        @Override
        public final Builder endpoint(String endpoint) {
            this.endpoint = endpoint;
            return this;
        }

        public final void setEndpoint(String endpoint) {
            this.endpoint = endpoint;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        public final VpcConfigResponse.Builder getResourcesVpcConfig() {
            return resourcesVpcConfig != null ? resourcesVpcConfig.toBuilder() : null;
        }

        @Override
        public final Builder resourcesVpcConfig(VpcConfigResponse resourcesVpcConfig) {
            this.resourcesVpcConfig = resourcesVpcConfig;
            return this;
        }

        public final void setResourcesVpcConfig(VpcConfigResponse.BuilderImpl resourcesVpcConfig) {
            this.resourcesVpcConfig = resourcesVpcConfig != null ? resourcesVpcConfig.build() : null;
        }

        public final Logging.Builder getLogging() {
            return logging != null ? logging.toBuilder() : null;
        }

        @Override
        public final Builder logging(Logging logging) {
            this.logging = logging;
            return this;
        }

        public final void setLogging(Logging.BuilderImpl logging) {
            this.logging = logging != null ? logging.build() : null;
        }

        public final Identity.Builder getIdentity() {
            return identity != null ? identity.toBuilder() : null;
        }

        @Override
        public final Builder identity(Identity identity) {
            this.identity = identity;
            return this;
        }

        public final void setIdentity(Identity.BuilderImpl identity) {
            this.identity = identity != null ? identity.build() : null;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ClusterStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final Certificate.Builder getCertificateAuthority() {
            return certificateAuthority != null ? certificateAuthority.toBuilder() : null;
        }

        @Override
        public final Builder certificateAuthority(Certificate certificateAuthority) {
            this.certificateAuthority = certificateAuthority;
            return this;
        }

        public final void setCertificateAuthority(Certificate.BuilderImpl certificateAuthority) {
            this.certificateAuthority = certificateAuthority != null ? certificateAuthority.build() : null;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        public final String getPlatformVersion() {
            return platformVersion;
        }

        @Override
        public final Builder platformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
            return this;
        }

        public final void setPlatformVersion(String platformVersion) {
            this.platformVersion = platformVersion;
        }

        public final Map<String, String> getTags() {
            return tags;
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        public final Collection<EncryptionConfig.Builder> getEncryptionConfig() {
            return encryptionConfig != null ? encryptionConfig.stream().map(EncryptionConfig::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder encryptionConfig(Collection<EncryptionConfig> encryptionConfig) {
            this.encryptionConfig = EncryptionConfigListCopier.copy(encryptionConfig);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionConfig(EncryptionConfig... encryptionConfig) {
            encryptionConfig(Arrays.asList(encryptionConfig));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder encryptionConfig(Consumer<EncryptionConfig.Builder>... encryptionConfig) {
            encryptionConfig(Stream.of(encryptionConfig).map(c -> EncryptionConfig.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setEncryptionConfig(Collection<EncryptionConfig.BuilderImpl> encryptionConfig) {
            this.encryptionConfig = EncryptionConfigListCopier.copyFromBuilder(encryptionConfig);
        }

        @Override
        public Cluster build() {
            return new Cluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
