/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.waiters;

import java.math.BigDecimal;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.ecs.EcsClient;
import software.amazon.awssdk.services.ecs.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.ecs.model.DescribeServicesRequest;
import software.amazon.awssdk.services.ecs.model.DescribeServicesResponse;
import software.amazon.awssdk.services.ecs.model.DescribeTasksRequest;
import software.amazon.awssdk.services.ecs.model.DescribeTasksResponse;
import software.amazon.awssdk.services.ecs.model.EcsRequest;
import software.amazon.awssdk.services.ecs.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultEcsWaiter implements EcsWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final EcsClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeTasksResponse> tasksRunningWaiter;

    private final Waiter<DescribeTasksResponse> tasksStoppedWaiter;

    private final Waiter<DescribeServicesResponse> servicesStableWaiter;

    private final Waiter<DescribeServicesResponse> servicesInactiveWaiter;

    private DefaultEcsWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = EcsClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.tasksRunningWaiter = Waiter.builder(DescribeTasksResponse.class).acceptors(tasksRunningWaiterAcceptors())
                .overrideConfiguration(tasksRunningWaiterConfig(builder.overrideConfiguration)).build();
        this.tasksStoppedWaiter = Waiter.builder(DescribeTasksResponse.class).acceptors(tasksStoppedWaiterAcceptors())
                .overrideConfiguration(tasksStoppedWaiterConfig(builder.overrideConfiguration)).build();
        this.servicesStableWaiter = Waiter.builder(DescribeServicesResponse.class).acceptors(servicesStableWaiterAcceptors())
                .overrideConfiguration(servicesStableWaiterConfig(builder.overrideConfiguration)).build();
        this.servicesInactiveWaiter = Waiter.builder(DescribeServicesResponse.class).acceptors(servicesInactiveWaiterAcceptors())
                .overrideConfiguration(servicesInactiveWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeServicesResponse> waitUntilServicesInactive(DescribeServicesRequest describeServicesRequest) {
        return servicesInactiveWaiter.run(() -> client.describeServices(applyWaitersUserAgent(describeServicesRequest)));
    }

    @Override
    public WaiterResponse<DescribeServicesResponse> waitUntilServicesInactive(DescribeServicesRequest describeServicesRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return servicesInactiveWaiter.run(() -> client.describeServices(applyWaitersUserAgent(describeServicesRequest)),
                servicesInactiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeServicesResponse> waitUntilServicesStable(DescribeServicesRequest describeServicesRequest) {
        return servicesStableWaiter.run(() -> client.describeServices(applyWaitersUserAgent(describeServicesRequest)));
    }

    @Override
    public WaiterResponse<DescribeServicesResponse> waitUntilServicesStable(DescribeServicesRequest describeServicesRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return servicesStableWaiter.run(() -> client.describeServices(applyWaitersUserAgent(describeServicesRequest)),
                servicesStableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTasksResponse> waitUntilTasksRunning(DescribeTasksRequest describeTasksRequest) {
        return tasksRunningWaiter.run(() -> client.describeTasks(applyWaitersUserAgent(describeTasksRequest)));
    }

    @Override
    public WaiterResponse<DescribeTasksResponse> waitUntilTasksRunning(DescribeTasksRequest describeTasksRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return tasksRunningWaiter.run(() -> client.describeTasks(applyWaitersUserAgent(describeTasksRequest)),
                tasksRunningWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeTasksResponse> waitUntilTasksStopped(DescribeTasksRequest describeTasksRequest) {
        return tasksStoppedWaiter.run(() -> client.describeTasks(applyWaitersUserAgent(describeTasksRequest)));
    }

    @Override
    public WaiterResponse<DescribeTasksResponse> waitUntilTasksStopped(DescribeTasksRequest describeTasksRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return tasksStoppedWaiter.run(() -> client.describeTasks(applyWaitersUserAgent(describeTasksRequest)),
                tasksStoppedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeTasksResponse>> tasksRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("tasks").flatten().field("lastStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "STOPPED"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (tasks[].lastStatus=STOPPED) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("failures").flatten().field("reason").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "MISSING"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (failures[].reason=MISSING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("tasks").flatten().field("lastStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "RUNNING"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeTasksResponse>> tasksStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("tasks").flatten().field("lastStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "STOPPED"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeServicesResponse>> servicesStableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeServicesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("failures").flatten().field("reason").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "MISSING"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (failures[].reason=MISSING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("services").flatten().field("status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "DRAINING"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (services[].status=DRAINING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("services").flatten().field("status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "INACTIVE"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (services[].status=INACTIVE) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(
                    input.field("services")
                            .filter(x0 -> x0.constant(x0.field("deployments").length()
                                    .compare("==", x0.constant(new BigDecimal("1")))
                                    .and(x0.field("runningCount").compare("==", x0.field("desiredCount"))).not())).length()
                            .compare("==", input.constant(new BigDecimal("0"))).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeServicesResponse>> servicesInactiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeServicesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("failures").flatten().field("reason").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "MISSING"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (failures[].reason=MISSING) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("services").flatten().field("status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "INACTIVE"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration tasksRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(100);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(6)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration tasksStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(100);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(6)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration servicesStableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration servicesInactiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static EcsWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends EcsRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements EcsWaiter.Builder {
        private EcsClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public EcsWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public EcsWaiter.Builder client(EcsClient client) {
            this.client = client;
            return this;
        }

        public EcsWaiter build() {
            return new DefaultEcsWaiter(this);
        }
    }
}
