/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The instance requirements for attribute-based instance type selection. Instead of specifying exact instance types,
 * you define requirements such as vCPU count, memory size, network performance, and accelerator specifications. Amazon
 * ECS automatically selects Amazon EC2 instance types that match these requirements, providing flexibility and helping
 * to mitigate capacity constraints.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceRequirementsRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceRequirementsRequest.Builder, InstanceRequirementsRequest> {
    private static final SdkField<VCpuCountRangeRequest> V_CPU_COUNT_FIELD = SdkField
            .<VCpuCountRangeRequest> builder(MarshallingType.SDK_POJO).memberName("vCpuCount")
            .getter(getter(InstanceRequirementsRequest::vCpuCount)).setter(setter(Builder::vCpuCount))
            .constructor(VCpuCountRangeRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vCpuCount").build()).build();

    private static final SdkField<MemoryMiBRequest> MEMORY_MIB_FIELD = SdkField
            .<MemoryMiBRequest> builder(MarshallingType.SDK_POJO).memberName("memoryMiB")
            .getter(getter(InstanceRequirementsRequest::memoryMiB)).setter(setter(Builder::memoryMiB))
            .constructor(MemoryMiBRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryMiB").build()).build();

    private static final SdkField<List<String>> CPU_MANUFACTURERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("cpuManufacturers")
            .getter(getter(InstanceRequirementsRequest::cpuManufacturersAsStrings))
            .setter(setter(Builder::cpuManufacturersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cpuManufacturers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<MemoryGiBPerVCpuRequest> MEMORY_GI_B_PER_V_CPU_FIELD = SdkField
            .<MemoryGiBPerVCpuRequest> builder(MarshallingType.SDK_POJO).memberName("memoryGiBPerVCpu")
            .getter(getter(InstanceRequirementsRequest::memoryGiBPerVCpu)).setter(setter(Builder::memoryGiBPerVCpu))
            .constructor(MemoryGiBPerVCpuRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memoryGiBPerVCpu").build()).build();

    private static final SdkField<List<String>> EXCLUDED_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("excludedInstanceTypes")
            .getter(getter(InstanceRequirementsRequest::excludedInstanceTypes))
            .setter(setter(Builder::excludedInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("excludedInstanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> INSTANCE_GENERATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("instanceGenerations")
            .getter(getter(InstanceRequirementsRequest::instanceGenerationsAsStrings))
            .setter(setter(Builder::instanceGenerationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceGenerations").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> SPOT_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("spotMaxPricePercentageOverLowestPrice")
            .getter(getter(InstanceRequirementsRequest::spotMaxPricePercentageOverLowestPrice))
            .setter(setter(Builder::spotMaxPricePercentageOverLowestPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("spotMaxPricePercentageOverLowestPrice").build()).build();

    private static final SdkField<Integer> ON_DEMAND_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("onDemandMaxPricePercentageOverLowestPrice")
            .getter(getter(InstanceRequirementsRequest::onDemandMaxPricePercentageOverLowestPrice))
            .setter(setter(Builder::onDemandMaxPricePercentageOverLowestPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("onDemandMaxPricePercentageOverLowestPrice").build()).build();

    private static final SdkField<String> BARE_METAL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("bareMetal").getter(getter(InstanceRequirementsRequest::bareMetalAsString))
            .setter(setter(Builder::bareMetal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bareMetal").build()).build();

    private static final SdkField<String> BURSTABLE_PERFORMANCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("burstablePerformance").getter(getter(InstanceRequirementsRequest::burstablePerformanceAsString))
            .setter(setter(Builder::burstablePerformance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("burstablePerformance").build())
            .build();

    private static final SdkField<Boolean> REQUIRE_HIBERNATE_SUPPORT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("requireHibernateSupport").getter(getter(InstanceRequirementsRequest::requireHibernateSupport))
            .setter(setter(Builder::requireHibernateSupport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("requireHibernateSupport").build())
            .build();

    private static final SdkField<NetworkInterfaceCountRequest> NETWORK_INTERFACE_COUNT_FIELD = SdkField
            .<NetworkInterfaceCountRequest> builder(MarshallingType.SDK_POJO).memberName("networkInterfaceCount")
            .getter(getter(InstanceRequirementsRequest::networkInterfaceCount)).setter(setter(Builder::networkInterfaceCount))
            .constructor(NetworkInterfaceCountRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkInterfaceCount").build())
            .build();

    private static final SdkField<String> LOCAL_STORAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("localStorage").getter(getter(InstanceRequirementsRequest::localStorageAsString))
            .setter(setter(Builder::localStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("localStorage").build()).build();

    private static final SdkField<List<String>> LOCAL_STORAGE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("localStorageTypes")
            .getter(getter(InstanceRequirementsRequest::localStorageTypesAsStrings))
            .setter(setter(Builder::localStorageTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("localStorageTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<TotalLocalStorageGBRequest> TOTAL_LOCAL_STORAGE_GB_FIELD = SdkField
            .<TotalLocalStorageGBRequest> builder(MarshallingType.SDK_POJO).memberName("totalLocalStorageGB")
            .getter(getter(InstanceRequirementsRequest::totalLocalStorageGB)).setter(setter(Builder::totalLocalStorageGB))
            .constructor(TotalLocalStorageGBRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("totalLocalStorageGB").build())
            .build();

    private static final SdkField<BaselineEbsBandwidthMbpsRequest> BASELINE_EBS_BANDWIDTH_MBPS_FIELD = SdkField
            .<BaselineEbsBandwidthMbpsRequest> builder(MarshallingType.SDK_POJO).memberName("baselineEbsBandwidthMbps")
            .getter(getter(InstanceRequirementsRequest::baselineEbsBandwidthMbps))
            .setter(setter(Builder::baselineEbsBandwidthMbps)).constructor(BaselineEbsBandwidthMbpsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("baselineEbsBandwidthMbps").build())
            .build();

    private static final SdkField<List<String>> ACCELERATOR_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("acceleratorTypes")
            .getter(getter(InstanceRequirementsRequest::acceleratorTypesAsStrings))
            .setter(setter(Builder::acceleratorTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("acceleratorTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AcceleratorCountRequest> ACCELERATOR_COUNT_FIELD = SdkField
            .<AcceleratorCountRequest> builder(MarshallingType.SDK_POJO).memberName("acceleratorCount")
            .getter(getter(InstanceRequirementsRequest::acceleratorCount)).setter(setter(Builder::acceleratorCount))
            .constructor(AcceleratorCountRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("acceleratorCount").build()).build();

    private static final SdkField<List<String>> ACCELERATOR_MANUFACTURERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("acceleratorManufacturers")
            .getter(getter(InstanceRequirementsRequest::acceleratorManufacturersAsStrings))
            .setter(setter(Builder::acceleratorManufacturersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("acceleratorManufacturers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ACCELERATOR_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("acceleratorNames")
            .getter(getter(InstanceRequirementsRequest::acceleratorNamesAsStrings))
            .setter(setter(Builder::acceleratorNamesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("acceleratorNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<AcceleratorTotalMemoryMiBRequest> ACCELERATOR_TOTAL_MEMORY_MIB_FIELD = SdkField
            .<AcceleratorTotalMemoryMiBRequest> builder(MarshallingType.SDK_POJO).memberName("acceleratorTotalMemoryMiB")
            .getter(getter(InstanceRequirementsRequest::acceleratorTotalMemoryMiB))
            .setter(setter(Builder::acceleratorTotalMemoryMiB)).constructor(AcceleratorTotalMemoryMiBRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("acceleratorTotalMemoryMiB").build())
            .build();

    private static final SdkField<NetworkBandwidthGbpsRequest> NETWORK_BANDWIDTH_GBPS_FIELD = SdkField
            .<NetworkBandwidthGbpsRequest> builder(MarshallingType.SDK_POJO).memberName("networkBandwidthGbps")
            .getter(getter(InstanceRequirementsRequest::networkBandwidthGbps)).setter(setter(Builder::networkBandwidthGbps))
            .constructor(NetworkBandwidthGbpsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkBandwidthGbps").build())
            .build();

    private static final SdkField<List<String>> ALLOWED_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("allowedInstanceTypes")
            .getter(getter(InstanceRequirementsRequest::allowedInstanceTypes))
            .setter(setter(Builder::allowedInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("allowedInstanceTypes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_SPOT_PRICE_AS_PERCENTAGE_OF_OPTIMAL_ON_DEMAND_PRICE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxSpotPriceAsPercentageOfOptimalOnDemandPrice")
            .getter(getter(InstanceRequirementsRequest::maxSpotPriceAsPercentageOfOptimalOnDemandPrice))
            .setter(setter(Builder::maxSpotPriceAsPercentageOfOptimalOnDemandPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("maxSpotPriceAsPercentageOfOptimalOnDemandPrice").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(V_CPU_COUNT_FIELD,
            MEMORY_MIB_FIELD, CPU_MANUFACTURERS_FIELD, MEMORY_GI_B_PER_V_CPU_FIELD, EXCLUDED_INSTANCE_TYPES_FIELD,
            INSTANCE_GENERATIONS_FIELD, SPOT_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD,
            ON_DEMAND_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD, BARE_METAL_FIELD, BURSTABLE_PERFORMANCE_FIELD,
            REQUIRE_HIBERNATE_SUPPORT_FIELD, NETWORK_INTERFACE_COUNT_FIELD, LOCAL_STORAGE_FIELD, LOCAL_STORAGE_TYPES_FIELD,
            TOTAL_LOCAL_STORAGE_GB_FIELD, BASELINE_EBS_BANDWIDTH_MBPS_FIELD, ACCELERATOR_TYPES_FIELD, ACCELERATOR_COUNT_FIELD,
            ACCELERATOR_MANUFACTURERS_FIELD, ACCELERATOR_NAMES_FIELD, ACCELERATOR_TOTAL_MEMORY_MIB_FIELD,
            NETWORK_BANDWIDTH_GBPS_FIELD, ALLOWED_INSTANCE_TYPES_FIELD,
            MAX_SPOT_PRICE_AS_PERCENTAGE_OF_OPTIMAL_ON_DEMAND_PRICE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final VCpuCountRangeRequest vCpuCount;

    private final MemoryMiBRequest memoryMiB;

    private final List<String> cpuManufacturers;

    private final MemoryGiBPerVCpuRequest memoryGiBPerVCpu;

    private final List<String> excludedInstanceTypes;

    private final List<String> instanceGenerations;

    private final Integer spotMaxPricePercentageOverLowestPrice;

    private final Integer onDemandMaxPricePercentageOverLowestPrice;

    private final String bareMetal;

    private final String burstablePerformance;

    private final Boolean requireHibernateSupport;

    private final NetworkInterfaceCountRequest networkInterfaceCount;

    private final String localStorage;

    private final List<String> localStorageTypes;

    private final TotalLocalStorageGBRequest totalLocalStorageGB;

    private final BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps;

    private final List<String> acceleratorTypes;

    private final AcceleratorCountRequest acceleratorCount;

    private final List<String> acceleratorManufacturers;

    private final List<String> acceleratorNames;

    private final AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB;

    private final NetworkBandwidthGbpsRequest networkBandwidthGbps;

    private final List<String> allowedInstanceTypes;

    private final Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice;

    private InstanceRequirementsRequest(BuilderImpl builder) {
        this.vCpuCount = builder.vCpuCount;
        this.memoryMiB = builder.memoryMiB;
        this.cpuManufacturers = builder.cpuManufacturers;
        this.memoryGiBPerVCpu = builder.memoryGiBPerVCpu;
        this.excludedInstanceTypes = builder.excludedInstanceTypes;
        this.instanceGenerations = builder.instanceGenerations;
        this.spotMaxPricePercentageOverLowestPrice = builder.spotMaxPricePercentageOverLowestPrice;
        this.onDemandMaxPricePercentageOverLowestPrice = builder.onDemandMaxPricePercentageOverLowestPrice;
        this.bareMetal = builder.bareMetal;
        this.burstablePerformance = builder.burstablePerformance;
        this.requireHibernateSupport = builder.requireHibernateSupport;
        this.networkInterfaceCount = builder.networkInterfaceCount;
        this.localStorage = builder.localStorage;
        this.localStorageTypes = builder.localStorageTypes;
        this.totalLocalStorageGB = builder.totalLocalStorageGB;
        this.baselineEbsBandwidthMbps = builder.baselineEbsBandwidthMbps;
        this.acceleratorTypes = builder.acceleratorTypes;
        this.acceleratorCount = builder.acceleratorCount;
        this.acceleratorManufacturers = builder.acceleratorManufacturers;
        this.acceleratorNames = builder.acceleratorNames;
        this.acceleratorTotalMemoryMiB = builder.acceleratorTotalMemoryMiB;
        this.networkBandwidthGbps = builder.networkBandwidthGbps;
        this.allowedInstanceTypes = builder.allowedInstanceTypes;
        this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = builder.maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
    }

    /**
     * <p>
     * The minimum and maximum number of vCPUs for the instance types. Amazon ECS selects instance types that have vCPU
     * counts within this range.
     * </p>
     * 
     * @return The minimum and maximum number of vCPUs for the instance types. Amazon ECS selects instance types that
     *         have vCPU counts within this range.
     */
    public final VCpuCountRangeRequest vCpuCount() {
        return vCpuCount;
    }

    /**
     * <p>
     * The minimum and maximum amount of memory in mebibytes (MiB) for the instance types. Amazon ECS selects instance
     * types that have memory within this range.
     * </p>
     * 
     * @return The minimum and maximum amount of memory in mebibytes (MiB) for the instance types. Amazon ECS selects
     *         instance types that have memory within this range.
     */
    public final MemoryMiBRequest memoryMiB() {
        return memoryMiB;
    }

    /**
     * <p>
     * The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
     * <code>amazon-web-services</code> to control which CPU types are used for your workloads.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCpuManufacturers} method.
     * </p>
     * 
     * @return The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
     *         <code>amazon-web-services</code> to control which CPU types are used for your workloads.
     */
    public final List<CpuManufacturer> cpuManufacturers() {
        return CpuManufacturerSetCopier.copyStringToEnum(cpuManufacturers);
    }

    /**
     * For responses, this returns true if the service returned a value for the CpuManufacturers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCpuManufacturers() {
        return cpuManufacturers != null && !(cpuManufacturers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
     * <code>amazon-web-services</code> to control which CPU types are used for your workloads.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCpuManufacturers} method.
     * </p>
     * 
     * @return The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
     *         <code>amazon-web-services</code> to control which CPU types are used for your workloads.
     */
    public final List<String> cpuManufacturersAsStrings() {
        return cpuManufacturers;
    }

    /**
     * <p>
     * The minimum and maximum amount of memory per vCPU in gibibytes (GiB). This helps ensure that instance types have
     * the appropriate memory-to-CPU ratio for your workloads.
     * </p>
     * 
     * @return The minimum and maximum amount of memory per vCPU in gibibytes (GiB). This helps ensure that instance
     *         types have the appropriate memory-to-CPU ratio for your workloads.
     */
    public final MemoryGiBPerVCpuRequest memoryGiBPerVCpu() {
        return memoryGiBPerVCpu;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedInstanceTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasExcludedInstanceTypes() {
        return excludedInstanceTypes != null && !(excludedInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance types to exclude from selection. Use this to prevent Amazon ECS from selecting specific instance
     * types that may not be suitable for your workloads.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedInstanceTypes} method.
     * </p>
     * 
     * @return The instance types to exclude from selection. Use this to prevent Amazon ECS from selecting specific
     *         instance types that may not be suitable for your workloads.
     */
    public final List<String> excludedInstanceTypes() {
        return excludedInstanceTypes;
    }

    /**
     * <p>
     * The instance generations to include. You can specify <code>current</code> to use the latest generation instances,
     * or <code>previous</code> to include previous generation instances for cost optimization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceGenerations} method.
     * </p>
     * 
     * @return The instance generations to include. You can specify <code>current</code> to use the latest generation
     *         instances, or <code>previous</code> to include previous generation instances for cost optimization.
     */
    public final List<InstanceGeneration> instanceGenerations() {
        return InstanceGenerationSetCopier.copyStringToEnum(instanceGenerations);
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceGenerations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceGenerations() {
        return instanceGenerations != null && !(instanceGenerations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance generations to include. You can specify <code>current</code> to use the latest generation instances,
     * or <code>previous</code> to include previous generation instances for cost optimization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceGenerations} method.
     * </p>
     * 
     * @return The instance generations to include. You can specify <code>current</code> to use the latest generation
     *         instances, or <code>previous</code> to include previous generation instances for cost optimization.
     */
    public final List<String> instanceGenerationsAsStrings() {
        return instanceGenerations;
    }

    /**
     * <p>
     * The maximum price for Spot instances as a percentage over the lowest priced On-Demand instance. This helps
     * control Spot instance costs while maintaining access to capacity.
     * </p>
     * 
     * @return The maximum price for Spot instances as a percentage over the lowest priced On-Demand instance. This
     *         helps control Spot instance costs while maintaining access to capacity.
     */
    public final Integer spotMaxPricePercentageOverLowestPrice() {
        return spotMaxPricePercentageOverLowestPrice;
    }

    /**
     * <p>
     * The price protection threshold for On-Demand Instances, as a percentage higher than an identified On-Demand
     * price. The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance
     * type with your specified attributes. If no current generation C, M, or R instance type matches your attributes,
     * then the identified price is from either the lowest priced current generation instance types or, failing that,
     * the lowest priced previous generation instance types that match your attributes. When Amazon ECS selects instance
     * types with your attributes, we will exclude instance types whose price exceeds your specified threshold.
     * </p>
     * 
     * @return The price protection threshold for On-Demand Instances, as a percentage higher than an identified
     *         On-Demand price. The identified On-Demand price is the price of the lowest priced current generation C,
     *         M, or R instance type with your specified attributes. If no current generation C, M, or R instance type
     *         matches your attributes, then the identified price is from either the lowest priced current generation
     *         instance types or, failing that, the lowest priced previous generation instance types that match your
     *         attributes. When Amazon ECS selects instance types with your attributes, we will exclude instance types
     *         whose price exceeds your specified threshold.
     */
    public final Integer onDemandMaxPricePercentageOverLowestPrice() {
        return onDemandMaxPricePercentageOverLowestPrice;
    }

    /**
     * <p>
     * Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare metal
     * instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare metal instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bareMetal} will
     * return {@link BareMetal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bareMetalAsString}.
     * </p>
     * 
     * @return Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare metal
     *         instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare metal
     *         instances.
     * @see BareMetal
     */
    public final BareMetal bareMetal() {
        return BareMetal.fromValue(bareMetal);
    }

    /**
     * <p>
     * Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare metal
     * instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare metal instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bareMetal} will
     * return {@link BareMetal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bareMetalAsString}.
     * </p>
     * 
     * @return Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare metal
     *         instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare metal
     *         instances.
     * @see BareMetal
     */
    public final String bareMetalAsString() {
        return bareMetal;
    }

    /**
     * <p>
     * Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
     * <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
     * <code>required</code> to use only burstable instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #burstablePerformance} will return {@link BurstablePerformance#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #burstablePerformanceAsString}.
     * </p>
     * 
     * @return Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
     *         <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
     *         <code>required</code> to use only burstable instances.
     * @see BurstablePerformance
     */
    public final BurstablePerformance burstablePerformance() {
        return BurstablePerformance.fromValue(burstablePerformance);
    }

    /**
     * <p>
     * Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
     * <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
     * <code>required</code> to use only burstable instances.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #burstablePerformance} will return {@link BurstablePerformance#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #burstablePerformanceAsString}.
     * </p>
     * 
     * @return Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
     *         <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
     *         <code>required</code> to use only burstable instances.
     * @see BurstablePerformance
     */
    public final String burstablePerformanceAsString() {
        return burstablePerformance;
    }

    /**
     * <p>
     * Indicates whether the instance types must support hibernation. When set to <code>true</code>, only instance types
     * that support hibernation are selected.
     * </p>
     * 
     * @return Indicates whether the instance types must support hibernation. When set to <code>true</code>, only
     *         instance types that support hibernation are selected.
     */
    public final Boolean requireHibernateSupport() {
        return requireHibernateSupport;
    }

    /**
     * <p>
     * The minimum and maximum number of network interfaces for the instance types. This is useful for workloads that
     * require multiple network interfaces.
     * </p>
     * 
     * @return The minimum and maximum number of network interfaces for the instance types. This is useful for workloads
     *         that require multiple network interfaces.
     */
    public final NetworkInterfaceCountRequest networkInterfaceCount() {
        return networkInterfaceCount;
    }

    /**
     * <p>
     * Indicates whether to include instance types with local storage. Set to <code>included</code> to allow local
     * storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances with local storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #localStorage} will
     * return {@link LocalStorage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #localStorageAsString}.
     * </p>
     * 
     * @return Indicates whether to include instance types with local storage. Set to <code>included</code> to allow
     *         local storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances with
     *         local storage.
     * @see LocalStorage
     */
    public final LocalStorage localStorage() {
        return LocalStorage.fromValue(localStorage);
    }

    /**
     * <p>
     * Indicates whether to include instance types with local storage. Set to <code>included</code> to allow local
     * storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances with local storage.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #localStorage} will
     * return {@link LocalStorage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #localStorageAsString}.
     * </p>
     * 
     * @return Indicates whether to include instance types with local storage. Set to <code>included</code> to allow
     *         local storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances with
     *         local storage.
     * @see LocalStorage
     */
    public final String localStorageAsString() {
        return localStorage;
    }

    /**
     * <p>
     * The local storage types to include. You can specify <code>hdd</code> for hard disk drives, <code>ssd</code> for
     * solid state drives, or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocalStorageTypes} method.
     * </p>
     * 
     * @return The local storage types to include. You can specify <code>hdd</code> for hard disk drives,
     *         <code>ssd</code> for solid state drives, or both.
     */
    public final List<LocalStorageType> localStorageTypes() {
        return LocalStorageTypeSetCopier.copyStringToEnum(localStorageTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the LocalStorageTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLocalStorageTypes() {
        return localStorageTypes != null && !(localStorageTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The local storage types to include. You can specify <code>hdd</code> for hard disk drives, <code>ssd</code> for
     * solid state drives, or both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocalStorageTypes} method.
     * </p>
     * 
     * @return The local storage types to include. You can specify <code>hdd</code> for hard disk drives,
     *         <code>ssd</code> for solid state drives, or both.
     */
    public final List<String> localStorageTypesAsStrings() {
        return localStorageTypes;
    }

    /**
     * <p>
     * The minimum and maximum total local storage in gigabytes (GB) for instance types with local storage.
     * </p>
     * 
     * @return The minimum and maximum total local storage in gigabytes (GB) for instance types with local storage.
     */
    public final TotalLocalStorageGBRequest totalLocalStorageGB() {
        return totalLocalStorageGB;
    }

    /**
     * <p>
     * The minimum and maximum baseline Amazon EBS bandwidth in megabits per second (Mbps). This is important for
     * workloads with high storage I/O requirements.
     * </p>
     * 
     * @return The minimum and maximum baseline Amazon EBS bandwidth in megabits per second (Mbps). This is important
     *         for workloads with high storage I/O requirements.
     */
    public final BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps() {
        return baselineEbsBandwidthMbps;
    }

    /**
     * <p>
     * The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
     * <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning inference
     * accelerators.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorTypes} method.
     * </p>
     * 
     * @return The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
     *         <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
     *         inference accelerators.
     */
    public final List<AcceleratorType> acceleratorTypes() {
        return AcceleratorTypeSetCopier.copyStringToEnum(acceleratorTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the AcceleratorTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAcceleratorTypes() {
        return acceleratorTypes != null && !(acceleratorTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
     * <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning inference
     * accelerators.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorTypes} method.
     * </p>
     * 
     * @return The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
     *         <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
     *         inference accelerators.
     */
    public final List<String> acceleratorTypesAsStrings() {
        return acceleratorTypes;
    }

    /**
     * <p>
     * The minimum and maximum number of accelerators for the instance types. This is used when you need instances with
     * specific numbers of GPUs or other accelerators.
     * </p>
     * 
     * @return The minimum and maximum number of accelerators for the instance types. This is used when you need
     *         instances with specific numbers of GPUs or other accelerators.
     */
    public final AcceleratorCountRequest acceleratorCount() {
        return acceleratorCount;
    }

    /**
     * <p>
     * The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
     * <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorManufacturers} method.
     * </p>
     * 
     * @return The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
     *         <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
     */
    public final List<AcceleratorManufacturer> acceleratorManufacturers() {
        return AcceleratorManufacturerSetCopier.copyStringToEnum(acceleratorManufacturers);
    }

    /**
     * For responses, this returns true if the service returned a value for the AcceleratorManufacturers property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAcceleratorManufacturers() {
        return acceleratorManufacturers != null && !(acceleratorManufacturers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
     * <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorManufacturers} method.
     * </p>
     * 
     * @return The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
     *         <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
     */
    public final List<String> acceleratorManufacturersAsStrings() {
        return acceleratorManufacturers;
    }

    /**
     * <p>
     * The specific accelerator names to include. For example, you can specify <code>a100</code>, <code>v100</code>,
     * <code>k80</code>, or other specific accelerator models.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorNames} method.
     * </p>
     * 
     * @return The specific accelerator names to include. For example, you can specify <code>a100</code>,
     *         <code>v100</code>, <code>k80</code>, or other specific accelerator models.
     */
    public final List<AcceleratorName> acceleratorNames() {
        return AcceleratorNameSetCopier.copyStringToEnum(acceleratorNames);
    }

    /**
     * For responses, this returns true if the service returned a value for the AcceleratorNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAcceleratorNames() {
        return acceleratorNames != null && !(acceleratorNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The specific accelerator names to include. For example, you can specify <code>a100</code>, <code>v100</code>,
     * <code>k80</code>, or other specific accelerator models.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorNames} method.
     * </p>
     * 
     * @return The specific accelerator names to include. For example, you can specify <code>a100</code>,
     *         <code>v100</code>, <code>k80</code>, or other specific accelerator models.
     */
    public final List<String> acceleratorNamesAsStrings() {
        return acceleratorNames;
    }

    /**
     * <p>
     * The minimum and maximum total accelerator memory in mebibytes (MiB). This is important for GPU workloads that
     * require specific amounts of video memory.
     * </p>
     * 
     * @return The minimum and maximum total accelerator memory in mebibytes (MiB). This is important for GPU workloads
     *         that require specific amounts of video memory.
     */
    public final AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB() {
        return acceleratorTotalMemoryMiB;
    }

    /**
     * <p>
     * The minimum and maximum network bandwidth in gigabits per second (Gbps). This is crucial for network-intensive
     * workloads that require high throughput.
     * </p>
     * 
     * @return The minimum and maximum network bandwidth in gigabits per second (Gbps). This is crucial for
     *         network-intensive workloads that require high throughput.
     */
    public final NetworkBandwidthGbpsRequest networkBandwidthGbps() {
        return networkBandwidthGbps;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedInstanceTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedInstanceTypes() {
        return allowedInstanceTypes != null && !(allowedInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance types to include in the selection. When specified, Amazon ECS only considers these instance types,
     * subject to the other requirements specified.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedInstanceTypes} method.
     * </p>
     * 
     * @return The instance types to include in the selection. When specified, Amazon ECS only considers these instance
     *         types, subject to the other requirements specified.
     */
    public final List<String> allowedInstanceTypes() {
        return allowedInstanceTypes;
    }

    /**
     * <p>
     * The maximum price for Spot instances as a percentage of the optimal On-Demand price. This provides more precise
     * cost control for Spot instance selection.
     * </p>
     * 
     * @return The maximum price for Spot instances as a percentage of the optimal On-Demand price. This provides more
     *         precise cost control for Spot instance selection.
     */
    public final Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice() {
        return maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vCpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(memoryMiB());
        hashCode = 31 * hashCode + Objects.hashCode(hasCpuManufacturers() ? cpuManufacturersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(memoryGiBPerVCpu());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedInstanceTypes() ? excludedInstanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceGenerations() ? instanceGenerationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(spotMaxPricePercentageOverLowestPrice());
        hashCode = 31 * hashCode + Objects.hashCode(onDemandMaxPricePercentageOverLowestPrice());
        hashCode = 31 * hashCode + Objects.hashCode(bareMetalAsString());
        hashCode = 31 * hashCode + Objects.hashCode(burstablePerformanceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requireHibernateSupport());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceCount());
        hashCode = 31 * hashCode + Objects.hashCode(localStorageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLocalStorageTypes() ? localStorageTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(totalLocalStorageGB());
        hashCode = 31 * hashCode + Objects.hashCode(baselineEbsBandwidthMbps());
        hashCode = 31 * hashCode + Objects.hashCode(hasAcceleratorTypes() ? acceleratorTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasAcceleratorManufacturers() ? acceleratorManufacturersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAcceleratorNames() ? acceleratorNamesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorTotalMemoryMiB());
        hashCode = 31 * hashCode + Objects.hashCode(networkBandwidthGbps());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedInstanceTypes() ? allowedInstanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxSpotPriceAsPercentageOfOptimalOnDemandPrice());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceRequirementsRequest)) {
            return false;
        }
        InstanceRequirementsRequest other = (InstanceRequirementsRequest) obj;
        return Objects.equals(vCpuCount(), other.vCpuCount())
                && Objects.equals(memoryMiB(), other.memoryMiB())
                && hasCpuManufacturers() == other.hasCpuManufacturers()
                && Objects.equals(cpuManufacturersAsStrings(), other.cpuManufacturersAsStrings())
                && Objects.equals(memoryGiBPerVCpu(), other.memoryGiBPerVCpu())
                && hasExcludedInstanceTypes() == other.hasExcludedInstanceTypes()
                && Objects.equals(excludedInstanceTypes(), other.excludedInstanceTypes())
                && hasInstanceGenerations() == other.hasInstanceGenerations()
                && Objects.equals(instanceGenerationsAsStrings(), other.instanceGenerationsAsStrings())
                && Objects.equals(spotMaxPricePercentageOverLowestPrice(), other.spotMaxPricePercentageOverLowestPrice())
                && Objects.equals(onDemandMaxPricePercentageOverLowestPrice(), other.onDemandMaxPricePercentageOverLowestPrice())
                && Objects.equals(bareMetalAsString(), other.bareMetalAsString())
                && Objects.equals(burstablePerformanceAsString(), other.burstablePerformanceAsString())
                && Objects.equals(requireHibernateSupport(), other.requireHibernateSupport())
                && Objects.equals(networkInterfaceCount(), other.networkInterfaceCount())
                && Objects.equals(localStorageAsString(), other.localStorageAsString())
                && hasLocalStorageTypes() == other.hasLocalStorageTypes()
                && Objects.equals(localStorageTypesAsStrings(), other.localStorageTypesAsStrings())
                && Objects.equals(totalLocalStorageGB(), other.totalLocalStorageGB())
                && Objects.equals(baselineEbsBandwidthMbps(), other.baselineEbsBandwidthMbps())
                && hasAcceleratorTypes() == other.hasAcceleratorTypes()
                && Objects.equals(acceleratorTypesAsStrings(), other.acceleratorTypesAsStrings())
                && Objects.equals(acceleratorCount(), other.acceleratorCount())
                && hasAcceleratorManufacturers() == other.hasAcceleratorManufacturers()
                && Objects.equals(acceleratorManufacturersAsStrings(), other.acceleratorManufacturersAsStrings())
                && hasAcceleratorNames() == other.hasAcceleratorNames()
                && Objects.equals(acceleratorNamesAsStrings(), other.acceleratorNamesAsStrings())
                && Objects.equals(acceleratorTotalMemoryMiB(), other.acceleratorTotalMemoryMiB())
                && Objects.equals(networkBandwidthGbps(), other.networkBandwidthGbps())
                && hasAllowedInstanceTypes() == other.hasAllowedInstanceTypes()
                && Objects.equals(allowedInstanceTypes(), other.allowedInstanceTypes())
                && Objects.equals(maxSpotPriceAsPercentageOfOptimalOnDemandPrice(),
                        other.maxSpotPriceAsPercentageOfOptimalOnDemandPrice());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceRequirementsRequest").add("VCpuCount", vCpuCount()).add("MemoryMiB", memoryMiB())
                .add("CpuManufacturers", hasCpuManufacturers() ? cpuManufacturersAsStrings() : null)
                .add("MemoryGiBPerVCpu", memoryGiBPerVCpu())
                .add("ExcludedInstanceTypes", hasExcludedInstanceTypes() ? excludedInstanceTypes() : null)
                .add("InstanceGenerations", hasInstanceGenerations() ? instanceGenerationsAsStrings() : null)
                .add("SpotMaxPricePercentageOverLowestPrice", spotMaxPricePercentageOverLowestPrice())
                .add("OnDemandMaxPricePercentageOverLowestPrice", onDemandMaxPricePercentageOverLowestPrice())
                .add("BareMetal", bareMetalAsString()).add("BurstablePerformance", burstablePerformanceAsString())
                .add("RequireHibernateSupport", requireHibernateSupport()).add("NetworkInterfaceCount", networkInterfaceCount())
                .add("LocalStorage", localStorageAsString())
                .add("LocalStorageTypes", hasLocalStorageTypes() ? localStorageTypesAsStrings() : null)
                .add("TotalLocalStorageGB", totalLocalStorageGB()).add("BaselineEbsBandwidthMbps", baselineEbsBandwidthMbps())
                .add("AcceleratorTypes", hasAcceleratorTypes() ? acceleratorTypesAsStrings() : null)
                .add("AcceleratorCount", acceleratorCount())
                .add("AcceleratorManufacturers", hasAcceleratorManufacturers() ? acceleratorManufacturersAsStrings() : null)
                .add("AcceleratorNames", hasAcceleratorNames() ? acceleratorNamesAsStrings() : null)
                .add("AcceleratorTotalMemoryMiB", acceleratorTotalMemoryMiB())
                .add("NetworkBandwidthGbps", networkBandwidthGbps())
                .add("AllowedInstanceTypes", hasAllowedInstanceTypes() ? allowedInstanceTypes() : null)
                .add("MaxSpotPriceAsPercentageOfOptimalOnDemandPrice", maxSpotPriceAsPercentageOfOptimalOnDemandPrice()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "vCpuCount":
            return Optional.ofNullable(clazz.cast(vCpuCount()));
        case "memoryMiB":
            return Optional.ofNullable(clazz.cast(memoryMiB()));
        case "cpuManufacturers":
            return Optional.ofNullable(clazz.cast(cpuManufacturersAsStrings()));
        case "memoryGiBPerVCpu":
            return Optional.ofNullable(clazz.cast(memoryGiBPerVCpu()));
        case "excludedInstanceTypes":
            return Optional.ofNullable(clazz.cast(excludedInstanceTypes()));
        case "instanceGenerations":
            return Optional.ofNullable(clazz.cast(instanceGenerationsAsStrings()));
        case "spotMaxPricePercentageOverLowestPrice":
            return Optional.ofNullable(clazz.cast(spotMaxPricePercentageOverLowestPrice()));
        case "onDemandMaxPricePercentageOverLowestPrice":
            return Optional.ofNullable(clazz.cast(onDemandMaxPricePercentageOverLowestPrice()));
        case "bareMetal":
            return Optional.ofNullable(clazz.cast(bareMetalAsString()));
        case "burstablePerformance":
            return Optional.ofNullable(clazz.cast(burstablePerformanceAsString()));
        case "requireHibernateSupport":
            return Optional.ofNullable(clazz.cast(requireHibernateSupport()));
        case "networkInterfaceCount":
            return Optional.ofNullable(clazz.cast(networkInterfaceCount()));
        case "localStorage":
            return Optional.ofNullable(clazz.cast(localStorageAsString()));
        case "localStorageTypes":
            return Optional.ofNullable(clazz.cast(localStorageTypesAsStrings()));
        case "totalLocalStorageGB":
            return Optional.ofNullable(clazz.cast(totalLocalStorageGB()));
        case "baselineEbsBandwidthMbps":
            return Optional.ofNullable(clazz.cast(baselineEbsBandwidthMbps()));
        case "acceleratorTypes":
            return Optional.ofNullable(clazz.cast(acceleratorTypesAsStrings()));
        case "acceleratorCount":
            return Optional.ofNullable(clazz.cast(acceleratorCount()));
        case "acceleratorManufacturers":
            return Optional.ofNullable(clazz.cast(acceleratorManufacturersAsStrings()));
        case "acceleratorNames":
            return Optional.ofNullable(clazz.cast(acceleratorNamesAsStrings()));
        case "acceleratorTotalMemoryMiB":
            return Optional.ofNullable(clazz.cast(acceleratorTotalMemoryMiB()));
        case "networkBandwidthGbps":
            return Optional.ofNullable(clazz.cast(networkBandwidthGbps()));
        case "allowedInstanceTypes":
            return Optional.ofNullable(clazz.cast(allowedInstanceTypes()));
        case "maxSpotPriceAsPercentageOfOptimalOnDemandPrice":
            return Optional.ofNullable(clazz.cast(maxSpotPriceAsPercentageOfOptimalOnDemandPrice()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("vCpuCount", V_CPU_COUNT_FIELD);
        map.put("memoryMiB", MEMORY_MIB_FIELD);
        map.put("cpuManufacturers", CPU_MANUFACTURERS_FIELD);
        map.put("memoryGiBPerVCpu", MEMORY_GI_B_PER_V_CPU_FIELD);
        map.put("excludedInstanceTypes", EXCLUDED_INSTANCE_TYPES_FIELD);
        map.put("instanceGenerations", INSTANCE_GENERATIONS_FIELD);
        map.put("spotMaxPricePercentageOverLowestPrice", SPOT_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD);
        map.put("onDemandMaxPricePercentageOverLowestPrice", ON_DEMAND_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD);
        map.put("bareMetal", BARE_METAL_FIELD);
        map.put("burstablePerformance", BURSTABLE_PERFORMANCE_FIELD);
        map.put("requireHibernateSupport", REQUIRE_HIBERNATE_SUPPORT_FIELD);
        map.put("networkInterfaceCount", NETWORK_INTERFACE_COUNT_FIELD);
        map.put("localStorage", LOCAL_STORAGE_FIELD);
        map.put("localStorageTypes", LOCAL_STORAGE_TYPES_FIELD);
        map.put("totalLocalStorageGB", TOTAL_LOCAL_STORAGE_GB_FIELD);
        map.put("baselineEbsBandwidthMbps", BASELINE_EBS_BANDWIDTH_MBPS_FIELD);
        map.put("acceleratorTypes", ACCELERATOR_TYPES_FIELD);
        map.put("acceleratorCount", ACCELERATOR_COUNT_FIELD);
        map.put("acceleratorManufacturers", ACCELERATOR_MANUFACTURERS_FIELD);
        map.put("acceleratorNames", ACCELERATOR_NAMES_FIELD);
        map.put("acceleratorTotalMemoryMiB", ACCELERATOR_TOTAL_MEMORY_MIB_FIELD);
        map.put("networkBandwidthGbps", NETWORK_BANDWIDTH_GBPS_FIELD);
        map.put("allowedInstanceTypes", ALLOWED_INSTANCE_TYPES_FIELD);
        map.put("maxSpotPriceAsPercentageOfOptimalOnDemandPrice", MAX_SPOT_PRICE_AS_PERCENTAGE_OF_OPTIMAL_ON_DEMAND_PRICE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<InstanceRequirementsRequest, T> g) {
        return obj -> g.apply((InstanceRequirementsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceRequirementsRequest> {
        /**
         * <p>
         * The minimum and maximum number of vCPUs for the instance types. Amazon ECS selects instance types that have
         * vCPU counts within this range.
         * </p>
         * 
         * @param vCpuCount
         *        The minimum and maximum number of vCPUs for the instance types. Amazon ECS selects instance types that
         *        have vCPU counts within this range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vCpuCount(VCpuCountRangeRequest vCpuCount);

        /**
         * <p>
         * The minimum and maximum number of vCPUs for the instance types. Amazon ECS selects instance types that have
         * vCPU counts within this range.
         * </p>
         * This is a convenience method that creates an instance of the {@link VCpuCountRangeRequest.Builder} avoiding
         * the need to create one manually via {@link VCpuCountRangeRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VCpuCountRangeRequest.Builder#build()} is called immediately and
         * its result is passed to {@link #vCpuCount(VCpuCountRangeRequest)}.
         * 
         * @param vCpuCount
         *        a consumer that will call methods on {@link VCpuCountRangeRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vCpuCount(VCpuCountRangeRequest)
         */
        default Builder vCpuCount(Consumer<VCpuCountRangeRequest.Builder> vCpuCount) {
            return vCpuCount(VCpuCountRangeRequest.builder().applyMutation(vCpuCount).build());
        }

        /**
         * <p>
         * The minimum and maximum amount of memory in mebibytes (MiB) for the instance types. Amazon ECS selects
         * instance types that have memory within this range.
         * </p>
         * 
         * @param memoryMiB
         *        The minimum and maximum amount of memory in mebibytes (MiB) for the instance types. Amazon ECS selects
         *        instance types that have memory within this range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryMiB(MemoryMiBRequest memoryMiB);

        /**
         * <p>
         * The minimum and maximum amount of memory in mebibytes (MiB) for the instance types. Amazon ECS selects
         * instance types that have memory within this range.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryMiBRequest.Builder} avoiding the
         * need to create one manually via {@link MemoryMiBRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryMiBRequest.Builder#build()} is called immediately and its
         * result is passed to {@link #memoryMiB(MemoryMiBRequest)}.
         * 
         * @param memoryMiB
         *        a consumer that will call methods on {@link MemoryMiBRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryMiB(MemoryMiBRequest)
         */
        default Builder memoryMiB(Consumer<MemoryMiBRequest.Builder> memoryMiB) {
            return memoryMiB(MemoryMiBRequest.builder().applyMutation(memoryMiB).build());
        }

        /**
         * <p>
         * The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         * <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         *        <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturersWithStrings(Collection<String> cpuManufacturers);

        /**
         * <p>
         * The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         * <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         *        <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturersWithStrings(String... cpuManufacturers);

        /**
         * <p>
         * The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         * <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         *        <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturers(Collection<CpuManufacturer> cpuManufacturers);

        /**
         * <p>
         * The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         * <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include or exclude. You can specify <code>intel</code>, <code>amd</code>, or
         *        <code>amazon-web-services</code> to control which CPU types are used for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturers(CpuManufacturer... cpuManufacturers);

        /**
         * <p>
         * The minimum and maximum amount of memory per vCPU in gibibytes (GiB). This helps ensure that instance types
         * have the appropriate memory-to-CPU ratio for your workloads.
         * </p>
         * 
         * @param memoryGiBPerVCpu
         *        The minimum and maximum amount of memory per vCPU in gibibytes (GiB). This helps ensure that instance
         *        types have the appropriate memory-to-CPU ratio for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryGiBPerVCpu(MemoryGiBPerVCpuRequest memoryGiBPerVCpu);

        /**
         * <p>
         * The minimum and maximum amount of memory per vCPU in gibibytes (GiB). This helps ensure that instance types
         * have the appropriate memory-to-CPU ratio for your workloads.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryGiBPerVCpuRequest.Builder} avoiding
         * the need to create one manually via {@link MemoryGiBPerVCpuRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryGiBPerVCpuRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #memoryGiBPerVCpu(MemoryGiBPerVCpuRequest)}.
         * 
         * @param memoryGiBPerVCpu
         *        a consumer that will call methods on {@link MemoryGiBPerVCpuRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryGiBPerVCpu(MemoryGiBPerVCpuRequest)
         */
        default Builder memoryGiBPerVCpu(Consumer<MemoryGiBPerVCpuRequest.Builder> memoryGiBPerVCpu) {
            return memoryGiBPerVCpu(MemoryGiBPerVCpuRequest.builder().applyMutation(memoryGiBPerVCpu).build());
        }

        /**
         * <p>
         * The instance types to exclude from selection. Use this to prevent Amazon ECS from selecting specific instance
         * types that may not be suitable for your workloads.
         * </p>
         * 
         * @param excludedInstanceTypes
         *        The instance types to exclude from selection. Use this to prevent Amazon ECS from selecting specific
         *        instance types that may not be suitable for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedInstanceTypes(Collection<String> excludedInstanceTypes);

        /**
         * <p>
         * The instance types to exclude from selection. Use this to prevent Amazon ECS from selecting specific instance
         * types that may not be suitable for your workloads.
         * </p>
         * 
         * @param excludedInstanceTypes
         *        The instance types to exclude from selection. Use this to prevent Amazon ECS from selecting specific
         *        instance types that may not be suitable for your workloads.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedInstanceTypes(String... excludedInstanceTypes);

        /**
         * <p>
         * The instance generations to include. You can specify <code>current</code> to use the latest generation
         * instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * </p>
         * 
         * @param instanceGenerations
         *        The instance generations to include. You can specify <code>current</code> to use the latest generation
         *        instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerationsWithStrings(Collection<String> instanceGenerations);

        /**
         * <p>
         * The instance generations to include. You can specify <code>current</code> to use the latest generation
         * instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * </p>
         * 
         * @param instanceGenerations
         *        The instance generations to include. You can specify <code>current</code> to use the latest generation
         *        instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerationsWithStrings(String... instanceGenerations);

        /**
         * <p>
         * The instance generations to include. You can specify <code>current</code> to use the latest generation
         * instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * </p>
         * 
         * @param instanceGenerations
         *        The instance generations to include. You can specify <code>current</code> to use the latest generation
         *        instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerations(Collection<InstanceGeneration> instanceGenerations);

        /**
         * <p>
         * The instance generations to include. You can specify <code>current</code> to use the latest generation
         * instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * </p>
         * 
         * @param instanceGenerations
         *        The instance generations to include. You can specify <code>current</code> to use the latest generation
         *        instances, or <code>previous</code> to include previous generation instances for cost optimization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerations(InstanceGeneration... instanceGenerations);

        /**
         * <p>
         * The maximum price for Spot instances as a percentage over the lowest priced On-Demand instance. This helps
         * control Spot instance costs while maintaining access to capacity.
         * </p>
         * 
         * @param spotMaxPricePercentageOverLowestPrice
         *        The maximum price for Spot instances as a percentage over the lowest priced On-Demand instance. This
         *        helps control Spot instance costs while maintaining access to capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotMaxPricePercentageOverLowestPrice(Integer spotMaxPricePercentageOverLowestPrice);

        /**
         * <p>
         * The price protection threshold for On-Demand Instances, as a percentage higher than an identified On-Demand
         * price. The identified On-Demand price is the price of the lowest priced current generation C, M, or R
         * instance type with your specified attributes. If no current generation C, M, or R instance type matches your
         * attributes, then the identified price is from either the lowest priced current generation instance types or,
         * failing that, the lowest priced previous generation instance types that match your attributes. When Amazon
         * ECS selects instance types with your attributes, we will exclude instance types whose price exceeds your
         * specified threshold.
         * </p>
         * 
         * @param onDemandMaxPricePercentageOverLowestPrice
         *        The price protection threshold for On-Demand Instances, as a percentage higher than an identified
         *        On-Demand price. The identified On-Demand price is the price of the lowest priced current generation
         *        C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance
         *        type matches your attributes, then the identified price is from either the lowest priced current
         *        generation instance types or, failing that, the lowest priced previous generation instance types that
         *        match your attributes. When Amazon ECS selects instance types with your attributes, we will exclude
         *        instance types whose price exceeds your specified threshold.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemandMaxPricePercentageOverLowestPrice(Integer onDemandMaxPricePercentageOverLowestPrice);

        /**
         * <p>
         * Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare metal
         * instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare metal instances.
         * </p>
         * 
         * @param bareMetal
         *        Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare
         *        metal instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare
         *        metal instances.
         * @see BareMetal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BareMetal
         */
        Builder bareMetal(String bareMetal);

        /**
         * <p>
         * Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare metal
         * instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare metal instances.
         * </p>
         * 
         * @param bareMetal
         *        Indicates whether to include bare metal instance types. Set to <code>included</code> to allow bare
         *        metal instances, <code>excluded</code> to exclude them, or <code>required</code> to use only bare
         *        metal instances.
         * @see BareMetal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BareMetal
         */
        Builder bareMetal(BareMetal bareMetal);

        /**
         * <p>
         * Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
         * <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
         * <code>required</code> to use only burstable instances.
         * </p>
         * 
         * @param burstablePerformance
         *        Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
         *        <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
         *        <code>required</code> to use only burstable instances.
         * @see BurstablePerformance
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BurstablePerformance
         */
        Builder burstablePerformance(String burstablePerformance);

        /**
         * <p>
         * Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
         * <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
         * <code>required</code> to use only burstable instances.
         * </p>
         * 
         * @param burstablePerformance
         *        Indicates whether to include burstable performance instance types (T2, T3, T3a, T4g). Set to
         *        <code>included</code> to allow burstable instances, <code>excluded</code> to exclude them, or
         *        <code>required</code> to use only burstable instances.
         * @see BurstablePerformance
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BurstablePerformance
         */
        Builder burstablePerformance(BurstablePerformance burstablePerformance);

        /**
         * <p>
         * Indicates whether the instance types must support hibernation. When set to <code>true</code>, only instance
         * types that support hibernation are selected.
         * </p>
         * 
         * @param requireHibernateSupport
         *        Indicates whether the instance types must support hibernation. When set to <code>true</code>, only
         *        instance types that support hibernation are selected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requireHibernateSupport(Boolean requireHibernateSupport);

        /**
         * <p>
         * The minimum and maximum number of network interfaces for the instance types. This is useful for workloads
         * that require multiple network interfaces.
         * </p>
         * 
         * @param networkInterfaceCount
         *        The minimum and maximum number of network interfaces for the instance types. This is useful for
         *        workloads that require multiple network interfaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceCount(NetworkInterfaceCountRequest networkInterfaceCount);

        /**
         * <p>
         * The minimum and maximum number of network interfaces for the instance types. This is useful for workloads
         * that require multiple network interfaces.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkInterfaceCountRequest.Builder}
         * avoiding the need to create one manually via {@link NetworkInterfaceCountRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkInterfaceCountRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #networkInterfaceCount(NetworkInterfaceCountRequest)}.
         * 
         * @param networkInterfaceCount
         *        a consumer that will call methods on {@link NetworkInterfaceCountRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaceCount(NetworkInterfaceCountRequest)
         */
        default Builder networkInterfaceCount(Consumer<NetworkInterfaceCountRequest.Builder> networkInterfaceCount) {
            return networkInterfaceCount(NetworkInterfaceCountRequest.builder().applyMutation(networkInterfaceCount).build());
        }

        /**
         * <p>
         * Indicates whether to include instance types with local storage. Set to <code>included</code> to allow local
         * storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances with local
         * storage.
         * </p>
         * 
         * @param localStorage
         *        Indicates whether to include instance types with local storage. Set to <code>included</code> to allow
         *        local storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances
         *        with local storage.
         * @see LocalStorage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalStorage
         */
        Builder localStorage(String localStorage);

        /**
         * <p>
         * Indicates whether to include instance types with local storage. Set to <code>included</code> to allow local
         * storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances with local
         * storage.
         * </p>
         * 
         * @param localStorage
         *        Indicates whether to include instance types with local storage. Set to <code>included</code> to allow
         *        local storage, <code>excluded</code> to exclude it, or <code>required</code> to use only instances
         *        with local storage.
         * @see LocalStorage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalStorage
         */
        Builder localStorage(LocalStorage localStorage);

        /**
         * <p>
         * The local storage types to include. You can specify <code>hdd</code> for hard disk drives, <code>ssd</code>
         * for solid state drives, or both.
         * </p>
         * 
         * @param localStorageTypes
         *        The local storage types to include. You can specify <code>hdd</code> for hard disk drives,
         *        <code>ssd</code> for solid state drives, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypesWithStrings(Collection<String> localStorageTypes);

        /**
         * <p>
         * The local storage types to include. You can specify <code>hdd</code> for hard disk drives, <code>ssd</code>
         * for solid state drives, or both.
         * </p>
         * 
         * @param localStorageTypes
         *        The local storage types to include. You can specify <code>hdd</code> for hard disk drives,
         *        <code>ssd</code> for solid state drives, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypesWithStrings(String... localStorageTypes);

        /**
         * <p>
         * The local storage types to include. You can specify <code>hdd</code> for hard disk drives, <code>ssd</code>
         * for solid state drives, or both.
         * </p>
         * 
         * @param localStorageTypes
         *        The local storage types to include. You can specify <code>hdd</code> for hard disk drives,
         *        <code>ssd</code> for solid state drives, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypes(Collection<LocalStorageType> localStorageTypes);

        /**
         * <p>
         * The local storage types to include. You can specify <code>hdd</code> for hard disk drives, <code>ssd</code>
         * for solid state drives, or both.
         * </p>
         * 
         * @param localStorageTypes
         *        The local storage types to include. You can specify <code>hdd</code> for hard disk drives,
         *        <code>ssd</code> for solid state drives, or both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypes(LocalStorageType... localStorageTypes);

        /**
         * <p>
         * The minimum and maximum total local storage in gigabytes (GB) for instance types with local storage.
         * </p>
         * 
         * @param totalLocalStorageGB
         *        The minimum and maximum total local storage in gigabytes (GB) for instance types with local storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalLocalStorageGB(TotalLocalStorageGBRequest totalLocalStorageGB);

        /**
         * <p>
         * The minimum and maximum total local storage in gigabytes (GB) for instance types with local storage.
         * </p>
         * This is a convenience method that creates an instance of the {@link TotalLocalStorageGBRequest.Builder}
         * avoiding the need to create one manually via {@link TotalLocalStorageGBRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TotalLocalStorageGBRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #totalLocalStorageGB(TotalLocalStorageGBRequest)}.
         * 
         * @param totalLocalStorageGB
         *        a consumer that will call methods on {@link TotalLocalStorageGBRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #totalLocalStorageGB(TotalLocalStorageGBRequest)
         */
        default Builder totalLocalStorageGB(Consumer<TotalLocalStorageGBRequest.Builder> totalLocalStorageGB) {
            return totalLocalStorageGB(TotalLocalStorageGBRequest.builder().applyMutation(totalLocalStorageGB).build());
        }

        /**
         * <p>
         * The minimum and maximum baseline Amazon EBS bandwidth in megabits per second (Mbps). This is important for
         * workloads with high storage I/O requirements.
         * </p>
         * 
         * @param baselineEbsBandwidthMbps
         *        The minimum and maximum baseline Amazon EBS bandwidth in megabits per second (Mbps). This is important
         *        for workloads with high storage I/O requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps);

        /**
         * <p>
         * The minimum and maximum baseline Amazon EBS bandwidth in megabits per second (Mbps). This is important for
         * workloads with high storage I/O requirements.
         * </p>
         * This is a convenience method that creates an instance of the {@link BaselineEbsBandwidthMbpsRequest.Builder}
         * avoiding the need to create one manually via {@link BaselineEbsBandwidthMbpsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BaselineEbsBandwidthMbpsRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest)}.
         * 
         * @param baselineEbsBandwidthMbps
         *        a consumer that will call methods on {@link BaselineEbsBandwidthMbpsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest)
         */
        default Builder baselineEbsBandwidthMbps(Consumer<BaselineEbsBandwidthMbpsRequest.Builder> baselineEbsBandwidthMbps) {
            return baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest.builder().applyMutation(baselineEbsBandwidthMbps)
                    .build());
        }

        /**
         * <p>
         * The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         * <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         * inference accelerators.
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         *        <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         *        inference accelerators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypesWithStrings(Collection<String> acceleratorTypes);

        /**
         * <p>
         * The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         * <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         * inference accelerators.
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         *        <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         *        inference accelerators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypesWithStrings(String... acceleratorTypes);

        /**
         * <p>
         * The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         * <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         * inference accelerators.
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         *        <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         *        inference accelerators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypes(Collection<AcceleratorType> acceleratorTypes);

        /**
         * <p>
         * The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         * <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         * inference accelerators.
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types to include. You can specify <code>gpu</code> for graphics processing units,
         *        <code>fpga</code> for field programmable gate arrays, or <code>inference</code> for machine learning
         *        inference accelerators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypes(AcceleratorType... acceleratorTypes);

        /**
         * <p>
         * The minimum and maximum number of accelerators for the instance types. This is used when you need instances
         * with specific numbers of GPUs or other accelerators.
         * </p>
         * 
         * @param acceleratorCount
         *        The minimum and maximum number of accelerators for the instance types. This is used when you need
         *        instances with specific numbers of GPUs or other accelerators.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorCount(AcceleratorCountRequest acceleratorCount);

        /**
         * <p>
         * The minimum and maximum number of accelerators for the instance types. This is used when you need instances
         * with specific numbers of GPUs or other accelerators.
         * </p>
         * This is a convenience method that creates an instance of the {@link AcceleratorCountRequest.Builder} avoiding
         * the need to create one manually via {@link AcceleratorCountRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AcceleratorCountRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #acceleratorCount(AcceleratorCountRequest)}.
         * 
         * @param acceleratorCount
         *        a consumer that will call methods on {@link AcceleratorCountRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #acceleratorCount(AcceleratorCountRequest)
         */
        default Builder acceleratorCount(Consumer<AcceleratorCountRequest.Builder> acceleratorCount) {
            return acceleratorCount(AcceleratorCountRequest.builder().applyMutation(acceleratorCount).build());
        }

        /**
         * <p>
         * The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         * <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * </p>
         * 
         * @param acceleratorManufacturers
         *        The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         *        <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturersWithStrings(Collection<String> acceleratorManufacturers);

        /**
         * <p>
         * The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         * <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * </p>
         * 
         * @param acceleratorManufacturers
         *        The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         *        <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturersWithStrings(String... acceleratorManufacturers);

        /**
         * <p>
         * The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         * <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * </p>
         * 
         * @param acceleratorManufacturers
         *        The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         *        <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturers(Collection<AcceleratorManufacturer> acceleratorManufacturers);

        /**
         * <p>
         * The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         * <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * </p>
         * 
         * @param acceleratorManufacturers
         *        The accelerator manufacturers to include. You can specify <code>nvidia</code>, <code>amd</code>,
         *        <code>amazon-web-services</code>, or <code>xilinx</code> depending on your accelerator requirements.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturers(AcceleratorManufacturer... acceleratorManufacturers);

        /**
         * <p>
         * The specific accelerator names to include. For example, you can specify <code>a100</code>, <code>v100</code>,
         * <code>k80</code>, or other specific accelerator models.
         * </p>
         * 
         * @param acceleratorNames
         *        The specific accelerator names to include. For example, you can specify <code>a100</code>,
         *        <code>v100</code>, <code>k80</code>, or other specific accelerator models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNamesWithStrings(Collection<String> acceleratorNames);

        /**
         * <p>
         * The specific accelerator names to include. For example, you can specify <code>a100</code>, <code>v100</code>,
         * <code>k80</code>, or other specific accelerator models.
         * </p>
         * 
         * @param acceleratorNames
         *        The specific accelerator names to include. For example, you can specify <code>a100</code>,
         *        <code>v100</code>, <code>k80</code>, or other specific accelerator models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNamesWithStrings(String... acceleratorNames);

        /**
         * <p>
         * The specific accelerator names to include. For example, you can specify <code>a100</code>, <code>v100</code>,
         * <code>k80</code>, or other specific accelerator models.
         * </p>
         * 
         * @param acceleratorNames
         *        The specific accelerator names to include. For example, you can specify <code>a100</code>,
         *        <code>v100</code>, <code>k80</code>, or other specific accelerator models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNames(Collection<AcceleratorName> acceleratorNames);

        /**
         * <p>
         * The specific accelerator names to include. For example, you can specify <code>a100</code>, <code>v100</code>,
         * <code>k80</code>, or other specific accelerator models.
         * </p>
         * 
         * @param acceleratorNames
         *        The specific accelerator names to include. For example, you can specify <code>a100</code>,
         *        <code>v100</code>, <code>k80</code>, or other specific accelerator models.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNames(AcceleratorName... acceleratorNames);

        /**
         * <p>
         * The minimum and maximum total accelerator memory in mebibytes (MiB). This is important for GPU workloads that
         * require specific amounts of video memory.
         * </p>
         * 
         * @param acceleratorTotalMemoryMiB
         *        The minimum and maximum total accelerator memory in mebibytes (MiB). This is important for GPU
         *        workloads that require specific amounts of video memory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB);

        /**
         * <p>
         * The minimum and maximum total accelerator memory in mebibytes (MiB). This is important for GPU workloads that
         * require specific amounts of video memory.
         * </p>
         * This is a convenience method that creates an instance of the {@link AcceleratorTotalMemoryMiBRequest.Builder}
         * avoiding the need to create one manually via {@link AcceleratorTotalMemoryMiBRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AcceleratorTotalMemoryMiBRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest)}.
         * 
         * @param acceleratorTotalMemoryMiB
         *        a consumer that will call methods on {@link AcceleratorTotalMemoryMiBRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest)
         */
        default Builder acceleratorTotalMemoryMiB(Consumer<AcceleratorTotalMemoryMiBRequest.Builder> acceleratorTotalMemoryMiB) {
            return acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest.builder().applyMutation(acceleratorTotalMemoryMiB)
                    .build());
        }

        /**
         * <p>
         * The minimum and maximum network bandwidth in gigabits per second (Gbps). This is crucial for
         * network-intensive workloads that require high throughput.
         * </p>
         * 
         * @param networkBandwidthGbps
         *        The minimum and maximum network bandwidth in gigabits per second (Gbps). This is crucial for
         *        network-intensive workloads that require high throughput.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBandwidthGbps(NetworkBandwidthGbpsRequest networkBandwidthGbps);

        /**
         * <p>
         * The minimum and maximum network bandwidth in gigabits per second (Gbps). This is crucial for
         * network-intensive workloads that require high throughput.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkBandwidthGbpsRequest.Builder}
         * avoiding the need to create one manually via {@link NetworkBandwidthGbpsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkBandwidthGbpsRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #networkBandwidthGbps(NetworkBandwidthGbpsRequest)}.
         * 
         * @param networkBandwidthGbps
         *        a consumer that will call methods on {@link NetworkBandwidthGbpsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkBandwidthGbps(NetworkBandwidthGbpsRequest)
         */
        default Builder networkBandwidthGbps(Consumer<NetworkBandwidthGbpsRequest.Builder> networkBandwidthGbps) {
            return networkBandwidthGbps(NetworkBandwidthGbpsRequest.builder().applyMutation(networkBandwidthGbps).build());
        }

        /**
         * <p>
         * The instance types to include in the selection. When specified, Amazon ECS only considers these instance
         * types, subject to the other requirements specified.
         * </p>
         * 
         * @param allowedInstanceTypes
         *        The instance types to include in the selection. When specified, Amazon ECS only considers these
         *        instance types, subject to the other requirements specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedInstanceTypes(Collection<String> allowedInstanceTypes);

        /**
         * <p>
         * The instance types to include in the selection. When specified, Amazon ECS only considers these instance
         * types, subject to the other requirements specified.
         * </p>
         * 
         * @param allowedInstanceTypes
         *        The instance types to include in the selection. When specified, Amazon ECS only considers these
         *        instance types, subject to the other requirements specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedInstanceTypes(String... allowedInstanceTypes);

        /**
         * <p>
         * The maximum price for Spot instances as a percentage of the optimal On-Demand price. This provides more
         * precise cost control for Spot instance selection.
         * </p>
         * 
         * @param maxSpotPriceAsPercentageOfOptimalOnDemandPrice
         *        The maximum price for Spot instances as a percentage of the optimal On-Demand price. This provides
         *        more precise cost control for Spot instance selection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSpotPriceAsPercentageOfOptimalOnDemandPrice(Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice);
    }

    static final class BuilderImpl implements Builder {
        private VCpuCountRangeRequest vCpuCount;

        private MemoryMiBRequest memoryMiB;

        private List<String> cpuManufacturers = DefaultSdkAutoConstructList.getInstance();

        private MemoryGiBPerVCpuRequest memoryGiBPerVCpu;

        private List<String> excludedInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> instanceGenerations = DefaultSdkAutoConstructList.getInstance();

        private Integer spotMaxPricePercentageOverLowestPrice;

        private Integer onDemandMaxPricePercentageOverLowestPrice;

        private String bareMetal;

        private String burstablePerformance;

        private Boolean requireHibernateSupport;

        private NetworkInterfaceCountRequest networkInterfaceCount;

        private String localStorage;

        private List<String> localStorageTypes = DefaultSdkAutoConstructList.getInstance();

        private TotalLocalStorageGBRequest totalLocalStorageGB;

        private BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps;

        private List<String> acceleratorTypes = DefaultSdkAutoConstructList.getInstance();

        private AcceleratorCountRequest acceleratorCount;

        private List<String> acceleratorManufacturers = DefaultSdkAutoConstructList.getInstance();

        private List<String> acceleratorNames = DefaultSdkAutoConstructList.getInstance();

        private AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB;

        private NetworkBandwidthGbpsRequest networkBandwidthGbps;

        private List<String> allowedInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceRequirementsRequest model) {
            vCpuCount(model.vCpuCount);
            memoryMiB(model.memoryMiB);
            cpuManufacturersWithStrings(model.cpuManufacturers);
            memoryGiBPerVCpu(model.memoryGiBPerVCpu);
            excludedInstanceTypes(model.excludedInstanceTypes);
            instanceGenerationsWithStrings(model.instanceGenerations);
            spotMaxPricePercentageOverLowestPrice(model.spotMaxPricePercentageOverLowestPrice);
            onDemandMaxPricePercentageOverLowestPrice(model.onDemandMaxPricePercentageOverLowestPrice);
            bareMetal(model.bareMetal);
            burstablePerformance(model.burstablePerformance);
            requireHibernateSupport(model.requireHibernateSupport);
            networkInterfaceCount(model.networkInterfaceCount);
            localStorage(model.localStorage);
            localStorageTypesWithStrings(model.localStorageTypes);
            totalLocalStorageGB(model.totalLocalStorageGB);
            baselineEbsBandwidthMbps(model.baselineEbsBandwidthMbps);
            acceleratorTypesWithStrings(model.acceleratorTypes);
            acceleratorCount(model.acceleratorCount);
            acceleratorManufacturersWithStrings(model.acceleratorManufacturers);
            acceleratorNamesWithStrings(model.acceleratorNames);
            acceleratorTotalMemoryMiB(model.acceleratorTotalMemoryMiB);
            networkBandwidthGbps(model.networkBandwidthGbps);
            allowedInstanceTypes(model.allowedInstanceTypes);
            maxSpotPriceAsPercentageOfOptimalOnDemandPrice(model.maxSpotPriceAsPercentageOfOptimalOnDemandPrice);
        }

        public final VCpuCountRangeRequest.Builder getVCpuCount() {
            return vCpuCount != null ? vCpuCount.toBuilder() : null;
        }

        public final void setVCpuCount(VCpuCountRangeRequest.BuilderImpl vCpuCount) {
            this.vCpuCount = vCpuCount != null ? vCpuCount.build() : null;
        }

        @Override
        public final Builder vCpuCount(VCpuCountRangeRequest vCpuCount) {
            this.vCpuCount = vCpuCount;
            return this;
        }

        public final MemoryMiBRequest.Builder getMemoryMiB() {
            return memoryMiB != null ? memoryMiB.toBuilder() : null;
        }

        public final void setMemoryMiB(MemoryMiBRequest.BuilderImpl memoryMiB) {
            this.memoryMiB = memoryMiB != null ? memoryMiB.build() : null;
        }

        @Override
        public final Builder memoryMiB(MemoryMiBRequest memoryMiB) {
            this.memoryMiB = memoryMiB;
            return this;
        }

        public final Collection<String> getCpuManufacturers() {
            if (cpuManufacturers instanceof SdkAutoConstructList) {
                return null;
            }
            return cpuManufacturers;
        }

        public final void setCpuManufacturers(Collection<String> cpuManufacturers) {
            this.cpuManufacturers = CpuManufacturerSetCopier.copy(cpuManufacturers);
        }

        @Override
        public final Builder cpuManufacturersWithStrings(Collection<String> cpuManufacturers) {
            this.cpuManufacturers = CpuManufacturerSetCopier.copy(cpuManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cpuManufacturersWithStrings(String... cpuManufacturers) {
            cpuManufacturersWithStrings(Arrays.asList(cpuManufacturers));
            return this;
        }

        @Override
        public final Builder cpuManufacturers(Collection<CpuManufacturer> cpuManufacturers) {
            this.cpuManufacturers = CpuManufacturerSetCopier.copyEnumToString(cpuManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cpuManufacturers(CpuManufacturer... cpuManufacturers) {
            cpuManufacturers(Arrays.asList(cpuManufacturers));
            return this;
        }

        public final MemoryGiBPerVCpuRequest.Builder getMemoryGiBPerVCpu() {
            return memoryGiBPerVCpu != null ? memoryGiBPerVCpu.toBuilder() : null;
        }

        public final void setMemoryGiBPerVCpu(MemoryGiBPerVCpuRequest.BuilderImpl memoryGiBPerVCpu) {
            this.memoryGiBPerVCpu = memoryGiBPerVCpu != null ? memoryGiBPerVCpu.build() : null;
        }

        @Override
        public final Builder memoryGiBPerVCpu(MemoryGiBPerVCpuRequest memoryGiBPerVCpu) {
            this.memoryGiBPerVCpu = memoryGiBPerVCpu;
            return this;
        }

        public final Collection<String> getExcludedInstanceTypes() {
            if (excludedInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return excludedInstanceTypes;
        }

        public final void setExcludedInstanceTypes(Collection<String> excludedInstanceTypes) {
            this.excludedInstanceTypes = ExcludedInstanceTypeSetCopier.copy(excludedInstanceTypes);
        }

        @Override
        public final Builder excludedInstanceTypes(Collection<String> excludedInstanceTypes) {
            this.excludedInstanceTypes = ExcludedInstanceTypeSetCopier.copy(excludedInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedInstanceTypes(String... excludedInstanceTypes) {
            excludedInstanceTypes(Arrays.asList(excludedInstanceTypes));
            return this;
        }

        public final Collection<String> getInstanceGenerations() {
            if (instanceGenerations instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceGenerations;
        }

        public final void setInstanceGenerations(Collection<String> instanceGenerations) {
            this.instanceGenerations = InstanceGenerationSetCopier.copy(instanceGenerations);
        }

        @Override
        public final Builder instanceGenerationsWithStrings(Collection<String> instanceGenerations) {
            this.instanceGenerations = InstanceGenerationSetCopier.copy(instanceGenerations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGenerationsWithStrings(String... instanceGenerations) {
            instanceGenerationsWithStrings(Arrays.asList(instanceGenerations));
            return this;
        }

        @Override
        public final Builder instanceGenerations(Collection<InstanceGeneration> instanceGenerations) {
            this.instanceGenerations = InstanceGenerationSetCopier.copyEnumToString(instanceGenerations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGenerations(InstanceGeneration... instanceGenerations) {
            instanceGenerations(Arrays.asList(instanceGenerations));
            return this;
        }

        public final Integer getSpotMaxPricePercentageOverLowestPrice() {
            return spotMaxPricePercentageOverLowestPrice;
        }

        public final void setSpotMaxPricePercentageOverLowestPrice(Integer spotMaxPricePercentageOverLowestPrice) {
            this.spotMaxPricePercentageOverLowestPrice = spotMaxPricePercentageOverLowestPrice;
        }

        @Override
        public final Builder spotMaxPricePercentageOverLowestPrice(Integer spotMaxPricePercentageOverLowestPrice) {
            this.spotMaxPricePercentageOverLowestPrice = spotMaxPricePercentageOverLowestPrice;
            return this;
        }

        public final Integer getOnDemandMaxPricePercentageOverLowestPrice() {
            return onDemandMaxPricePercentageOverLowestPrice;
        }

        public final void setOnDemandMaxPricePercentageOverLowestPrice(Integer onDemandMaxPricePercentageOverLowestPrice) {
            this.onDemandMaxPricePercentageOverLowestPrice = onDemandMaxPricePercentageOverLowestPrice;
        }

        @Override
        public final Builder onDemandMaxPricePercentageOverLowestPrice(Integer onDemandMaxPricePercentageOverLowestPrice) {
            this.onDemandMaxPricePercentageOverLowestPrice = onDemandMaxPricePercentageOverLowestPrice;
            return this;
        }

        public final String getBareMetal() {
            return bareMetal;
        }

        public final void setBareMetal(String bareMetal) {
            this.bareMetal = bareMetal;
        }

        @Override
        public final Builder bareMetal(String bareMetal) {
            this.bareMetal = bareMetal;
            return this;
        }

        @Override
        public final Builder bareMetal(BareMetal bareMetal) {
            this.bareMetal(bareMetal == null ? null : bareMetal.toString());
            return this;
        }

        public final String getBurstablePerformance() {
            return burstablePerformance;
        }

        public final void setBurstablePerformance(String burstablePerformance) {
            this.burstablePerformance = burstablePerformance;
        }

        @Override
        public final Builder burstablePerformance(String burstablePerformance) {
            this.burstablePerformance = burstablePerformance;
            return this;
        }

        @Override
        public final Builder burstablePerformance(BurstablePerformance burstablePerformance) {
            this.burstablePerformance(burstablePerformance == null ? null : burstablePerformance.toString());
            return this;
        }

        public final Boolean getRequireHibernateSupport() {
            return requireHibernateSupport;
        }

        public final void setRequireHibernateSupport(Boolean requireHibernateSupport) {
            this.requireHibernateSupport = requireHibernateSupport;
        }

        @Override
        public final Builder requireHibernateSupport(Boolean requireHibernateSupport) {
            this.requireHibernateSupport = requireHibernateSupport;
            return this;
        }

        public final NetworkInterfaceCountRequest.Builder getNetworkInterfaceCount() {
            return networkInterfaceCount != null ? networkInterfaceCount.toBuilder() : null;
        }

        public final void setNetworkInterfaceCount(NetworkInterfaceCountRequest.BuilderImpl networkInterfaceCount) {
            this.networkInterfaceCount = networkInterfaceCount != null ? networkInterfaceCount.build() : null;
        }

        @Override
        public final Builder networkInterfaceCount(NetworkInterfaceCountRequest networkInterfaceCount) {
            this.networkInterfaceCount = networkInterfaceCount;
            return this;
        }

        public final String getLocalStorage() {
            return localStorage;
        }

        public final void setLocalStorage(String localStorage) {
            this.localStorage = localStorage;
        }

        @Override
        public final Builder localStorage(String localStorage) {
            this.localStorage = localStorage;
            return this;
        }

        @Override
        public final Builder localStorage(LocalStorage localStorage) {
            this.localStorage(localStorage == null ? null : localStorage.toString());
            return this;
        }

        public final Collection<String> getLocalStorageTypes() {
            if (localStorageTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return localStorageTypes;
        }

        public final void setLocalStorageTypes(Collection<String> localStorageTypes) {
            this.localStorageTypes = LocalStorageTypeSetCopier.copy(localStorageTypes);
        }

        @Override
        public final Builder localStorageTypesWithStrings(Collection<String> localStorageTypes) {
            this.localStorageTypes = LocalStorageTypeSetCopier.copy(localStorageTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localStorageTypesWithStrings(String... localStorageTypes) {
            localStorageTypesWithStrings(Arrays.asList(localStorageTypes));
            return this;
        }

        @Override
        public final Builder localStorageTypes(Collection<LocalStorageType> localStorageTypes) {
            this.localStorageTypes = LocalStorageTypeSetCopier.copyEnumToString(localStorageTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localStorageTypes(LocalStorageType... localStorageTypes) {
            localStorageTypes(Arrays.asList(localStorageTypes));
            return this;
        }

        public final TotalLocalStorageGBRequest.Builder getTotalLocalStorageGB() {
            return totalLocalStorageGB != null ? totalLocalStorageGB.toBuilder() : null;
        }

        public final void setTotalLocalStorageGB(TotalLocalStorageGBRequest.BuilderImpl totalLocalStorageGB) {
            this.totalLocalStorageGB = totalLocalStorageGB != null ? totalLocalStorageGB.build() : null;
        }

        @Override
        public final Builder totalLocalStorageGB(TotalLocalStorageGBRequest totalLocalStorageGB) {
            this.totalLocalStorageGB = totalLocalStorageGB;
            return this;
        }

        public final BaselineEbsBandwidthMbpsRequest.Builder getBaselineEbsBandwidthMbps() {
            return baselineEbsBandwidthMbps != null ? baselineEbsBandwidthMbps.toBuilder() : null;
        }

        public final void setBaselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest.BuilderImpl baselineEbsBandwidthMbps) {
            this.baselineEbsBandwidthMbps = baselineEbsBandwidthMbps != null ? baselineEbsBandwidthMbps.build() : null;
        }

        @Override
        public final Builder baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps) {
            this.baselineEbsBandwidthMbps = baselineEbsBandwidthMbps;
            return this;
        }

        public final Collection<String> getAcceleratorTypes() {
            if (acceleratorTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return acceleratorTypes;
        }

        public final void setAcceleratorTypes(Collection<String> acceleratorTypes) {
            this.acceleratorTypes = AcceleratorTypeSetCopier.copy(acceleratorTypes);
        }

        @Override
        public final Builder acceleratorTypesWithStrings(Collection<String> acceleratorTypes) {
            this.acceleratorTypes = AcceleratorTypeSetCopier.copy(acceleratorTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorTypesWithStrings(String... acceleratorTypes) {
            acceleratorTypesWithStrings(Arrays.asList(acceleratorTypes));
            return this;
        }

        @Override
        public final Builder acceleratorTypes(Collection<AcceleratorType> acceleratorTypes) {
            this.acceleratorTypes = AcceleratorTypeSetCopier.copyEnumToString(acceleratorTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorTypes(AcceleratorType... acceleratorTypes) {
            acceleratorTypes(Arrays.asList(acceleratorTypes));
            return this;
        }

        public final AcceleratorCountRequest.Builder getAcceleratorCount() {
            return acceleratorCount != null ? acceleratorCount.toBuilder() : null;
        }

        public final void setAcceleratorCount(AcceleratorCountRequest.BuilderImpl acceleratorCount) {
            this.acceleratorCount = acceleratorCount != null ? acceleratorCount.build() : null;
        }

        @Override
        public final Builder acceleratorCount(AcceleratorCountRequest acceleratorCount) {
            this.acceleratorCount = acceleratorCount;
            return this;
        }

        public final Collection<String> getAcceleratorManufacturers() {
            if (acceleratorManufacturers instanceof SdkAutoConstructList) {
                return null;
            }
            return acceleratorManufacturers;
        }

        public final void setAcceleratorManufacturers(Collection<String> acceleratorManufacturers) {
            this.acceleratorManufacturers = AcceleratorManufacturerSetCopier.copy(acceleratorManufacturers);
        }

        @Override
        public final Builder acceleratorManufacturersWithStrings(Collection<String> acceleratorManufacturers) {
            this.acceleratorManufacturers = AcceleratorManufacturerSetCopier.copy(acceleratorManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorManufacturersWithStrings(String... acceleratorManufacturers) {
            acceleratorManufacturersWithStrings(Arrays.asList(acceleratorManufacturers));
            return this;
        }

        @Override
        public final Builder acceleratorManufacturers(Collection<AcceleratorManufacturer> acceleratorManufacturers) {
            this.acceleratorManufacturers = AcceleratorManufacturerSetCopier.copyEnumToString(acceleratorManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorManufacturers(AcceleratorManufacturer... acceleratorManufacturers) {
            acceleratorManufacturers(Arrays.asList(acceleratorManufacturers));
            return this;
        }

        public final Collection<String> getAcceleratorNames() {
            if (acceleratorNames instanceof SdkAutoConstructList) {
                return null;
            }
            return acceleratorNames;
        }

        public final void setAcceleratorNames(Collection<String> acceleratorNames) {
            this.acceleratorNames = AcceleratorNameSetCopier.copy(acceleratorNames);
        }

        @Override
        public final Builder acceleratorNamesWithStrings(Collection<String> acceleratorNames) {
            this.acceleratorNames = AcceleratorNameSetCopier.copy(acceleratorNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorNamesWithStrings(String... acceleratorNames) {
            acceleratorNamesWithStrings(Arrays.asList(acceleratorNames));
            return this;
        }

        @Override
        public final Builder acceleratorNames(Collection<AcceleratorName> acceleratorNames) {
            this.acceleratorNames = AcceleratorNameSetCopier.copyEnumToString(acceleratorNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorNames(AcceleratorName... acceleratorNames) {
            acceleratorNames(Arrays.asList(acceleratorNames));
            return this;
        }

        public final AcceleratorTotalMemoryMiBRequest.Builder getAcceleratorTotalMemoryMiB() {
            return acceleratorTotalMemoryMiB != null ? acceleratorTotalMemoryMiB.toBuilder() : null;
        }

        public final void setAcceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest.BuilderImpl acceleratorTotalMemoryMiB) {
            this.acceleratorTotalMemoryMiB = acceleratorTotalMemoryMiB != null ? acceleratorTotalMemoryMiB.build() : null;
        }

        @Override
        public final Builder acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB) {
            this.acceleratorTotalMemoryMiB = acceleratorTotalMemoryMiB;
            return this;
        }

        public final NetworkBandwidthGbpsRequest.Builder getNetworkBandwidthGbps() {
            return networkBandwidthGbps != null ? networkBandwidthGbps.toBuilder() : null;
        }

        public final void setNetworkBandwidthGbps(NetworkBandwidthGbpsRequest.BuilderImpl networkBandwidthGbps) {
            this.networkBandwidthGbps = networkBandwidthGbps != null ? networkBandwidthGbps.build() : null;
        }

        @Override
        public final Builder networkBandwidthGbps(NetworkBandwidthGbpsRequest networkBandwidthGbps) {
            this.networkBandwidthGbps = networkBandwidthGbps;
            return this;
        }

        public final Collection<String> getAllowedInstanceTypes() {
            if (allowedInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return allowedInstanceTypes;
        }

        public final void setAllowedInstanceTypes(Collection<String> allowedInstanceTypes) {
            this.allowedInstanceTypes = AllowedInstanceTypeSetCopier.copy(allowedInstanceTypes);
        }

        @Override
        public final Builder allowedInstanceTypes(Collection<String> allowedInstanceTypes) {
            this.allowedInstanceTypes = AllowedInstanceTypeSetCopier.copy(allowedInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedInstanceTypes(String... allowedInstanceTypes) {
            allowedInstanceTypes(Arrays.asList(allowedInstanceTypes));
            return this;
        }

        public final Integer getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice() {
            return maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        }

        public final void setMaxSpotPriceAsPercentageOfOptimalOnDemandPrice(Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice) {
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        }

        @Override
        public final Builder maxSpotPriceAsPercentageOfOptimalOnDemandPrice(Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice) {
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
            return this;
        }

        @Override
        public InstanceRequirementsRequest build() {
            return new InstanceRequirementsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
