/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details on the network bindings between a container and its host container instance. After a task reaches the
 * <code>RUNNING</code> status, manual and automatic host and container port assignments are visible in the
 * <code>networkBindings</code> section of <a>DescribeTasks</a> API responses.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetworkBinding implements SdkPojo, Serializable, ToCopyableBuilder<NetworkBinding.Builder, NetworkBinding> {
    private static final SdkField<String> BIND_IP_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("bindIP")
            .getter(getter(NetworkBinding::bindIP)).setter(setter(Builder::bindIP))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bindIP").build()).build();

    private static final SdkField<Integer> CONTAINER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("containerPort").getter(getter(NetworkBinding::containerPort)).setter(setter(Builder::containerPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPort").build()).build();

    private static final SdkField<Integer> HOST_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("hostPort").getter(getter(NetworkBinding::hostPort)).setter(setter(Builder::hostPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPort").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("protocol").getter(getter(NetworkBinding::protocolAsString)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protocol").build()).build();

    private static final SdkField<String> CONTAINER_PORT_RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("containerPortRange").getter(getter(NetworkBinding::containerPortRange))
            .setter(setter(Builder::containerPortRange))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("containerPortRange").build())
            .build();

    private static final SdkField<String> HOST_PORT_RANGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("hostPortRange").getter(getter(NetworkBinding::hostPortRange)).setter(setter(Builder::hostPortRange))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("hostPortRange").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(BIND_IP_FIELD,
            CONTAINER_PORT_FIELD, HOST_PORT_FIELD, PROTOCOL_FIELD, CONTAINER_PORT_RANGE_FIELD, HOST_PORT_RANGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String bindIP;

    private final Integer containerPort;

    private final Integer hostPort;

    private final String protocol;

    private final String containerPortRange;

    private final String hostPortRange;

    private NetworkBinding(BuilderImpl builder) {
        this.bindIP = builder.bindIP;
        this.containerPort = builder.containerPort;
        this.hostPort = builder.hostPort;
        this.protocol = builder.protocol;
        this.containerPortRange = builder.containerPortRange;
        this.hostPortRange = builder.hostPortRange;
    }

    /**
     * <p>
     * The IP address that the container is bound to on the container instance.
     * </p>
     * 
     * @return The IP address that the container is bound to on the container instance.
     */
    public final String bindIP() {
        return bindIP;
    }

    /**
     * <p>
     * The port number on the container that's used with the network binding.
     * </p>
     * 
     * @return The port number on the container that's used with the network binding.
     */
    public final Integer containerPort() {
        return containerPort;
    }

    /**
     * <p>
     * The port number on the host that's used with the network binding.
     * </p>
     * 
     * @return The port number on the host that's used with the network binding.
     */
    public final Integer hostPort() {
        return hostPort;
    }

    /**
     * <p>
     * The protocol used for the network binding.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the network binding.
     * @see TransportProtocol
     */
    public final TransportProtocol protocol() {
        return TransportProtocol.fromValue(protocol);
    }

    /**
     * <p>
     * The protocol used for the network binding.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #protocol} will
     * return {@link TransportProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #protocolAsString}.
     * </p>
     * 
     * @return The protocol used for the network binding.
     * @see TransportProtocol
     */
    public final String protocolAsString() {
        return protocol;
    }

    /**
     * <p>
     * The port number range on the container that's bound to the dynamically mapped host port range.
     * </p>
     * <p>
     * The following rules apply when you specify a <code>containerPortRange</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
     * </p>
     * </li>
     * <li>
     * <p>
     * This parameter is available for both the EC2 and Fargate launch types.
     * </p>
     * </li>
     * <li>
     * <p>
     * This parameter is available for both the Linux and Windows operating systems.
     * </p>
     * </li>
     * <li>
     * <p>
     * The container instance must have at least version 1.67.0 of the container agent and at least version 1.67.0-1 of
     * the <code>ecs-init</code> package
     * </p>
     * </li>
     * <li>
     * <p>
     * You can specify a maximum of 100 port ranges per container.
     * </p>
     * </li>
     * <li>
     * <p>
     * You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is set to the
     * same value as the <code>containerPortRange</code>. This is a static mapping strategy.
     * </p>
     * </li>
     * <li>
     * <p>
     * For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open host ports
     * from the default ephemeral range and passes it to docker to bind them to the container ports.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * The <code>containerPortRange</code> valid values are between 1 and 65535.
     * </p>
     * </li>
     * <li>
     * <p>
     * A port can only be included in one port mapping per container.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot specify overlapping port ranges.
     * </p>
     * </li>
     * <li>
     * <p>
     * The first port in the range must be less than last port in the range.
     * </p>
     * </li>
     * <li>
     * <p>
     * Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a large
     * number of ports.
     * </p>
     * <p>
     * For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the Github
     * website.
     * </p>
     * <p>
     * For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
     * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
     * >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can call <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
     * <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that are bound to
     * the container ports.
     * </p>
     * 
     * @return The port number range on the container that's bound to the dynamically mapped host port range.</p>
     *         <p>
     *         The following rules apply when you specify a <code>containerPortRange</code>:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         This parameter is available for both the EC2 and Fargate launch types.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         This parameter is available for both the Linux and Windows operating systems.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The container instance must have at least version 1.67.0 of the container agent and at least version
     *         1.67.0-1 of the <code>ecs-init</code> package
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You can specify a maximum of 100 port ranges per container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as
     *         follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is set
     *         to the same value as the <code>containerPortRange</code>. This is a static mapping strategy.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open host
     *         ports from the default ephemeral range and passes it to docker to bind them to the container ports.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         The <code>containerPortRange</code> valid values are between 1 and 65535.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A port can only be included in one port mapping per container.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You cannot specify overlapping port ranges.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The first port in the range must be less than last port in the range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a
     *         large number of ports.
     *         </p>
     *         <p>
     *         For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the
     *         Github website.
     *         </p>
     *         <p>
     *         For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
     *         "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
     *         >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can call <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
     *         <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that are
     *         bound to the container ports.
     */
    public final String containerPortRange() {
        return containerPortRange;
    }

    /**
     * <p>
     * The port number range on the host that's used with the network binding. This is assigned is assigned by Docker
     * and delivered by the Amazon ECS agent.
     * </p>
     * 
     * @return The port number range on the host that's used with the network binding. This is assigned is assigned by
     *         Docker and delivered by the Amazon ECS agent.
     */
    public final String hostPortRange() {
        return hostPortRange;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(bindIP());
        hashCode = 31 * hashCode + Objects.hashCode(containerPort());
        hashCode = 31 * hashCode + Objects.hashCode(hostPort());
        hashCode = 31 * hashCode + Objects.hashCode(protocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(containerPortRange());
        hashCode = 31 * hashCode + Objects.hashCode(hostPortRange());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetworkBinding)) {
            return false;
        }
        NetworkBinding other = (NetworkBinding) obj;
        return Objects.equals(bindIP(), other.bindIP()) && Objects.equals(containerPort(), other.containerPort())
                && Objects.equals(hostPort(), other.hostPort()) && Objects.equals(protocolAsString(), other.protocolAsString())
                && Objects.equals(containerPortRange(), other.containerPortRange())
                && Objects.equals(hostPortRange(), other.hostPortRange());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetworkBinding").add("BindIP", bindIP()).add("ContainerPort", containerPort())
                .add("HostPort", hostPort()).add("Protocol", protocolAsString()).add("ContainerPortRange", containerPortRange())
                .add("HostPortRange", hostPortRange()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "bindIP":
            return Optional.ofNullable(clazz.cast(bindIP()));
        case "containerPort":
            return Optional.ofNullable(clazz.cast(containerPort()));
        case "hostPort":
            return Optional.ofNullable(clazz.cast(hostPort()));
        case "protocol":
            return Optional.ofNullable(clazz.cast(protocolAsString()));
        case "containerPortRange":
            return Optional.ofNullable(clazz.cast(containerPortRange()));
        case "hostPortRange":
            return Optional.ofNullable(clazz.cast(hostPortRange()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<NetworkBinding, T> g) {
        return obj -> g.apply((NetworkBinding) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetworkBinding> {
        /**
         * <p>
         * The IP address that the container is bound to on the container instance.
         * </p>
         * 
         * @param bindIP
         *        The IP address that the container is bound to on the container instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bindIP(String bindIP);

        /**
         * <p>
         * The port number on the container that's used with the network binding.
         * </p>
         * 
         * @param containerPort
         *        The port number on the container that's used with the network binding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPort(Integer containerPort);

        /**
         * <p>
         * The port number on the host that's used with the network binding.
         * </p>
         * 
         * @param hostPort
         *        The port number on the host that's used with the network binding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPort(Integer hostPort);

        /**
         * <p>
         * The protocol used for the network binding.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the network binding.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * The protocol used for the network binding.
         * </p>
         * 
         * @param protocol
         *        The protocol used for the network binding.
         * @see TransportProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see TransportProtocol
         */
        Builder protocol(TransportProtocol protocol);

        /**
         * <p>
         * The port number range on the container that's bound to the dynamically mapped host port range.
         * </p>
         * <p>
         * The following rules apply when you specify a <code>containerPortRange</code>:
         * </p>
         * <ul>
         * <li>
         * <p>
         * You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
         * </p>
         * </li>
         * <li>
         * <p>
         * This parameter is available for both the EC2 and Fargate launch types.
         * </p>
         * </li>
         * <li>
         * <p>
         * This parameter is available for both the Linux and Windows operating systems.
         * </p>
         * </li>
         * <li>
         * <p>
         * The container instance must have at least version 1.67.0 of the container agent and at least version 1.67.0-1
         * of the <code>ecs-init</code> package
         * </p>
         * </li>
         * <li>
         * <p>
         * You can specify a maximum of 100 port ranges per container.
         * </p>
         * </li>
         * <li>
         * <p>
         * You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as
         * follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is set to
         * the same value as the <code>containerPortRange</code>. This is a static mapping strategy.
         * </p>
         * </li>
         * <li>
         * <p>
         * For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open host
         * ports from the default ephemeral range and passes it to docker to bind them to the container ports.
         * </p>
         * </li>
         * </ul>
         * </li>
         * <li>
         * <p>
         * The <code>containerPortRange</code> valid values are between 1 and 65535.
         * </p>
         * </li>
         * <li>
         * <p>
         * A port can only be included in one port mapping per container.
         * </p>
         * </li>
         * <li>
         * <p>
         * You cannot specify overlapping port ranges.
         * </p>
         * </li>
         * <li>
         * <p>
         * The first port in the range must be less than last port in the range.
         * </p>
         * </li>
         * <li>
         * <p>
         * Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a large
         * number of ports.
         * </p>
         * <p>
         * For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the Github
         * website.
         * </p>
         * <p>
         * For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
         * "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
         * >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can call <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
         * <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that are
         * bound to the container ports.
         * </p>
         * 
         * @param containerPortRange
         *        The port number range on the container that's bound to the dynamically mapped host port range.</p>
         *        <p>
         *        The following rules apply when you specify a <code>containerPortRange</code>:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        You must use either the <code>bridge</code> network mode or the <code>awsvpc</code> network mode.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        This parameter is available for both the EC2 and Fargate launch types.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        This parameter is available for both the Linux and Windows operating systems.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The container instance must have at least version 1.67.0 of the container agent and at least version
         *        1.67.0-1 of the <code>ecs-init</code> package
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You can specify a maximum of 100 port ranges per container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You do not specify a <code>hostPortRange</code>. The value of the <code>hostPortRange</code> is set as
         *        follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        For containers in a task with the <code>awsvpc</code> network mode, the <code>hostPortRange</code> is
         *        set to the same value as the <code>containerPortRange</code>. This is a static mapping strategy.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For containers in a task with the <code>bridge</code> network mode, the Amazon ECS agent finds open
         *        host ports from the default ephemeral range and passes it to docker to bind them to the container
         *        ports.
         *        </p>
         *        </li>
         *        </ul>
         *        </li>
         *        <li>
         *        <p>
         *        The <code>containerPortRange</code> valid values are between 1 and 65535.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        A port can only be included in one port mapping per container.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        You cannot specify overlapping port ranges.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The first port in the range must be less than last port in the range.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Docker recommends that you turn off the docker-proxy in the Docker daemon config file when you have a
         *        large number of ports.
         *        </p>
         *        <p>
         *        For more information, see <a href="https://github.com/moby/moby/issues/11185"> Issue #11185</a> on the
         *        Github website.
         *        </p>
         *        <p>
         *        For information about how to turn off the docker-proxy in the Docker daemon config file, see <a href=
         *        "https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#bootstrap_docker_daemon"
         *        >Docker daemon</a> in the <i>Amazon ECS Developer Guide</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can call <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_DescribeTasks.html">
         *        <code>DescribeTasks</code> </a> to view the <code>hostPortRange</code> which are the host ports that
         *        are bound to the container ports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder containerPortRange(String containerPortRange);

        /**
         * <p>
         * The port number range on the host that's used with the network binding. This is assigned is assigned by
         * Docker and delivered by the Amazon ECS agent.
         * </p>
         * 
         * @param hostPortRange
         *        The port number range on the host that's used with the network binding. This is assigned is assigned
         *        by Docker and delivered by the Amazon ECS agent.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostPortRange(String hostPortRange);
    }

    static final class BuilderImpl implements Builder {
        private String bindIP;

        private Integer containerPort;

        private Integer hostPort;

        private String protocol;

        private String containerPortRange;

        private String hostPortRange;

        private BuilderImpl() {
        }

        private BuilderImpl(NetworkBinding model) {
            bindIP(model.bindIP);
            containerPort(model.containerPort);
            hostPort(model.hostPort);
            protocol(model.protocol);
            containerPortRange(model.containerPortRange);
            hostPortRange(model.hostPortRange);
        }

        public final String getBindIP() {
            return bindIP;
        }

        public final void setBindIP(String bindIP) {
            this.bindIP = bindIP;
        }

        @Override
        public final Builder bindIP(String bindIP) {
            this.bindIP = bindIP;
            return this;
        }

        public final Integer getContainerPort() {
            return containerPort;
        }

        public final void setContainerPort(Integer containerPort) {
            this.containerPort = containerPort;
        }

        @Override
        public final Builder containerPort(Integer containerPort) {
            this.containerPort = containerPort;
            return this;
        }

        public final Integer getHostPort() {
            return hostPort;
        }

        public final void setHostPort(Integer hostPort) {
            this.hostPort = hostPort;
        }

        @Override
        public final Builder hostPort(Integer hostPort) {
            this.hostPort = hostPort;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        @Override
        public final Builder protocol(TransportProtocol protocol) {
            this.protocol(protocol == null ? null : protocol.toString());
            return this;
        }

        public final String getContainerPortRange() {
            return containerPortRange;
        }

        public final void setContainerPortRange(String containerPortRange) {
            this.containerPortRange = containerPortRange;
        }

        @Override
        public final Builder containerPortRange(String containerPortRange) {
            this.containerPortRange = containerPortRange;
            return this;
        }

        public final String getHostPortRange() {
            return hostPortRange;
        }

        public final void setHostPortRange(String hostPortRange) {
            this.hostPortRange = hostPortRange;
        }

        @Override
        public final Builder hostPortRange(String hostPortRange) {
            this.hostPortRange = hostPortRange;
            return this;
        }

        @Override
        public NetworkBinding build() {
            return new NetworkBinding(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
