/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an IPAM policy.
 * </p>
 * <p>
 * An IPAM policy is a set of rules that define how public IPv4 addresses from IPAM pools are allocated to Amazon Web
 * Services resources. Each rule maps an Amazon Web Services service to IPAM pools that the service will use to get IP
 * addresses. A single policy can have multiple rules and be applied to multiple Amazon Web Services Regions. If the
 * IPAM pool run out of addresses then the services fallback to Amazon-provided IP addresses. A policy can be applied to
 * an individual Amazon Web Services account or an entity within Amazon Web Services Organizations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamPolicy implements SdkPojo, Serializable, ToCopyableBuilder<IpamPolicy.Builder, IpamPolicy> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(IpamPolicy::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_POLICY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPolicyId")
            .getter(getter(IpamPolicy::ipamPolicyId))
            .setter(setter(Builder::ipamPolicyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPolicyId")
                    .unmarshallLocationName("ipamPolicyId").build()).build();

    private static final SdkField<String> IPAM_POLICY_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPolicyArn")
            .getter(getter(IpamPolicy::ipamPolicyArn))
            .setter(setter(Builder::ipamPolicyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPolicyArn")
                    .unmarshallLocationName("ipamPolicyArn").build()).build();

    private static final SdkField<String> IPAM_POLICY_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPolicyRegion")
            .getter(getter(IpamPolicy::ipamPolicyRegion))
            .setter(setter(Builder::ipamPolicyRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPolicyRegion")
                    .unmarshallLocationName("ipamPolicyRegion").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(IpamPolicy::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> STATE_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StateMessage")
            .getter(getter(IpamPolicy::stateMessage))
            .setter(setter(Builder::stateMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateMessage")
                    .unmarshallLocationName("stateMessage").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(IpamPolicy::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> IPAM_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamId")
            .getter(getter(IpamPolicy::ipamId))
            .setter(setter(Builder::ipamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamId")
                    .unmarshallLocationName("ipamId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            IPAM_POLICY_ID_FIELD, IPAM_POLICY_ARN_FIELD, IPAM_POLICY_REGION_FIELD, STATE_FIELD, STATE_MESSAGE_FIELD, TAGS_FIELD,
            IPAM_ID_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamPolicyId;

    private final String ipamPolicyArn;

    private final String ipamPolicyRegion;

    private final String state;

    private final String stateMessage;

    private final List<Tag> tags;

    private final String ipamId;

    private IpamPolicy(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamPolicyId = builder.ipamPolicyId;
        this.ipamPolicyArn = builder.ipamPolicyArn;
        this.ipamPolicyRegion = builder.ipamPolicyRegion;
        this.state = builder.state;
        this.stateMessage = builder.stateMessage;
        this.tags = builder.tags;
        this.ipamId = builder.ipamId;
    }

    /**
     * <p>
     * The account ID that owns the IPAM policy.
     * </p>
     * 
     * @return The account ID that owns the IPAM policy.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the IPAM policy.
     * </p>
     * 
     * @return The ID of the IPAM policy.
     */
    public final String ipamPolicyId() {
        return ipamPolicyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IPAM policy.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IPAM policy.
     */
    public final String ipamPolicyArn() {
        return ipamPolicyArn;
    }

    /**
     * <p>
     * The Region of the IPAM policy.
     * </p>
     * 
     * @return The Region of the IPAM policy.
     */
    public final String ipamPolicyRegion() {
        return ipamPolicyRegion;
    }

    /**
     * <p>
     * The state of the IPAM policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamPolicyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM policy.
     * @see IpamPolicyState
     */
    public final IpamPolicyState state() {
        return IpamPolicyState.fromValue(state);
    }

    /**
     * <p>
     * The state of the IPAM policy.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamPolicyState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM policy.
     * @see IpamPolicyState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A message about the state of the IPAM policy.
     * </p>
     * 
     * @return A message about the state of the IPAM policy.
     */
    public final String stateMessage() {
        return stateMessage;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the IPAM policy.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the IPAM policy.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The ID of the IPAM this policy belongs to.
     * </p>
     * 
     * @return The ID of the IPAM this policy belongs to.
     */
    public final String ipamId() {
        return ipamId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPolicyId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPolicyArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPolicyRegion());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateMessage());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ipamId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamPolicy)) {
            return false;
        }
        IpamPolicy other = (IpamPolicy) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(ipamPolicyId(), other.ipamPolicyId())
                && Objects.equals(ipamPolicyArn(), other.ipamPolicyArn())
                && Objects.equals(ipamPolicyRegion(), other.ipamPolicyRegion())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(stateMessage(), other.stateMessage())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(ipamId(), other.ipamId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamPolicy").add("OwnerId", ownerId()).add("IpamPolicyId", ipamPolicyId())
                .add("IpamPolicyArn", ipamPolicyArn()).add("IpamPolicyRegion", ipamPolicyRegion()).add("State", stateAsString())
                .add("StateMessage", stateMessage()).add("Tags", hasTags() ? tags() : null).add("IpamId", ipamId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamPolicyId":
            return Optional.ofNullable(clazz.cast(ipamPolicyId()));
        case "IpamPolicyArn":
            return Optional.ofNullable(clazz.cast(ipamPolicyArn()));
        case "IpamPolicyRegion":
            return Optional.ofNullable(clazz.cast(ipamPolicyRegion()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateMessage":
            return Optional.ofNullable(clazz.cast(stateMessage()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "IpamId":
            return Optional.ofNullable(clazz.cast(ipamId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("OwnerId", OWNER_ID_FIELD);
        map.put("IpamPolicyId", IPAM_POLICY_ID_FIELD);
        map.put("IpamPolicyArn", IPAM_POLICY_ARN_FIELD);
        map.put("IpamPolicyRegion", IPAM_POLICY_REGION_FIELD);
        map.put("State", STATE_FIELD);
        map.put("StateMessage", STATE_MESSAGE_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("IpamId", IPAM_ID_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IpamPolicy, T> g) {
        return obj -> g.apply((IpamPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamPolicy> {
        /**
         * <p>
         * The account ID that owns the IPAM policy.
         * </p>
         * 
         * @param ownerId
         *        The account ID that owns the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the IPAM policy.
         * </p>
         * 
         * @param ipamPolicyId
         *        The ID of the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPolicyId(String ipamPolicyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IPAM policy.
         * </p>
         * 
         * @param ipamPolicyArn
         *        The Amazon Resource Name (ARN) of the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPolicyArn(String ipamPolicyArn);

        /**
         * <p>
         * The Region of the IPAM policy.
         * </p>
         * 
         * @param ipamPolicyRegion
         *        The Region of the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPolicyRegion(String ipamPolicyRegion);

        /**
         * <p>
         * The state of the IPAM policy.
         * </p>
         * 
         * @param state
         *        The state of the IPAM policy.
         * @see IpamPolicyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPolicyState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the IPAM policy.
         * </p>
         * 
         * @param state
         *        The state of the IPAM policy.
         * @see IpamPolicyState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPolicyState
         */
        Builder state(IpamPolicyState state);

        /**
         * <p>
         * A message about the state of the IPAM policy.
         * </p>
         * 
         * @param stateMessage
         *        A message about the state of the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateMessage(String stateMessage);

        /**
         * <p>
         * The tags assigned to the IPAM policy.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the IPAM policy.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the IPAM policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the IPAM policy.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The ID of the IPAM this policy belongs to.
         * </p>
         * 
         * @param ipamId
         *        The ID of the IPAM this policy belongs to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamId(String ipamId);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamPolicyId;

        private String ipamPolicyArn;

        private String ipamPolicyRegion;

        private String state;

        private String stateMessage;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String ipamId;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamPolicy model) {
            ownerId(model.ownerId);
            ipamPolicyId(model.ipamPolicyId);
            ipamPolicyArn(model.ipamPolicyArn);
            ipamPolicyRegion(model.ipamPolicyRegion);
            state(model.state);
            stateMessage(model.stateMessage);
            tags(model.tags);
            ipamId(model.ipamId);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamPolicyId() {
            return ipamPolicyId;
        }

        public final void setIpamPolicyId(String ipamPolicyId) {
            this.ipamPolicyId = ipamPolicyId;
        }

        @Override
        public final Builder ipamPolicyId(String ipamPolicyId) {
            this.ipamPolicyId = ipamPolicyId;
            return this;
        }

        public final String getIpamPolicyArn() {
            return ipamPolicyArn;
        }

        public final void setIpamPolicyArn(String ipamPolicyArn) {
            this.ipamPolicyArn = ipamPolicyArn;
        }

        @Override
        public final Builder ipamPolicyArn(String ipamPolicyArn) {
            this.ipamPolicyArn = ipamPolicyArn;
            return this;
        }

        public final String getIpamPolicyRegion() {
            return ipamPolicyRegion;
        }

        public final void setIpamPolicyRegion(String ipamPolicyRegion) {
            this.ipamPolicyRegion = ipamPolicyRegion;
        }

        @Override
        public final Builder ipamPolicyRegion(String ipamPolicyRegion) {
            this.ipamPolicyRegion = ipamPolicyRegion;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamPolicyState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateMessage() {
            return stateMessage;
        }

        public final void setStateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
        }

        @Override
        public final Builder stateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getIpamId() {
            return ipamId;
        }

        public final void setIpamId(String ipamId) {
            this.ipamId = ipamId;
        }

        @Override
        public final Builder ipamId(String ipamId) {
            this.ipamId = ipamId;
            return this;
        }

        @Override
        public IpamPolicy build() {
            return new IpamPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
