/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration that links an Amazon VPC IPAM scope to an external authority system. It specifies the type of
 * external system and the external resource identifier that identifies your account or instance in that system.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/vpc/latest/ipam/integrate-infoblox-ipam.html">Integrate VPC IPAM with Infoblox
 * infrastructure</a> in the <i>Amazon VPC IPAM User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExternalAuthorityConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<ExternalAuthorityConfiguration.Builder, ExternalAuthorityConfiguration> {
    private static final SdkField<String> TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Type")
            .getter(getter(ExternalAuthorityConfiguration::typeAsString))
            .setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type")
                    .unmarshallLocationName("Type").build()).build();

    private static final SdkField<String> EXTERNAL_RESOURCE_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ExternalResourceIdentifier")
            .getter(getter(ExternalAuthorityConfiguration::externalResourceIdentifier))
            .setter(setter(Builder::externalResourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExternalResourceIdentifier")
                    .unmarshallLocationName("ExternalResourceIdentifier").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD,
            EXTERNAL_RESOURCE_IDENTIFIER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String externalResourceIdentifier;

    private ExternalAuthorityConfiguration(BuilderImpl builder) {
        this.type = builder.type;
        this.externalResourceIdentifier = builder.externalResourceIdentifier;
    }

    /**
     * <p>
     * The type of external authority.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IpamScopeExternalAuthorityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The type of external authority.
     * @see IpamScopeExternalAuthorityType
     */
    public final IpamScopeExternalAuthorityType type() {
        return IpamScopeExternalAuthorityType.fromValue(type);
    }

    /**
     * <p>
     * The type of external authority.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link IpamScopeExternalAuthorityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #typeAsString}.
     * </p>
     * 
     * @return The type of external authority.
     * @see IpamScopeExternalAuthorityType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The identifier for the external resource managing this scope. For Infoblox integrations, this is the Infoblox
     * resource identifier in the format
     * <code>&lt;version&gt;.identity.account.&lt;entity_realm&gt;.&lt;entity_id&gt;</code>.
     * </p>
     * 
     * @return The identifier for the external resource managing this scope. For Infoblox integrations, this is the
     *         Infoblox resource identifier in the format
     *         <code>&lt;version&gt;.identity.account.&lt;entity_realm&gt;.&lt;entity_id&gt;</code>.
     */
    public final String externalResourceIdentifier() {
        return externalResourceIdentifier;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(externalResourceIdentifier());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExternalAuthorityConfiguration)) {
            return false;
        }
        ExternalAuthorityConfiguration other = (ExternalAuthorityConfiguration) obj;
        return Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(externalResourceIdentifier(), other.externalResourceIdentifier());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExternalAuthorityConfiguration").add("Type", typeAsString())
                .add("ExternalResourceIdentifier", externalResourceIdentifier()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ExternalResourceIdentifier":
            return Optional.ofNullable(clazz.cast(externalResourceIdentifier()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Type", TYPE_FIELD);
        map.put("ExternalResourceIdentifier", EXTERNAL_RESOURCE_IDENTIFIER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ExternalAuthorityConfiguration, T> g) {
        return obj -> g.apply((ExternalAuthorityConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExternalAuthorityConfiguration> {
        /**
         * <p>
         * The type of external authority.
         * </p>
         * 
         * @param type
         *        The type of external authority.
         * @see IpamScopeExternalAuthorityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeExternalAuthorityType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of external authority.
         * </p>
         * 
         * @param type
         *        The type of external authority.
         * @see IpamScopeExternalAuthorityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeExternalAuthorityType
         */
        Builder type(IpamScopeExternalAuthorityType type);

        /**
         * <p>
         * The identifier for the external resource managing this scope. For Infoblox integrations, this is the Infoblox
         * resource identifier in the format
         * <code>&lt;version&gt;.identity.account.&lt;entity_realm&gt;.&lt;entity_id&gt;</code>.
         * </p>
         * 
         * @param externalResourceIdentifier
         *        The identifier for the external resource managing this scope. For Infoblox integrations, this is the
         *        Infoblox resource identifier in the format
         *        <code>&lt;version&gt;.identity.account.&lt;entity_realm&gt;.&lt;entity_id&gt;</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalResourceIdentifier(String externalResourceIdentifier);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String externalResourceIdentifier;

        private BuilderImpl() {
        }

        private BuilderImpl(ExternalAuthorityConfiguration model) {
            type(model.type);
            externalResourceIdentifier(model.externalResourceIdentifier);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(IpamScopeExternalAuthorityType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getExternalResourceIdentifier() {
            return externalResourceIdentifier;
        }

        public final void setExternalResourceIdentifier(String externalResourceIdentifier) {
            this.externalResourceIdentifier = externalResourceIdentifier;
        }

        @Override
        public final Builder externalResourceIdentifier(String externalResourceIdentifier) {
            this.externalResourceIdentifier = externalResourceIdentifier;
            return this;
        }

        @Override
        public ExternalAuthorityConfiguration build() {
            return new ExternalAuthorityConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
