/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An IPAM resource discovery association. An associated resource discovery is a resource discovery that has been
 * associated with an IPAM. IPAM aggregates the resource CIDRs discovered by the associated resource discovery.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamResourceDiscoveryAssociation implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamResourceDiscoveryAssociation.Builder, IpamResourceDiscoveryAssociation> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(IpamResourceDiscoveryAssociation::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ASSOCIATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryAssociationId")
            .getter(getter(IpamResourceDiscoveryAssociation::ipamResourceDiscoveryAssociationId))
            .setter(setter(Builder::ipamResourceDiscoveryAssociationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryAssociationId")
                    .unmarshallLocationName("ipamResourceDiscoveryAssociationId").build()).build();

    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ASSOCIATION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryAssociationArn")
            .getter(getter(IpamResourceDiscoveryAssociation::ipamResourceDiscoveryAssociationArn))
            .setter(setter(Builder::ipamResourceDiscoveryAssociationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("IpamResourceDiscoveryAssociationArn")
                    .unmarshallLocationName("ipamResourceDiscoveryAssociationArn").build()).build();

    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryId")
            .getter(getter(IpamResourceDiscoveryAssociation::ipamResourceDiscoveryId))
            .setter(setter(Builder::ipamResourceDiscoveryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryId")
                    .unmarshallLocationName("ipamResourceDiscoveryId").build()).build();

    private static final SdkField<String> IPAM_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamId")
            .getter(getter(IpamResourceDiscoveryAssociation::ipamId))
            .setter(setter(Builder::ipamId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamId")
                    .unmarshallLocationName("ipamId").build()).build();

    private static final SdkField<String> IPAM_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamArn")
            .getter(getter(IpamResourceDiscoveryAssociation::ipamArn))
            .setter(setter(Builder::ipamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamArn")
                    .unmarshallLocationName("ipamArn").build()).build();

    private static final SdkField<String> IPAM_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamRegion")
            .getter(getter(IpamResourceDiscoveryAssociation::ipamRegion))
            .setter(setter(Builder::ipamRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamRegion")
                    .unmarshallLocationName("ipamRegion").build()).build();

    private static final SdkField<Boolean> IS_DEFAULT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsDefault")
            .getter(getter(IpamResourceDiscoveryAssociation::isDefault))
            .setter(setter(Builder::isDefault))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsDefault")
                    .unmarshallLocationName("isDefault").build()).build();

    private static final SdkField<String> RESOURCE_DISCOVERY_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceDiscoveryStatus")
            .getter(getter(IpamResourceDiscoveryAssociation::resourceDiscoveryStatusAsString))
            .setter(setter(Builder::resourceDiscoveryStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceDiscoveryStatus")
                    .unmarshallLocationName("resourceDiscoveryStatus").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(IpamResourceDiscoveryAssociation::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(IpamResourceDiscoveryAssociation::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            IPAM_RESOURCE_DISCOVERY_ASSOCIATION_ID_FIELD, IPAM_RESOURCE_DISCOVERY_ASSOCIATION_ARN_FIELD,
            IPAM_RESOURCE_DISCOVERY_ID_FIELD, IPAM_ID_FIELD, IPAM_ARN_FIELD, IPAM_REGION_FIELD, IS_DEFAULT_FIELD,
            RESOURCE_DISCOVERY_STATUS_FIELD, STATE_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("OwnerId", OWNER_ID_FIELD);
                    put("IpamResourceDiscoveryAssociationId", IPAM_RESOURCE_DISCOVERY_ASSOCIATION_ID_FIELD);
                    put("IpamResourceDiscoveryAssociationArn", IPAM_RESOURCE_DISCOVERY_ASSOCIATION_ARN_FIELD);
                    put("IpamResourceDiscoveryId", IPAM_RESOURCE_DISCOVERY_ID_FIELD);
                    put("IpamId", IPAM_ID_FIELD);
                    put("IpamArn", IPAM_ARN_FIELD);
                    put("IpamRegion", IPAM_REGION_FIELD);
                    put("IsDefault", IS_DEFAULT_FIELD);
                    put("ResourceDiscoveryStatus", RESOURCE_DISCOVERY_STATUS_FIELD);
                    put("State", STATE_FIELD);
                    put("TagSet", TAGS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamResourceDiscoveryAssociationId;

    private final String ipamResourceDiscoveryAssociationArn;

    private final String ipamResourceDiscoveryId;

    private final String ipamId;

    private final String ipamArn;

    private final String ipamRegion;

    private final Boolean isDefault;

    private final String resourceDiscoveryStatus;

    private final String state;

    private final List<Tag> tags;

    private IpamResourceDiscoveryAssociation(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamResourceDiscoveryAssociationId = builder.ipamResourceDiscoveryAssociationId;
        this.ipamResourceDiscoveryAssociationArn = builder.ipamResourceDiscoveryAssociationArn;
        this.ipamResourceDiscoveryId = builder.ipamResourceDiscoveryId;
        this.ipamId = builder.ipamId;
        this.ipamArn = builder.ipamArn;
        this.ipamRegion = builder.ipamRegion;
        this.isDefault = builder.isDefault;
        this.resourceDiscoveryStatus = builder.resourceDiscoveryStatus;
        this.state = builder.state;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the resource discovery owner.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the resource discovery owner.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The resource discovery association ID.
     * </p>
     * 
     * @return The resource discovery association ID.
     */
    public final String ipamResourceDiscoveryAssociationId() {
        return ipamResourceDiscoveryAssociationId;
    }

    /**
     * <p>
     * The resource discovery association Amazon Resource Name (ARN).
     * </p>
     * 
     * @return The resource discovery association Amazon Resource Name (ARN).
     */
    public final String ipamResourceDiscoveryAssociationArn() {
        return ipamResourceDiscoveryAssociationArn;
    }

    /**
     * <p>
     * The resource discovery ID.
     * </p>
     * 
     * @return The resource discovery ID.
     */
    public final String ipamResourceDiscoveryId() {
        return ipamResourceDiscoveryId;
    }

    /**
     * <p>
     * The IPAM ID.
     * </p>
     * 
     * @return The IPAM ID.
     */
    public final String ipamId() {
        return ipamId;
    }

    /**
     * <p>
     * The IPAM ARN.
     * </p>
     * 
     * @return The IPAM ARN.
     */
    public final String ipamArn() {
        return ipamArn;
    }

    /**
     * <p>
     * The IPAM home Region.
     * </p>
     * 
     * @return The IPAM home Region.
     */
    public final String ipamRegion() {
        return ipamRegion;
    }

    /**
     * <p>
     * Defines if the resource discovery is the default. When you create an IPAM, a default resource discovery is
     * created for your IPAM and it's associated with your IPAM.
     * </p>
     * 
     * @return Defines if the resource discovery is the default. When you create an IPAM, a default resource discovery
     *         is created for your IPAM and it's associated with your IPAM.
     */
    public final Boolean isDefault() {
        return isDefault;
    }

    /**
     * <p>
     * The resource discovery status.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>active</code> - Connection or permissions required to read the results of the resource discovery are
     * intact.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>not-found</code> - Connection or permissions required to read the results of the resource discovery are
     * broken. This may happen if the owner of the resource discovery stopped sharing it or deleted the resource
     * discovery. Verify the resource discovery still exists and the Amazon Web Services RAM resource share is still
     * intact.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resourceDiscoveryStatus} will return {@link IpamAssociatedResourceDiscoveryStatus#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #resourceDiscoveryStatusAsString}.
     * </p>
     * 
     * @return The resource discovery status.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>active</code> - Connection or permissions required to read the results of the resource discovery
     *         are intact.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>not-found</code> - Connection or permissions required to read the results of the resource discovery
     *         are broken. This may happen if the owner of the resource discovery stopped sharing it or deleted the
     *         resource discovery. Verify the resource discovery still exists and the Amazon Web Services RAM resource
     *         share is still intact.
     *         </p>
     *         </li>
     * @see IpamAssociatedResourceDiscoveryStatus
     */
    public final IpamAssociatedResourceDiscoveryStatus resourceDiscoveryStatus() {
        return IpamAssociatedResourceDiscoveryStatus.fromValue(resourceDiscoveryStatus);
    }

    /**
     * <p>
     * The resource discovery status.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>active</code> - Connection or permissions required to read the results of the resource discovery are
     * intact.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>not-found</code> - Connection or permissions required to read the results of the resource discovery are
     * broken. This may happen if the owner of the resource discovery stopped sharing it or deleted the resource
     * discovery. Verify the resource discovery still exists and the Amazon Web Services RAM resource share is still
     * intact.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #resourceDiscoveryStatus} will return {@link IpamAssociatedResourceDiscoveryStatus#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #resourceDiscoveryStatusAsString}.
     * </p>
     * 
     * @return The resource discovery status.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>active</code> - Connection or permissions required to read the results of the resource discovery
     *         are intact.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>not-found</code> - Connection or permissions required to read the results of the resource discovery
     *         are broken. This may happen if the owner of the resource discovery stopped sharing it or deleted the
     *         resource discovery. Verify the resource discovery still exists and the Amazon Web Services RAM resource
     *         share is still intact.
     *         </p>
     *         </li>
     * @see IpamAssociatedResourceDiscoveryStatus
     */
    public final String resourceDiscoveryStatusAsString() {
        return resourceDiscoveryStatus;
    }

    /**
     * <p>
     * The lifecycle state of the association when you associate or disassociate a resource discovery.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>associate-in-progress</code> - Resource discovery is being associated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>associate-complete</code> - Resource discovery association is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>associate-failed</code> - Resource discovery association has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>disassociate-complete</code> - Resource discovery disassociation is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>disassociate-failed </code> - Resource discovery disassociation has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery association
     * has been removed and the resource discovery associatation is being isolated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-complete</code> - Resource discovery isolation is complete..
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>restore-in-progress</code> - Resource discovery is being restored.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamResourceDiscoveryAssociationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The lifecycle state of the association when you associate or disassociate a resource discovery.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>associate-in-progress</code> - Resource discovery is being associated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>associate-complete</code> - Resource discovery association is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>associate-failed</code> - Resource discovery association has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>disassociate-complete</code> - Resource discovery disassociation is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>disassociate-failed </code> - Resource discovery disassociation has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery
     *         association has been removed and the resource discovery associatation is being isolated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-complete</code> - Resource discovery isolation is complete..
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>restore-in-progress</code> - Resource discovery is being restored.
     *         </p>
     *         </li>
     * @see IpamResourceDiscoveryAssociationState
     */
    public final IpamResourceDiscoveryAssociationState state() {
        return IpamResourceDiscoveryAssociationState.fromValue(state);
    }

    /**
     * <p>
     * The lifecycle state of the association when you associate or disassociate a resource discovery.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>associate-in-progress</code> - Resource discovery is being associated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>associate-complete</code> - Resource discovery association is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>associate-failed</code> - Resource discovery association has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>disassociate-complete</code> - Resource discovery disassociation is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>disassociate-failed </code> - Resource discovery disassociation has failed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery association
     * has been removed and the resource discovery associatation is being isolated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>isolate-complete</code> - Resource discovery isolation is complete..
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>restore-in-progress</code> - Resource discovery is being restored.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamResourceDiscoveryAssociationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #stateAsString}.
     * </p>
     * 
     * @return The lifecycle state of the association when you associate or disassociate a resource discovery.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>associate-in-progress</code> - Resource discovery is being associated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>associate-complete</code> - Resource discovery association is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>associate-failed</code> - Resource discovery association has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>disassociate-complete</code> - Resource discovery disassociation is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>disassociate-failed </code> - Resource discovery disassociation has failed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery
     *         association has been removed and the resource discovery associatation is being isolated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>isolate-complete</code> - Resource discovery isolation is complete..
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>restore-in-progress</code> - Resource discovery is being restored.
     *         </p>
     *         </li>
     * @see IpamResourceDiscoveryAssociationState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an optional
     * value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
     *         optional value. You can use tags to search and filter your resources or track your Amazon Web Services
     *         costs.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryAssociationId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryAssociationArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamRegion());
        hashCode = 31 * hashCode + Objects.hashCode(isDefault());
        hashCode = 31 * hashCode + Objects.hashCode(resourceDiscoveryStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamResourceDiscoveryAssociation)) {
            return false;
        }
        IpamResourceDiscoveryAssociation other = (IpamResourceDiscoveryAssociation) obj;
        return Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(ipamResourceDiscoveryAssociationId(), other.ipamResourceDiscoveryAssociationId())
                && Objects.equals(ipamResourceDiscoveryAssociationArn(), other.ipamResourceDiscoveryAssociationArn())
                && Objects.equals(ipamResourceDiscoveryId(), other.ipamResourceDiscoveryId())
                && Objects.equals(ipamId(), other.ipamId()) && Objects.equals(ipamArn(), other.ipamArn())
                && Objects.equals(ipamRegion(), other.ipamRegion()) && Objects.equals(isDefault(), other.isDefault())
                && Objects.equals(resourceDiscoveryStatusAsString(), other.resourceDiscoveryStatusAsString())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamResourceDiscoveryAssociation").add("OwnerId", ownerId())
                .add("IpamResourceDiscoveryAssociationId", ipamResourceDiscoveryAssociationId())
                .add("IpamResourceDiscoveryAssociationArn", ipamResourceDiscoveryAssociationArn())
                .add("IpamResourceDiscoveryId", ipamResourceDiscoveryId()).add("IpamId", ipamId()).add("IpamArn", ipamArn())
                .add("IpamRegion", ipamRegion()).add("IsDefault", isDefault())
                .add("ResourceDiscoveryStatus", resourceDiscoveryStatusAsString()).add("State", stateAsString())
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamResourceDiscoveryAssociationId":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryAssociationId()));
        case "IpamResourceDiscoveryAssociationArn":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryAssociationArn()));
        case "IpamResourceDiscoveryId":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryId()));
        case "IpamId":
            return Optional.ofNullable(clazz.cast(ipamId()));
        case "IpamArn":
            return Optional.ofNullable(clazz.cast(ipamArn()));
        case "IpamRegion":
            return Optional.ofNullable(clazz.cast(ipamRegion()));
        case "IsDefault":
            return Optional.ofNullable(clazz.cast(isDefault()));
        case "ResourceDiscoveryStatus":
            return Optional.ofNullable(clazz.cast(resourceDiscoveryStatusAsString()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<IpamResourceDiscoveryAssociation, T> g) {
        return obj -> g.apply((IpamResourceDiscoveryAssociation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamResourceDiscoveryAssociation> {
        /**
         * <p>
         * The Amazon Web Services account ID of the resource discovery owner.
         * </p>
         * 
         * @param ownerId
         *        The Amazon Web Services account ID of the resource discovery owner.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The resource discovery association ID.
         * </p>
         * 
         * @param ipamResourceDiscoveryAssociationId
         *        The resource discovery association ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryAssociationId(String ipamResourceDiscoveryAssociationId);

        /**
         * <p>
         * The resource discovery association Amazon Resource Name (ARN).
         * </p>
         * 
         * @param ipamResourceDiscoveryAssociationArn
         *        The resource discovery association Amazon Resource Name (ARN).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryAssociationArn(String ipamResourceDiscoveryAssociationArn);

        /**
         * <p>
         * The resource discovery ID.
         * </p>
         * 
         * @param ipamResourceDiscoveryId
         *        The resource discovery ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId);

        /**
         * <p>
         * The IPAM ID.
         * </p>
         * 
         * @param ipamId
         *        The IPAM ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamId(String ipamId);

        /**
         * <p>
         * The IPAM ARN.
         * </p>
         * 
         * @param ipamArn
         *        The IPAM ARN.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamArn(String ipamArn);

        /**
         * <p>
         * The IPAM home Region.
         * </p>
         * 
         * @param ipamRegion
         *        The IPAM home Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamRegion(String ipamRegion);

        /**
         * <p>
         * Defines if the resource discovery is the default. When you create an IPAM, a default resource discovery is
         * created for your IPAM and it's associated with your IPAM.
         * </p>
         * 
         * @param isDefault
         *        Defines if the resource discovery is the default. When you create an IPAM, a default resource
         *        discovery is created for your IPAM and it's associated with your IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDefault(Boolean isDefault);

        /**
         * <p>
         * The resource discovery status.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>active</code> - Connection or permissions required to read the results of the resource discovery are
         * intact.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>not-found</code> - Connection or permissions required to read the results of the resource discovery are
         * broken. This may happen if the owner of the resource discovery stopped sharing it or deleted the resource
         * discovery. Verify the resource discovery still exists and the Amazon Web Services RAM resource share is still
         * intact.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceDiscoveryStatus
         *        The resource discovery status.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>active</code> - Connection or permissions required to read the results of the resource discovery
         *        are intact.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>not-found</code> - Connection or permissions required to read the results of the resource
         *        discovery are broken. This may happen if the owner of the resource discovery stopped sharing it or
         *        deleted the resource discovery. Verify the resource discovery still exists and the Amazon Web Services
         *        RAM resource share is still intact.
         *        </p>
         *        </li>
         * @see IpamAssociatedResourceDiscoveryStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamAssociatedResourceDiscoveryStatus
         */
        Builder resourceDiscoveryStatus(String resourceDiscoveryStatus);

        /**
         * <p>
         * The resource discovery status.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>active</code> - Connection or permissions required to read the results of the resource discovery are
         * intact.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>not-found</code> - Connection or permissions required to read the results of the resource discovery are
         * broken. This may happen if the owner of the resource discovery stopped sharing it or deleted the resource
         * discovery. Verify the resource discovery still exists and the Amazon Web Services RAM resource share is still
         * intact.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceDiscoveryStatus
         *        The resource discovery status.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>active</code> - Connection or permissions required to read the results of the resource discovery
         *        are intact.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>not-found</code> - Connection or permissions required to read the results of the resource
         *        discovery are broken. This may happen if the owner of the resource discovery stopped sharing it or
         *        deleted the resource discovery. Verify the resource discovery still exists and the Amazon Web Services
         *        RAM resource share is still intact.
         *        </p>
         *        </li>
         * @see IpamAssociatedResourceDiscoveryStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamAssociatedResourceDiscoveryStatus
         */
        Builder resourceDiscoveryStatus(IpamAssociatedResourceDiscoveryStatus resourceDiscoveryStatus);

        /**
         * <p>
         * The lifecycle state of the association when you associate or disassociate a resource discovery.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>associate-in-progress</code> - Resource discovery is being associated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>associate-complete</code> - Resource discovery association is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>associate-failed</code> - Resource discovery association has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>disassociate-complete</code> - Resource discovery disassociation is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>disassociate-failed </code> - Resource discovery disassociation has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery
         * association has been removed and the resource discovery associatation is being isolated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-complete</code> - Resource discovery isolation is complete..
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>restore-in-progress</code> - Resource discovery is being restored.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The lifecycle state of the association when you associate or disassociate a resource discovery.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>associate-in-progress</code> - Resource discovery is being associated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>associate-complete</code> - Resource discovery association is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>associate-failed</code> - Resource discovery association has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>disassociate-complete</code> - Resource discovery disassociation is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>disassociate-failed </code> - Resource discovery disassociation has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery
         *        association has been removed and the resource discovery associatation is being isolated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-complete</code> - Resource discovery isolation is complete..
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>restore-in-progress</code> - Resource discovery is being restored.
         *        </p>
         *        </li>
         * @see IpamResourceDiscoveryAssociationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceDiscoveryAssociationState
         */
        Builder state(String state);

        /**
         * <p>
         * The lifecycle state of the association when you associate or disassociate a resource discovery.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>associate-in-progress</code> - Resource discovery is being associated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>associate-complete</code> - Resource discovery association is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>associate-failed</code> - Resource discovery association has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>disassociate-complete</code> - Resource discovery disassociation is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>disassociate-failed </code> - Resource discovery disassociation has failed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery
         * association has been removed and the resource discovery associatation is being isolated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>isolate-complete</code> - Resource discovery isolation is complete..
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>restore-in-progress</code> - Resource discovery is being restored.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The lifecycle state of the association when you associate or disassociate a resource discovery.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>associate-in-progress</code> - Resource discovery is being associated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>associate-complete</code> - Resource discovery association is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>associate-failed</code> - Resource discovery association has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>disassociate-in-progress</code> - Resource discovery is being disassociated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>disassociate-complete</code> - Resource discovery disassociation is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>disassociate-failed </code> - Resource discovery disassociation has failed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-in-progress</code> - Amazon Web Services account that created the resource discovery
         *        association has been removed and the resource discovery associatation is being isolated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>isolate-complete</code> - Resource discovery isolation is complete..
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>restore-in-progress</code> - Resource discovery is being restored.
         *        </p>
         *        </li>
         * @see IpamResourceDiscoveryAssociationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceDiscoveryAssociationState
         */
        Builder state(IpamResourceDiscoveryAssociationState state);

        /**
         * <p>
         * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         * optional value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
         * </p>
         * 
         * @param tags
         *        A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         *        optional value. You can use tags to search and filter your resources or track your Amazon Web Services
         *        costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         * optional value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
         * </p>
         * 
         * @param tags
         *        A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         *        optional value. You can use tags to search and filter your resources or track your Amazon Web Services
         *        costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * A tag is a label that you assign to an Amazon Web Services resource. Each tag consists of a key and an
         * optional value. You can use tags to search and filter your resources or track your Amazon Web Services costs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamResourceDiscoveryAssociationId;

        private String ipamResourceDiscoveryAssociationArn;

        private String ipamResourceDiscoveryId;

        private String ipamId;

        private String ipamArn;

        private String ipamRegion;

        private Boolean isDefault;

        private String resourceDiscoveryStatus;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IpamResourceDiscoveryAssociation model) {
            ownerId(model.ownerId);
            ipamResourceDiscoveryAssociationId(model.ipamResourceDiscoveryAssociationId);
            ipamResourceDiscoveryAssociationArn(model.ipamResourceDiscoveryAssociationArn);
            ipamResourceDiscoveryId(model.ipamResourceDiscoveryId);
            ipamId(model.ipamId);
            ipamArn(model.ipamArn);
            ipamRegion(model.ipamRegion);
            isDefault(model.isDefault);
            resourceDiscoveryStatus(model.resourceDiscoveryStatus);
            state(model.state);
            tags(model.tags);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamResourceDiscoveryAssociationId() {
            return ipamResourceDiscoveryAssociationId;
        }

        public final void setIpamResourceDiscoveryAssociationId(String ipamResourceDiscoveryAssociationId) {
            this.ipamResourceDiscoveryAssociationId = ipamResourceDiscoveryAssociationId;
        }

        @Override
        public final Builder ipamResourceDiscoveryAssociationId(String ipamResourceDiscoveryAssociationId) {
            this.ipamResourceDiscoveryAssociationId = ipamResourceDiscoveryAssociationId;
            return this;
        }

        public final String getIpamResourceDiscoveryAssociationArn() {
            return ipamResourceDiscoveryAssociationArn;
        }

        public final void setIpamResourceDiscoveryAssociationArn(String ipamResourceDiscoveryAssociationArn) {
            this.ipamResourceDiscoveryAssociationArn = ipamResourceDiscoveryAssociationArn;
        }

        @Override
        public final Builder ipamResourceDiscoveryAssociationArn(String ipamResourceDiscoveryAssociationArn) {
            this.ipamResourceDiscoveryAssociationArn = ipamResourceDiscoveryAssociationArn;
            return this;
        }

        public final String getIpamResourceDiscoveryId() {
            return ipamResourceDiscoveryId;
        }

        public final void setIpamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
        }

        @Override
        public final Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
            return this;
        }

        public final String getIpamId() {
            return ipamId;
        }

        public final void setIpamId(String ipamId) {
            this.ipamId = ipamId;
        }

        @Override
        public final Builder ipamId(String ipamId) {
            this.ipamId = ipamId;
            return this;
        }

        public final String getIpamArn() {
            return ipamArn;
        }

        public final void setIpamArn(String ipamArn) {
            this.ipamArn = ipamArn;
        }

        @Override
        public final Builder ipamArn(String ipamArn) {
            this.ipamArn = ipamArn;
            return this;
        }

        public final String getIpamRegion() {
            return ipamRegion;
        }

        public final void setIpamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
        }

        @Override
        public final Builder ipamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
            return this;
        }

        public final Boolean getIsDefault() {
            return isDefault;
        }

        public final void setIsDefault(Boolean isDefault) {
            this.isDefault = isDefault;
        }

        @Override
        public final Builder isDefault(Boolean isDefault) {
            this.isDefault = isDefault;
            return this;
        }

        public final String getResourceDiscoveryStatus() {
            return resourceDiscoveryStatus;
        }

        public final void setResourceDiscoveryStatus(String resourceDiscoveryStatus) {
            this.resourceDiscoveryStatus = resourceDiscoveryStatus;
        }

        @Override
        public final Builder resourceDiscoveryStatus(String resourceDiscoveryStatus) {
            this.resourceDiscoveryStatus = resourceDiscoveryStatus;
            return this;
        }

        @Override
        public final Builder resourceDiscoveryStatus(IpamAssociatedResourceDiscoveryStatus resourceDiscoveryStatus) {
            this.resourceDiscoveryStatus(resourceDiscoveryStatus == null ? null : resourceDiscoveryStatus.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamResourceDiscoveryAssociationState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public IpamResourceDiscoveryAssociation build() {
            return new IpamResourceDiscoveryAssociation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
