/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ebs;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ebs.model.EbsException;
import software.amazon.awssdk.services.ebs.model.EbsRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockRequest;
import software.amazon.awssdk.services.ebs.model.GetSnapshotBlockResponse;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest;
import software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse;
import software.amazon.awssdk.services.ebs.model.ResourceNotFoundException;
import software.amazon.awssdk.services.ebs.model.ValidationException;
import software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable;
import software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable;
import software.amazon.awssdk.services.ebs.transform.GetSnapshotBlockRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.ListChangedBlocksRequestMarshaller;
import software.amazon.awssdk.services.ebs.transform.ListSnapshotBlocksRequestMarshaller;

/**
 * Internal implementation of {@link EbsClient}.
 *
 * @see EbsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEbsClient implements EbsClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEbsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Returns the data in a block in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param getSnapshotBlockRequest
     * @param responseTransformer
     *        Functional interface for processing the streamed response content. The unmarshalled
     *        GetSnapshotBlockResponse and an InputStream to the response content are provided as parameters to the
     *        callback. The callback may return a transformed type which will be the return value of this method. See
     *        {@link software.amazon.awssdk.core.sync.ResponseTransformer} for details on implementing this interface
     *        and for links to pre-canned implementations for common scenarios like downloading to a file. The service
     *        documentation for the response content is as follows '
     *        <p>
     *        The data content of the block.
     *        </p>
     *        '.
     * @return The transformed result of the ResponseTransformer.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.GetSnapshotBlock
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/GetSnapshotBlock" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public <ReturnT> ReturnT getSnapshotBlock(GetSnapshotBlockRequest getSnapshotBlockRequest,
            ResponseTransformer<GetSnapshotBlockResponse, ReturnT> responseTransformer) throws ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(true)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetSnapshotBlockResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetSnapshotBlockResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(
                new ClientExecutionParams<GetSnapshotBlockRequest, GetSnapshotBlockResponse>()
                        .withOperationName("GetSnapshotBlock").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getSnapshotBlockRequest)
                        .withMarshaller(new GetSnapshotBlockRequestMarshaller(protocolFactory)), responseTransformer);
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store
     * snapshots of the same volume/snapshot lineage.
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return Result of the ListChangedBlocks operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChangedBlocksResponse listChangedBlocks(ListChangedBlocksRequest listChangedBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListChangedBlocksResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListChangedBlocksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListChangedBlocksRequest, ListChangedBlocksResponse>()
                .withOperationName("ListChangedBlocks").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listChangedBlocksRequest)
                .withMarshaller(new ListChangedBlocksRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks that are different between two Amazon Elastic Block Store
     * snapshots of the same volume/snapshot lineage.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client
     *             .listChangedBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListChangedBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListChangedBlocksIterable responses = client.listChangedBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listChangedBlocks(software.amazon.awssdk.services.ebs.model.ListChangedBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listChangedBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListChangedBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListChangedBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListChangedBlocksIterable listChangedBlocksPaginator(ListChangedBlocksRequest listChangedBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return new ListChangedBlocksIterable(this, applyPaginatorUserAgent(listChangedBlocksRequest));
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return Result of the ListSnapshotBlocks operation returned by the service.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSnapshotBlocksResponse listSnapshotBlocks(ListSnapshotBlocksRequest listSnapshotBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSnapshotBlocksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSnapshotBlocksResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<ListSnapshotBlocksRequest, ListSnapshotBlocksResponse>()
                .withOperationName("ListSnapshotBlocks").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(listSnapshotBlocksRequest)
                .withMarshaller(new ListSnapshotBlocksRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Returns the block indexes and block tokens for blocks in an Amazon Elastic Block Store snapshot.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client
     *             .listSnapshotBlocksPaginator(request);
     *     for (software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.ebs.paginators.ListSnapshotBlocksIterable responses = client.listSnapshotBlocksPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listSnapshotBlocks(software.amazon.awssdk.services.ebs.model.ListSnapshotBlocksRequest)} operation.</b>
     * </p>
     *
     * @param listSnapshotBlocksRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ValidationException
     *         The input fails to satisfy the constraints of the EBS direct APIs.
     * @throws ResourceNotFoundException
     *         The specified resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws EbsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample EbsClient.ListSnapshotBlocks
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ebs-2019-11-02/ListSnapshotBlocks" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListSnapshotBlocksIterable listSnapshotBlocksPaginator(ListSnapshotBlocksRequest listSnapshotBlocksRequest)
            throws ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException, EbsException {
        return new ListSnapshotBlocksIterable(this, applyPaginatorUserAgent(listSnapshotBlocksRequest));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(EbsException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends EbsRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
