/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.dynamodb.DynamoDbAsyncClient;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface DynamoDbAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link DynamoDbAsyncClient#describeContributorInsights} API until the desired condition
     * {@code ContributorInsightsEnabled} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeContributorInsightsRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeContributorInsightsResponse>> waitUntilContributorInsightsEnabled(
            DescribeContributorInsightsRequest describeContributorInsightsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeContributorInsights} API until the desired condition
     * {@code ContributorInsightsEnabled} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeContributorInsightsRequest#builder()}
     *
     * @param describeContributorInsightsRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeContributorInsightsResponse>> waitUntilContributorInsightsEnabled(
            Consumer<DescribeContributorInsightsRequest.Builder> describeContributorInsightsRequest) {
        return waitUntilContributorInsightsEnabled(DescribeContributorInsightsRequest.builder()
                .applyMutation(describeContributorInsightsRequest).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeContributorInsights} API until the desired condition
     * {@code ContributorInsightsEnabled} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeContributorInsightsRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeContributorInsightsResponse>> waitUntilContributorInsightsEnabled(
            DescribeContributorInsightsRequest describeContributorInsightsRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeContributorInsights} API until the desired condition
     * {@code ContributorInsightsEnabled} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeContributorInsightsRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeContributorInsightsResponse>> waitUntilContributorInsightsEnabled(
            Consumer<DescribeContributorInsightsRequest.Builder> describeContributorInsightsRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilContributorInsightsEnabled(
                DescribeContributorInsightsRequest.builder().applyMutation(describeContributorInsightsRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeExport} API until the desired condition {@code ExportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeExportRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilExportCompleted(
            DescribeExportRequest describeExportRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeExport} API until the desired condition {@code ExportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeExportRequest#builder()}
     *
     * @param describeExportRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilExportCompleted(
            Consumer<DescribeExportRequest.Builder> describeExportRequest) {
        return waitUntilExportCompleted(DescribeExportRequest.builder().applyMutation(describeExportRequest).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeExport} API until the desired condition {@code ExportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeExportRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilExportCompleted(
            DescribeExportRequest describeExportRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeExport} API until the desired condition {@code ExportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeExportRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeExportResponse>> waitUntilExportCompleted(
            Consumer<DescribeExportRequest.Builder> describeExportRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilExportCompleted(DescribeExportRequest.builder().applyMutation(describeExportRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeImport} API until the desired condition {@code ImportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeImportRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilImportCompleted(
            DescribeImportRequest describeImportRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeImport} API until the desired condition {@code ImportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeImportRequest#builder()}
     *
     * @param describeImportRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilImportCompleted(
            Consumer<DescribeImportRequest.Builder> describeImportRequest) {
        return waitUntilImportCompleted(DescribeImportRequest.builder().applyMutation(describeImportRequest).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeImport} API until the desired condition {@code ImportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeImportRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilImportCompleted(
            DescribeImportRequest describeImportRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeImport} API until the desired condition {@code ImportCompleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeImportRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeImportResponse>> waitUntilImportCompleted(
            Consumer<DescribeImportRequest.Builder> describeImportRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilImportCompleted(DescribeImportRequest.builder().applyMutation(describeImportRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeKinesisStreamingDestination} API until the desired condition
     * {@code KinesisStreamingDestinationActive} is met, or until it is determined that the resource will never enter
     * into the desired state
     *
     * @param describeKinesisStreamingDestinationRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeKinesisStreamingDestinationResponse>> waitUntilKinesisStreamingDestinationActive(
            DescribeKinesisStreamingDestinationRequest describeKinesisStreamingDestinationRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeKinesisStreamingDestination} API until the desired condition
     * {@code KinesisStreamingDestinationActive} is met, or until it is determined that the resource will never enter
     * into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeKinesisStreamingDestinationRequest#builder()}
     *
     * @param describeKinesisStreamingDestinationRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeKinesisStreamingDestinationResponse>> waitUntilKinesisStreamingDestinationActive(
            Consumer<DescribeKinesisStreamingDestinationRequest.Builder> describeKinesisStreamingDestinationRequest) {
        return waitUntilKinesisStreamingDestinationActive(DescribeKinesisStreamingDestinationRequest.builder()
                .applyMutation(describeKinesisStreamingDestinationRequest).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeKinesisStreamingDestination} API until the desired condition
     * {@code KinesisStreamingDestinationActive} is met, or until it is determined that the resource will never enter
     * into the desired state
     *
     * @param describeKinesisStreamingDestinationRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeKinesisStreamingDestinationResponse>> waitUntilKinesisStreamingDestinationActive(
            DescribeKinesisStreamingDestinationRequest describeKinesisStreamingDestinationRequest,
            WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeKinesisStreamingDestination} API until the desired condition
     * {@code KinesisStreamingDestinationActive} is met, or until it is determined that the resource will never enter
     * into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeKinesisStreamingDestinationRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeKinesisStreamingDestinationResponse>> waitUntilKinesisStreamingDestinationActive(
            Consumer<DescribeKinesisStreamingDestinationRequest.Builder> describeKinesisStreamingDestinationRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilKinesisStreamingDestinationActive(
                DescribeKinesisStreamingDestinationRequest.builder().applyMutation(describeKinesisStreamingDestinationRequest)
                        .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableExists} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeTableRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableExists(
            DescribeTableRequest describeTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableExists} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeTableRequest#builder()}
     *
     * @param describeTableRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableExists(
            Consumer<DescribeTableRequest.Builder> describeTableRequest) {
        return waitUntilTableExists(DescribeTableRequest.builder().applyMutation(describeTableRequest).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableExists} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeTableRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableExists(
            DescribeTableRequest describeTableRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableExists} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeTableRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableExists(
            Consumer<DescribeTableRequest.Builder> describeTableRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTableExists(DescribeTableRequest.builder().applyMutation(describeTableRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableNotExists} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeTableRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableNotExists(
            DescribeTableRequest describeTableRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableNotExists} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeTableRequest#builder()}
     *
     * @param describeTableRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableNotExists(
            Consumer<DescribeTableRequest.Builder> describeTableRequest) {
        return waitUntilTableNotExists(DescribeTableRequest.builder().applyMutation(describeTableRequest).build());
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableNotExists} is met, or
     * until it is determined that the resource will never enter into the desired state
     *
     * @param describeTableRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableNotExists(
            DescribeTableRequest describeTableRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link DynamoDbAsyncClient#describeTable} API until the desired condition {@code TableNotExists} is met, or
     * until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeTableRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeTableResponse>> waitUntilTableNotExists(
            Consumer<DescribeTableRequest.Builder> describeTableRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilTableNotExists(DescribeTableRequest.builder().applyMutation(describeTableRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link DynamoDbAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultDynamoDbAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link DynamoDbAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link DynamoDbAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link DynamoDbAsyncWaiter}
     */
    static DynamoDbAsyncWaiter create() {
        return DefaultDynamoDbAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link DynamoDbAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(DynamoDbAsyncClient client);

        /**
         * Builds an instance of {@link DynamoDbAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link DynamoDbAsyncWaiter}
         */
        DynamoDbAsyncWaiter build();
    }
}
