/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the auto scaling settings to be modified for a global table or global secondary index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingSettingsUpdate implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingSettingsUpdate.Builder, AutoScalingSettingsUpdate> {
    private static final SdkField<Long> MINIMUM_UNITS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MinimumUnits").getter(getter(AutoScalingSettingsUpdate::minimumUnits))
            .setter(setter(Builder::minimumUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimumUnits").build()).build();

    private static final SdkField<Long> MAXIMUM_UNITS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("MaximumUnits").getter(getter(AutoScalingSettingsUpdate::maximumUnits))
            .setter(setter(Builder::maximumUnits))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumUnits").build()).build();

    private static final SdkField<Boolean> AUTO_SCALING_DISABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoScalingDisabled").getter(getter(AutoScalingSettingsUpdate::autoScalingDisabled))
            .setter(setter(Builder::autoScalingDisabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingDisabled").build())
            .build();

    private static final SdkField<String> AUTO_SCALING_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AutoScalingRoleArn").getter(getter(AutoScalingSettingsUpdate::autoScalingRoleArn))
            .setter(setter(Builder::autoScalingRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoScalingRoleArn").build())
            .build();

    private static final SdkField<AutoScalingPolicyUpdate> SCALING_POLICY_UPDATE_FIELD = SdkField
            .<AutoScalingPolicyUpdate> builder(MarshallingType.SDK_POJO).memberName("ScalingPolicyUpdate")
            .getter(getter(AutoScalingSettingsUpdate::scalingPolicyUpdate)).setter(setter(Builder::scalingPolicyUpdate))
            .constructor(AutoScalingPolicyUpdate::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalingPolicyUpdate").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MINIMUM_UNITS_FIELD,
            MAXIMUM_UNITS_FIELD, AUTO_SCALING_DISABLED_FIELD, AUTO_SCALING_ROLE_ARN_FIELD, SCALING_POLICY_UPDATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Long minimumUnits;

    private final Long maximumUnits;

    private final Boolean autoScalingDisabled;

    private final String autoScalingRoleArn;

    private final AutoScalingPolicyUpdate scalingPolicyUpdate;

    private AutoScalingSettingsUpdate(BuilderImpl builder) {
        this.minimumUnits = builder.minimumUnits;
        this.maximumUnits = builder.maximumUnits;
        this.autoScalingDisabled = builder.autoScalingDisabled;
        this.autoScalingRoleArn = builder.autoScalingRoleArn;
        this.scalingPolicyUpdate = builder.scalingPolicyUpdate;
    }

    /**
     * <p>
     * The minimum capacity units that a global table or global secondary index should be scaled down to.
     * </p>
     * 
     * @return The minimum capacity units that a global table or global secondary index should be scaled down to.
     */
    public final Long minimumUnits() {
        return minimumUnits;
    }

    /**
     * <p>
     * The maximum capacity units that a global table or global secondary index should be scaled up to.
     * </p>
     * 
     * @return The maximum capacity units that a global table or global secondary index should be scaled up to.
     */
    public final Long maximumUnits() {
        return maximumUnits;
    }

    /**
     * <p>
     * Disabled auto scaling for this global table or global secondary index.
     * </p>
     * 
     * @return Disabled auto scaling for this global table or global secondary index.
     */
    public final Boolean autoScalingDisabled() {
        return autoScalingDisabled;
    }

    /**
     * <p>
     * Role ARN used for configuring auto scaling policy.
     * </p>
     * 
     * @return Role ARN used for configuring auto scaling policy.
     */
    public final String autoScalingRoleArn() {
        return autoScalingRoleArn;
    }

    /**
     * <p>
     * The scaling policy to apply for scaling target global table or global secondary index capacity units.
     * </p>
     * 
     * @return The scaling policy to apply for scaling target global table or global secondary index capacity units.
     */
    public final AutoScalingPolicyUpdate scalingPolicyUpdate() {
        return scalingPolicyUpdate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(minimumUnits());
        hashCode = 31 * hashCode + Objects.hashCode(maximumUnits());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingDisabled());
        hashCode = 31 * hashCode + Objects.hashCode(autoScalingRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(scalingPolicyUpdate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingSettingsUpdate)) {
            return false;
        }
        AutoScalingSettingsUpdate other = (AutoScalingSettingsUpdate) obj;
        return Objects.equals(minimumUnits(), other.minimumUnits()) && Objects.equals(maximumUnits(), other.maximumUnits())
                && Objects.equals(autoScalingDisabled(), other.autoScalingDisabled())
                && Objects.equals(autoScalingRoleArn(), other.autoScalingRoleArn())
                && Objects.equals(scalingPolicyUpdate(), other.scalingPolicyUpdate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AutoScalingSettingsUpdate").add("MinimumUnits", minimumUnits())
                .add("MaximumUnits", maximumUnits()).add("AutoScalingDisabled", autoScalingDisabled())
                .add("AutoScalingRoleArn", autoScalingRoleArn()).add("ScalingPolicyUpdate", scalingPolicyUpdate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MinimumUnits":
            return Optional.ofNullable(clazz.cast(minimumUnits()));
        case "MaximumUnits":
            return Optional.ofNullable(clazz.cast(maximumUnits()));
        case "AutoScalingDisabled":
            return Optional.ofNullable(clazz.cast(autoScalingDisabled()));
        case "AutoScalingRoleArn":
            return Optional.ofNullable(clazz.cast(autoScalingRoleArn()));
        case "ScalingPolicyUpdate":
            return Optional.ofNullable(clazz.cast(scalingPolicyUpdate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingSettingsUpdate, T> g) {
        return obj -> g.apply((AutoScalingSettingsUpdate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingSettingsUpdate> {
        /**
         * <p>
         * The minimum capacity units that a global table or global secondary index should be scaled down to.
         * </p>
         * 
         * @param minimumUnits
         *        The minimum capacity units that a global table or global secondary index should be scaled down to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumUnits(Long minimumUnits);

        /**
         * <p>
         * The maximum capacity units that a global table or global secondary index should be scaled up to.
         * </p>
         * 
         * @param maximumUnits
         *        The maximum capacity units that a global table or global secondary index should be scaled up to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumUnits(Long maximumUnits);

        /**
         * <p>
         * Disabled auto scaling for this global table or global secondary index.
         * </p>
         * 
         * @param autoScalingDisabled
         *        Disabled auto scaling for this global table or global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingDisabled(Boolean autoScalingDisabled);

        /**
         * <p>
         * Role ARN used for configuring auto scaling policy.
         * </p>
         * 
         * @param autoScalingRoleArn
         *        Role ARN used for configuring auto scaling policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoScalingRoleArn(String autoScalingRoleArn);

        /**
         * <p>
         * The scaling policy to apply for scaling target global table or global secondary index capacity units.
         * </p>
         * 
         * @param scalingPolicyUpdate
         *        The scaling policy to apply for scaling target global table or global secondary index capacity units.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scalingPolicyUpdate(AutoScalingPolicyUpdate scalingPolicyUpdate);

        /**
         * <p>
         * The scaling policy to apply for scaling target global table or global secondary index capacity units.
         * </p>
         * This is a convenience method that creates an instance of the {@link AutoScalingPolicyUpdate.Builder} avoiding
         * the need to create one manually via {@link AutoScalingPolicyUpdate#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoScalingPolicyUpdate.Builder#build()} is called immediately
         * and its result is passed to {@link #scalingPolicyUpdate(AutoScalingPolicyUpdate)}.
         * 
         * @param scalingPolicyUpdate
         *        a consumer that will call methods on {@link AutoScalingPolicyUpdate.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scalingPolicyUpdate(AutoScalingPolicyUpdate)
         */
        default Builder scalingPolicyUpdate(Consumer<AutoScalingPolicyUpdate.Builder> scalingPolicyUpdate) {
            return scalingPolicyUpdate(AutoScalingPolicyUpdate.builder().applyMutation(scalingPolicyUpdate).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Long minimumUnits;

        private Long maximumUnits;

        private Boolean autoScalingDisabled;

        private String autoScalingRoleArn;

        private AutoScalingPolicyUpdate scalingPolicyUpdate;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingSettingsUpdate model) {
            minimumUnits(model.minimumUnits);
            maximumUnits(model.maximumUnits);
            autoScalingDisabled(model.autoScalingDisabled);
            autoScalingRoleArn(model.autoScalingRoleArn);
            scalingPolicyUpdate(model.scalingPolicyUpdate);
        }

        public final Long getMinimumUnits() {
            return minimumUnits;
        }

        public final void setMinimumUnits(Long minimumUnits) {
            this.minimumUnits = minimumUnits;
        }

        @Override
        public final Builder minimumUnits(Long minimumUnits) {
            this.minimumUnits = minimumUnits;
            return this;
        }

        public final Long getMaximumUnits() {
            return maximumUnits;
        }

        public final void setMaximumUnits(Long maximumUnits) {
            this.maximumUnits = maximumUnits;
        }

        @Override
        public final Builder maximumUnits(Long maximumUnits) {
            this.maximumUnits = maximumUnits;
            return this;
        }

        public final Boolean getAutoScalingDisabled() {
            return autoScalingDisabled;
        }

        public final void setAutoScalingDisabled(Boolean autoScalingDisabled) {
            this.autoScalingDisabled = autoScalingDisabled;
        }

        @Override
        public final Builder autoScalingDisabled(Boolean autoScalingDisabled) {
            this.autoScalingDisabled = autoScalingDisabled;
            return this;
        }

        public final String getAutoScalingRoleArn() {
            return autoScalingRoleArn;
        }

        public final void setAutoScalingRoleArn(String autoScalingRoleArn) {
            this.autoScalingRoleArn = autoScalingRoleArn;
        }

        @Override
        public final Builder autoScalingRoleArn(String autoScalingRoleArn) {
            this.autoScalingRoleArn = autoScalingRoleArn;
            return this;
        }

        public final AutoScalingPolicyUpdate.Builder getScalingPolicyUpdate() {
            return scalingPolicyUpdate != null ? scalingPolicyUpdate.toBuilder() : null;
        }

        public final void setScalingPolicyUpdate(AutoScalingPolicyUpdate.BuilderImpl scalingPolicyUpdate) {
            this.scalingPolicyUpdate = scalingPolicyUpdate != null ? scalingPolicyUpdate.build() : null;
        }

        @Override
        public final Builder scalingPolicyUpdate(AutoScalingPolicyUpdate scalingPolicyUpdate) {
            this.scalingPolicyUpdate = scalingPolicyUpdate;
            return this;
        }

        @Override
        public AutoScalingSettingsUpdate build() {
            return new AutoScalingSettingsUpdate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
