/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents <i>a single element</i> of a key schema. A key schema specifies the attributes that make up the primary
 * key of a table, or the key attributes of an index.
 * </p>
 * <p>
 * A <code>KeySchemaElement</code> represents exactly one attribute of the primary key. For example, a simple primary
 * key would be represented by one <code>KeySchemaElement</code> (for the partition key). A composite primary key would
 * require one <code>KeySchemaElement</code> for the partition key, and another <code>KeySchemaElement</code> for the
 * sort key.
 * </p>
 * <p>
 * A <code>KeySchemaElement</code> must be a scalar, top-level attribute (not a nested attribute). The data type must be
 * one of String, Number, or Binary. The attribute cannot be nested within a List or a Map.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class KeySchemaElement implements SdkPojo, Serializable,
        ToCopyableBuilder<KeySchemaElement.Builder, KeySchemaElement> {
    private static final SdkField<String> ATTRIBUTE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeySchemaElement::attributeName)).setter(setter(Builder::attributeName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeName").build()).build();

    private static final SdkField<String> KEY_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(KeySchemaElement::keyTypeAsString)).setter(setter(Builder::keyType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeyType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTRIBUTE_NAME_FIELD,
            KEY_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String attributeName;

    private final String keyType;

    private KeySchemaElement(BuilderImpl builder) {
        this.attributeName = builder.attributeName;
        this.keyType = builder.keyType;
    }

    /**
     * <p>
     * The name of a key attribute.
     * </p>
     * 
     * @return The name of a key attribute.
     */
    public String attributeName() {
        return attributeName;
    }

    /**
     * <p>
     * The role that this key attribute will assume:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyType} will
     * return {@link KeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyTypeAsString}.
     * </p>
     * 
     * @return The role that this key attribute will assume:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     * @see KeyType
     */
    public KeyType keyType() {
        return KeyType.fromValue(keyType);
    }

    /**
     * <p>
     * The role that this key attribute will assume:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #keyType} will
     * return {@link KeyType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #keyTypeAsString}.
     * </p>
     * 
     * @return The role that this key attribute will assume:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     * @see KeyType
     */
    public String keyTypeAsString() {
        return keyType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(attributeName());
        hashCode = 31 * hashCode + Objects.hashCode(keyTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof KeySchemaElement)) {
            return false;
        }
        KeySchemaElement other = (KeySchemaElement) obj;
        return Objects.equals(attributeName(), other.attributeName())
                && Objects.equals(keyTypeAsString(), other.keyTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("KeySchemaElement").add("AttributeName", attributeName()).add("KeyType", keyTypeAsString())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AttributeName":
            return Optional.ofNullable(clazz.cast(attributeName()));
        case "KeyType":
            return Optional.ofNullable(clazz.cast(keyTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<KeySchemaElement, T> g) {
        return obj -> g.apply((KeySchemaElement) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, KeySchemaElement> {
        /**
         * <p>
         * The name of a key attribute.
         * </p>
         * 
         * @param attributeName
         *        The name of a key attribute.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeName(String attributeName);

        /**
         * <p>
         * The role that this key attribute will assume:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keyType
         *        The role that this key attribute will assume:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @see KeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyType
         */
        Builder keyType(String keyType);

        /**
         * <p>
         * The role that this key attribute will assume:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keyType
         *        The role that this key attribute will assume:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @see KeyType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see KeyType
         */
        Builder keyType(KeyType keyType);
    }

    static final class BuilderImpl implements Builder {
        private String attributeName;

        private String keyType;

        private BuilderImpl() {
        }

        private BuilderImpl(KeySchemaElement model) {
            attributeName(model.attributeName);
            keyType(model.keyType);
        }

        public final String getAttributeName() {
            return attributeName;
        }

        @Override
        public final Builder attributeName(String attributeName) {
            this.attributeName = attributeName;
            return this;
        }

        public final void setAttributeName(String attributeName) {
            this.attributeName = attributeName;
        }

        public final String getKeyTypeAsString() {
            return keyType;
        }

        @Override
        public final Builder keyType(String keyType) {
            this.keyType = keyType;
            return this;
        }

        @Override
        public final Builder keyType(KeyType keyType) {
            this.keyType(keyType == null ? null : keyType.toString());
            return this;
        }

        public final void setKeyType(String keyType) {
            this.keyType = keyType;
        }

        @Override
        public KeySchemaElement build() {
            return new KeySchemaElement(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
