/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.datasync.model;

import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateLocationObjectStorageRequest extends DataSyncRequest implements
        ToCopyableBuilder<UpdateLocationObjectStorageRequest.Builder, UpdateLocationObjectStorageRequest> {
    private static final SdkField<String> LOCATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LocationArn").getter(getter(UpdateLocationObjectStorageRequest::locationArn))
            .setter(setter(Builder::locationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocationArn").build()).build();

    private static final SdkField<Integer> SERVER_PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ServerPort").getter(getter(UpdateLocationObjectStorageRequest::serverPort))
            .setter(setter(Builder::serverPort))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerPort").build()).build();

    private static final SdkField<String> SERVER_PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerProtocol").getter(getter(UpdateLocationObjectStorageRequest::serverProtocolAsString))
            .setter(setter(Builder::serverProtocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerProtocol").build()).build();

    private static final SdkField<String> SUBDIRECTORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Subdirectory").getter(getter(UpdateLocationObjectStorageRequest::subdirectory))
            .setter(setter(Builder::subdirectory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Subdirectory").build()).build();

    private static final SdkField<String> SERVER_HOSTNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServerHostname").getter(getter(UpdateLocationObjectStorageRequest::serverHostname))
            .setter(setter(Builder::serverHostname))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerHostname").build()).build();

    private static final SdkField<String> ACCESS_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccessKey").getter(getter(UpdateLocationObjectStorageRequest::accessKey))
            .setter(setter(Builder::accessKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessKey").build()).build();

    private static final SdkField<String> SECRET_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SecretKey").getter(getter(UpdateLocationObjectStorageRequest::secretKey))
            .setter(setter(Builder::secretKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecretKey").build()).build();

    private static final SdkField<List<String>> AGENT_ARNS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AgentArns")
            .getter(getter(UpdateLocationObjectStorageRequest::agentArns))
            .setter(setter(Builder::agentArns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AgentArns").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<SdkBytes> SERVER_CERTIFICATE_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
            .memberName("ServerCertificate").getter(getter(UpdateLocationObjectStorageRequest::serverCertificate))
            .setter(setter(Builder::serverCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerCertificate").build()).build();

    private static final SdkField<CmkSecretConfig> CMK_SECRET_CONFIG_FIELD = SdkField
            .<CmkSecretConfig> builder(MarshallingType.SDK_POJO).memberName("CmkSecretConfig")
            .getter(getter(UpdateLocationObjectStorageRequest::cmkSecretConfig)).setter(setter(Builder::cmkSecretConfig))
            .constructor(CmkSecretConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CmkSecretConfig").build()).build();

    private static final SdkField<CustomSecretConfig> CUSTOM_SECRET_CONFIG_FIELD = SdkField
            .<CustomSecretConfig> builder(MarshallingType.SDK_POJO).memberName("CustomSecretConfig")
            .getter(getter(UpdateLocationObjectStorageRequest::customSecretConfig)).setter(setter(Builder::customSecretConfig))
            .constructor(CustomSecretConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomSecretConfig").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LOCATION_ARN_FIELD,
            SERVER_PORT_FIELD, SERVER_PROTOCOL_FIELD, SUBDIRECTORY_FIELD, SERVER_HOSTNAME_FIELD, ACCESS_KEY_FIELD,
            SECRET_KEY_FIELD, AGENT_ARNS_FIELD, SERVER_CERTIFICATE_FIELD, CMK_SECRET_CONFIG_FIELD, CUSTOM_SECRET_CONFIG_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String locationArn;

    private final Integer serverPort;

    private final String serverProtocol;

    private final String subdirectory;

    private final String serverHostname;

    private final String accessKey;

    private final String secretKey;

    private final List<String> agentArns;

    private final SdkBytes serverCertificate;

    private final CmkSecretConfig cmkSecretConfig;

    private final CustomSecretConfig customSecretConfig;

    private UpdateLocationObjectStorageRequest(BuilderImpl builder) {
        super(builder);
        this.locationArn = builder.locationArn;
        this.serverPort = builder.serverPort;
        this.serverProtocol = builder.serverProtocol;
        this.subdirectory = builder.subdirectory;
        this.serverHostname = builder.serverHostname;
        this.accessKey = builder.accessKey;
        this.secretKey = builder.secretKey;
        this.agentArns = builder.agentArns;
        this.serverCertificate = builder.serverCertificate;
        this.cmkSecretConfig = builder.cmkSecretConfig;
        this.customSecretConfig = builder.customSecretConfig;
    }

    /**
     * <p>
     * Specifies the ARN of the object storage system location that you're updating.
     * </p>
     * 
     * @return Specifies the ARN of the object storage system location that you're updating.
     */
    public final String locationArn() {
        return locationArn;
    }

    /**
     * <p>
     * Specifies the port that your object storage server accepts inbound network traffic on (for example, port 443).
     * </p>
     * 
     * @return Specifies the port that your object storage server accepts inbound network traffic on (for example, port
     *         443).
     */
    public final Integer serverPort() {
        return serverPort;
    }

    /**
     * <p>
     * Specifies the protocol that your object storage server uses to communicate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serverProtocol}
     * will return {@link ObjectStorageServerProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #serverProtocolAsString}.
     * </p>
     * 
     * @return Specifies the protocol that your object storage server uses to communicate.
     * @see ObjectStorageServerProtocol
     */
    public final ObjectStorageServerProtocol serverProtocol() {
        return ObjectStorageServerProtocol.fromValue(serverProtocol);
    }

    /**
     * <p>
     * Specifies the protocol that your object storage server uses to communicate.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serverProtocol}
     * will return {@link ObjectStorageServerProtocol#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #serverProtocolAsString}.
     * </p>
     * 
     * @return Specifies the protocol that your object storage server uses to communicate.
     * @see ObjectStorageServerProtocol
     */
    public final String serverProtocolAsString() {
        return serverProtocol;
    }

    /**
     * <p>
     * Specifies the object prefix for your object storage server. If this is a source location, DataSync only copies
     * objects with this prefix. If this is a destination location, DataSync writes all objects with this prefix.
     * </p>
     * 
     * @return Specifies the object prefix for your object storage server. If this is a source location, DataSync only
     *         copies objects with this prefix. If this is a destination location, DataSync writes all objects with this
     *         prefix.
     */
    public final String subdirectory() {
        return subdirectory;
    }

    /**
     * <p>
     * Specifies the domain name or IP address (IPv4 or IPv6) of the object storage server that your DataSync agent
     * connects to.
     * </p>
     * 
     * @return Specifies the domain name or IP address (IPv4 or IPv6) of the object storage server that your DataSync
     *         agent connects to.
     */
    public final String serverHostname() {
        return serverHostname;
    }

    /**
     * <p>
     * Specifies the access key (for example, a user name) if credentials are required to authenticate with the object
     * storage server.
     * </p>
     * 
     * @return Specifies the access key (for example, a user name) if credentials are required to authenticate with the
     *         object storage server.
     */
    public final String accessKey() {
        return accessKey;
    }

    /**
     * <p>
     * Specifies the secret key (for example, a password) if credentials are required to authenticate with the object
     * storage server.
     * </p>
     * <note>
     * <p>
     * If you provide a secret using <code>SecretKey</code>, but do not provide secret configuration details using
     * <code>CmkSecretConfig</code> or <code>CustomSecretConfig</code>, then DataSync stores the token using your Amazon
     * Web Services account's Secrets Manager secret.
     * </p>
     * </note>
     * 
     * @return Specifies the secret key (for example, a password) if credentials are required to authenticate with the
     *         object storage server.</p> <note>
     *         <p>
     *         If you provide a secret using <code>SecretKey</code>, but do not provide secret configuration details
     *         using <code>CmkSecretConfig</code> or <code>CustomSecretConfig</code>, then DataSync stores the token
     *         using your Amazon Web Services account's Secrets Manager secret.
     *         </p>
     */
    public final String secretKey() {
        return secretKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the AgentArns property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasAgentArns() {
        return agentArns != null && !(agentArns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * (Optional) Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can connect with your object
     * storage system. If you are setting up an agentless cross-cloud transfer, you do not need to specify a value for
     * this parameter.
     * </p>
     * <note>
     * <p>
     * You cannot add or remove agents from a storage location after you initially create it.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAgentArns} method.
     * </p>
     * 
     * @return (Optional) Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can connect with your
     *         object storage system. If you are setting up an agentless cross-cloud transfer, you do not need to
     *         specify a value for this parameter.</p> <note>
     *         <p>
     *         You cannot add or remove agents from a storage location after you initially create it.
     *         </p>
     */
    public final List<String> agentArns() {
        return agentArns;
    }

    /**
     * <p>
     * Specifies a certificate chain for DataSync to authenticate with your object storage system if the system uses a
     * private or self-signed certificate authority (CA). You must specify a single <code>.pem</code> file with a full
     * certificate chain (for example, <code>file:///home/user/.ssh/object_storage_certificates.pem</code>).
     * </p>
     * <p>
     * The certificate chain might include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The object storage system's certificate
     * </p>
     * </li>
     * <li>
     * <p>
     * All intermediate certificates (if there are any)
     * </p>
     * </li>
     * <li>
     * <p>
     * The root certificate of the signing CA
     * </p>
     * </li>
     * </ul>
     * <p>
     * You can concatenate your certificates into a <code>.pem</code> file (which can be up to 32768 bytes before base64
     * encoding). The following example <code>cat</code> command creates an <code>object_storage_certificates.pem</code>
     * file that includes three certificates:
     * </p>
     * <p>
     * <code>cat object_server_certificate.pem intermediate_certificate.pem ca_root_certificate.pem &gt; object_storage_certificates.pem</code>
     * </p>
     * <p>
     * To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
     * </p>
     * <p>
     * Updating this parameter doesn't interfere with tasks that you have in progress.
     * </p>
     * 
     * @return Specifies a certificate chain for DataSync to authenticate with your object storage system if the system
     *         uses a private or self-signed certificate authority (CA). You must specify a single <code>.pem</code>
     *         file with a full certificate chain (for example,
     *         <code>file:///home/user/.ssh/object_storage_certificates.pem</code>).</p>
     *         <p>
     *         The certificate chain might include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The object storage system's certificate
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         All intermediate certificates (if there are any)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The root certificate of the signing CA
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         You can concatenate your certificates into a <code>.pem</code> file (which can be up to 32768 bytes
     *         before base64 encoding). The following example <code>cat</code> command creates an
     *         <code>object_storage_certificates.pem</code> file that includes three certificates:
     *         </p>
     *         <p>
     *         <code>cat object_server_certificate.pem intermediate_certificate.pem ca_root_certificate.pem &gt; object_storage_certificates.pem</code>
     *         </p>
     *         <p>
     *         To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
     *         </p>
     *         <p>
     *         Updating this parameter doesn't interfere with tasks that you have in progress.
     */
    public final SdkBytes serverCertificate() {
        return serverCertificate;
    }

    /**
     * <p>
     * Specifies configuration information for a DataSync-managed secret, such as an authentication token or set of
     * credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
     * </p>
     * 
     * @return Specifies configuration information for a DataSync-managed secret, such as an authentication token or set
     *         of credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
     */
    public final CmkSecretConfig cmkSecretConfig() {
        return cmkSecretConfig;
    }

    /**
     * <p>
     * Specifies configuration information for a customer-managed secret, such as an authentication token or set of
     * credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
     * </p>
     * 
     * @return Specifies configuration information for a customer-managed secret, such as an authentication token or set
     *         of credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
     */
    public final CustomSecretConfig customSecretConfig() {
        return customSecretConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(locationArn());
        hashCode = 31 * hashCode + Objects.hashCode(serverPort());
        hashCode = 31 * hashCode + Objects.hashCode(serverProtocolAsString());
        hashCode = 31 * hashCode + Objects.hashCode(subdirectory());
        hashCode = 31 * hashCode + Objects.hashCode(serverHostname());
        hashCode = 31 * hashCode + Objects.hashCode(accessKey());
        hashCode = 31 * hashCode + Objects.hashCode(secretKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasAgentArns() ? agentArns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(serverCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(cmkSecretConfig());
        hashCode = 31 * hashCode + Objects.hashCode(customSecretConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateLocationObjectStorageRequest)) {
            return false;
        }
        UpdateLocationObjectStorageRequest other = (UpdateLocationObjectStorageRequest) obj;
        return Objects.equals(locationArn(), other.locationArn()) && Objects.equals(serverPort(), other.serverPort())
                && Objects.equals(serverProtocolAsString(), other.serverProtocolAsString())
                && Objects.equals(subdirectory(), other.subdirectory())
                && Objects.equals(serverHostname(), other.serverHostname()) && Objects.equals(accessKey(), other.accessKey())
                && Objects.equals(secretKey(), other.secretKey()) && hasAgentArns() == other.hasAgentArns()
                && Objects.equals(agentArns(), other.agentArns())
                && Objects.equals(serverCertificate(), other.serverCertificate())
                && Objects.equals(cmkSecretConfig(), other.cmkSecretConfig())
                && Objects.equals(customSecretConfig(), other.customSecretConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateLocationObjectStorageRequest").add("LocationArn", locationArn())
                .add("ServerPort", serverPort()).add("ServerProtocol", serverProtocolAsString())
                .add("Subdirectory", subdirectory()).add("ServerHostname", serverHostname()).add("AccessKey", accessKey())
                .add("SecretKey", secretKey() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AgentArns", hasAgentArns() ? agentArns() : null).add("ServerCertificate", serverCertificate())
                .add("CmkSecretConfig", cmkSecretConfig()).add("CustomSecretConfig", customSecretConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LocationArn":
            return Optional.ofNullable(clazz.cast(locationArn()));
        case "ServerPort":
            return Optional.ofNullable(clazz.cast(serverPort()));
        case "ServerProtocol":
            return Optional.ofNullable(clazz.cast(serverProtocolAsString()));
        case "Subdirectory":
            return Optional.ofNullable(clazz.cast(subdirectory()));
        case "ServerHostname":
            return Optional.ofNullable(clazz.cast(serverHostname()));
        case "AccessKey":
            return Optional.ofNullable(clazz.cast(accessKey()));
        case "SecretKey":
            return Optional.ofNullable(clazz.cast(secretKey()));
        case "AgentArns":
            return Optional.ofNullable(clazz.cast(agentArns()));
        case "ServerCertificate":
            return Optional.ofNullable(clazz.cast(serverCertificate()));
        case "CmkSecretConfig":
            return Optional.ofNullable(clazz.cast(cmkSecretConfig()));
        case "CustomSecretConfig":
            return Optional.ofNullable(clazz.cast(customSecretConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("LocationArn", LOCATION_ARN_FIELD);
        map.put("ServerPort", SERVER_PORT_FIELD);
        map.put("ServerProtocol", SERVER_PROTOCOL_FIELD);
        map.put("Subdirectory", SUBDIRECTORY_FIELD);
        map.put("ServerHostname", SERVER_HOSTNAME_FIELD);
        map.put("AccessKey", ACCESS_KEY_FIELD);
        map.put("SecretKey", SECRET_KEY_FIELD);
        map.put("AgentArns", AGENT_ARNS_FIELD);
        map.put("ServerCertificate", SERVER_CERTIFICATE_FIELD);
        map.put("CmkSecretConfig", CMK_SECRET_CONFIG_FIELD);
        map.put("CustomSecretConfig", CUSTOM_SECRET_CONFIG_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<UpdateLocationObjectStorageRequest, T> g) {
        return obj -> g.apply((UpdateLocationObjectStorageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends DataSyncRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, UpdateLocationObjectStorageRequest> {
        /**
         * <p>
         * Specifies the ARN of the object storage system location that you're updating.
         * </p>
         * 
         * @param locationArn
         *        Specifies the ARN of the object storage system location that you're updating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locationArn(String locationArn);

        /**
         * <p>
         * Specifies the port that your object storage server accepts inbound network traffic on (for example, port
         * 443).
         * </p>
         * 
         * @param serverPort
         *        Specifies the port that your object storage server accepts inbound network traffic on (for example,
         *        port 443).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverPort(Integer serverPort);

        /**
         * <p>
         * Specifies the protocol that your object storage server uses to communicate.
         * </p>
         * 
         * @param serverProtocol
         *        Specifies the protocol that your object storage server uses to communicate.
         * @see ObjectStorageServerProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObjectStorageServerProtocol
         */
        Builder serverProtocol(String serverProtocol);

        /**
         * <p>
         * Specifies the protocol that your object storage server uses to communicate.
         * </p>
         * 
         * @param serverProtocol
         *        Specifies the protocol that your object storage server uses to communicate.
         * @see ObjectStorageServerProtocol
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ObjectStorageServerProtocol
         */
        Builder serverProtocol(ObjectStorageServerProtocol serverProtocol);

        /**
         * <p>
         * Specifies the object prefix for your object storage server. If this is a source location, DataSync only
         * copies objects with this prefix. If this is a destination location, DataSync writes all objects with this
         * prefix.
         * </p>
         * 
         * @param subdirectory
         *        Specifies the object prefix for your object storage server. If this is a source location, DataSync
         *        only copies objects with this prefix. If this is a destination location, DataSync writes all objects
         *        with this prefix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subdirectory(String subdirectory);

        /**
         * <p>
         * Specifies the domain name or IP address (IPv4 or IPv6) of the object storage server that your DataSync agent
         * connects to.
         * </p>
         * 
         * @param serverHostname
         *        Specifies the domain name or IP address (IPv4 or IPv6) of the object storage server that your DataSync
         *        agent connects to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverHostname(String serverHostname);

        /**
         * <p>
         * Specifies the access key (for example, a user name) if credentials are required to authenticate with the
         * object storage server.
         * </p>
         * 
         * @param accessKey
         *        Specifies the access key (for example, a user name) if credentials are required to authenticate with
         *        the object storage server.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessKey(String accessKey);

        /**
         * <p>
         * Specifies the secret key (for example, a password) if credentials are required to authenticate with the
         * object storage server.
         * </p>
         * <note>
         * <p>
         * If you provide a secret using <code>SecretKey</code>, but do not provide secret configuration details using
         * <code>CmkSecretConfig</code> or <code>CustomSecretConfig</code>, then DataSync stores the token using your
         * Amazon Web Services account's Secrets Manager secret.
         * </p>
         * </note>
         * 
         * @param secretKey
         *        Specifies the secret key (for example, a password) if credentials are required to authenticate with
         *        the object storage server.</p> <note>
         *        <p>
         *        If you provide a secret using <code>SecretKey</code>, but do not provide secret configuration details
         *        using <code>CmkSecretConfig</code> or <code>CustomSecretConfig</code>, then DataSync stores the token
         *        using your Amazon Web Services account's Secrets Manager secret.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secretKey(String secretKey);

        /**
         * <p>
         * (Optional) Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can connect with your
         * object storage system. If you are setting up an agentless cross-cloud transfer, you do not need to specify a
         * value for this parameter.
         * </p>
         * <note>
         * <p>
         * You cannot add or remove agents from a storage location after you initially create it.
         * </p>
         * </note>
         * 
         * @param agentArns
         *        (Optional) Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can connect with
         *        your object storage system. If you are setting up an agentless cross-cloud transfer, you do not need
         *        to specify a value for this parameter.</p> <note>
         *        <p>
         *        You cannot add or remove agents from a storage location after you initially create it.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(Collection<String> agentArns);

        /**
         * <p>
         * (Optional) Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can connect with your
         * object storage system. If you are setting up an agentless cross-cloud transfer, you do not need to specify a
         * value for this parameter.
         * </p>
         * <note>
         * <p>
         * You cannot add or remove agents from a storage location after you initially create it.
         * </p>
         * </note>
         * 
         * @param agentArns
         *        (Optional) Specifies the Amazon Resource Names (ARNs) of the DataSync agents that can connect with
         *        your object storage system. If you are setting up an agentless cross-cloud transfer, you do not need
         *        to specify a value for this parameter.</p> <note>
         *        <p>
         *        You cannot add or remove agents from a storage location after you initially create it.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder agentArns(String... agentArns);

        /**
         * <p>
         * Specifies a certificate chain for DataSync to authenticate with your object storage system if the system uses
         * a private or self-signed certificate authority (CA). You must specify a single <code>.pem</code> file with a
         * full certificate chain (for example, <code>file:///home/user/.ssh/object_storage_certificates.pem</code>).
         * </p>
         * <p>
         * The certificate chain might include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The object storage system's certificate
         * </p>
         * </li>
         * <li>
         * <p>
         * All intermediate certificates (if there are any)
         * </p>
         * </li>
         * <li>
         * <p>
         * The root certificate of the signing CA
         * </p>
         * </li>
         * </ul>
         * <p>
         * You can concatenate your certificates into a <code>.pem</code> file (which can be up to 32768 bytes before
         * base64 encoding). The following example <code>cat</code> command creates an
         * <code>object_storage_certificates.pem</code> file that includes three certificates:
         * </p>
         * <p>
         * <code>cat object_server_certificate.pem intermediate_certificate.pem ca_root_certificate.pem &gt; object_storage_certificates.pem</code>
         * </p>
         * <p>
         * To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
         * </p>
         * <p>
         * Updating this parameter doesn't interfere with tasks that you have in progress.
         * </p>
         * 
         * @param serverCertificate
         *        Specifies a certificate chain for DataSync to authenticate with your object storage system if the
         *        system uses a private or self-signed certificate authority (CA). You must specify a single
         *        <code>.pem</code> file with a full certificate chain (for example,
         *        <code>file:///home/user/.ssh/object_storage_certificates.pem</code>).</p>
         *        <p>
         *        The certificate chain might include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        The object storage system's certificate
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        All intermediate certificates (if there are any)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The root certificate of the signing CA
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        You can concatenate your certificates into a <code>.pem</code> file (which can be up to 32768 bytes
         *        before base64 encoding). The following example <code>cat</code> command creates an
         *        <code>object_storage_certificates.pem</code> file that includes three certificates:
         *        </p>
         *        <p>
         *        <code>cat object_server_certificate.pem intermediate_certificate.pem ca_root_certificate.pem &gt; object_storage_certificates.pem</code>
         *        </p>
         *        <p>
         *        To use this parameter, configure <code>ServerProtocol</code> to <code>HTTPS</code>.
         *        </p>
         *        <p>
         *        Updating this parameter doesn't interfere with tasks that you have in progress.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverCertificate(SdkBytes serverCertificate);

        /**
         * <p>
         * Specifies configuration information for a DataSync-managed secret, such as an authentication token or set of
         * credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
         * </p>
         * 
         * @param cmkSecretConfig
         *        Specifies configuration information for a DataSync-managed secret, such as an authentication token or
         *        set of credentials that DataSync uses to access a specific transfer location, and a customer-managed
         *        KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cmkSecretConfig(CmkSecretConfig cmkSecretConfig);

        /**
         * <p>
         * Specifies configuration information for a DataSync-managed secret, such as an authentication token or set of
         * credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
         * </p>
         * This is a convenience method that creates an instance of the {@link CmkSecretConfig.Builder} avoiding the
         * need to create one manually via {@link CmkSecretConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CmkSecretConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #cmkSecretConfig(CmkSecretConfig)}.
         * 
         * @param cmkSecretConfig
         *        a consumer that will call methods on {@link CmkSecretConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cmkSecretConfig(CmkSecretConfig)
         */
        default Builder cmkSecretConfig(Consumer<CmkSecretConfig.Builder> cmkSecretConfig) {
            return cmkSecretConfig(CmkSecretConfig.builder().applyMutation(cmkSecretConfig).build());
        }

        /**
         * <p>
         * Specifies configuration information for a customer-managed secret, such as an authentication token or set of
         * credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
         * </p>
         * 
         * @param customSecretConfig
         *        Specifies configuration information for a customer-managed secret, such as an authentication token or
         *        set of credentials that DataSync uses to access a specific transfer location, and a customer-managed
         *        KMS key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customSecretConfig(CustomSecretConfig customSecretConfig);

        /**
         * <p>
         * Specifies configuration information for a customer-managed secret, such as an authentication token or set of
         * credentials that DataSync uses to access a specific transfer location, and a customer-managed KMS key.
         * </p>
         * This is a convenience method that creates an instance of the {@link CustomSecretConfig.Builder} avoiding the
         * need to create one manually via {@link CustomSecretConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CustomSecretConfig.Builder#build()} is called immediately and its
         * result is passed to {@link #customSecretConfig(CustomSecretConfig)}.
         * 
         * @param customSecretConfig
         *        a consumer that will call methods on {@link CustomSecretConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #customSecretConfig(CustomSecretConfig)
         */
        default Builder customSecretConfig(Consumer<CustomSecretConfig.Builder> customSecretConfig) {
            return customSecretConfig(CustomSecretConfig.builder().applyMutation(customSecretConfig).build());
        }

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DataSyncRequest.BuilderImpl implements Builder {
        private String locationArn;

        private Integer serverPort;

        private String serverProtocol;

        private String subdirectory;

        private String serverHostname;

        private String accessKey;

        private String secretKey;

        private List<String> agentArns = DefaultSdkAutoConstructList.getInstance();

        private SdkBytes serverCertificate;

        private CmkSecretConfig cmkSecretConfig;

        private CustomSecretConfig customSecretConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateLocationObjectStorageRequest model) {
            super(model);
            locationArn(model.locationArn);
            serverPort(model.serverPort);
            serverProtocol(model.serverProtocol);
            subdirectory(model.subdirectory);
            serverHostname(model.serverHostname);
            accessKey(model.accessKey);
            secretKey(model.secretKey);
            agentArns(model.agentArns);
            serverCertificate(model.serverCertificate);
            cmkSecretConfig(model.cmkSecretConfig);
            customSecretConfig(model.customSecretConfig);
        }

        public final String getLocationArn() {
            return locationArn;
        }

        public final void setLocationArn(String locationArn) {
            this.locationArn = locationArn;
        }

        @Override
        public final Builder locationArn(String locationArn) {
            this.locationArn = locationArn;
            return this;
        }

        public final Integer getServerPort() {
            return serverPort;
        }

        public final void setServerPort(Integer serverPort) {
            this.serverPort = serverPort;
        }

        @Override
        public final Builder serverPort(Integer serverPort) {
            this.serverPort = serverPort;
            return this;
        }

        public final String getServerProtocol() {
            return serverProtocol;
        }

        public final void setServerProtocol(String serverProtocol) {
            this.serverProtocol = serverProtocol;
        }

        @Override
        public final Builder serverProtocol(String serverProtocol) {
            this.serverProtocol = serverProtocol;
            return this;
        }

        @Override
        public final Builder serverProtocol(ObjectStorageServerProtocol serverProtocol) {
            this.serverProtocol(serverProtocol == null ? null : serverProtocol.toString());
            return this;
        }

        public final String getSubdirectory() {
            return subdirectory;
        }

        public final void setSubdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
        }

        @Override
        public final Builder subdirectory(String subdirectory) {
            this.subdirectory = subdirectory;
            return this;
        }

        public final String getServerHostname() {
            return serverHostname;
        }

        public final void setServerHostname(String serverHostname) {
            this.serverHostname = serverHostname;
        }

        @Override
        public final Builder serverHostname(String serverHostname) {
            this.serverHostname = serverHostname;
            return this;
        }

        public final String getAccessKey() {
            return accessKey;
        }

        public final void setAccessKey(String accessKey) {
            this.accessKey = accessKey;
        }

        @Override
        public final Builder accessKey(String accessKey) {
            this.accessKey = accessKey;
            return this;
        }

        public final String getSecretKey() {
            return secretKey;
        }

        public final void setSecretKey(String secretKey) {
            this.secretKey = secretKey;
        }

        @Override
        public final Builder secretKey(String secretKey) {
            this.secretKey = secretKey;
            return this;
        }

        public final Collection<String> getAgentArns() {
            if (agentArns instanceof SdkAutoConstructList) {
                return null;
            }
            return agentArns;
        }

        public final void setAgentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
        }

        @Override
        public final Builder agentArns(Collection<String> agentArns) {
            this.agentArns = AgentArnListCopier.copy(agentArns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder agentArns(String... agentArns) {
            agentArns(Arrays.asList(agentArns));
            return this;
        }

        public final ByteBuffer getServerCertificate() {
            return serverCertificate == null ? null : serverCertificate.asByteBuffer();
        }

        public final void setServerCertificate(ByteBuffer serverCertificate) {
            serverCertificate(serverCertificate == null ? null : SdkBytes.fromByteBuffer(serverCertificate));
        }

        @Override
        public final Builder serverCertificate(SdkBytes serverCertificate) {
            this.serverCertificate = serverCertificate;
            return this;
        }

        public final CmkSecretConfig.Builder getCmkSecretConfig() {
            return cmkSecretConfig != null ? cmkSecretConfig.toBuilder() : null;
        }

        public final void setCmkSecretConfig(CmkSecretConfig.BuilderImpl cmkSecretConfig) {
            this.cmkSecretConfig = cmkSecretConfig != null ? cmkSecretConfig.build() : null;
        }

        @Override
        public final Builder cmkSecretConfig(CmkSecretConfig cmkSecretConfig) {
            this.cmkSecretConfig = cmkSecretConfig;
            return this;
        }

        public final CustomSecretConfig.Builder getCustomSecretConfig() {
            return customSecretConfig != null ? customSecretConfig.toBuilder() : null;
        }

        public final void setCustomSecretConfig(CustomSecretConfig.BuilderImpl customSecretConfig) {
            this.customSecretConfig = customSecretConfig != null ? customSecretConfig.build() : null;
        }

        @Override
        public final Builder customSecretConfig(CustomSecretConfig customSecretConfig) {
            this.customSecretConfig = customSecretConfig;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateLocationObjectStorageRequest build() {
            return new UpdateLocationObjectStorageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
