/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Provides information that describes a serverless replication created by the <code>CreateReplication</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Replication implements SdkPojo, Serializable, ToCopyableBuilder<Replication.Builder, Replication> {
    private static final SdkField<String> REPLICATION_CONFIG_IDENTIFIER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReplicationConfigIdentifier")
            .getter(getter(Replication::replicationConfigIdentifier))
            .setter(setter(Builder::replicationConfigIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationConfigIdentifier")
                    .build()).build();

    private static final SdkField<String> REPLICATION_CONFIG_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicationConfigArn").getter(getter(Replication::replicationConfigArn))
            .setter(setter(Builder::replicationConfigArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationConfigArn").build())
            .build();

    private static final SdkField<String> SOURCE_ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceEndpointArn").getter(getter(Replication::sourceEndpointArn))
            .setter(setter(Builder::sourceEndpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceEndpointArn").build()).build();

    private static final SdkField<String> TARGET_ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TargetEndpointArn").getter(getter(Replication::targetEndpointArn))
            .setter(setter(Builder::targetEndpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TargetEndpointArn").build()).build();

    private static final SdkField<String> REPLICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicationType").getter(getter(Replication::replicationTypeAsString))
            .setter(setter(Builder::replicationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Replication::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<ProvisionData> PROVISION_DATA_FIELD = SdkField
            .<ProvisionData> builder(MarshallingType.SDK_POJO).memberName("ProvisionData")
            .getter(getter(Replication::provisionData)).setter(setter(Builder::provisionData))
            .constructor(ProvisionData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionData").build()).build();

    private static final SdkField<List<PremigrationAssessmentStatus>> PREMIGRATION_ASSESSMENT_STATUSES_FIELD = SdkField
            .<List<PremigrationAssessmentStatus>> builder(MarshallingType.LIST)
            .memberName("PremigrationAssessmentStatuses")
            .getter(getter(Replication::premigrationAssessmentStatuses))
            .setter(setter(Builder::premigrationAssessmentStatuses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PremigrationAssessmentStatuses")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PremigrationAssessmentStatus> builder(MarshallingType.SDK_POJO)
                                            .constructor(PremigrationAssessmentStatus::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STOP_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StopReason").getter(getter(Replication::stopReason)).setter(setter(Builder::stopReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StopReason").build()).build();

    private static final SdkField<List<String>> FAILURE_MESSAGES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("FailureMessages")
            .getter(getter(Replication::failureMessages))
            .setter(setter(Builder::failureMessages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureMessages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ReplicationStats> REPLICATION_STATS_FIELD = SdkField
            .<ReplicationStats> builder(MarshallingType.SDK_POJO).memberName("ReplicationStats")
            .getter(getter(Replication::replicationStats)).setter(setter(Builder::replicationStats))
            .constructor(ReplicationStats::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationStats").build()).build();

    private static final SdkField<String> START_REPLICATION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StartReplicationType").getter(getter(Replication::startReplicationType))
            .setter(setter(Builder::startReplicationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartReplicationType").build())
            .build();

    private static final SdkField<Instant> CDC_START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CdcStartTime").getter(getter(Replication::cdcStartTime)).setter(setter(Builder::cdcStartTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CdcStartTime").build()).build();

    private static final SdkField<String> CDC_START_POSITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CdcStartPosition").getter(getter(Replication::cdcStartPosition))
            .setter(setter(Builder::cdcStartPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CdcStartPosition").build()).build();

    private static final SdkField<String> CDC_STOP_POSITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CdcStopPosition").getter(getter(Replication::cdcStopPosition)).setter(setter(Builder::cdcStopPosition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CdcStopPosition").build()).build();

    private static final SdkField<String> RECOVERY_CHECKPOINT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecoveryCheckpoint").getter(getter(Replication::recoveryCheckpoint))
            .setter(setter(Builder::recoveryCheckpoint))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryCheckpoint").build())
            .build();

    private static final SdkField<Instant> REPLICATION_CREATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReplicationCreateTime").getter(getter(Replication::replicationCreateTime))
            .setter(setter(Builder::replicationCreateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationCreateTime").build())
            .build();

    private static final SdkField<Instant> REPLICATION_UPDATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReplicationUpdateTime").getter(getter(Replication::replicationUpdateTime))
            .setter(setter(Builder::replicationUpdateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationUpdateTime").build())
            .build();

    private static final SdkField<Instant> REPLICATION_LAST_STOP_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReplicationLastStopTime").getter(getter(Replication::replicationLastStopTime))
            .setter(setter(Builder::replicationLastStopTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationLastStopTime").build())
            .build();

    private static final SdkField<Instant> REPLICATION_DEPROVISION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReplicationDeprovisionTime")
            .getter(getter(Replication::replicationDeprovisionTime))
            .setter(setter(Builder::replicationDeprovisionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicationDeprovisionTime").build())
            .build();

    private static final SdkField<Boolean> IS_READ_ONLY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsReadOnly").getter(getter(Replication::isReadOnly)).setter(setter(Builder::isReadOnly))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsReadOnly").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            REPLICATION_CONFIG_IDENTIFIER_FIELD, REPLICATION_CONFIG_ARN_FIELD, SOURCE_ENDPOINT_ARN_FIELD,
            TARGET_ENDPOINT_ARN_FIELD, REPLICATION_TYPE_FIELD, STATUS_FIELD, PROVISION_DATA_FIELD,
            PREMIGRATION_ASSESSMENT_STATUSES_FIELD, STOP_REASON_FIELD, FAILURE_MESSAGES_FIELD, REPLICATION_STATS_FIELD,
            START_REPLICATION_TYPE_FIELD, CDC_START_TIME_FIELD, CDC_START_POSITION_FIELD, CDC_STOP_POSITION_FIELD,
            RECOVERY_CHECKPOINT_FIELD, REPLICATION_CREATE_TIME_FIELD, REPLICATION_UPDATE_TIME_FIELD,
            REPLICATION_LAST_STOP_TIME_FIELD, REPLICATION_DEPROVISION_TIME_FIELD, IS_READ_ONLY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String replicationConfigIdentifier;

    private final String replicationConfigArn;

    private final String sourceEndpointArn;

    private final String targetEndpointArn;

    private final String replicationType;

    private final String status;

    private final ProvisionData provisionData;

    private final List<PremigrationAssessmentStatus> premigrationAssessmentStatuses;

    private final String stopReason;

    private final List<String> failureMessages;

    private final ReplicationStats replicationStats;

    private final String startReplicationType;

    private final Instant cdcStartTime;

    private final String cdcStartPosition;

    private final String cdcStopPosition;

    private final String recoveryCheckpoint;

    private final Instant replicationCreateTime;

    private final Instant replicationUpdateTime;

    private final Instant replicationLastStopTime;

    private final Instant replicationDeprovisionTime;

    private final Boolean isReadOnly;

    private Replication(BuilderImpl builder) {
        this.replicationConfigIdentifier = builder.replicationConfigIdentifier;
        this.replicationConfigArn = builder.replicationConfigArn;
        this.sourceEndpointArn = builder.sourceEndpointArn;
        this.targetEndpointArn = builder.targetEndpointArn;
        this.replicationType = builder.replicationType;
        this.status = builder.status;
        this.provisionData = builder.provisionData;
        this.premigrationAssessmentStatuses = builder.premigrationAssessmentStatuses;
        this.stopReason = builder.stopReason;
        this.failureMessages = builder.failureMessages;
        this.replicationStats = builder.replicationStats;
        this.startReplicationType = builder.startReplicationType;
        this.cdcStartTime = builder.cdcStartTime;
        this.cdcStartPosition = builder.cdcStartPosition;
        this.cdcStopPosition = builder.cdcStopPosition;
        this.recoveryCheckpoint = builder.recoveryCheckpoint;
        this.replicationCreateTime = builder.replicationCreateTime;
        this.replicationUpdateTime = builder.replicationUpdateTime;
        this.replicationLastStopTime = builder.replicationLastStopTime;
        this.replicationDeprovisionTime = builder.replicationDeprovisionTime;
        this.isReadOnly = builder.isReadOnly;
    }

    /**
     * <p>
     * The identifier for the <code>ReplicationConfig</code> associated with the replication.
     * </p>
     * 
     * @return The identifier for the <code>ReplicationConfig</code> associated with the replication.
     */
    public final String replicationConfigIdentifier() {
        return replicationConfigIdentifier;
    }

    /**
     * <p>
     * The Amazon Resource Name for the <code>ReplicationConfig</code> associated with the replication.
     * </p>
     * 
     * @return The Amazon Resource Name for the <code>ReplicationConfig</code> associated with the replication.
     */
    public final String replicationConfigArn() {
        return replicationConfigArn;
    }

    /**
     * <p>
     * The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its data
     * source.
     * </p>
     * 
     * @return The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its
     *         data source.
     */
    public final String sourceEndpointArn() {
        return sourceEndpointArn;
    }

    /**
     * <p>
     * The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its data
     * target.
     * </p>
     * 
     * @return The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its
     *         data target.
     */
    public final String targetEndpointArn() {
        return targetEndpointArn;
    }

    /**
     * <p>
     * The type of the serverless replication.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replicationType}
     * will return {@link MigrationTypeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #replicationTypeAsString}.
     * </p>
     * 
     * @return The type of the serverless replication.
     * @see MigrationTypeValue
     */
    public final MigrationTypeValue replicationType() {
        return MigrationTypeValue.fromValue(replicationType);
    }

    /**
     * <p>
     * The type of the serverless replication.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replicationType}
     * will return {@link MigrationTypeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #replicationTypeAsString}.
     * </p>
     * 
     * @return The type of the serverless replication.
     * @see MigrationTypeValue
     */
    public final String replicationTypeAsString() {
        return replicationType;
    }

    /**
     * <p>
     * The current status of the serverless replication.
     * </p>
     * 
     * @return The current status of the serverless replication.
     */
    public final String status() {
        return status;
    }

    /**
     * <p>
     * Information about provisioning resources for an DMS serverless replication.
     * </p>
     * 
     * @return Information about provisioning resources for an DMS serverless replication.
     */
    public final ProvisionData provisionData() {
        return provisionData;
    }

    /**
     * For responses, this returns true if the service returned a value for the PremigrationAssessmentStatuses property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasPremigrationAssessmentStatuses() {
        return premigrationAssessmentStatuses != null && !(premigrationAssessmentStatuses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The status output of premigration assessment in describe-replications.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasPremigrationAssessmentStatuses} method.
     * </p>
     * 
     * @return The status output of premigration assessment in describe-replications.
     */
    public final List<PremigrationAssessmentStatus> premigrationAssessmentStatuses() {
        return premigrationAssessmentStatuses;
    }

    /**
     * <p>
     * The reason the replication task was stopped. This response parameter can return one of the following values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>"Stop Reason NORMAL"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason RECOVERABLE_ERROR"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason FATAL_ERROR"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason FULL_LOAD_ONLY_FINISHED"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_AFTER_FULL_LOAD"</code> – Full load completed, with cached changes not applied
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_AFTER_CACHED_EVENTS"</code> – Full load completed, with cached changes applied
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason EXPRESS_LICENSE_LIMITS_REACHED"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_AFTER_DDL_APPLY"</code> – User-defined stop task after DDL applied
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_DUE_TO_LOW_MEMORY"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_DUE_TO_LOW_DISK"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_AT_SERVER_TIME"</code> – User-defined server time for stopping task
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason STOPPED_AT_COMMIT_TIME"</code> – User-defined commit time for stopping task
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason RECONFIGURATION_RESTART"</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>"Stop Reason RECYCLE_TASK"</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return The reason the replication task was stopped. This response parameter can return one of the following
     *         values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>"Stop Reason NORMAL"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason RECOVERABLE_ERROR"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason FATAL_ERROR"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason FULL_LOAD_ONLY_FINISHED"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_AFTER_FULL_LOAD"</code> – Full load completed, with cached changes not applied
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_AFTER_CACHED_EVENTS"</code> – Full load completed, with cached changes applied
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason EXPRESS_LICENSE_LIMITS_REACHED"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_AFTER_DDL_APPLY"</code> – User-defined stop task after DDL applied
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_DUE_TO_LOW_MEMORY"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_DUE_TO_LOW_DISK"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_AT_SERVER_TIME"</code> – User-defined server time for stopping task
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason STOPPED_AT_COMMIT_TIME"</code> – User-defined commit time for stopping task
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason RECONFIGURATION_RESTART"</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>"Stop Reason RECYCLE_TASK"</code>
     *         </p>
     *         </li>
     */
    public final String stopReason() {
        return stopReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the FailureMessages property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasFailureMessages() {
        return failureMessages != null && !(failureMessages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Error and other information about why a serverless replication failed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFailureMessages} method.
     * </p>
     * 
     * @return Error and other information about why a serverless replication failed.
     */
    public final List<String> failureMessages() {
        return failureMessages;
    }

    /**
     * <p>
     * This object provides a collection of statistics about a serverless replication.
     * </p>
     * 
     * @return This object provides a collection of statistics about a serverless replication.
     */
    public final ReplicationStats replicationStats() {
        return replicationStats;
    }

    /**
     * <p>
     * The type of replication to start.
     * </p>
     * 
     * @return The type of replication to start.
     */
    public final String startReplicationType() {
        return startReplicationType;
    }

    /**
     * <p>
     * Indicates the start time for a change data capture (CDC) operation. Use either <code>CdcStartTime</code> or
     * <code>CdcStartPosition</code> to specify when you want a CDC operation to start. Specifying both values results
     * in an error.
     * </p>
     * 
     * @return Indicates the start time for a change data capture (CDC) operation. Use either <code>CdcStartTime</code>
     *         or <code>CdcStartPosition</code> to specify when you want a CDC operation to start. Specifying both
     *         values results in an error.
     */
    public final Instant cdcStartTime() {
        return cdcStartTime;
    }

    /**
     * <p>
     * Indicates the start time for a change data capture (CDC) operation. Use either <code>CdcStartTime</code> or
     * <code>CdcStartPosition</code> to specify when you want a CDC operation to start. Specifying both values results
     * in an error.
     * </p>
     * 
     * @return Indicates the start time for a change data capture (CDC) operation. Use either <code>CdcStartTime</code>
     *         or <code>CdcStartPosition</code> to specify when you want a CDC operation to start. Specifying both
     *         values results in an error.
     */
    public final String cdcStartPosition() {
        return cdcStartPosition;
    }

    /**
     * <p>
     * Indicates when you want a change data capture (CDC) operation to stop. The value can be either server time or
     * commit time.
     * </p>
     * 
     * @return Indicates when you want a change data capture (CDC) operation to stop. The value can be either server
     *         time or commit time.
     */
    public final String cdcStopPosition() {
        return cdcStopPosition;
    }

    /**
     * <p>
     * Indicates the last checkpoint that occurred during a change data capture (CDC) operation. You can provide this
     * value to the <code>CdcStartPosition</code> parameter to start a CDC operation that begins at that checkpoint.
     * </p>
     * 
     * @return Indicates the last checkpoint that occurred during a change data capture (CDC) operation. You can provide
     *         this value to the <code>CdcStartPosition</code> parameter to start a CDC operation that begins at that
     *         checkpoint.
     */
    public final String recoveryCheckpoint() {
        return recoveryCheckpoint;
    }

    /**
     * <p>
     * The time the serverless replication was created.
     * </p>
     * 
     * @return The time the serverless replication was created.
     */
    public final Instant replicationCreateTime() {
        return replicationCreateTime;
    }

    /**
     * <p>
     * The time the serverless replication was updated.
     * </p>
     * 
     * @return The time the serverless replication was updated.
     */
    public final Instant replicationUpdateTime() {
        return replicationUpdateTime;
    }

    /**
     * <p>
     * The timestamp when replication was last stopped.
     * </p>
     * 
     * @return The timestamp when replication was last stopped.
     */
    public final Instant replicationLastStopTime() {
        return replicationLastStopTime;
    }

    /**
     * <p>
     * The timestamp when DMS will deprovision the replication.
     * </p>
     * 
     * @return The timestamp when DMS will deprovision the replication.
     */
    public final Instant replicationDeprovisionTime() {
        return replicationDeprovisionTime;
    }

    /**
     * <p>
     * Indicates whether the serverless replication is read-only. When set to <code>true</code>, this replication is
     * managed by DMS as part of a zero-ETL integration and cannot be modified or deleted directly. You can only modify
     * or delete read-only replications through their associated zero-ETL integration.
     * </p>
     * 
     * @return Indicates whether the serverless replication is read-only. When set to <code>true</code>, this
     *         replication is managed by DMS as part of a zero-ETL integration and cannot be modified or deleted
     *         directly. You can only modify or delete read-only replications through their associated zero-ETL
     *         integration.
     */
    public final Boolean isReadOnly() {
        return isReadOnly;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(replicationConfigIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(replicationConfigArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceEndpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(targetEndpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(replicationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(provisionData());
        hashCode = 31 * hashCode
                + Objects.hashCode(hasPremigrationAssessmentStatuses() ? premigrationAssessmentStatuses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stopReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasFailureMessages() ? failureMessages() : null);
        hashCode = 31 * hashCode + Objects.hashCode(replicationStats());
        hashCode = 31 * hashCode + Objects.hashCode(startReplicationType());
        hashCode = 31 * hashCode + Objects.hashCode(cdcStartTime());
        hashCode = 31 * hashCode + Objects.hashCode(cdcStartPosition());
        hashCode = 31 * hashCode + Objects.hashCode(cdcStopPosition());
        hashCode = 31 * hashCode + Objects.hashCode(recoveryCheckpoint());
        hashCode = 31 * hashCode + Objects.hashCode(replicationCreateTime());
        hashCode = 31 * hashCode + Objects.hashCode(replicationUpdateTime());
        hashCode = 31 * hashCode + Objects.hashCode(replicationLastStopTime());
        hashCode = 31 * hashCode + Objects.hashCode(replicationDeprovisionTime());
        hashCode = 31 * hashCode + Objects.hashCode(isReadOnly());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Replication)) {
            return false;
        }
        Replication other = (Replication) obj;
        return Objects.equals(replicationConfigIdentifier(), other.replicationConfigIdentifier())
                && Objects.equals(replicationConfigArn(), other.replicationConfigArn())
                && Objects.equals(sourceEndpointArn(), other.sourceEndpointArn())
                && Objects.equals(targetEndpointArn(), other.targetEndpointArn())
                && Objects.equals(replicationTypeAsString(), other.replicationTypeAsString())
                && Objects.equals(status(), other.status()) && Objects.equals(provisionData(), other.provisionData())
                && hasPremigrationAssessmentStatuses() == other.hasPremigrationAssessmentStatuses()
                && Objects.equals(premigrationAssessmentStatuses(), other.premigrationAssessmentStatuses())
                && Objects.equals(stopReason(), other.stopReason()) && hasFailureMessages() == other.hasFailureMessages()
                && Objects.equals(failureMessages(), other.failureMessages())
                && Objects.equals(replicationStats(), other.replicationStats())
                && Objects.equals(startReplicationType(), other.startReplicationType())
                && Objects.equals(cdcStartTime(), other.cdcStartTime())
                && Objects.equals(cdcStartPosition(), other.cdcStartPosition())
                && Objects.equals(cdcStopPosition(), other.cdcStopPosition())
                && Objects.equals(recoveryCheckpoint(), other.recoveryCheckpoint())
                && Objects.equals(replicationCreateTime(), other.replicationCreateTime())
                && Objects.equals(replicationUpdateTime(), other.replicationUpdateTime())
                && Objects.equals(replicationLastStopTime(), other.replicationLastStopTime())
                && Objects.equals(replicationDeprovisionTime(), other.replicationDeprovisionTime())
                && Objects.equals(isReadOnly(), other.isReadOnly());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("Replication")
                .add("ReplicationConfigIdentifier", replicationConfigIdentifier())
                .add("ReplicationConfigArn", replicationConfigArn())
                .add("SourceEndpointArn", sourceEndpointArn())
                .add("TargetEndpointArn", targetEndpointArn())
                .add("ReplicationType", replicationTypeAsString())
                .add("Status", status())
                .add("ProvisionData", provisionData())
                .add("PremigrationAssessmentStatuses",
                        hasPremigrationAssessmentStatuses() ? premigrationAssessmentStatuses() : null)
                .add("StopReason", stopReason()).add("FailureMessages", hasFailureMessages() ? failureMessages() : null)
                .add("ReplicationStats", replicationStats()).add("StartReplicationType", startReplicationType())
                .add("CdcStartTime", cdcStartTime()).add("CdcStartPosition", cdcStartPosition())
                .add("CdcStopPosition", cdcStopPosition()).add("RecoveryCheckpoint", recoveryCheckpoint())
                .add("ReplicationCreateTime", replicationCreateTime()).add("ReplicationUpdateTime", replicationUpdateTime())
                .add("ReplicationLastStopTime", replicationLastStopTime())
                .add("ReplicationDeprovisionTime", replicationDeprovisionTime()).add("IsReadOnly", isReadOnly()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ReplicationConfigIdentifier":
            return Optional.ofNullable(clazz.cast(replicationConfigIdentifier()));
        case "ReplicationConfigArn":
            return Optional.ofNullable(clazz.cast(replicationConfigArn()));
        case "SourceEndpointArn":
            return Optional.ofNullable(clazz.cast(sourceEndpointArn()));
        case "TargetEndpointArn":
            return Optional.ofNullable(clazz.cast(targetEndpointArn()));
        case "ReplicationType":
            return Optional.ofNullable(clazz.cast(replicationTypeAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "ProvisionData":
            return Optional.ofNullable(clazz.cast(provisionData()));
        case "PremigrationAssessmentStatuses":
            return Optional.ofNullable(clazz.cast(premigrationAssessmentStatuses()));
        case "StopReason":
            return Optional.ofNullable(clazz.cast(stopReason()));
        case "FailureMessages":
            return Optional.ofNullable(clazz.cast(failureMessages()));
        case "ReplicationStats":
            return Optional.ofNullable(clazz.cast(replicationStats()));
        case "StartReplicationType":
            return Optional.ofNullable(clazz.cast(startReplicationType()));
        case "CdcStartTime":
            return Optional.ofNullable(clazz.cast(cdcStartTime()));
        case "CdcStartPosition":
            return Optional.ofNullable(clazz.cast(cdcStartPosition()));
        case "CdcStopPosition":
            return Optional.ofNullable(clazz.cast(cdcStopPosition()));
        case "RecoveryCheckpoint":
            return Optional.ofNullable(clazz.cast(recoveryCheckpoint()));
        case "ReplicationCreateTime":
            return Optional.ofNullable(clazz.cast(replicationCreateTime()));
        case "ReplicationUpdateTime":
            return Optional.ofNullable(clazz.cast(replicationUpdateTime()));
        case "ReplicationLastStopTime":
            return Optional.ofNullable(clazz.cast(replicationLastStopTime()));
        case "ReplicationDeprovisionTime":
            return Optional.ofNullable(clazz.cast(replicationDeprovisionTime()));
        case "IsReadOnly":
            return Optional.ofNullable(clazz.cast(isReadOnly()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ReplicationConfigIdentifier", REPLICATION_CONFIG_IDENTIFIER_FIELD);
        map.put("ReplicationConfigArn", REPLICATION_CONFIG_ARN_FIELD);
        map.put("SourceEndpointArn", SOURCE_ENDPOINT_ARN_FIELD);
        map.put("TargetEndpointArn", TARGET_ENDPOINT_ARN_FIELD);
        map.put("ReplicationType", REPLICATION_TYPE_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("ProvisionData", PROVISION_DATA_FIELD);
        map.put("PremigrationAssessmentStatuses", PREMIGRATION_ASSESSMENT_STATUSES_FIELD);
        map.put("StopReason", STOP_REASON_FIELD);
        map.put("FailureMessages", FAILURE_MESSAGES_FIELD);
        map.put("ReplicationStats", REPLICATION_STATS_FIELD);
        map.put("StartReplicationType", START_REPLICATION_TYPE_FIELD);
        map.put("CdcStartTime", CDC_START_TIME_FIELD);
        map.put("CdcStartPosition", CDC_START_POSITION_FIELD);
        map.put("CdcStopPosition", CDC_STOP_POSITION_FIELD);
        map.put("RecoveryCheckpoint", RECOVERY_CHECKPOINT_FIELD);
        map.put("ReplicationCreateTime", REPLICATION_CREATE_TIME_FIELD);
        map.put("ReplicationUpdateTime", REPLICATION_UPDATE_TIME_FIELD);
        map.put("ReplicationLastStopTime", REPLICATION_LAST_STOP_TIME_FIELD);
        map.put("ReplicationDeprovisionTime", REPLICATION_DEPROVISION_TIME_FIELD);
        map.put("IsReadOnly", IS_READ_ONLY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Replication, T> g) {
        return obj -> g.apply((Replication) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Replication> {
        /**
         * <p>
         * The identifier for the <code>ReplicationConfig</code> associated with the replication.
         * </p>
         * 
         * @param replicationConfigIdentifier
         *        The identifier for the <code>ReplicationConfig</code> associated with the replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationConfigIdentifier(String replicationConfigIdentifier);

        /**
         * <p>
         * The Amazon Resource Name for the <code>ReplicationConfig</code> associated with the replication.
         * </p>
         * 
         * @param replicationConfigArn
         *        The Amazon Resource Name for the <code>ReplicationConfig</code> associated with the replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationConfigArn(String replicationConfigArn);

        /**
         * <p>
         * The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its data
         * source.
         * </p>
         * 
         * @param sourceEndpointArn
         *        The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its
         *        data source.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceEndpointArn(String sourceEndpointArn);

        /**
         * <p>
         * The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its data
         * target.
         * </p>
         * 
         * @param targetEndpointArn
         *        The Amazon Resource Name for an existing <code>Endpoint</code> the serverless replication uses for its
         *        data target.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder targetEndpointArn(String targetEndpointArn);

        /**
         * <p>
         * The type of the serverless replication.
         * </p>
         * 
         * @param replicationType
         *        The type of the serverless replication.
         * @see MigrationTypeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationTypeValue
         */
        Builder replicationType(String replicationType);

        /**
         * <p>
         * The type of the serverless replication.
         * </p>
         * 
         * @param replicationType
         *        The type of the serverless replication.
         * @see MigrationTypeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MigrationTypeValue
         */
        Builder replicationType(MigrationTypeValue replicationType);

        /**
         * <p>
         * The current status of the serverless replication.
         * </p>
         * 
         * @param status
         *        The current status of the serverless replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * Information about provisioning resources for an DMS serverless replication.
         * </p>
         * 
         * @param provisionData
         *        Information about provisioning resources for an DMS serverless replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionData(ProvisionData provisionData);

        /**
         * <p>
         * Information about provisioning resources for an DMS serverless replication.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProvisionData.Builder} avoiding the need
         * to create one manually via {@link ProvisionData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProvisionData.Builder#build()} is called immediately and its
         * result is passed to {@link #provisionData(ProvisionData)}.
         * 
         * @param provisionData
         *        a consumer that will call methods on {@link ProvisionData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionData(ProvisionData)
         */
        default Builder provisionData(Consumer<ProvisionData.Builder> provisionData) {
            return provisionData(ProvisionData.builder().applyMutation(provisionData).build());
        }

        /**
         * <p>
         * The status output of premigration assessment in describe-replications.
         * </p>
         * 
         * @param premigrationAssessmentStatuses
         *        The status output of premigration assessment in describe-replications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder premigrationAssessmentStatuses(Collection<PremigrationAssessmentStatus> premigrationAssessmentStatuses);

        /**
         * <p>
         * The status output of premigration assessment in describe-replications.
         * </p>
         * 
         * @param premigrationAssessmentStatuses
         *        The status output of premigration assessment in describe-replications.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder premigrationAssessmentStatuses(PremigrationAssessmentStatus... premigrationAssessmentStatuses);

        /**
         * <p>
         * The status output of premigration assessment in describe-replications.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.databasemigration.model.PremigrationAssessmentStatus.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.databasemigration.model.PremigrationAssessmentStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.databasemigration.model.PremigrationAssessmentStatus.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #premigrationAssessmentStatuses(List<PremigrationAssessmentStatus>)}.
         * 
         * @param premigrationAssessmentStatuses
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.databasemigration.model.PremigrationAssessmentStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #premigrationAssessmentStatuses(java.util.Collection<PremigrationAssessmentStatus>)
         */
        Builder premigrationAssessmentStatuses(Consumer<PremigrationAssessmentStatus.Builder>... premigrationAssessmentStatuses);

        /**
         * <p>
         * The reason the replication task was stopped. This response parameter can return one of the following values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>"Stop Reason NORMAL"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason RECOVERABLE_ERROR"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason FATAL_ERROR"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason FULL_LOAD_ONLY_FINISHED"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_AFTER_FULL_LOAD"</code> – Full load completed, with cached changes not applied
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_AFTER_CACHED_EVENTS"</code> – Full load completed, with cached changes applied
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason EXPRESS_LICENSE_LIMITS_REACHED"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_AFTER_DDL_APPLY"</code> – User-defined stop task after DDL applied
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_DUE_TO_LOW_MEMORY"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_DUE_TO_LOW_DISK"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_AT_SERVER_TIME"</code> – User-defined server time for stopping task
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason STOPPED_AT_COMMIT_TIME"</code> – User-defined commit time for stopping task
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason RECONFIGURATION_RESTART"</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>"Stop Reason RECYCLE_TASK"</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param stopReason
         *        The reason the replication task was stopped. This response parameter can return one of the following
         *        values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>"Stop Reason NORMAL"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason RECOVERABLE_ERROR"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason FATAL_ERROR"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason FULL_LOAD_ONLY_FINISHED"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_AFTER_FULL_LOAD"</code> – Full load completed, with cached changes not
         *        applied
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_AFTER_CACHED_EVENTS"</code> – Full load completed, with cached changes
         *        applied
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason EXPRESS_LICENSE_LIMITS_REACHED"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_AFTER_DDL_APPLY"</code> – User-defined stop task after DDL applied
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_DUE_TO_LOW_MEMORY"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_DUE_TO_LOW_DISK"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_AT_SERVER_TIME"</code> – User-defined server time for stopping task
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason STOPPED_AT_COMMIT_TIME"</code> – User-defined commit time for stopping task
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason RECONFIGURATION_RESTART"</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>"Stop Reason RECYCLE_TASK"</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stopReason(String stopReason);

        /**
         * <p>
         * Error and other information about why a serverless replication failed.
         * </p>
         * 
         * @param failureMessages
         *        Error and other information about why a serverless replication failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessages(Collection<String> failureMessages);

        /**
         * <p>
         * Error and other information about why a serverless replication failed.
         * </p>
         * 
         * @param failureMessages
         *        Error and other information about why a serverless replication failed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessages(String... failureMessages);

        /**
         * <p>
         * This object provides a collection of statistics about a serverless replication.
         * </p>
         * 
         * @param replicationStats
         *        This object provides a collection of statistics about a serverless replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationStats(ReplicationStats replicationStats);

        /**
         * <p>
         * This object provides a collection of statistics about a serverless replication.
         * </p>
         * This is a convenience method that creates an instance of the {@link ReplicationStats.Builder} avoiding the
         * need to create one manually via {@link ReplicationStats#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ReplicationStats.Builder#build()} is called immediately and its
         * result is passed to {@link #replicationStats(ReplicationStats)}.
         * 
         * @param replicationStats
         *        a consumer that will call methods on {@link ReplicationStats.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #replicationStats(ReplicationStats)
         */
        default Builder replicationStats(Consumer<ReplicationStats.Builder> replicationStats) {
            return replicationStats(ReplicationStats.builder().applyMutation(replicationStats).build());
        }

        /**
         * <p>
         * The type of replication to start.
         * </p>
         * 
         * @param startReplicationType
         *        The type of replication to start.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startReplicationType(String startReplicationType);

        /**
         * <p>
         * Indicates the start time for a change data capture (CDC) operation. Use either <code>CdcStartTime</code> or
         * <code>CdcStartPosition</code> to specify when you want a CDC operation to start. Specifying both values
         * results in an error.
         * </p>
         * 
         * @param cdcStartTime
         *        Indicates the start time for a change data capture (CDC) operation. Use either
         *        <code>CdcStartTime</code> or <code>CdcStartPosition</code> to specify when you want a CDC operation to
         *        start. Specifying both values results in an error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cdcStartTime(Instant cdcStartTime);

        /**
         * <p>
         * Indicates the start time for a change data capture (CDC) operation. Use either <code>CdcStartTime</code> or
         * <code>CdcStartPosition</code> to specify when you want a CDC operation to start. Specifying both values
         * results in an error.
         * </p>
         * 
         * @param cdcStartPosition
         *        Indicates the start time for a change data capture (CDC) operation. Use either
         *        <code>CdcStartTime</code> or <code>CdcStartPosition</code> to specify when you want a CDC operation to
         *        start. Specifying both values results in an error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cdcStartPosition(String cdcStartPosition);

        /**
         * <p>
         * Indicates when you want a change data capture (CDC) operation to stop. The value can be either server time or
         * commit time.
         * </p>
         * 
         * @param cdcStopPosition
         *        Indicates when you want a change data capture (CDC) operation to stop. The value can be either server
         *        time or commit time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cdcStopPosition(String cdcStopPosition);

        /**
         * <p>
         * Indicates the last checkpoint that occurred during a change data capture (CDC) operation. You can provide
         * this value to the <code>CdcStartPosition</code> parameter to start a CDC operation that begins at that
         * checkpoint.
         * </p>
         * 
         * @param recoveryCheckpoint
         *        Indicates the last checkpoint that occurred during a change data capture (CDC) operation. You can
         *        provide this value to the <code>CdcStartPosition</code> parameter to start a CDC operation that begins
         *        at that checkpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryCheckpoint(String recoveryCheckpoint);

        /**
         * <p>
         * The time the serverless replication was created.
         * </p>
         * 
         * @param replicationCreateTime
         *        The time the serverless replication was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationCreateTime(Instant replicationCreateTime);

        /**
         * <p>
         * The time the serverless replication was updated.
         * </p>
         * 
         * @param replicationUpdateTime
         *        The time the serverless replication was updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationUpdateTime(Instant replicationUpdateTime);

        /**
         * <p>
         * The timestamp when replication was last stopped.
         * </p>
         * 
         * @param replicationLastStopTime
         *        The timestamp when replication was last stopped.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationLastStopTime(Instant replicationLastStopTime);

        /**
         * <p>
         * The timestamp when DMS will deprovision the replication.
         * </p>
         * 
         * @param replicationDeprovisionTime
         *        The timestamp when DMS will deprovision the replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicationDeprovisionTime(Instant replicationDeprovisionTime);

        /**
         * <p>
         * Indicates whether the serverless replication is read-only. When set to <code>true</code>, this replication is
         * managed by DMS as part of a zero-ETL integration and cannot be modified or deleted directly. You can only
         * modify or delete read-only replications through their associated zero-ETL integration.
         * </p>
         * 
         * @param isReadOnly
         *        Indicates whether the serverless replication is read-only. When set to <code>true</code>, this
         *        replication is managed by DMS as part of a zero-ETL integration and cannot be modified or deleted
         *        directly. You can only modify or delete read-only replications through their associated zero-ETL
         *        integration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isReadOnly(Boolean isReadOnly);
    }

    static final class BuilderImpl implements Builder {
        private String replicationConfigIdentifier;

        private String replicationConfigArn;

        private String sourceEndpointArn;

        private String targetEndpointArn;

        private String replicationType;

        private String status;

        private ProvisionData provisionData;

        private List<PremigrationAssessmentStatus> premigrationAssessmentStatuses = DefaultSdkAutoConstructList.getInstance();

        private String stopReason;

        private List<String> failureMessages = DefaultSdkAutoConstructList.getInstance();

        private ReplicationStats replicationStats;

        private String startReplicationType;

        private Instant cdcStartTime;

        private String cdcStartPosition;

        private String cdcStopPosition;

        private String recoveryCheckpoint;

        private Instant replicationCreateTime;

        private Instant replicationUpdateTime;

        private Instant replicationLastStopTime;

        private Instant replicationDeprovisionTime;

        private Boolean isReadOnly;

        private BuilderImpl() {
        }

        private BuilderImpl(Replication model) {
            replicationConfigIdentifier(model.replicationConfigIdentifier);
            replicationConfigArn(model.replicationConfigArn);
            sourceEndpointArn(model.sourceEndpointArn);
            targetEndpointArn(model.targetEndpointArn);
            replicationType(model.replicationType);
            status(model.status);
            provisionData(model.provisionData);
            premigrationAssessmentStatuses(model.premigrationAssessmentStatuses);
            stopReason(model.stopReason);
            failureMessages(model.failureMessages);
            replicationStats(model.replicationStats);
            startReplicationType(model.startReplicationType);
            cdcStartTime(model.cdcStartTime);
            cdcStartPosition(model.cdcStartPosition);
            cdcStopPosition(model.cdcStopPosition);
            recoveryCheckpoint(model.recoveryCheckpoint);
            replicationCreateTime(model.replicationCreateTime);
            replicationUpdateTime(model.replicationUpdateTime);
            replicationLastStopTime(model.replicationLastStopTime);
            replicationDeprovisionTime(model.replicationDeprovisionTime);
            isReadOnly(model.isReadOnly);
        }

        public final String getReplicationConfigIdentifier() {
            return replicationConfigIdentifier;
        }

        public final void setReplicationConfigIdentifier(String replicationConfigIdentifier) {
            this.replicationConfigIdentifier = replicationConfigIdentifier;
        }

        @Override
        public final Builder replicationConfigIdentifier(String replicationConfigIdentifier) {
            this.replicationConfigIdentifier = replicationConfigIdentifier;
            return this;
        }

        public final String getReplicationConfigArn() {
            return replicationConfigArn;
        }

        public final void setReplicationConfigArn(String replicationConfigArn) {
            this.replicationConfigArn = replicationConfigArn;
        }

        @Override
        public final Builder replicationConfigArn(String replicationConfigArn) {
            this.replicationConfigArn = replicationConfigArn;
            return this;
        }

        public final String getSourceEndpointArn() {
            return sourceEndpointArn;
        }

        public final void setSourceEndpointArn(String sourceEndpointArn) {
            this.sourceEndpointArn = sourceEndpointArn;
        }

        @Override
        public final Builder sourceEndpointArn(String sourceEndpointArn) {
            this.sourceEndpointArn = sourceEndpointArn;
            return this;
        }

        public final String getTargetEndpointArn() {
            return targetEndpointArn;
        }

        public final void setTargetEndpointArn(String targetEndpointArn) {
            this.targetEndpointArn = targetEndpointArn;
        }

        @Override
        public final Builder targetEndpointArn(String targetEndpointArn) {
            this.targetEndpointArn = targetEndpointArn;
            return this;
        }

        public final String getReplicationType() {
            return replicationType;
        }

        public final void setReplicationType(String replicationType) {
            this.replicationType = replicationType;
        }

        @Override
        public final Builder replicationType(String replicationType) {
            this.replicationType = replicationType;
            return this;
        }

        @Override
        public final Builder replicationType(MigrationTypeValue replicationType) {
            this.replicationType(replicationType == null ? null : replicationType.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final ProvisionData.Builder getProvisionData() {
            return provisionData != null ? provisionData.toBuilder() : null;
        }

        public final void setProvisionData(ProvisionData.BuilderImpl provisionData) {
            this.provisionData = provisionData != null ? provisionData.build() : null;
        }

        @Override
        public final Builder provisionData(ProvisionData provisionData) {
            this.provisionData = provisionData;
            return this;
        }

        public final List<PremigrationAssessmentStatus.Builder> getPremigrationAssessmentStatuses() {
            List<PremigrationAssessmentStatus.Builder> result = PremigrationAssessmentStatusListCopier
                    .copyToBuilder(this.premigrationAssessmentStatuses);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setPremigrationAssessmentStatuses(
                Collection<PremigrationAssessmentStatus.BuilderImpl> premigrationAssessmentStatuses) {
            this.premigrationAssessmentStatuses = PremigrationAssessmentStatusListCopier
                    .copyFromBuilder(premigrationAssessmentStatuses);
        }

        @Override
        public final Builder premigrationAssessmentStatuses(
                Collection<PremigrationAssessmentStatus> premigrationAssessmentStatuses) {
            this.premigrationAssessmentStatuses = PremigrationAssessmentStatusListCopier.copy(premigrationAssessmentStatuses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder premigrationAssessmentStatuses(PremigrationAssessmentStatus... premigrationAssessmentStatuses) {
            premigrationAssessmentStatuses(Arrays.asList(premigrationAssessmentStatuses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder premigrationAssessmentStatuses(
                Consumer<PremigrationAssessmentStatus.Builder>... premigrationAssessmentStatuses) {
            premigrationAssessmentStatuses(Stream.of(premigrationAssessmentStatuses)
                    .map(c -> PremigrationAssessmentStatus.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStopReason() {
            return stopReason;
        }

        public final void setStopReason(String stopReason) {
            this.stopReason = stopReason;
        }

        @Override
        public final Builder stopReason(String stopReason) {
            this.stopReason = stopReason;
            return this;
        }

        public final Collection<String> getFailureMessages() {
            if (failureMessages instanceof SdkAutoConstructList) {
                return null;
            }
            return failureMessages;
        }

        public final void setFailureMessages(Collection<String> failureMessages) {
            this.failureMessages = StringListCopier.copy(failureMessages);
        }

        @Override
        public final Builder failureMessages(Collection<String> failureMessages) {
            this.failureMessages = StringListCopier.copy(failureMessages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder failureMessages(String... failureMessages) {
            failureMessages(Arrays.asList(failureMessages));
            return this;
        }

        public final ReplicationStats.Builder getReplicationStats() {
            return replicationStats != null ? replicationStats.toBuilder() : null;
        }

        public final void setReplicationStats(ReplicationStats.BuilderImpl replicationStats) {
            this.replicationStats = replicationStats != null ? replicationStats.build() : null;
        }

        @Override
        public final Builder replicationStats(ReplicationStats replicationStats) {
            this.replicationStats = replicationStats;
            return this;
        }

        public final String getStartReplicationType() {
            return startReplicationType;
        }

        public final void setStartReplicationType(String startReplicationType) {
            this.startReplicationType = startReplicationType;
        }

        @Override
        public final Builder startReplicationType(String startReplicationType) {
            this.startReplicationType = startReplicationType;
            return this;
        }

        public final Instant getCdcStartTime() {
            return cdcStartTime;
        }

        public final void setCdcStartTime(Instant cdcStartTime) {
            this.cdcStartTime = cdcStartTime;
        }

        @Override
        public final Builder cdcStartTime(Instant cdcStartTime) {
            this.cdcStartTime = cdcStartTime;
            return this;
        }

        public final String getCdcStartPosition() {
            return cdcStartPosition;
        }

        public final void setCdcStartPosition(String cdcStartPosition) {
            this.cdcStartPosition = cdcStartPosition;
        }

        @Override
        public final Builder cdcStartPosition(String cdcStartPosition) {
            this.cdcStartPosition = cdcStartPosition;
            return this;
        }

        public final String getCdcStopPosition() {
            return cdcStopPosition;
        }

        public final void setCdcStopPosition(String cdcStopPosition) {
            this.cdcStopPosition = cdcStopPosition;
        }

        @Override
        public final Builder cdcStopPosition(String cdcStopPosition) {
            this.cdcStopPosition = cdcStopPosition;
            return this;
        }

        public final String getRecoveryCheckpoint() {
            return recoveryCheckpoint;
        }

        public final void setRecoveryCheckpoint(String recoveryCheckpoint) {
            this.recoveryCheckpoint = recoveryCheckpoint;
        }

        @Override
        public final Builder recoveryCheckpoint(String recoveryCheckpoint) {
            this.recoveryCheckpoint = recoveryCheckpoint;
            return this;
        }

        public final Instant getReplicationCreateTime() {
            return replicationCreateTime;
        }

        public final void setReplicationCreateTime(Instant replicationCreateTime) {
            this.replicationCreateTime = replicationCreateTime;
        }

        @Override
        public final Builder replicationCreateTime(Instant replicationCreateTime) {
            this.replicationCreateTime = replicationCreateTime;
            return this;
        }

        public final Instant getReplicationUpdateTime() {
            return replicationUpdateTime;
        }

        public final void setReplicationUpdateTime(Instant replicationUpdateTime) {
            this.replicationUpdateTime = replicationUpdateTime;
        }

        @Override
        public final Builder replicationUpdateTime(Instant replicationUpdateTime) {
            this.replicationUpdateTime = replicationUpdateTime;
            return this;
        }

        public final Instant getReplicationLastStopTime() {
            return replicationLastStopTime;
        }

        public final void setReplicationLastStopTime(Instant replicationLastStopTime) {
            this.replicationLastStopTime = replicationLastStopTime;
        }

        @Override
        public final Builder replicationLastStopTime(Instant replicationLastStopTime) {
            this.replicationLastStopTime = replicationLastStopTime;
            return this;
        }

        public final Instant getReplicationDeprovisionTime() {
            return replicationDeprovisionTime;
        }

        public final void setReplicationDeprovisionTime(Instant replicationDeprovisionTime) {
            this.replicationDeprovisionTime = replicationDeprovisionTime;
        }

        @Override
        public final Builder replicationDeprovisionTime(Instant replicationDeprovisionTime) {
            this.replicationDeprovisionTime = replicationDeprovisionTime;
            return this;
        }

        public final Boolean getIsReadOnly() {
            return isReadOnly;
        }

        public final void setIsReadOnly(Boolean isReadOnly) {
            this.isReadOnly = isReadOnly;
        }

        @Override
        public final Builder isReadOnly(Boolean isReadOnly) {
            this.isReadOnly = isReadOnly;
            return this;
        }

        @Override
        public Replication build() {
            return new Replication(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
