/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costexplorer;

import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.costexplorer.model.BillExpirationException;
import software.amazon.awssdk.services.costexplorer.model.CostExplorerException;
import software.amazon.awssdk.services.costexplorer.model.CostExplorerRequest;
import software.amazon.awssdk.services.costexplorer.model.CreateCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.CreateCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.model.DataUnavailableException;
import software.amazon.awssdk.services.costexplorer.model.DeleteCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.DeleteCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.model.DescribeCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.DescribeCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageRequest;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageResponse;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageWithResourcesRequest;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageWithResourcesResponse;
import software.amazon.awssdk.services.costexplorer.model.GetCostForecastRequest;
import software.amazon.awssdk.services.costexplorer.model.GetCostForecastResponse;
import software.amazon.awssdk.services.costexplorer.model.GetDimensionValuesRequest;
import software.amazon.awssdk.services.costexplorer.model.GetDimensionValuesResponse;
import software.amazon.awssdk.services.costexplorer.model.GetReservationCoverageRequest;
import software.amazon.awssdk.services.costexplorer.model.GetReservationCoverageResponse;
import software.amazon.awssdk.services.costexplorer.model.GetReservationPurchaseRecommendationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetReservationPurchaseRecommendationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetReservationUtilizationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetReservationUtilizationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetRightsizingRecommendationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetRightsizingRecommendationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansPurchaseRecommendationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansPurchaseRecommendationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetTagsRequest;
import software.amazon.awssdk.services.costexplorer.model.GetTagsResponse;
import software.amazon.awssdk.services.costexplorer.model.GetUsageForecastRequest;
import software.amazon.awssdk.services.costexplorer.model.GetUsageForecastResponse;
import software.amazon.awssdk.services.costexplorer.model.InvalidNextTokenException;
import software.amazon.awssdk.services.costexplorer.model.LimitExceededException;
import software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsRequest;
import software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsResponse;
import software.amazon.awssdk.services.costexplorer.model.RequestChangedException;
import software.amazon.awssdk.services.costexplorer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.costexplorer.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.costexplorer.model.UnresolvableUsageUnitException;
import software.amazon.awssdk.services.costexplorer.model.UpdateCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.UpdateCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoveragePublisher;
import software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsPublisher;
import software.amazon.awssdk.services.costexplorer.paginators.ListCostCategoryDefinitionsPublisher;
import software.amazon.awssdk.services.costexplorer.transform.CreateCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.DeleteCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.DescribeCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetCostAndUsageRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetCostAndUsageWithResourcesRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetCostForecastRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetDimensionValuesRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetReservationCoverageRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetReservationPurchaseRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetReservationUtilizationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetRightsizingRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansCoverageRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansPurchaseRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansUtilizationDetailsRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansUtilizationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetTagsRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetUsageForecastRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.ListCostCategoryDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.UpdateCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CostExplorerAsyncClient}.
 *
 * @see CostExplorerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCostExplorerAsyncClient implements CostExplorerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCostExplorerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCostExplorerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a new Cost Category with the requested name and rules.
     * </p>
     *
     * @param createCostCategoryDefinitionRequest
     * @return A Java Future containing the result of the CreateCostCategoryDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServiceQuotaExceededException You've reached the limit on the number of resources you can create, or
     *         exceeded the size of an individual resources.</li>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.CreateCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/CreateCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCostCategoryDefinitionResponse> createCostCategoryDefinition(
            CreateCostCategoryDefinitionRequest createCostCategoryDefinitionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCostCategoryDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCostCategoryDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCostCategoryDefinitionRequest, CreateCostCategoryDefinitionResponse>()
                            .withOperationName("CreateCostCategoryDefinition")
                            .withMarshaller(new CreateCostCategoryDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(createCostCategoryDefinitionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Cost Category. Expenses from this month going forward will no longer be categorized with this Cost
     * Category.
     * </p>
     *
     * @param deleteCostCategoryDefinitionRequest
     * @return A Java Future containing the result of the DeleteCostCategoryDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ARN in the request doesn't exist.</li>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.DeleteCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/DeleteCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCostCategoryDefinitionResponse> deleteCostCategoryDefinition(
            DeleteCostCategoryDefinitionRequest deleteCostCategoryDefinitionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCostCategoryDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCostCategoryDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCostCategoryDefinitionRequest, DeleteCostCategoryDefinitionResponse>()
                            .withOperationName("DeleteCostCategoryDefinition")
                            .withMarshaller(new DeleteCostCategoryDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(deleteCostCategoryDefinitionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the name, ARN, rules, definition, and effective dates of a Cost Category that's defined in the account.
     * </p>
     * <p>
     * You have the option to use <code>EffectiveOn</code> to return a Cost Category that is active on a specific date.
     * If there is no <code>EffectiveOn</code> specified, you’ll see a Cost Category that is effective on the current
     * date. If Cost Category is still effective, <code>EffectiveEnd</code> is omitted in the response.
     * </p>
     *
     * @param describeCostCategoryDefinitionRequest
     * @return A Java Future containing the result of the DescribeCostCategoryDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ARN in the request doesn't exist.</li>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.DescribeCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/DescribeCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCostCategoryDefinitionResponse> describeCostCategoryDefinition(
            DescribeCostCategoryDefinitionRequest describeCostCategoryDefinitionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCostCategoryDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCostCategoryDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCostCategoryDefinitionRequest, DescribeCostCategoryDefinitionResponse>()
                            .withOperationName("DescribeCostCategoryDefinition")
                            .withMarshaller(new DescribeCostCategoryDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(describeCostCategoryDefinitionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves cost and usage metrics for your account. You can specify which cost and usage-related metric, such as
     * <code>BlendedCosts</code> or <code>UsageQuantity</code>, that you want the request to return. You can also filter
     * and group your data by various dimensions, such as <code>SERVICE</code> or <code>AZ</code>, in a specific time
     * range. For a complete list of valid dimensions, see the <a
     * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html"
     * >GetDimensionValues</a> operation. Master accounts in an organization in AWS Organizations have access to all
     * member accounts.
     * </p>
     *
     * @param getCostAndUsageRequest
     * @return A Java Future containing the result of the GetCostAndUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>BillExpirationException The requested report expired. Update the date interval and try again.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>RequestChangedException Your request parameters changed between pages. Try again with the old
     *         parameters or without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetCostAndUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetCostAndUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCostAndUsageResponse> getCostAndUsage(GetCostAndUsageRequest getCostAndUsageRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCostAndUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCostAndUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCostAndUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCostAndUsageRequest, GetCostAndUsageResponse>()
                            .withOperationName("GetCostAndUsage")
                            .withMarshaller(new GetCostAndUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getCostAndUsageRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves cost and usage metrics with resources for your account. You can specify which cost and usage-related
     * metric, such as <code>BlendedCosts</code> or <code>UsageQuantity</code>, that you want the request to return. You
     * can also filter and group your data by various dimensions, such as <code>SERVICE</code> or <code>AZ</code>, in a
     * specific time range. For a complete list of valid dimensions, see the <a
     * href="https://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html"
     * >GetDimensionValues</a> operation. Master accounts in an organization in AWS Organizations have access to all
     * member accounts. This API is currently available for the Amazon Elastic Compute Cloud – Compute service only.
     * </p>
     * <note>
     * <p>
     * This is an opt-in only feature. You can enable this feature from the Cost Explorer Settings page. For information
     * on how to access the Settings page, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/ce-access.html">Controlling Access for Cost
     * Explorer</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * </note>
     *
     * @param getCostAndUsageWithResourcesRequest
     * @return A Java Future containing the result of the GetCostAndUsageWithResources operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>BillExpirationException The requested report expired. Update the date interval and try again.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>RequestChangedException Your request parameters changed between pages. Try again with the old
     *         parameters or without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetCostAndUsageWithResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetCostAndUsageWithResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCostAndUsageWithResourcesResponse> getCostAndUsageWithResources(
            GetCostAndUsageWithResourcesRequest getCostAndUsageWithResourcesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCostAndUsageWithResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCostAndUsageWithResourcesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCostAndUsageWithResourcesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCostAndUsageWithResourcesRequest, GetCostAndUsageWithResourcesResponse>()
                            .withOperationName("GetCostAndUsageWithResources")
                            .withMarshaller(new GetCostAndUsageWithResourcesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getCostAndUsageWithResourcesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a forecast for how much Amazon Web Services predicts that you will spend over the forecast time period
     * that you select, based on your past costs.
     * </p>
     *
     * @param getCostForecastRequest
     * @return A Java Future containing the result of the GetCostForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetCostForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetCostForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCostForecastResponse> getCostForecast(GetCostForecastRequest getCostForecastRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCostForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCostForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCostForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCostForecastRequest, GetCostForecastResponse>()
                            .withOperationName("GetCostForecast")
                            .withMarshaller(new GetCostForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getCostForecastRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves all available filter values for a specified filter over a period of time. You can search the dimension
     * values for an arbitrary string.
     * </p>
     *
     * @param getDimensionValuesRequest
     * @return A Java Future containing the result of the GetDimensionValues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>BillExpirationException The requested report expired. Update the date interval and try again.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>RequestChangedException Your request parameters changed between pages. Try again with the old
     *         parameters or without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetDimensionValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetDimensionValues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetDimensionValuesResponse> getDimensionValues(GetDimensionValuesRequest getDimensionValuesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetDimensionValuesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetDimensionValuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetDimensionValuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetDimensionValuesRequest, GetDimensionValuesResponse>()
                            .withOperationName("GetDimensionValues")
                            .withMarshaller(new GetDimensionValuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getDimensionValuesRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the reservation coverage for your account. This enables you to see how much of your Amazon Elastic
     * Compute Cloud, Amazon ElastiCache, Amazon Relational Database Service, or Amazon Redshift usage is covered by a
     * reservation. An organization's master account can see the coverage of the associated member accounts. This
     * supports dimensions, Cost Categories, and nested expressions. For any time period, you can filter data about
     * reservation usage by the following dimensions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AZ
     * </p>
     * </li>
     * <li>
     * <p>
     * CACHE_ENGINE
     * </p>
     * </li>
     * <li>
     * <p>
     * DATABASE_ENGINE
     * </p>
     * </li>
     * <li>
     * <p>
     * DEPLOYMENT_OPTION
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_TYPE
     * </p>
     * </li>
     * <li>
     * <p>
     * LINKED_ACCOUNT
     * </p>
     * </li>
     * <li>
     * <p>
     * OPERATING_SYSTEM
     * </p>
     * </li>
     * <li>
     * <p>
     * PLATFORM
     * </p>
     * </li>
     * <li>
     * <p>
     * REGION
     * </p>
     * </li>
     * <li>
     * <p>
     * SERVICE
     * </p>
     * </li>
     * <li>
     * <p>
     * TAG
     * </p>
     * </li>
     * <li>
     * <p>
     * TENANCY
     * </p>
     * </li>
     * </ul>
     * <p>
     * To determine valid values for a dimension, use the <code>GetDimensionValues</code> operation.
     * </p>
     *
     * @param getReservationCoverageRequest
     *        You can use the following request parameters to query for how much of your instance usage a reservation
     *        covered.
     * @return A Java Future containing the result of the GetReservationCoverage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetReservationCoverage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetReservationCoverage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetReservationCoverageResponse> getReservationCoverage(
            GetReservationCoverageRequest getReservationCoverageRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReservationCoverageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReservationCoverageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReservationCoverageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReservationCoverageRequest, GetReservationCoverageResponse>()
                            .withOperationName("GetReservationCoverage")
                            .withMarshaller(new GetReservationCoverageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getReservationCoverageRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets recommendations for which reservations to purchase. These recommendations could help you reduce your costs.
     * Reservations provide a discounted hourly rate (up to 75%) compared to On-Demand pricing.
     * </p>
     * <p>
     * AWS generates your recommendations by identifying your On-Demand usage during a specific time period and
     * collecting your usage into categories that are eligible for a reservation. After AWS has these categories, it
     * simulates every combination of reservations in each category of usage to identify the best number of each type of
     * RI to purchase to maximize your estimated savings.
     * </p>
     * <p>
     * For example, AWS automatically aggregates your Amazon EC2 Linux, shared tenancy, and c4 family usage in the US
     * West (Oregon) Region and recommends that you buy size-flexible regional reservations to apply to the c4 family
     * usage. AWS recommends the smallest size instance in an instance family. This makes it easier to purchase a
     * size-flexible RI. AWS also shows the equal number of normalized units so that you can purchase any instance size
     * that you want. For this example, your RI recommendation would be for <code>c4.large</code> because that is the
     * smallest size instance in the c4 instance family.
     * </p>
     *
     * @param getReservationPurchaseRecommendationRequest
     * @return A Java Future containing the result of the GetReservationPurchaseRecommendation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetReservationPurchaseRecommendation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetReservationPurchaseRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReservationPurchaseRecommendationResponse> getReservationPurchaseRecommendation(
            GetReservationPurchaseRecommendationRequest getReservationPurchaseRecommendationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReservationPurchaseRecommendationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetReservationPurchaseRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReservationPurchaseRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReservationPurchaseRecommendationRequest, GetReservationPurchaseRecommendationResponse>()
                            .withOperationName("GetReservationPurchaseRecommendation")
                            .withMarshaller(new GetReservationPurchaseRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getReservationPurchaseRecommendationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the reservation utilization for your account. Master accounts in an organization have access to member
     * accounts. You can filter data by dimensions in a time period. You can use <code>GetDimensionValues</code> to
     * determine the possible dimension values. Currently, you can group only by <code>SUBSCRIPTION_ID</code>.
     * </p>
     *
     * @param getReservationUtilizationRequest
     * @return A Java Future containing the result of the GetReservationUtilization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetReservationUtilization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetReservationUtilization" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetReservationUtilizationResponse> getReservationUtilization(
            GetReservationUtilizationRequest getReservationUtilizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetReservationUtilizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetReservationUtilizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetReservationUtilizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetReservationUtilizationRequest, GetReservationUtilizationResponse>()
                            .withOperationName("GetReservationUtilization")
                            .withMarshaller(new GetReservationUtilizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getReservationUtilizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates recommendations that helps you save cost by identifying idle and underutilized Amazon EC2 instances.
     * </p>
     * <p>
     * Recommendations are generated to either downsize or terminate instances, along with providing savings detail and
     * metrics. For details on calculation and function, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/ce-what-is.html">Optimizing Your Cost with
     * Rightsizing Recommendations</a>.
     * </p>
     *
     * @param getRightsizingRecommendationRequest
     * @return A Java Future containing the result of the GetRightsizingRecommendation operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetRightsizingRecommendation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetRightsizingRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRightsizingRecommendationResponse> getRightsizingRecommendation(
            GetRightsizingRecommendationRequest getRightsizingRecommendationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRightsizingRecommendationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRightsizingRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRightsizingRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRightsizingRecommendationRequest, GetRightsizingRecommendationResponse>()
                            .withOperationName("GetRightsizingRecommendation")
                            .withMarshaller(new GetRightsizingRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getRightsizingRecommendationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Savings Plans covered for your account. This enables you to see how much of your cost is covered by
     * a Savings Plan. An organization’s master account can see the coverage of the associated member accounts. This
     * supports dimensions, Cost Categories, and nested expressions. For any time period, you can filter data for
     * Savings Plans usage with the following dimensions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LINKED_ACCOUNT</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REGION</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSTANCE_FAMILY</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To determine valid values for a dimension, use the <code>GetDimensionValues</code> operation.
     * </p>
     *
     * @param getSavingsPlansCoverageRequest
     * @return A Java Future containing the result of the GetSavingsPlansCoverage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetSavingsPlansCoverage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansCoverage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetSavingsPlansCoverageResponse> getSavingsPlansCoverage(
            GetSavingsPlansCoverageRequest getSavingsPlansCoverageRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSavingsPlansCoverageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSavingsPlansCoverageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSavingsPlansCoverageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSavingsPlansCoverageRequest, GetSavingsPlansCoverageResponse>()
                            .withOperationName("GetSavingsPlansCoverage")
                            .withMarshaller(new GetSavingsPlansCoverageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSavingsPlansCoverageRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Savings Plans covered for your account. This enables you to see how much of your cost is covered by
     * a Savings Plan. An organization’s master account can see the coverage of the associated member accounts. This
     * supports dimensions, Cost Categories, and nested expressions. For any time period, you can filter data for
     * Savings Plans usage with the following dimensions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LINKED_ACCOUNT</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REGION</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSTANCE_FAMILY</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To determine valid values for a dimension, use the <code>GetDimensionValues</code> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getSavingsPlansCoverage(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoveragePublisher publisher = client.getSavingsPlansCoveragePaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoveragePublisher publisher = client.getSavingsPlansCoveragePaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSavingsPlansCoverage(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageRequest)}
     * operation.</b>
     * </p>
     *
     * @param getSavingsPlansCoverageRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetSavingsPlansCoverage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansCoverage" target="_top">AWS API
     *      Documentation</a>
     */
    public GetSavingsPlansCoveragePublisher getSavingsPlansCoveragePaginator(
            GetSavingsPlansCoverageRequest getSavingsPlansCoverageRequest) {
        return new GetSavingsPlansCoveragePublisher(this, applyPaginatorUserAgent(getSavingsPlansCoverageRequest));
    }

    /**
     * <p>
     * Retrieves your request parameters, Savings Plan Recommendations Summary and Details.
     * </p>
     *
     * @param getSavingsPlansPurchaseRecommendationRequest
     * @return A Java Future containing the result of the GetSavingsPlansPurchaseRecommendation operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetSavingsPlansPurchaseRecommendation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansPurchaseRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSavingsPlansPurchaseRecommendationResponse> getSavingsPlansPurchaseRecommendation(
            GetSavingsPlansPurchaseRecommendationRequest getSavingsPlansPurchaseRecommendationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSavingsPlansPurchaseRecommendationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetSavingsPlansPurchaseRecommendationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSavingsPlansPurchaseRecommendationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSavingsPlansPurchaseRecommendationRequest, GetSavingsPlansPurchaseRecommendationResponse>()
                            .withOperationName("GetSavingsPlansPurchaseRecommendation")
                            .withMarshaller(new GetSavingsPlansPurchaseRecommendationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSavingsPlansPurchaseRecommendationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the Savings Plans utilization for your account across date ranges with daily or monthly granularity.
     * Master accounts in an organization have access to member accounts. You can use <code>GetDimensionValues</code> in
     * <code>SAVINGS_PLANS</code> to determine the possible dimension values.
     * </p>
     * <note>
     * <p>
     * You cannot group by any dimension values for <code>GetSavingsPlansUtilization</code>.
     * </p>
     * </note>
     *
     * @param getSavingsPlansUtilizationRequest
     * @return A Java Future containing the result of the GetSavingsPlansUtilization operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetSavingsPlansUtilization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansUtilization" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSavingsPlansUtilizationResponse> getSavingsPlansUtilization(
            GetSavingsPlansUtilizationRequest getSavingsPlansUtilizationRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSavingsPlansUtilizationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetSavingsPlansUtilizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSavingsPlansUtilizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSavingsPlansUtilizationRequest, GetSavingsPlansUtilizationResponse>()
                            .withOperationName("GetSavingsPlansUtilization")
                            .withMarshaller(new GetSavingsPlansUtilizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSavingsPlansUtilizationRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves attribute data along with aggregate utilization and savings data for a given time period. This doesn't
     * support granular or grouped data (daily/monthly) in response. You can't retrieve data by dates in a single
     * response similar to <code>GetSavingsPlanUtilization</code>, but you have the option to make multiple calls to
     * <code>GetSavingsPlanUtilizationDetails</code> by providing individual dates. You can use
     * <code>GetDimensionValues</code> in <code>SAVINGS_PLANS</code> to determine the possible dimension values.
     * </p>
     * <note>
     * <p>
     * <code>GetSavingsPlanUtilizationDetails</code> internally groups data by <code>SavingsPlansArn</code>.
     * </p>
     * </note>
     *
     * @param getSavingsPlansUtilizationDetailsRequest
     * @return A Java Future containing the result of the GetSavingsPlansUtilizationDetails operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetSavingsPlansUtilizationDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansUtilizationDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetSavingsPlansUtilizationDetailsResponse> getSavingsPlansUtilizationDetails(
            GetSavingsPlansUtilizationDetailsRequest getSavingsPlansUtilizationDetailsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetSavingsPlansUtilizationDetailsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetSavingsPlansUtilizationDetailsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetSavingsPlansUtilizationDetailsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetSavingsPlansUtilizationDetailsRequest, GetSavingsPlansUtilizationDetailsResponse>()
                            .withOperationName("GetSavingsPlansUtilizationDetails")
                            .withMarshaller(new GetSavingsPlansUtilizationDetailsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getSavingsPlansUtilizationDetailsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves attribute data along with aggregate utilization and savings data for a given time period. This doesn't
     * support granular or grouped data (daily/monthly) in response. You can't retrieve data by dates in a single
     * response similar to <code>GetSavingsPlanUtilization</code>, but you have the option to make multiple calls to
     * <code>GetSavingsPlanUtilizationDetails</code> by providing individual dates. You can use
     * <code>GetDimensionValues</code> in <code>SAVINGS_PLANS</code> to determine the possible dimension values.
     * </p>
     * <note>
     * <p>
     * <code>GetSavingsPlanUtilizationDetails</code> internally groups data by <code>SavingsPlansArn</code>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getSavingsPlansUtilizationDetails(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsPublisher publisher = client.getSavingsPlansUtilizationDetailsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsPublisher publisher = client.getSavingsPlansUtilizationDetailsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSavingsPlansUtilizationDetails(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getSavingsPlansUtilizationDetailsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetSavingsPlansUtilizationDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansUtilizationDetails"
     *      target="_top">AWS API Documentation</a>
     */
    public GetSavingsPlansUtilizationDetailsPublisher getSavingsPlansUtilizationDetailsPaginator(
            GetSavingsPlansUtilizationDetailsRequest getSavingsPlansUtilizationDetailsRequest) {
        return new GetSavingsPlansUtilizationDetailsPublisher(this,
                applyPaginatorUserAgent(getSavingsPlansUtilizationDetailsRequest));
    }

    /**
     * <p>
     * Queries for available tag keys and tag values for a specified period. You can search the tag values for an
     * arbitrary string.
     * </p>
     *
     * @param getTagsRequest
     * @return A Java Future containing the result of the GetTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>BillExpirationException The requested report expired. Update the date interval and try again.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>InvalidNextTokenException The pagination token is invalid. Try again without a pagination token.</li>
     *         <li>RequestChangedException Your request parameters changed between pages. Try again with the old
     *         parameters or without a pagination token.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetTagsResponse> getTags(GetTagsRequest getTagsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetTagsRequest, GetTagsResponse>().withOperationName("GetTags")
                            .withMarshaller(new GetTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getTagsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a forecast for how much Amazon Web Services predicts that you will use over the forecast time period
     * that you select, based on your past usage.
     * </p>
     *
     * @param getUsageForecastRequest
     * @return A Java Future containing the result of the GetUsageForecast operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>DataUnavailableException The requested data is unavailable.</li>
     *         <li>UnresolvableUsageUnitException Cost Explorer was unable to identify the usage unit. Provide
     *         <code>UsageType/UsageTypeGroup</code> filter selections that contain matching units, for example:
     *         <code>hours</code>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.GetUsageForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetUsageForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetUsageForecastResponse> getUsageForecast(GetUsageForecastRequest getUsageForecastRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetUsageForecastResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetUsageForecastResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetUsageForecastResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetUsageForecastRequest, GetUsageForecastResponse>()
                            .withOperationName("GetUsageForecast")
                            .withMarshaller(new GetUsageForecastRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(getUsageForecastRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the name, ARN, <code>NumberOfRules</code> and effective dates of all Cost Categories defined in the
     * account. You have the option to use <code>EffectiveOn</code> to return a list of Cost Categories that were active
     * on a specific date. If there is no <code>EffectiveOn</code> specified, you’ll see Cost Categories that are
     * effective on the current date. If Cost Category is still effective, <code>EffectiveEnd</code> is omitted in the
     * response. <code>ListCostCategoryDefinitions</code> supports pagination. The request can have a
     * <code>MaxResults</code> range up to 100.
     * </p>
     *
     * @param listCostCategoryDefinitionsRequest
     * @return A Java Future containing the result of the ListCostCategoryDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.ListCostCategoryDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/ListCostCategoryDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCostCategoryDefinitionsResponse> listCostCategoryDefinitions(
            ListCostCategoryDefinitionsRequest listCostCategoryDefinitionsRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCostCategoryDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCostCategoryDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCostCategoryDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCostCategoryDefinitionsRequest, ListCostCategoryDefinitionsResponse>()
                            .withOperationName("ListCostCategoryDefinitions")
                            .withMarshaller(new ListCostCategoryDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(listCostCategoryDefinitionsRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the name, ARN, <code>NumberOfRules</code> and effective dates of all Cost Categories defined in the
     * account. You have the option to use <code>EffectiveOn</code> to return a list of Cost Categories that were active
     * on a specific date. If there is no <code>EffectiveOn</code> specified, you’ll see Cost Categories that are
     * effective on the current date. If Cost Category is still effective, <code>EffectiveEnd</code> is omitted in the
     * response. <code>ListCostCategoryDefinitions</code> supports pagination. The request can have a
     * <code>MaxResults</code> range up to 100.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCostCategoryDefinitions(software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.ListCostCategoryDefinitionsPublisher publisher = client.listCostCategoryDefinitionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.ListCostCategoryDefinitionsPublisher publisher = client.listCostCategoryDefinitionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCostCategoryDefinitions(software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCostCategoryDefinitionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.ListCostCategoryDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/ListCostCategoryDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    public ListCostCategoryDefinitionsPublisher listCostCategoryDefinitionsPaginator(
            ListCostCategoryDefinitionsRequest listCostCategoryDefinitionsRequest) {
        return new ListCostCategoryDefinitionsPublisher(this, applyPaginatorUserAgent(listCostCategoryDefinitionsRequest));
    }

    /**
     * <p>
     * Updates an existing Cost Category. Changes made to the Cost Category rules will be used to categorize the current
     * month’s expenses and future expenses. This won’t change categorization for the previous months.
     * </p>
     *
     * @param updateCostCategoryDefinitionRequest
     * @return A Java Future containing the result of the UpdateCostCategoryDefinition operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The specified ARN in the request doesn't exist.</li>
     *         <li>ServiceQuotaExceededException You've reached the limit on the number of resources you can create, or
     *         exceeded the size of an individual resources.</li>
     *         <li>LimitExceededException You made too many calls in a short period of time. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostExplorerException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample CostExplorerAsyncClient.UpdateCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/UpdateCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCostCategoryDefinitionResponse> updateCostCategoryDefinition(
            UpdateCostCategoryDefinitionRequest updateCostCategoryDefinitionRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCostCategoryDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCostCategoryDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCostCategoryDefinitionRequest, UpdateCostCategoryDefinitionResponse>()
                            .withOperationName("UpdateCostCategoryDefinition")
                            .withMarshaller(new UpdateCostCategoryDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withInput(updateCostCategoryDefinitionRequest));
            return executeFuture;
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CostExplorerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnresolvableUsageUnitException")
                                .exceptionBuilderSupplier(UnresolvableUsageUnitException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestChangedException")
                                .exceptionBuilderSupplier(RequestChangedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataUnavailableException")
                                .exceptionBuilderSupplier(DataUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BillExpirationException")
                                .exceptionBuilderSupplier(BillExpirationException::builder).build());
    }

    private <T extends CostExplorerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
