/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costexplorer;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.costexplorer.model.BillExpirationException;
import software.amazon.awssdk.services.costexplorer.model.CostExplorerException;
import software.amazon.awssdk.services.costexplorer.model.CostExplorerRequest;
import software.amazon.awssdk.services.costexplorer.model.CreateCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.CreateCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.model.DataUnavailableException;
import software.amazon.awssdk.services.costexplorer.model.DeleteCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.DeleteCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.model.DescribeCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.DescribeCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageRequest;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageResponse;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageWithResourcesRequest;
import software.amazon.awssdk.services.costexplorer.model.GetCostAndUsageWithResourcesResponse;
import software.amazon.awssdk.services.costexplorer.model.GetCostForecastRequest;
import software.amazon.awssdk.services.costexplorer.model.GetCostForecastResponse;
import software.amazon.awssdk.services.costexplorer.model.GetDimensionValuesRequest;
import software.amazon.awssdk.services.costexplorer.model.GetDimensionValuesResponse;
import software.amazon.awssdk.services.costexplorer.model.GetReservationCoverageRequest;
import software.amazon.awssdk.services.costexplorer.model.GetReservationCoverageResponse;
import software.amazon.awssdk.services.costexplorer.model.GetReservationPurchaseRecommendationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetReservationPurchaseRecommendationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetReservationUtilizationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetReservationUtilizationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetRightsizingRecommendationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetRightsizingRecommendationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansPurchaseRecommendationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansPurchaseRecommendationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsResponse;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationRequest;
import software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationResponse;
import software.amazon.awssdk.services.costexplorer.model.GetTagsRequest;
import software.amazon.awssdk.services.costexplorer.model.GetTagsResponse;
import software.amazon.awssdk.services.costexplorer.model.GetUsageForecastRequest;
import software.amazon.awssdk.services.costexplorer.model.GetUsageForecastResponse;
import software.amazon.awssdk.services.costexplorer.model.InvalidNextTokenException;
import software.amazon.awssdk.services.costexplorer.model.LimitExceededException;
import software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsRequest;
import software.amazon.awssdk.services.costexplorer.model.ListCostCategoryDefinitionsResponse;
import software.amazon.awssdk.services.costexplorer.model.RequestChangedException;
import software.amazon.awssdk.services.costexplorer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.costexplorer.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.costexplorer.model.UnresolvableUsageUnitException;
import software.amazon.awssdk.services.costexplorer.model.UpdateCostCategoryDefinitionRequest;
import software.amazon.awssdk.services.costexplorer.model.UpdateCostCategoryDefinitionResponse;
import software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoverageIterable;
import software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsIterable;
import software.amazon.awssdk.services.costexplorer.transform.CreateCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.DeleteCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.DescribeCostCategoryDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetCostAndUsageRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetCostAndUsageWithResourcesRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetCostForecastRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetDimensionValuesRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetReservationCoverageRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetReservationPurchaseRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetReservationUtilizationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetRightsizingRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansCoverageRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansPurchaseRecommendationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansUtilizationDetailsRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetSavingsPlansUtilizationRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetTagsRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.GetUsageForecastRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.ListCostCategoryDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.costexplorer.transform.UpdateCostCategoryDefinitionRequestMarshaller;

/**
 * Internal implementation of {@link CostExplorerClient}.
 *
 * @see CostExplorerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCostExplorerClient implements CostExplorerClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCostExplorerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <important>
     * <p>
     * <i> <b>Cost Category is in public beta for AWS Billing and Cost Management and is subject to change. Your use of
     * Cost Categories is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * </important>
     * <p>
     * Creates a new Cost Category with the requested name and rules.
     * </p>
     *
     * @param createCostCategoryDefinitionRequest
     * @return Result of the CreateCostCategoryDefinition operation returned by the service.
     * @throws ServiceQuotaExceededException
     *         You've reached the limit on the number of resources you can create, or exceeded the size of an individual
     *         resources.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.CreateCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/CreateCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateCostCategoryDefinitionResponse createCostCategoryDefinition(
            CreateCostCategoryDefinitionRequest createCostCategoryDefinitionRequest) throws ServiceQuotaExceededException,
            LimitExceededException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateCostCategoryDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<CreateCostCategoryDefinitionRequest, CreateCostCategoryDefinitionResponse>()
                        .withOperationName("CreateCostCategoryDefinition").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(createCostCategoryDefinitionRequest)
                        .withMarshaller(new CreateCostCategoryDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <important>
     * <p>
     * <i> <b>Cost Category is in public beta for AWS Billing and Cost Management and is subject to change. Your use of
     * Cost Categories is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * </important>
     * <p>
     * Deletes a Cost Category. Expenses from this month going forward will no longer be categorized with this Cost
     * Category.
     * </p>
     *
     * @param deleteCostCategoryDefinitionRequest
     * @return Result of the DeleteCostCategoryDefinition operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.DeleteCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/DeleteCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteCostCategoryDefinitionResponse deleteCostCategoryDefinition(
            DeleteCostCategoryDefinitionRequest deleteCostCategoryDefinitionRequest) throws ResourceNotFoundException,
            LimitExceededException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteCostCategoryDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DeleteCostCategoryDefinitionRequest, DeleteCostCategoryDefinitionResponse>()
                        .withOperationName("DeleteCostCategoryDefinition").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(deleteCostCategoryDefinitionRequest)
                        .withMarshaller(new DeleteCostCategoryDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <important>
     * <p>
     * <i> <b>Cost Category is in public beta for AWS Billing and Cost Management and is subject to change. Your use of
     * Cost Categories is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * </important>
     * <p>
     * Returns the name, ARN, rules, definition, and effective dates of a Cost Category that's defined in the account.
     * </p>
     * <p>
     * You have the option to use <code>EffectiveOn</code> to return a Cost Category that is active on a specific date.
     * If there is no <code>EffectiveOn</code> specified, you’ll see a Cost Category that is effective on the current
     * date. If Cost Category is still effective, <code>EffectiveEnd</code> is omitted in the response.
     * </p>
     *
     * @param describeCostCategoryDefinitionRequest
     * @return Result of the DescribeCostCategoryDefinition operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.DescribeCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/DescribeCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeCostCategoryDefinitionResponse describeCostCategoryDefinition(
            DescribeCostCategoryDefinitionRequest describeCostCategoryDefinitionRequest) throws ResourceNotFoundException,
            LimitExceededException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeCostCategoryDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<DescribeCostCategoryDefinitionRequest, DescribeCostCategoryDefinitionResponse>()
                        .withOperationName("DescribeCostCategoryDefinition").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(describeCostCategoryDefinitionRequest)
                        .withMarshaller(new DescribeCostCategoryDefinitionRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves cost and usage metrics for your account. You can specify which cost and usage-related metric, such as
     * <code>BlendedCosts</code> or <code>UsageQuantity</code>, that you want the request to return. You can also filter
     * and group your data by various dimensions, such as <code>SERVICE</code> or <code>AZ</code>, in a specific time
     * range. For a complete list of valid dimensions, see the <a
     * href="http://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html"
     * >GetDimensionValues</a> operation. Master accounts in an organization in AWS Organizations have access to all
     * member accounts.
     * </p>
     *
     * @param getCostAndUsageRequest
     * @return Result of the GetCostAndUsage operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws BillExpirationException
     *         The requested report expired. Update the date interval and try again.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws RequestChangedException
     *         Your request parameters changed between pages. Try again with the old parameters or without a pagination
     *         token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetCostAndUsage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetCostAndUsage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCostAndUsageResponse getCostAndUsage(GetCostAndUsageRequest getCostAndUsageRequest) throws LimitExceededException,
            BillExpirationException, DataUnavailableException, InvalidNextTokenException, RequestChangedException,
            AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCostAndUsageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCostAndUsageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetCostAndUsageRequest, GetCostAndUsageResponse>()
                .withOperationName("GetCostAndUsage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getCostAndUsageRequest)
                .withMarshaller(new GetCostAndUsageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves cost and usage metrics with resources for your account. You can specify which cost and usage-related
     * metric, such as <code>BlendedCosts</code> or <code>UsageQuantity</code>, that you want the request to return. You
     * can also filter and group your data by various dimensions, such as <code>SERVICE</code> or <code>AZ</code>, in a
     * specific time range. For a complete list of valid dimensions, see the <a
     * href="http://docs.aws.amazon.com/aws-cost-management/latest/APIReference/API_GetDimensionValues.html"
     * >GetDimensionValues</a> operation. Master accounts in an organization in AWS Organizations have access to all
     * member accounts. This API is currently available for the Amazon Elastic Compute Cloud – Compute service only.
     * </p>
     * <note>
     * <p>
     * This is an opt-in only feature. You can enable this feature from the Cost Explorer Settings page. For information
     * on how to access the Settings page, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/ce-access.html">Controlling Access for Cost
     * Explorer</a> in the <i>AWS Billing and Cost Management User Guide</i>.
     * </p>
     * </note>
     *
     * @param getCostAndUsageWithResourcesRequest
     * @return Result of the GetCostAndUsageWithResources operation returned by the service.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws BillExpirationException
     *         The requested report expired. Update the date interval and try again.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws RequestChangedException
     *         Your request parameters changed between pages. Try again with the old parameters or without a pagination
     *         token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetCostAndUsageWithResources
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetCostAndUsageWithResources"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetCostAndUsageWithResourcesResponse getCostAndUsageWithResources(
            GetCostAndUsageWithResourcesRequest getCostAndUsageWithResourcesRequest) throws DataUnavailableException,
            LimitExceededException, BillExpirationException, InvalidNextTokenException, RequestChangedException,
            AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCostAndUsageWithResourcesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetCostAndUsageWithResourcesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetCostAndUsageWithResourcesRequest, GetCostAndUsageWithResourcesResponse>()
                        .withOperationName("GetCostAndUsageWithResources").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getCostAndUsageWithResourcesRequest)
                        .withMarshaller(new GetCostAndUsageWithResourcesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a forecast for how much Amazon Web Services predicts that you will spend over the forecast time period
     * that you select, based on your past costs.
     * </p>
     *
     * @param getCostForecastRequest
     * @return Result of the GetCostForecast operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetCostForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetCostForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetCostForecastResponse getCostForecast(GetCostForecastRequest getCostForecastRequest) throws LimitExceededException,
            DataUnavailableException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetCostForecastResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetCostForecastResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetCostForecastRequest, GetCostForecastResponse>()
                .withOperationName("GetCostForecast").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getCostForecastRequest)
                .withMarshaller(new GetCostForecastRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves all available filter values for a specified filter over a period of time. You can search the dimension
     * values for an arbitrary string.
     * </p>
     *
     * @param getDimensionValuesRequest
     * @return Result of the GetDimensionValues operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws BillExpirationException
     *         The requested report expired. Update the date interval and try again.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws RequestChangedException
     *         Your request parameters changed between pages. Try again with the old parameters or without a pagination
     *         token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetDimensionValues
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetDimensionValues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDimensionValuesResponse getDimensionValues(GetDimensionValuesRequest getDimensionValuesRequest)
            throws LimitExceededException, BillExpirationException, DataUnavailableException, InvalidNextTokenException,
            RequestChangedException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDimensionValuesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDimensionValuesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetDimensionValuesRequest, GetDimensionValuesResponse>()
                .withOperationName("GetDimensionValues").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getDimensionValuesRequest)
                .withMarshaller(new GetDimensionValuesRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the reservation coverage for your account. This enables you to see how much of your Amazon Elastic
     * Compute Cloud, Amazon ElastiCache, Amazon Relational Database Service, or Amazon Redshift usage is covered by a
     * reservation. An organization's master account can see the coverage of the associated member accounts. For any
     * time period, you can filter data about reservation usage by the following dimensions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * AZ
     * </p>
     * </li>
     * <li>
     * <p>
     * CACHE_ENGINE
     * </p>
     * </li>
     * <li>
     * <p>
     * DATABASE_ENGINE
     * </p>
     * </li>
     * <li>
     * <p>
     * DEPLOYMENT_OPTION
     * </p>
     * </li>
     * <li>
     * <p>
     * INSTANCE_TYPE
     * </p>
     * </li>
     * <li>
     * <p>
     * LINKED_ACCOUNT
     * </p>
     * </li>
     * <li>
     * <p>
     * OPERATING_SYSTEM
     * </p>
     * </li>
     * <li>
     * <p>
     * PLATFORM
     * </p>
     * </li>
     * <li>
     * <p>
     * REGION
     * </p>
     * </li>
     * <li>
     * <p>
     * SERVICE
     * </p>
     * </li>
     * <li>
     * <p>
     * TAG
     * </p>
     * </li>
     * <li>
     * <p>
     * TENANCY
     * </p>
     * </li>
     * </ul>
     * <p>
     * To determine valid values for a dimension, use the <code>GetDimensionValues</code> operation.
     * </p>
     *
     * @param getReservationCoverageRequest
     *        You can use the following request parameters to query for how much of your instance usage a reservation
     *        covered.
     * @return Result of the GetReservationCoverage operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetReservationCoverage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetReservationCoverage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetReservationCoverageResponse getReservationCoverage(GetReservationCoverageRequest getReservationCoverageRequest)
            throws LimitExceededException, DataUnavailableException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReservationCoverageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReservationCoverageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetReservationCoverageRequest, GetReservationCoverageResponse>()
                .withOperationName("GetReservationCoverage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getReservationCoverageRequest)
                .withMarshaller(new GetReservationCoverageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Gets recommendations for which reservations to purchase. These recommendations could help you reduce your costs.
     * Reservations provide a discounted hourly rate (up to 75%) compared to On-Demand pricing.
     * </p>
     * <p>
     * AWS generates your recommendations by identifying your On-Demand usage during a specific time period and
     * collecting your usage into categories that are eligible for a reservation. After AWS has these categories, it
     * simulates every combination of reservations in each category of usage to identify the best number of each type of
     * RI to purchase to maximize your estimated savings.
     * </p>
     * <p>
     * For example, AWS automatically aggregates your Amazon EC2 Linux, shared tenancy, and c4 family usage in the US
     * West (Oregon) Region and recommends that you buy size-flexible regional reservations to apply to the c4 family
     * usage. AWS recommends the smallest size instance in an instance family. This makes it easier to purchase a
     * size-flexible RI. AWS also shows the equal number of normalized units so that you can purchase any instance size
     * that you want. For this example, your RI recommendation would be for <code>c4.large</code> because that is the
     * smallest size instance in the c4 instance family.
     * </p>
     *
     * @param getReservationPurchaseRecommendationRequest
     * @return Result of the GetReservationPurchaseRecommendation operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetReservationPurchaseRecommendation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetReservationPurchaseRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetReservationPurchaseRecommendationResponse getReservationPurchaseRecommendation(
            GetReservationPurchaseRecommendationRequest getReservationPurchaseRecommendationRequest)
            throws LimitExceededException, DataUnavailableException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReservationPurchaseRecommendationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetReservationPurchaseRecommendationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetReservationPurchaseRecommendationRequest, GetReservationPurchaseRecommendationResponse>()
                        .withOperationName("GetReservationPurchaseRecommendation").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getReservationPurchaseRecommendationRequest)
                        .withMarshaller(new GetReservationPurchaseRecommendationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the reservation utilization for your account. Master accounts in an organization have access to member
     * accounts. You can filter data by dimensions in a time period. You can use <code>GetDimensionValues</code> to
     * determine the possible dimension values. Currently, you can group only by <code>SUBSCRIPTION_ID</code>.
     * </p>
     *
     * @param getReservationUtilizationRequest
     * @return Result of the GetReservationUtilization operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetReservationUtilization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetReservationUtilization" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetReservationUtilizationResponse getReservationUtilization(
            GetReservationUtilizationRequest getReservationUtilizationRequest) throws LimitExceededException,
            DataUnavailableException, InvalidNextTokenException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetReservationUtilizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetReservationUtilizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetReservationUtilizationRequest, GetReservationUtilizationResponse>()
                        .withOperationName("GetReservationUtilization").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getReservationUtilizationRequest)
                        .withMarshaller(new GetReservationUtilizationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Creates recommendations that helps you save cost by identifying idle and underutilized Amazon EC2 instances.
     * </p>
     * <p>
     * Recommendations are generated to either downsize or terminate instances, along with providing savings detail and
     * metrics. For details on calculation and function, see <a
     * href="https://docs.aws.amazon.com/awsaccountbilling/latest/aboutv2/ce-what-is.html">Optimizing Your Cost with
     * Rightsizing Recommendations</a>.
     * </p>
     *
     * @param getRightsizingRecommendationRequest
     * @return Result of the GetRightsizingRecommendation operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetRightsizingRecommendation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetRightsizingRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRightsizingRecommendationResponse getRightsizingRecommendation(
            GetRightsizingRecommendationRequest getRightsizingRecommendationRequest) throws LimitExceededException,
            InvalidNextTokenException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRightsizingRecommendationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRightsizingRecommendationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetRightsizingRecommendationRequest, GetRightsizingRecommendationResponse>()
                        .withOperationName("GetRightsizingRecommendation").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getRightsizingRecommendationRequest)
                        .withMarshaller(new GetRightsizingRecommendationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the Savings Plans covered for your account. This enables you to see how much of your cost is covered by
     * a Savings Plan. An organization’s master account can see the coverage of the associated member accounts. For any
     * time period, you can filter data for Savings Plans usage with the following dimensions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LINKED_ACCOUNT</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REGION</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSTANCE_FAMILY</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To determine valid values for a dimension, use the <code>GetDimensionValues</code> operation.
     * </p>
     *
     * @param getSavingsPlansCoverageRequest
     * @return Result of the GetSavingsPlansCoverage operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetSavingsPlansCoverage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansCoverage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSavingsPlansCoverageResponse getSavingsPlansCoverage(GetSavingsPlansCoverageRequest getSavingsPlansCoverageRequest)
            throws LimitExceededException, DataUnavailableException, InvalidNextTokenException, AwsServiceException,
            SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSavingsPlansCoverageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSavingsPlansCoverageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetSavingsPlansCoverageRequest, GetSavingsPlansCoverageResponse>()
                .withOperationName("GetSavingsPlansCoverage").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getSavingsPlansCoverageRequest)
                .withMarshaller(new GetSavingsPlansCoverageRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the Savings Plans covered for your account. This enables you to see how much of your cost is covered by
     * a Savings Plan. An organization’s master account can see the coverage of the associated member accounts. For any
     * time period, you can filter data for Savings Plans usage with the following dimensions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>LINKED_ACCOUNT</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REGION</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SERVICE</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INSTANCE_FAMILY</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * To determine valid values for a dimension, use the <code>GetDimensionValues</code> operation.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getSavingsPlansCoverage(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoverageIterable responses = client.getSavingsPlansCoveragePaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoverageIterable responses = client
     *             .getSavingsPlansCoveragePaginator(request);
     *     for (software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansCoverageIterable responses = client.getSavingsPlansCoveragePaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSavingsPlansCoverage(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansCoverageRequest)}
     * operation.</b>
     * </p>
     *
     * @param getSavingsPlansCoverageRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetSavingsPlansCoverage
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansCoverage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetSavingsPlansCoverageIterable getSavingsPlansCoveragePaginator(
            GetSavingsPlansCoverageRequest getSavingsPlansCoverageRequest) throws LimitExceededException,
            DataUnavailableException, InvalidNextTokenException, AwsServiceException, SdkClientException, CostExplorerException {
        return new GetSavingsPlansCoverageIterable(this, applyPaginatorUserAgent(getSavingsPlansCoverageRequest));
    }

    /**
     * <p>
     * Retrieves your request parameters, Savings Plan Recommendations Summary and Details.
     * </p>
     *
     * @param getSavingsPlansPurchaseRecommendationRequest
     * @return Result of the GetSavingsPlansPurchaseRecommendation operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetSavingsPlansPurchaseRecommendation
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansPurchaseRecommendation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSavingsPlansPurchaseRecommendationResponse getSavingsPlansPurchaseRecommendation(
            GetSavingsPlansPurchaseRecommendationRequest getSavingsPlansPurchaseRecommendationRequest)
            throws LimitExceededException, InvalidNextTokenException, AwsServiceException, SdkClientException,
            CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSavingsPlansPurchaseRecommendationResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetSavingsPlansPurchaseRecommendationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetSavingsPlansPurchaseRecommendationRequest, GetSavingsPlansPurchaseRecommendationResponse>()
                        .withOperationName("GetSavingsPlansPurchaseRecommendation").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getSavingsPlansPurchaseRecommendationRequest)
                        .withMarshaller(new GetSavingsPlansPurchaseRecommendationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves the Savings Plans utilization for your account across date ranges with daily or monthly granularity.
     * Master accounts in an organization have access to member accounts. You can use <code>GetDimensionValues</code> in
     * <code>SAVINGS_PLANS</code> to determine the possible dimension values.
     * </p>
     * <note>
     * <p>
     * You cannot group by any dimension values for <code>GetSavingsPlansUtilization</code>.
     * </p>
     * </note>
     *
     * @param getSavingsPlansUtilizationRequest
     * @return Result of the GetSavingsPlansUtilization operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetSavingsPlansUtilization
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansUtilization" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetSavingsPlansUtilizationResponse getSavingsPlansUtilization(
            GetSavingsPlansUtilizationRequest getSavingsPlansUtilizationRequest) throws LimitExceededException,
            DataUnavailableException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSavingsPlansUtilizationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSavingsPlansUtilizationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetSavingsPlansUtilizationRequest, GetSavingsPlansUtilizationResponse>()
                        .withOperationName("GetSavingsPlansUtilization").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getSavingsPlansUtilizationRequest)
                        .withMarshaller(new GetSavingsPlansUtilizationRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves attribute data along with aggregate utilization and savings data for a given time period. This doesn't
     * support granular or grouped data (daily/monthly) in response. You can't retrieve data by dates in a single
     * response similar to <code>GetSavingsPlanUtilization</code>, but you have the option to make multiple calls to
     * <code>GetSavingsPlanUtilizationDetails</code> by providing individual dates. You can use
     * <code>GetDimensionValues</code> in <code>SAVINGS_PLANS</code> to determine the possible dimension values.
     * </p>
     * <note>
     * <p>
     * <code>GetSavingsPlanUtilizationDetails</code> internally groups data by <code>SavingsPlansArn</code>.
     * </p>
     * </note>
     *
     * @param getSavingsPlansUtilizationDetailsRequest
     * @return Result of the GetSavingsPlansUtilizationDetails operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetSavingsPlansUtilizationDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansUtilizationDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSavingsPlansUtilizationDetailsResponse getSavingsPlansUtilizationDetails(
            GetSavingsPlansUtilizationDetailsRequest getSavingsPlansUtilizationDetailsRequest) throws LimitExceededException,
            DataUnavailableException, InvalidNextTokenException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetSavingsPlansUtilizationDetailsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetSavingsPlansUtilizationDetailsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<GetSavingsPlansUtilizationDetailsRequest, GetSavingsPlansUtilizationDetailsResponse>()
                        .withOperationName("GetSavingsPlansUtilizationDetails").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(getSavingsPlansUtilizationDetailsRequest)
                        .withMarshaller(new GetSavingsPlansUtilizationDetailsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves attribute data along with aggregate utilization and savings data for a given time period. This doesn't
     * support granular or grouped data (daily/monthly) in response. You can't retrieve data by dates in a single
     * response similar to <code>GetSavingsPlanUtilization</code>, but you have the option to make multiple calls to
     * <code>GetSavingsPlanUtilizationDetails</code> by providing individual dates. You can use
     * <code>GetDimensionValues</code> in <code>SAVINGS_PLANS</code> to determine the possible dimension values.
     * </p>
     * <note>
     * <p>
     * <code>GetSavingsPlanUtilizationDetails</code> internally groups data by <code>SavingsPlansArn</code>.
     * </p>
     * </note><br/>
     * <p>
     * This is a variant of
     * {@link #getSavingsPlansUtilizationDetails(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsIterable responses = client.getSavingsPlansUtilizationDetailsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsIterable responses = client
     *             .getSavingsPlansUtilizationDetailsPaginator(request);
     *     for (software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.costexplorer.paginators.GetSavingsPlansUtilizationDetailsIterable responses = client.getSavingsPlansUtilizationDetailsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getSavingsPlansUtilizationDetails(software.amazon.awssdk.services.costexplorer.model.GetSavingsPlansUtilizationDetailsRequest)}
     * operation.</b>
     * </p>
     *
     * @param getSavingsPlansUtilizationDetailsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetSavingsPlansUtilizationDetails
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetSavingsPlansUtilizationDetails"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetSavingsPlansUtilizationDetailsIterable getSavingsPlansUtilizationDetailsPaginator(
            GetSavingsPlansUtilizationDetailsRequest getSavingsPlansUtilizationDetailsRequest) throws LimitExceededException,
            DataUnavailableException, InvalidNextTokenException, AwsServiceException, SdkClientException, CostExplorerException {
        return new GetSavingsPlansUtilizationDetailsIterable(this,
                applyPaginatorUserAgent(getSavingsPlansUtilizationDetailsRequest));
    }

    /**
     * <p>
     * Queries for available tag keys and tag values for a specified period. You can search the tag values for an
     * arbitrary string.
     * </p>
     *
     * @param getTagsRequest
     * @return Result of the GetTags operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws BillExpirationException
     *         The requested report expired. Update the date interval and try again.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws InvalidNextTokenException
     *         The pagination token is invalid. Try again without a pagination token.
     * @throws RequestChangedException
     *         Your request parameters changed between pages. Try again with the old parameters or without a pagination
     *         token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetTagsResponse getTags(GetTagsRequest getTagsRequest) throws LimitExceededException, BillExpirationException,
            DataUnavailableException, InvalidNextTokenException, RequestChangedException, AwsServiceException,
            SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetTagsRequest, GetTagsResponse>().withOperationName("GetTags")
                .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler).withInput(getTagsRequest)
                .withMarshaller(new GetTagsRequestMarshaller(protocolFactory)));
    }

    /**
     * <p>
     * Retrieves a forecast for how much Amazon Web Services predicts that you will use over the forecast time period
     * that you select, based on your past usage.
     * </p>
     *
     * @param getUsageForecastRequest
     * @return Result of the GetUsageForecast operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws DataUnavailableException
     *         The requested data is unavailable.
     * @throws UnresolvableUsageUnitException
     *         Cost Explorer was unable to identify the usage unit. Provide <code>UsageType/UsageTypeGroup</code> filter
     *         selections that contain matching units, for example: <code>hours</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.GetUsageForecast
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/GetUsageForecast" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetUsageForecastResponse getUsageForecast(GetUsageForecastRequest getUsageForecastRequest)
            throws LimitExceededException, DataUnavailableException, UnresolvableUsageUnitException, AwsServiceException,
            SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUsageForecastResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUsageForecastResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<GetUsageForecastRequest, GetUsageForecastResponse>()
                .withOperationName("GetUsageForecast").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(getUsageForecastRequest)
                .withMarshaller(new GetUsageForecastRequestMarshaller(protocolFactory)));
    }

    /**
     * <important>
     * <p>
     * <i> <b>Cost Category is in public beta for AWS Billing and Cost Management and is subject to change. Your use of
     * Cost Categories is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * </important>
     * <p>
     * Returns the name, ARN and effective dates of all Cost Categories defined in the account. You have the option to
     * use <code>EffectiveOn</code> to return a list of Cost Categories that were active on a specific date. If there is
     * no <code>EffectiveOn</code> specified, you’ll see Cost Categories that are effective on the current date. If Cost
     * Category is still effective, <code>EffectiveEnd</code> is omitted in the response.
     * </p>
     *
     * @param listCostCategoryDefinitionsRequest
     * @return Result of the ListCostCategoryDefinitions operation returned by the service.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.ListCostCategoryDefinitions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/ListCostCategoryDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListCostCategoryDefinitionsResponse listCostCategoryDefinitions(
            ListCostCategoryDefinitionsRequest listCostCategoryDefinitionsRequest) throws LimitExceededException,
            AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListCostCategoryDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListCostCategoryDefinitionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<ListCostCategoryDefinitionsRequest, ListCostCategoryDefinitionsResponse>()
                        .withOperationName("ListCostCategoryDefinitions").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(listCostCategoryDefinitionsRequest)
                        .withMarshaller(new ListCostCategoryDefinitionsRequestMarshaller(protocolFactory)));
    }

    /**
     * <important>
     * <p>
     * <i> <b>Cost Category is in public beta for AWS Billing and Cost Management and is subject to change. Your use of
     * Cost Categories is subject to the Beta Service Participation terms of the <a
     * href="https://aws.amazon.com/service-terms/">AWS Service Terms</a> (Section 1.10).</b> </i>
     * </p>
     * </important>
     * <p>
     * Updates an existing Cost Category. Changes made to the Cost Category rules will be used to categorize the current
     * month’s expenses and future expenses. This won’t change categorization for the previous months.
     * </p>
     *
     * @param updateCostCategoryDefinitionRequest
     * @return Result of the UpdateCostCategoryDefinition operation returned by the service.
     * @throws ResourceNotFoundException
     *         The specified ARN in the request doesn't exist.
     * @throws ServiceQuotaExceededException
     *         You've reached the limit on the number of resources you can create, or exceeded the size of an individual
     *         resources.
     * @throws LimitExceededException
     *         You made too many calls in a short period of time. Try again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CostExplorerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CostExplorerClient.UpdateCostCategoryDefinition
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ce-2017-10-25/UpdateCostCategoryDefinition"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateCostCategoryDefinitionResponse updateCostCategoryDefinition(
            UpdateCostCategoryDefinitionRequest updateCostCategoryDefinitionRequest) throws ResourceNotFoundException,
            ServiceQuotaExceededException, LimitExceededException, AwsServiceException, SdkClientException, CostExplorerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateCostCategoryDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateCostCategoryDefinitionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler
                .execute(new ClientExecutionParams<UpdateCostCategoryDefinitionRequest, UpdateCostCategoryDefinitionResponse>()
                        .withOperationName("UpdateCostCategoryDefinition").withResponseHandler(responseHandler)
                        .withErrorResponseHandler(errorResponseHandler).withInput(updateCostCategoryDefinitionRequest)
                        .withMarshaller(new UpdateCostCategoryDefinitionRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CostExplorerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnresolvableUsageUnitException")
                                .exceptionBuilderSupplier(UnresolvableUsageUnitException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestChangedException")
                                .exceptionBuilderSupplier(RequestChangedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DataUnavailableException")
                                .exceptionBuilderSupplier(DataUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BillExpirationException")
                                .exceptionBuilderSupplier(BillExpirationException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CostExplorerRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
