/*
 * Copyright 2010-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://aws.amazon.com/apache2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package software.amazon.awssdk.config.defaults;

import java.net.URI;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import java.util.function.Supplier;
import software.amazon.awssdk.annotation.ReviewBeforeRelease;
import software.amazon.awssdk.annotation.SdkProtectedApi;
import software.amazon.awssdk.auth.AwsCredentialsProvider;
import software.amazon.awssdk.client.builder.ClientBuilder;
import software.amazon.awssdk.config.ClientConfiguration;
import software.amazon.awssdk.config.ClientOverrideConfiguration;
import software.amazon.awssdk.config.MutableClientConfiguration;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * A class that makes it easy to apply default values to a {@link MutableClientConfiguration} if they weren't already
 * specified by the customer (or a higher precedent configuration). This is used by extending it and overriding optional
 * "apply default" methods to check if the customer has specified a piece of configuration. If not, the subclass can inject its
 * own configuration to be added to the customer's.
 *
 * <p>Configurations are implementing using different levels of precedents. For example, the customer's configuration has the
 * highest priority, followed by configuration generated by a builder, followed by service-specific overrides, followed by
 * generic global configurations. This configuration hierarchy is managed by the {@link ClientBuilder} implementation.</p>
 *
 * @see GlobalClientConfigurationDefaults
 */
@SdkProtectedApi
public abstract class ClientConfigurationDefaults {
    /**
     * Apply sync defaults to a mutable configuration object. This will potentially override any values that are absent from the
     * provided configuration, but should never override something that is already present.
     */
    public final void applySyncDefaults(MutableClientConfiguration config) {
        applyDefaultValues(config);
        // There are currently no sync-specific fields.
    }

    /**
     * Apply async defaults to a mutable configuration object. This will potentially override any values that are absent from the
     * provided configuration, but should never override something that is already present.
     */
    @ReviewBeforeRelease("Hardcodes thread pool default but that should go away anyways when we switch to NIO")
    public final void applyAsyncDefaults(MutableClientConfiguration config) {
        applyDefaultValues(config);

        config.asyncExecutorService(applyDefault(config.asyncExecutorService(), this::getAsyncExecutorDefault));
    }

    /**
     * Apply {@link ClientConfiguration} defaults to the provided configuration. This will potentially override any values that
     * are absent from the provided configuration, but should never override something that is already present.
     */
    private void applyDefaultValues(MutableClientConfiguration config) {
        config.overrideConfiguration(applyDefaultValues(config.overrideConfiguration(), this::applyOverrideDefaults));
        config.credentialsProvider(applyDefault(config.credentialsProvider(), this::getCredentialsDefault));
        config.endpoint(applyDefault(config.endpoint(), this::getEndpointDefault));
    }

    /**
     * Optionally overridden by the child class to add additional configuration defaults to the builder, assuming those fields
     * are not already set. Implementations of this method shouldn't override values already set in the builder.
     */
    protected void applyOverrideDefaults(ClientOverrideConfiguration.Builder builder) {

    }

    /**
     * Optionally overridden by the child class to return a default credentials provider.
     */
    protected AwsCredentialsProvider getCredentialsDefault() {
        return null;
    }

    /**
     * Optionally overridden by the child class to return a default endpoint.
     */
    protected URI getEndpointDefault() {
        return null;
    }

    /**
     * Optionally overridden by the child class to return a default executor service.
     */
    protected ScheduledExecutorService getAsyncExecutorDefault() {
        return null;
    }

    /**
     * Apply a default value to an optional field if the customer did not specify it.
     *
     * @param currentValue The current value of the field.
     * @param defaultValue The supplier for the default value, assuming there is not a value currently set.
     */
    protected final <T> T applyDefault(T currentValue, Supplier<T> defaultValue) {
        return currentValue != null ? currentValue : defaultValue.get();
    }

    /**
     * Apply default values to an immutable ({@link ToCopyableBuilder}) object by converting it to a builder, applying the
     * provided mutation functions to the builder, and then returning the {@link ToCopyableBuilder} constructed from that
     * builder.
     *
     * @param toBuildable    The immutable object that should have default values applied to it.
     * @param defaultApplier The function that will apply defaults to the builder for the new object.
     * @return The new {@link ToCopyableBuilder} with additional default values added by the builderOverrider.
     */
    private <T extends ToCopyableBuilder<B, T>, B extends CopyableBuilder<B, T>>
            T applyDefaultValues(T toBuildable, Consumer<B> defaultApplier) {
        B builder = toBuildable.toBuilder();
        defaultApplier.accept(builder);
        return builder.build();
    }
}
