/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a data table in Amazon Connect. A data table is a JSON-like data structure where attributes and values are
 * dynamically set by customers. Customers can reference table values within call flows, applications, views, and
 * workspaces to pinpoint dynamic configuration that changes their contact center's behavior in a predetermined and safe
 * way.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DataTable implements SdkPojo, Serializable, ToCopyableBuilder<DataTable.Builder, DataTable> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(DataTable::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(DataTable::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(DataTable::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> TIME_ZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TimeZone").getter(getter(DataTable::timeZone)).setter(setter(Builder::timeZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TimeZone").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(DataTable::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> VALUE_LOCK_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ValueLockLevel").getter(getter(DataTable::valueLockLevelAsString))
            .setter(setter(Builder::valueLockLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValueLockLevel").build()).build();

    private static final SdkField<DataTableLockVersion> LOCK_VERSION_FIELD = SdkField
            .<DataTableLockVersion> builder(MarshallingType.SDK_POJO).memberName("LockVersion")
            .getter(getter(DataTable::lockVersion)).setter(setter(Builder::lockVersion))
            .constructor(DataTableLockVersion::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LockVersion").build()).build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Version")
            .getter(getter(DataTable::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Version").build()).build();

    private static final SdkField<String> VERSION_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VersionDescription").getter(getter(DataTable::versionDescription))
            .setter(setter(Builder::versionDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VersionDescription").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(DataTable::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(DataTable::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(DataTable::lastModifiedTime)).setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<String> LAST_MODIFIED_REGION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastModifiedRegion").getter(getter(DataTable::lastModifiedRegion))
            .setter(setter(Builder::lastModifiedRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedRegion").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Tags")
            .getter(getter(DataTable::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ID_FIELD,
            ARN_FIELD, TIME_ZONE_FIELD, DESCRIPTION_FIELD, VALUE_LOCK_LEVEL_FIELD, LOCK_VERSION_FIELD, VERSION_FIELD,
            VERSION_DESCRIPTION_FIELD, STATUS_FIELD, CREATED_TIME_FIELD, LAST_MODIFIED_TIME_FIELD, LAST_MODIFIED_REGION_FIELD,
            TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String id;

    private final String arn;

    private final String timeZone;

    private final String description;

    private final String valueLockLevel;

    private final DataTableLockVersion lockVersion;

    private final String version;

    private final String versionDescription;

    private final String status;

    private final Instant createdTime;

    private final Instant lastModifiedTime;

    private final String lastModifiedRegion;

    private final Map<String, String> tags;

    private DataTable(BuilderImpl builder) {
        this.name = builder.name;
        this.id = builder.id;
        this.arn = builder.arn;
        this.timeZone = builder.timeZone;
        this.description = builder.description;
        this.valueLockLevel = builder.valueLockLevel;
        this.lockVersion = builder.lockVersion;
        this.version = builder.version;
        this.versionDescription = builder.versionDescription;
        this.status = builder.status;
        this.createdTime = builder.createdTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.lastModifiedRegion = builder.lastModifiedRegion;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The human-readable name of the data table. Must be unique within the instance and conform to Connect naming
     * standards.
     * </p>
     * 
     * @return The human-readable name of the data table. Must be unique within the instance and conform to Connect
     *         naming standards.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The unique identifier for the data table. Does not include version aliases.
     * </p>
     * 
     * @return The unique identifier for the data table. Does not include version aliases.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the data table. Does not include version aliases.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the data table. Does not include version aliases.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The IANA timezone identifier used when resolving time based dynamic values. Required even if no time slices are
     * specified.
     * </p>
     * 
     * @return The IANA timezone identifier used when resolving time based dynamic values. Required even if no time
     *         slices are specified.
     */
    public final String timeZone() {
        return timeZone;
    }

    /**
     * <p>
     * An optional description of the data table's purpose and contents.
     * </p>
     * 
     * @return An optional description of the data table's purpose and contents.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE, VALUE, and
     * NONE. Determines how concurrent edits are handled when multiple users attempt to modify values simultaneously.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #valueLockLevel}
     * will return {@link DataTableLockLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #valueLockLevelAsString}.
     * </p>
     * 
     * @return The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE,
     *         VALUE, and NONE. Determines how concurrent edits are handled when multiple users attempt to modify values
     *         simultaneously.
     * @see DataTableLockLevel
     */
    public final DataTableLockLevel valueLockLevel() {
        return DataTableLockLevel.fromValue(valueLockLevel);
    }

    /**
     * <p>
     * The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE, VALUE, and
     * NONE. Determines how concurrent edits are handled when multiple users attempt to modify values simultaneously.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #valueLockLevel}
     * will return {@link DataTableLockLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #valueLockLevelAsString}.
     * </p>
     * 
     * @return The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE,
     *         VALUE, and NONE. Determines how concurrent edits are handled when multiple users attempt to modify values
     *         simultaneously.
     * @see DataTableLockLevel
     */
    public final String valueLockLevelAsString() {
        return valueLockLevel;
    }

    /**
     * <p>
     * The lock version information used for optimistic locking and table versioning. Changes with each update to
     * prevent concurrent modification conflicts.
     * </p>
     * 
     * @return The lock version information used for optimistic locking and table versioning. Changes with each update
     *         to prevent concurrent modification conflicts.
     */
    public final DataTableLockVersion lockVersion() {
        return lockVersion;
    }

    /**
     * <p>
     * A unique identifier and alias for customer managed versions (not $LATEST or $SAVED).
     * </p>
     * 
     * @return A unique identifier and alias for customer managed versions (not $LATEST or $SAVED).
     */
    public final String version() {
        return version;
    }

    /**
     * <p>
     * A description of the customer managed version.
     * </p>
     * 
     * @return A description of the customer managed version.
     */
    public final String versionDescription() {
        return versionDescription;
    }

    /**
     * <p>
     * The current status of the data table. One of PUBLISHED or SAVED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DataTableStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the data table. One of PUBLISHED or SAVED.
     * @see DataTableStatus
     */
    public final DataTableStatus status() {
        return DataTableStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the data table. One of PUBLISHED or SAVED.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link DataTableStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the data table. One of PUBLISHED or SAVED.
     * @see DataTableStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The timestamp when the data table was created.
     * </p>
     * 
     * @return The timestamp when the data table was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The timestamp when the data table or any of its properties were last modified.
     * </p>
     * 
     * @return The timestamp when the data table or any of its properties were last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The AWS region where the data table was last modified, used for region replication.
     * </p>
     * 
     * @return The AWS region where the data table was last modified, used for region replication.
     */
    public final String lastModifiedRegion() {
        return lastModifiedRegion;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Key-value pairs for attribute based access control (TBAC or ABAC) and organization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Key-value pairs for attribute based access control (TBAC or ABAC) and organization.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(timeZone());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(valueLockLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lockVersion());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(versionDescription());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedRegion());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DataTable)) {
            return false;
        }
        DataTable other = (DataTable) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(id(), other.id()) && Objects.equals(arn(), other.arn())
                && Objects.equals(timeZone(), other.timeZone()) && Objects.equals(description(), other.description())
                && Objects.equals(valueLockLevelAsString(), other.valueLockLevelAsString())
                && Objects.equals(lockVersion(), other.lockVersion()) && Objects.equals(version(), other.version())
                && Objects.equals(versionDescription(), other.versionDescription())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(lastModifiedRegion(), other.lastModifiedRegion()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DataTable").add("Name", name()).add("Id", id()).add("Arn", arn()).add("TimeZone", timeZone())
                .add("Description", description()).add("ValueLockLevel", valueLockLevelAsString())
                .add("LockVersion", lockVersion()).add("Version", version()).add("VersionDescription", versionDescription())
                .add("Status", statusAsString()).add("CreatedTime", createdTime()).add("LastModifiedTime", lastModifiedTime())
                .add("LastModifiedRegion", lastModifiedRegion()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "TimeZone":
            return Optional.ofNullable(clazz.cast(timeZone()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "ValueLockLevel":
            return Optional.ofNullable(clazz.cast(valueLockLevelAsString()));
        case "LockVersion":
            return Optional.ofNullable(clazz.cast(lockVersion()));
        case "Version":
            return Optional.ofNullable(clazz.cast(version()));
        case "VersionDescription":
            return Optional.ofNullable(clazz.cast(versionDescription()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "LastModifiedRegion":
            return Optional.ofNullable(clazz.cast(lastModifiedRegion()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Name", NAME_FIELD);
        map.put("Id", ID_FIELD);
        map.put("Arn", ARN_FIELD);
        map.put("TimeZone", TIME_ZONE_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("ValueLockLevel", VALUE_LOCK_LEVEL_FIELD);
        map.put("LockVersion", LOCK_VERSION_FIELD);
        map.put("Version", VERSION_FIELD);
        map.put("VersionDescription", VERSION_DESCRIPTION_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("LastModifiedTime", LAST_MODIFIED_TIME_FIELD);
        map.put("LastModifiedRegion", LAST_MODIFIED_REGION_FIELD);
        map.put("Tags", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DataTable, T> g) {
        return obj -> g.apply((DataTable) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DataTable> {
        /**
         * <p>
         * The human-readable name of the data table. Must be unique within the instance and conform to Connect naming
         * standards.
         * </p>
         * 
         * @param name
         *        The human-readable name of the data table. Must be unique within the instance and conform to Connect
         *        naming standards.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The unique identifier for the data table. Does not include version aliases.
         * </p>
         * 
         * @param id
         *        The unique identifier for the data table. Does not include version aliases.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The Amazon Resource Name (ARN) for the data table. Does not include version aliases.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) for the data table. Does not include version aliases.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The IANA timezone identifier used when resolving time based dynamic values. Required even if no time slices
         * are specified.
         * </p>
         * 
         * @param timeZone
         *        The IANA timezone identifier used when resolving time based dynamic values. Required even if no time
         *        slices are specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timeZone(String timeZone);

        /**
         * <p>
         * An optional description of the data table's purpose and contents.
         * </p>
         * 
         * @param description
         *        An optional description of the data table's purpose and contents.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE, VALUE,
         * and NONE. Determines how concurrent edits are handled when multiple users attempt to modify values
         * simultaneously.
         * </p>
         * 
         * @param valueLockLevel
         *        The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE,
         *        VALUE, and NONE. Determines how concurrent edits are handled when multiple users attempt to modify
         *        values simultaneously.
         * @see DataTableLockLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataTableLockLevel
         */
        Builder valueLockLevel(String valueLockLevel);

        /**
         * <p>
         * The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE, VALUE,
         * and NONE. Determines how concurrent edits are handled when multiple users attempt to modify values
         * simultaneously.
         * </p>
         * 
         * @param valueLockLevel
         *        The data level that concurrent value edits are locked on. One of DATA_TABLE, PRIMARY_VALUE, ATTRIBUTE,
         *        VALUE, and NONE. Determines how concurrent edits are handled when multiple users attempt to modify
         *        values simultaneously.
         * @see DataTableLockLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataTableLockLevel
         */
        Builder valueLockLevel(DataTableLockLevel valueLockLevel);

        /**
         * <p>
         * The lock version information used for optimistic locking and table versioning. Changes with each update to
         * prevent concurrent modification conflicts.
         * </p>
         * 
         * @param lockVersion
         *        The lock version information used for optimistic locking and table versioning. Changes with each
         *        update to prevent concurrent modification conflicts.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lockVersion(DataTableLockVersion lockVersion);

        /**
         * <p>
         * The lock version information used for optimistic locking and table versioning. Changes with each update to
         * prevent concurrent modification conflicts.
         * </p>
         * This is a convenience method that creates an instance of the {@link DataTableLockVersion.Builder} avoiding
         * the need to create one manually via {@link DataTableLockVersion#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DataTableLockVersion.Builder#build()} is called immediately and
         * its result is passed to {@link #lockVersion(DataTableLockVersion)}.
         * 
         * @param lockVersion
         *        a consumer that will call methods on {@link DataTableLockVersion.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lockVersion(DataTableLockVersion)
         */
        default Builder lockVersion(Consumer<DataTableLockVersion.Builder> lockVersion) {
            return lockVersion(DataTableLockVersion.builder().applyMutation(lockVersion).build());
        }

        /**
         * <p>
         * A unique identifier and alias for customer managed versions (not $LATEST or $SAVED).
         * </p>
         * 
         * @param version
         *        A unique identifier and alias for customer managed versions (not $LATEST or $SAVED).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * A description of the customer managed version.
         * </p>
         * 
         * @param versionDescription
         *        A description of the customer managed version.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder versionDescription(String versionDescription);

        /**
         * <p>
         * The current status of the data table. One of PUBLISHED or SAVED.
         * </p>
         * 
         * @param status
         *        The current status of the data table. One of PUBLISHED or SAVED.
         * @see DataTableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataTableStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the data table. One of PUBLISHED or SAVED.
         * </p>
         * 
         * @param status
         *        The current status of the data table. One of PUBLISHED or SAVED.
         * @see DataTableStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DataTableStatus
         */
        Builder status(DataTableStatus status);

        /**
         * <p>
         * The timestamp when the data table was created.
         * </p>
         * 
         * @param createdTime
         *        The timestamp when the data table was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The timestamp when the data table or any of its properties were last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The timestamp when the data table or any of its properties were last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The AWS region where the data table was last modified, used for region replication.
         * </p>
         * 
         * @param lastModifiedRegion
         *        The AWS region where the data table was last modified, used for region replication.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedRegion(String lastModifiedRegion);

        /**
         * <p>
         * Key-value pairs for attribute based access control (TBAC or ABAC) and organization.
         * </p>
         * 
         * @param tags
         *        Key-value pairs for attribute based access control (TBAC or ABAC) and organization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String id;

        private String arn;

        private String timeZone;

        private String description;

        private String valueLockLevel;

        private DataTableLockVersion lockVersion;

        private String version;

        private String versionDescription;

        private String status;

        private Instant createdTime;

        private Instant lastModifiedTime;

        private String lastModifiedRegion;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DataTable model) {
            name(model.name);
            id(model.id);
            arn(model.arn);
            timeZone(model.timeZone);
            description(model.description);
            valueLockLevel(model.valueLockLevel);
            lockVersion(model.lockVersion);
            version(model.version);
            versionDescription(model.versionDescription);
            status(model.status);
            createdTime(model.createdTime);
            lastModifiedTime(model.lastModifiedTime);
            lastModifiedRegion(model.lastModifiedRegion);
            tags(model.tags);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getTimeZone() {
            return timeZone;
        }

        public final void setTimeZone(String timeZone) {
            this.timeZone = timeZone;
        }

        @Override
        public final Builder timeZone(String timeZone) {
            this.timeZone = timeZone;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getValueLockLevel() {
            return valueLockLevel;
        }

        public final void setValueLockLevel(String valueLockLevel) {
            this.valueLockLevel = valueLockLevel;
        }

        @Override
        public final Builder valueLockLevel(String valueLockLevel) {
            this.valueLockLevel = valueLockLevel;
            return this;
        }

        @Override
        public final Builder valueLockLevel(DataTableLockLevel valueLockLevel) {
            this.valueLockLevel(valueLockLevel == null ? null : valueLockLevel.toString());
            return this;
        }

        public final DataTableLockVersion.Builder getLockVersion() {
            return lockVersion != null ? lockVersion.toBuilder() : null;
        }

        public final void setLockVersion(DataTableLockVersion.BuilderImpl lockVersion) {
            this.lockVersion = lockVersion != null ? lockVersion.build() : null;
        }

        @Override
        public final Builder lockVersion(DataTableLockVersion lockVersion) {
            this.lockVersion = lockVersion;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final String getVersionDescription() {
            return versionDescription;
        }

        public final void setVersionDescription(String versionDescription) {
            this.versionDescription = versionDescription;
        }

        @Override
        public final Builder versionDescription(String versionDescription) {
            this.versionDescription = versionDescription;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(DataTableStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final String getLastModifiedRegion() {
            return lastModifiedRegion;
        }

        public final void setLastModifiedRegion(String lastModifiedRegion) {
            this.lastModifiedRegion = lastModifiedRegion;
        }

        @Override
        public final Builder lastModifiedRegion(String lastModifiedRegion) {
            this.lastModifiedRegion = lastModifiedRegion;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        @Override
        public DataTable build() {
            return new DataTable(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
