/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.config.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Config rules evaluate the configuration settings of your Amazon Web Services resources. A rule can run when Config
 * detects a configuration change to an Amazon Web Services resource or at a periodic frequency that you choose (for
 * example, every 24 hours). There are two types of rules: <i>Config Managed Rules</i> and <i>Config Custom Rules</i>.
 * </p>
 * <p>
 * Config Managed Rules are predefined, customizable rules created by Config. For a list of managed rules, see <a
 * href="https://docs.aws.amazon.com/config/latest/developerguide/managed-rules-by-aws-config.html">List of Config
 * Managed Rules</a>.
 * </p>
 * <p>
 * Config Custom Rules are rules that you create from scratch. There are two ways to create Config custom rules: with
 * Lambda functions (<a href=
 * "https://docs.aws.amazon.com/config/latest/developerguide/gettingstarted-concepts.html#gettingstarted-concepts-function"
 * > Lambda Developer Guide</a>) and with Guard (<a
 * href="https://github.com/aws-cloudformation/cloudformation-guard">Guard GitHub Repository</a>), a policy-as-code
 * language. Config custom rules created with Lambda are called <i>Config Custom Lambda Rules</i> and Config custom
 * rules created with Guard are called <i>Config Custom Policy Rules</i>.
 * </p>
 * <p>
 * For more information about developing and using Config rules, see <a
 * href="https://docs.aws.amazon.com/config/latest/developerguide/evaluate-config.html">Evaluating Resource with Config
 * Rules</a> in the <i>Config Developer Guide</i>.
 * </p>
 * <note>
 * <p>
 * You can use the Amazon Web Services CLI and Amazon Web Services SDKs if you want to create a rule that triggers
 * evaluations for your resources when Config delivers the configuration snapshot. For more information, see
 * <a>ConfigSnapshotDeliveryProperties</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ConfigRule implements SdkPojo, Serializable, ToCopyableBuilder<ConfigRule.Builder, ConfigRule> {
    private static final SdkField<String> CONFIG_RULE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigRuleName").getter(getter(ConfigRule::configRuleName)).setter(setter(Builder::configRuleName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigRuleName").build()).build();

    private static final SdkField<String> CONFIG_RULE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigRuleArn").getter(getter(ConfigRule::configRuleArn)).setter(setter(Builder::configRuleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigRuleArn").build()).build();

    private static final SdkField<String> CONFIG_RULE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigRuleId").getter(getter(ConfigRule::configRuleId)).setter(setter(Builder::configRuleId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigRuleId").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ConfigRule::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<Scope> SCOPE_FIELD = SdkField.<Scope> builder(MarshallingType.SDK_POJO).memberName("Scope")
            .getter(getter(ConfigRule::scope)).setter(setter(Builder::scope)).constructor(Scope::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scope").build()).build();

    private static final SdkField<Source> SOURCE_FIELD = SdkField.<Source> builder(MarshallingType.SDK_POJO).memberName("Source")
            .getter(getter(ConfigRule::source)).setter(setter(Builder::source)).constructor(Source::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Source").build()).build();

    private static final SdkField<String> INPUT_PARAMETERS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InputParameters").getter(getter(ConfigRule::inputParameters)).setter(setter(Builder::inputParameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InputParameters").build()).build();

    private static final SdkField<String> MAXIMUM_EXECUTION_FREQUENCY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MaximumExecutionFrequency").getter(getter(ConfigRule::maximumExecutionFrequencyAsString))
            .setter(setter(Builder::maximumExecutionFrequency))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaximumExecutionFrequency").build())
            .build();

    private static final SdkField<String> CONFIG_RULE_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ConfigRuleState").getter(getter(ConfigRule::configRuleStateAsString))
            .setter(setter(Builder::configRuleState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConfigRuleState").build()).build();

    private static final SdkField<String> CREATED_BY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CreatedBy").getter(getter(ConfigRule::createdBy)).setter(setter(Builder::createdBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<List<EvaluationModeConfiguration>> EVALUATION_MODES_FIELD = SdkField
            .<List<EvaluationModeConfiguration>> builder(MarshallingType.LIST)
            .memberName("EvaluationModes")
            .getter(getter(ConfigRule::evaluationModes))
            .setter(setter(Builder::evaluationModes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EvaluationModes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EvaluationModeConfiguration> builder(MarshallingType.SDK_POJO)
                                            .constructor(EvaluationModeConfiguration::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIG_RULE_NAME_FIELD,
            CONFIG_RULE_ARN_FIELD, CONFIG_RULE_ID_FIELD, DESCRIPTION_FIELD, SCOPE_FIELD, SOURCE_FIELD, INPUT_PARAMETERS_FIELD,
            MAXIMUM_EXECUTION_FREQUENCY_FIELD, CONFIG_RULE_STATE_FIELD, CREATED_BY_FIELD, EVALUATION_MODES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String configRuleName;

    private final String configRuleArn;

    private final String configRuleId;

    private final String description;

    private final Scope scope;

    private final Source source;

    private final String inputParameters;

    private final String maximumExecutionFrequency;

    private final String configRuleState;

    private final String createdBy;

    private final List<EvaluationModeConfiguration> evaluationModes;

    private ConfigRule(BuilderImpl builder) {
        this.configRuleName = builder.configRuleName;
        this.configRuleArn = builder.configRuleArn;
        this.configRuleId = builder.configRuleId;
        this.description = builder.description;
        this.scope = builder.scope;
        this.source = builder.source;
        this.inputParameters = builder.inputParameters;
        this.maximumExecutionFrequency = builder.maximumExecutionFrequency;
        this.configRuleState = builder.configRuleState;
        this.createdBy = builder.createdBy;
        this.evaluationModes = builder.evaluationModes;
    }

    /**
     * <p>
     * The name that you assign to the Config rule. The name is required if you are adding a new rule.
     * </p>
     * 
     * @return The name that you assign to the Config rule. The name is required if you are adding a new rule.
     */
    public final String configRuleName() {
        return configRuleName;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Config rule.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Config rule.
     */
    public final String configRuleArn() {
        return configRuleArn;
    }

    /**
     * <p>
     * The ID of the Config rule.
     * </p>
     * 
     * @return The ID of the Config rule.
     */
    public final String configRuleId() {
        return configRuleId;
    }

    /**
     * <p>
     * The description that you provide for the Config rule.
     * </p>
     * 
     * @return The description that you provide for the Config rule.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Defines which resources can trigger an evaluation for the rule. The scope can include one or more resource types,
     * a combination of one resource type and one resource ID, or a combination of a tag key and value. Specify a scope
     * to constrain the resources that can trigger an evaluation for the rule. If you do not specify a scope,
     * evaluations are triggered when any resource in the recording group changes.
     * </p>
     * 
     * @return Defines which resources can trigger an evaluation for the rule. The scope can include one or more
     *         resource types, a combination of one resource type and one resource ID, or a combination of a tag key and
     *         value. Specify a scope to constrain the resources that can trigger an evaluation for the rule. If you do
     *         not specify a scope, evaluations are triggered when any resource in the recording group changes.
     */
    public final Scope scope() {
        return scope;
    }

    /**
     * <p>
     * Provides the rule owner (<code>Amazon Web Services</code> for managed rules, <code>CUSTOM_POLICY</code> for
     * Custom Policy rules, and <code>CUSTOM_LAMBDA</code> for Custom Lambda rules), the rule identifier, and the
     * notifications that cause the function to evaluate your Amazon Web Services resources.
     * </p>
     * 
     * @return Provides the rule owner (<code>Amazon Web Services</code> for managed rules, <code>CUSTOM_POLICY</code>
     *         for Custom Policy rules, and <code>CUSTOM_LAMBDA</code> for Custom Lambda rules), the rule identifier,
     *         and the notifications that cause the function to evaluate your Amazon Web Services resources.
     */
    public final Source source() {
        return source;
    }

    /**
     * <p>
     * A string, in JSON format, that is passed to the Config rule Lambda function.
     * </p>
     * 
     * @return A string, in JSON format, that is passed to the Config rule Lambda function.
     */
    public final String inputParameters() {
        return inputParameters;
    }

    /**
     * <p>
     * The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
     * <code>MaximumExecutionFrequency</code> when:
     * </p>
     * <ul>
     * <li>
     * <p>
     * This is for an Config managed rule that is triggered at a periodic frequency.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your custom rule is triggered when Config delivers the configuration snapshot. For more information, see
     * <a>ConfigSnapshotDeliveryProperties</a>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a valid
     * value for the <code>MaximumExecutionFrequency</code> parameter.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
     *         <code>MaximumExecutionFrequency</code> when:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         This is for an Config managed rule that is triggered at a periodic frequency.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your custom rule is triggered when Config delivers the configuration snapshot. For more information, see
     *         <a>ConfigSnapshotDeliveryProperties</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify
     *         a valid value for the <code>MaximumExecutionFrequency</code> parameter.
     *         </p>
     * @see MaximumExecutionFrequency
     */
    public final MaximumExecutionFrequency maximumExecutionFrequency() {
        return MaximumExecutionFrequency.fromValue(maximumExecutionFrequency);
    }

    /**
     * <p>
     * The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
     * <code>MaximumExecutionFrequency</code> when:
     * </p>
     * <ul>
     * <li>
     * <p>
     * This is for an Config managed rule that is triggered at a periodic frequency.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your custom rule is triggered when Config delivers the configuration snapshot. For more information, see
     * <a>ConfigSnapshotDeliveryProperties</a>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a valid
     * value for the <code>MaximumExecutionFrequency</code> parameter.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #maximumExecutionFrequency} will return {@link MaximumExecutionFrequency#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #maximumExecutionFrequencyAsString}.
     * </p>
     * 
     * @return The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
     *         <code>MaximumExecutionFrequency</code> when:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         This is for an Config managed rule that is triggered at a periodic frequency.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Your custom rule is triggered when Config delivers the configuration snapshot. For more information, see
     *         <a>ConfigSnapshotDeliveryProperties</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify
     *         a valid value for the <code>MaximumExecutionFrequency</code> parameter.
     *         </p>
     * @see MaximumExecutionFrequency
     */
    public final String maximumExecutionFrequencyAsString() {
        return maximumExecutionFrequency;
    }

    /**
     * <p>
     * Indicates whether the Config rule is active or is currently being deleted by Config. It can also indicate the
     * evaluation status for the Config rule.
     * </p>
     * <p>
     * Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
     * <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
     * </p>
     * <p>
     * Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
     * <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config rule.
     * </p>
     * <p>
     * Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
     * <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and all of its
     * evaluations are erased and are no longer available.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #configRuleState}
     * will return {@link ConfigRuleState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #configRuleStateAsString}.
     * </p>
     * 
     * @return Indicates whether the Config rule is active or is currently being deleted by Config. It can also indicate
     *         the evaluation status for the Config rule.</p>
     *         <p>
     *         Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
     *         <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
     *         </p>
     *         <p>
     *         Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
     *         <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config
     *         rule.
     *         </p>
     *         <p>
     *         Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
     *         <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and all
     *         of its evaluations are erased and are no longer available.
     * @see ConfigRuleState
     */
    public final ConfigRuleState configRuleState() {
        return ConfigRuleState.fromValue(configRuleState);
    }

    /**
     * <p>
     * Indicates whether the Config rule is active or is currently being deleted by Config. It can also indicate the
     * evaluation status for the Config rule.
     * </p>
     * <p>
     * Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
     * <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
     * </p>
     * <p>
     * Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
     * <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config rule.
     * </p>
     * <p>
     * Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
     * <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and all of its
     * evaluations are erased and are no longer available.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #configRuleState}
     * will return {@link ConfigRuleState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #configRuleStateAsString}.
     * </p>
     * 
     * @return Indicates whether the Config rule is active or is currently being deleted by Config. It can also indicate
     *         the evaluation status for the Config rule.</p>
     *         <p>
     *         Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
     *         <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
     *         </p>
     *         <p>
     *         Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
     *         <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config
     *         rule.
     *         </p>
     *         <p>
     *         Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
     *         <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and all
     *         of its evaluations are erased and are no longer available.
     * @see ConfigRuleState
     */
    public final String configRuleStateAsString() {
        return configRuleState;
    }

    /**
     * <p>
     * Service principal name of the service that created the rule.
     * </p>
     * <note>
     * <p>
     * The field is populated only if the service-linked rule is created by a service. The field is empty if you create
     * your own rule.
     * </p>
     * </note>
     * 
     * @return Service principal name of the service that created the rule.</p> <note>
     *         <p>
     *         The field is populated only if the service-linked rule is created by a service. The field is empty if you
     *         create your own rule.
     *         </p>
     */
    public final String createdBy() {
        return createdBy;
    }

    /**
     * For responses, this returns true if the service returned a value for the EvaluationModes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEvaluationModes() {
        return evaluationModes != null && !(evaluationModes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the value is
     * Detective evaluation mode only.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEvaluationModes} method.
     * </p>
     * 
     * @return The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the
     *         value is Detective evaluation mode only.
     */
    public final List<EvaluationModeConfiguration> evaluationModes() {
        return evaluationModes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(configRuleName());
        hashCode = 31 * hashCode + Objects.hashCode(configRuleArn());
        hashCode = 31 * hashCode + Objects.hashCode(configRuleId());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(scope());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(inputParameters());
        hashCode = 31 * hashCode + Objects.hashCode(maximumExecutionFrequencyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(configRuleStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(hasEvaluationModes() ? evaluationModes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ConfigRule)) {
            return false;
        }
        ConfigRule other = (ConfigRule) obj;
        return Objects.equals(configRuleName(), other.configRuleName()) && Objects.equals(configRuleArn(), other.configRuleArn())
                && Objects.equals(configRuleId(), other.configRuleId()) && Objects.equals(description(), other.description())
                && Objects.equals(scope(), other.scope()) && Objects.equals(source(), other.source())
                && Objects.equals(inputParameters(), other.inputParameters())
                && Objects.equals(maximumExecutionFrequencyAsString(), other.maximumExecutionFrequencyAsString())
                && Objects.equals(configRuleStateAsString(), other.configRuleStateAsString())
                && Objects.equals(createdBy(), other.createdBy()) && hasEvaluationModes() == other.hasEvaluationModes()
                && Objects.equals(evaluationModes(), other.evaluationModes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ConfigRule").add("ConfigRuleName", configRuleName()).add("ConfigRuleArn", configRuleArn())
                .add("ConfigRuleId", configRuleId()).add("Description", description()).add("Scope", scope())
                .add("Source", source()).add("InputParameters", inputParameters())
                .add("MaximumExecutionFrequency", maximumExecutionFrequencyAsString())
                .add("ConfigRuleState", configRuleStateAsString()).add("CreatedBy", createdBy())
                .add("EvaluationModes", hasEvaluationModes() ? evaluationModes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ConfigRuleName":
            return Optional.ofNullable(clazz.cast(configRuleName()));
        case "ConfigRuleArn":
            return Optional.ofNullable(clazz.cast(configRuleArn()));
        case "ConfigRuleId":
            return Optional.ofNullable(clazz.cast(configRuleId()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Scope":
            return Optional.ofNullable(clazz.cast(scope()));
        case "Source":
            return Optional.ofNullable(clazz.cast(source()));
        case "InputParameters":
            return Optional.ofNullable(clazz.cast(inputParameters()));
        case "MaximumExecutionFrequency":
            return Optional.ofNullable(clazz.cast(maximumExecutionFrequencyAsString()));
        case "ConfigRuleState":
            return Optional.ofNullable(clazz.cast(configRuleStateAsString()));
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "EvaluationModes":
            return Optional.ofNullable(clazz.cast(evaluationModes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ConfigRuleName", CONFIG_RULE_NAME_FIELD);
        map.put("ConfigRuleArn", CONFIG_RULE_ARN_FIELD);
        map.put("ConfigRuleId", CONFIG_RULE_ID_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("Scope", SCOPE_FIELD);
        map.put("Source", SOURCE_FIELD);
        map.put("InputParameters", INPUT_PARAMETERS_FIELD);
        map.put("MaximumExecutionFrequency", MAXIMUM_EXECUTION_FREQUENCY_FIELD);
        map.put("ConfigRuleState", CONFIG_RULE_STATE_FIELD);
        map.put("CreatedBy", CREATED_BY_FIELD);
        map.put("EvaluationModes", EVALUATION_MODES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ConfigRule, T> g) {
        return obj -> g.apply((ConfigRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ConfigRule> {
        /**
         * <p>
         * The name that you assign to the Config rule. The name is required if you are adding a new rule.
         * </p>
         * 
         * @param configRuleName
         *        The name that you assign to the Config rule. The name is required if you are adding a new rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configRuleName(String configRuleName);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Config rule.
         * </p>
         * 
         * @param configRuleArn
         *        The Amazon Resource Name (ARN) of the Config rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configRuleArn(String configRuleArn);

        /**
         * <p>
         * The ID of the Config rule.
         * </p>
         * 
         * @param configRuleId
         *        The ID of the Config rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configRuleId(String configRuleId);

        /**
         * <p>
         * The description that you provide for the Config rule.
         * </p>
         * 
         * @param description
         *        The description that you provide for the Config rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Defines which resources can trigger an evaluation for the rule. The scope can include one or more resource
         * types, a combination of one resource type and one resource ID, or a combination of a tag key and value.
         * Specify a scope to constrain the resources that can trigger an evaluation for the rule. If you do not specify
         * a scope, evaluations are triggered when any resource in the recording group changes.
         * </p>
         * 
         * @param scope
         *        Defines which resources can trigger an evaluation for the rule. The scope can include one or more
         *        resource types, a combination of one resource type and one resource ID, or a combination of a tag key
         *        and value. Specify a scope to constrain the resources that can trigger an evaluation for the rule. If
         *        you do not specify a scope, evaluations are triggered when any resource in the recording group
         *        changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scope(Scope scope);

        /**
         * <p>
         * Defines which resources can trigger an evaluation for the rule. The scope can include one or more resource
         * types, a combination of one resource type and one resource ID, or a combination of a tag key and value.
         * Specify a scope to constrain the resources that can trigger an evaluation for the rule. If you do not specify
         * a scope, evaluations are triggered when any resource in the recording group changes.
         * </p>
         * This is a convenience method that creates an instance of the {@link Scope.Builder} avoiding the need to
         * create one manually via {@link Scope#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Scope.Builder#build()} is called immediately and its result is
         * passed to {@link #scope(Scope)}.
         * 
         * @param scope
         *        a consumer that will call methods on {@link Scope.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scope(Scope)
         */
        default Builder scope(Consumer<Scope.Builder> scope) {
            return scope(Scope.builder().applyMutation(scope).build());
        }

        /**
         * <p>
         * Provides the rule owner (<code>Amazon Web Services</code> for managed rules, <code>CUSTOM_POLICY</code> for
         * Custom Policy rules, and <code>CUSTOM_LAMBDA</code> for Custom Lambda rules), the rule identifier, and the
         * notifications that cause the function to evaluate your Amazon Web Services resources.
         * </p>
         * 
         * @param source
         *        Provides the rule owner (<code>Amazon Web Services</code> for managed rules,
         *        <code>CUSTOM_POLICY</code> for Custom Policy rules, and <code>CUSTOM_LAMBDA</code> for Custom Lambda
         *        rules), the rule identifier, and the notifications that cause the function to evaluate your Amazon Web
         *        Services resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(Source source);

        /**
         * <p>
         * Provides the rule owner (<code>Amazon Web Services</code> for managed rules, <code>CUSTOM_POLICY</code> for
         * Custom Policy rules, and <code>CUSTOM_LAMBDA</code> for Custom Lambda rules), the rule identifier, and the
         * notifications that cause the function to evaluate your Amazon Web Services resources.
         * </p>
         * This is a convenience method that creates an instance of the {@link Source.Builder} avoiding the need to
         * create one manually via {@link Source#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Source.Builder#build()} is called immediately and its result is
         * passed to {@link #source(Source)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link Source.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(Source)
         */
        default Builder source(Consumer<Source.Builder> source) {
            return source(Source.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * A string, in JSON format, that is passed to the Config rule Lambda function.
         * </p>
         * 
         * @param inputParameters
         *        A string, in JSON format, that is passed to the Config rule Lambda function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inputParameters(String inputParameters);

        /**
         * <p>
         * The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
         * <code>MaximumExecutionFrequency</code> when:
         * </p>
         * <ul>
         * <li>
         * <p>
         * This is for an Config managed rule that is triggered at a periodic frequency.
         * </p>
         * </li>
         * <li>
         * <p>
         * Your custom rule is triggered when Config delivers the configuration snapshot. For more information, see
         * <a>ConfigSnapshotDeliveryProperties</a>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a
         * valid value for the <code>MaximumExecutionFrequency</code> parameter.
         * </p>
         * </note>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
         *        <code>MaximumExecutionFrequency</code> when:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        This is for an Config managed rule that is triggered at a periodic frequency.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Your custom rule is triggered when Config delivers the configuration snapshot. For more information,
         *        see <a>ConfigSnapshotDeliveryProperties</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency,
         *        specify a valid value for the <code>MaximumExecutionFrequency</code> parameter.
         *        </p>
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(String maximumExecutionFrequency);

        /**
         * <p>
         * The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
         * <code>MaximumExecutionFrequency</code> when:
         * </p>
         * <ul>
         * <li>
         * <p>
         * This is for an Config managed rule that is triggered at a periodic frequency.
         * </p>
         * </li>
         * <li>
         * <p>
         * Your custom rule is triggered when Config delivers the configuration snapshot. For more information, see
         * <a>ConfigSnapshotDeliveryProperties</a>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency, specify a
         * valid value for the <code>MaximumExecutionFrequency</code> parameter.
         * </p>
         * </note>
         * 
         * @param maximumExecutionFrequency
         *        The maximum frequency with which Config runs evaluations for a rule. You can specify a value for
         *        <code>MaximumExecutionFrequency</code> when:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        This is for an Config managed rule that is triggered at a periodic frequency.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Your custom rule is triggered when Config delivers the configuration snapshot. For more information,
         *        see <a>ConfigSnapshotDeliveryProperties</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        By default, rules with a periodic trigger are evaluated every 24 hours. To change the frequency,
         *        specify a valid value for the <code>MaximumExecutionFrequency</code> parameter.
         *        </p>
         * @see MaximumExecutionFrequency
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MaximumExecutionFrequency
         */
        Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency);

        /**
         * <p>
         * Indicates whether the Config rule is active or is currently being deleted by Config. It can also indicate the
         * evaluation status for the Config rule.
         * </p>
         * <p>
         * Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
         * <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
         * </p>
         * <p>
         * Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
         * <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config rule.
         * </p>
         * <p>
         * Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
         * <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and all of
         * its evaluations are erased and are no longer available.
         * </p>
         * 
         * @param configRuleState
         *        Indicates whether the Config rule is active or is currently being deleted by Config. It can also
         *        indicate the evaluation status for the Config rule.</p>
         *        <p>
         *        Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
         *        <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
         *        </p>
         *        <p>
         *        Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
         *        <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config
         *        rule.
         *        </p>
         *        <p>
         *        Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
         *        <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and
         *        all of its evaluations are erased and are no longer available.
         * @see ConfigRuleState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigRuleState
         */
        Builder configRuleState(String configRuleState);

        /**
         * <p>
         * Indicates whether the Config rule is active or is currently being deleted by Config. It can also indicate the
         * evaluation status for the Config rule.
         * </p>
         * <p>
         * Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
         * <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
         * </p>
         * <p>
         * Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
         * <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config rule.
         * </p>
         * <p>
         * Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
         * <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and all of
         * its evaluations are erased and are no longer available.
         * </p>
         * 
         * @param configRuleState
         *        Indicates whether the Config rule is active or is currently being deleted by Config. It can also
         *        indicate the evaluation status for the Config rule.</p>
         *        <p>
         *        Config sets the state of the rule to <code>EVALUATING</code> temporarily after you use the
         *        <code>StartConfigRulesEvaluation</code> request to evaluate your resources against the Config rule.
         *        </p>
         *        <p>
         *        Config sets the state of the rule to <code>DELETING_RESULTS</code> temporarily after you use the
         *        <code>DeleteEvaluationResults</code> request to delete the current evaluation results for the Config
         *        rule.
         *        </p>
         *        <p>
         *        Config temporarily sets the state of a rule to <code>DELETING</code> after you use the
         *        <code>DeleteConfigRule</code> request to delete the rule. After Config deletes the rule, the rule and
         *        all of its evaluations are erased and are no longer available.
         * @see ConfigRuleState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ConfigRuleState
         */
        Builder configRuleState(ConfigRuleState configRuleState);

        /**
         * <p>
         * Service principal name of the service that created the rule.
         * </p>
         * <note>
         * <p>
         * The field is populated only if the service-linked rule is created by a service. The field is empty if you
         * create your own rule.
         * </p>
         * </note>
         * 
         * @param createdBy
         *        Service principal name of the service that created the rule.</p> <note>
         *        <p>
         *        The field is populated only if the service-linked rule is created by a service. The field is empty if
         *        you create your own rule.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(String createdBy);

        /**
         * <p>
         * The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the value
         * is Detective evaluation mode only.
         * </p>
         * 
         * @param evaluationModes
         *        The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the
         *        value is Detective evaluation mode only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationModes(Collection<EvaluationModeConfiguration> evaluationModes);

        /**
         * <p>
         * The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the value
         * is Detective evaluation mode only.
         * </p>
         * 
         * @param evaluationModes
         *        The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the
         *        value is Detective evaluation mode only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder evaluationModes(EvaluationModeConfiguration... evaluationModes);

        /**
         * <p>
         * The modes the Config rule can be evaluated in. The valid values are distinct objects. By default, the value
         * is Detective evaluation mode only.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.config.model.EvaluationModeConfiguration.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.config.model.EvaluationModeConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.config.model.EvaluationModeConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #evaluationModes(List<EvaluationModeConfiguration>)}.
         * 
         * @param evaluationModes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.config.model.EvaluationModeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #evaluationModes(java.util.Collection<EvaluationModeConfiguration>)
         */
        Builder evaluationModes(Consumer<EvaluationModeConfiguration.Builder>... evaluationModes);
    }

    static final class BuilderImpl implements Builder {
        private String configRuleName;

        private String configRuleArn;

        private String configRuleId;

        private String description;

        private Scope scope;

        private Source source;

        private String inputParameters;

        private String maximumExecutionFrequency;

        private String configRuleState;

        private String createdBy;

        private List<EvaluationModeConfiguration> evaluationModes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ConfigRule model) {
            configRuleName(model.configRuleName);
            configRuleArn(model.configRuleArn);
            configRuleId(model.configRuleId);
            description(model.description);
            scope(model.scope);
            source(model.source);
            inputParameters(model.inputParameters);
            maximumExecutionFrequency(model.maximumExecutionFrequency);
            configRuleState(model.configRuleState);
            createdBy(model.createdBy);
            evaluationModes(model.evaluationModes);
        }

        public final String getConfigRuleName() {
            return configRuleName;
        }

        public final void setConfigRuleName(String configRuleName) {
            this.configRuleName = configRuleName;
        }

        @Override
        public final Builder configRuleName(String configRuleName) {
            this.configRuleName = configRuleName;
            return this;
        }

        public final String getConfigRuleArn() {
            return configRuleArn;
        }

        public final void setConfigRuleArn(String configRuleArn) {
            this.configRuleArn = configRuleArn;
        }

        @Override
        public final Builder configRuleArn(String configRuleArn) {
            this.configRuleArn = configRuleArn;
            return this;
        }

        public final String getConfigRuleId() {
            return configRuleId;
        }

        public final void setConfigRuleId(String configRuleId) {
            this.configRuleId = configRuleId;
        }

        @Override
        public final Builder configRuleId(String configRuleId) {
            this.configRuleId = configRuleId;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Scope.Builder getScope() {
            return scope != null ? scope.toBuilder() : null;
        }

        public final void setScope(Scope.BuilderImpl scope) {
            this.scope = scope != null ? scope.build() : null;
        }

        @Override
        public final Builder scope(Scope scope) {
            this.scope = scope;
            return this;
        }

        public final Source.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        public final void setSource(Source.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        public final Builder source(Source source) {
            this.source = source;
            return this;
        }

        public final String getInputParameters() {
            return inputParameters;
        }

        public final void setInputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
        }

        @Override
        public final Builder inputParameters(String inputParameters) {
            this.inputParameters = inputParameters;
            return this;
        }

        public final String getMaximumExecutionFrequency() {
            return maximumExecutionFrequency;
        }

        public final void setMaximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
        }

        @Override
        public final Builder maximumExecutionFrequency(String maximumExecutionFrequency) {
            this.maximumExecutionFrequency = maximumExecutionFrequency;
            return this;
        }

        @Override
        public final Builder maximumExecutionFrequency(MaximumExecutionFrequency maximumExecutionFrequency) {
            this.maximumExecutionFrequency(maximumExecutionFrequency == null ? null : maximumExecutionFrequency.toString());
            return this;
        }

        public final String getConfigRuleState() {
            return configRuleState;
        }

        public final void setConfigRuleState(String configRuleState) {
            this.configRuleState = configRuleState;
        }

        @Override
        public final Builder configRuleState(String configRuleState) {
            this.configRuleState = configRuleState;
            return this;
        }

        @Override
        public final Builder configRuleState(ConfigRuleState configRuleState) {
            this.configRuleState(configRuleState == null ? null : configRuleState.toString());
            return this;
        }

        public final String getCreatedBy() {
            return createdBy;
        }

        public final void setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
        }

        @Override
        public final Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final List<EvaluationModeConfiguration.Builder> getEvaluationModes() {
            List<EvaluationModeConfiguration.Builder> result = EvaluationModesCopier.copyToBuilder(this.evaluationModes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEvaluationModes(Collection<EvaluationModeConfiguration.BuilderImpl> evaluationModes) {
            this.evaluationModes = EvaluationModesCopier.copyFromBuilder(evaluationModes);
        }

        @Override
        public final Builder evaluationModes(Collection<EvaluationModeConfiguration> evaluationModes) {
            this.evaluationModes = EvaluationModesCopier.copy(evaluationModes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder evaluationModes(EvaluationModeConfiguration... evaluationModes) {
            evaluationModes(Arrays.asList(evaluationModes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder evaluationModes(Consumer<EvaluationModeConfiguration.Builder>... evaluationModes) {
            evaluationModes(Stream.of(evaluationModes).map(c -> EvaluationModeConfiguration.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ConfigRule build() {
            return new ConfigRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
