/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The preference to control which resource type values are considered when generating rightsizing recommendations. You
 * can specify this preference as a combination of include and exclude lists. You must specify either an
 * <code>includeList</code> or <code>excludeList</code>. If the preference is an empty set of resource type values, an
 * error occurs. For more information, see <a
 * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/rightsizing-preferences.html"> Rightsizing
 * recommendation preferences</a> in the <i>Compute Optimizer User Guide</i>.
 * </p>
 * <note>
 * <ul>
 * <li>
 * <p>
 * This preference is only available for the Amazon EC2 instance and Amazon EC2 Auto Scaling group resource types.
 * </p>
 * </li>
 * <li>
 * <p>
 * Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
 * </p>
 * </li>
 * </ul>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PreferredResource implements SdkPojo, Serializable,
        ToCopyableBuilder<PreferredResource.Builder, PreferredResource> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(PreferredResource::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<String>> INCLUDE_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("includeList")
            .getter(getter(PreferredResource::includeList))
            .setter(setter(Builder::includeList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("includeList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXCLUDE_LIST_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("excludeList")
            .getter(getter(PreferredResource::excludeList))
            .setter(setter(Builder::excludeList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("excludeList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            INCLUDE_LIST_FIELD, EXCLUDE_LIST_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<String> includeList;

    private final List<String> excludeList;

    private PreferredResource(BuilderImpl builder) {
        this.name = builder.name;
        this.includeList = builder.includeList;
        this.excludeList = builder.excludeList;
    }

    /**
     * <p>
     * The type of preferred resource to customize.
     * </p>
     * <note>
     * <p>
     * Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link PreferredResourceName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The type of preferred resource to customize. </p> <note>
     *         <p>
     *         Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
     *         </p>
     * @see PreferredResourceName
     */
    public final PreferredResourceName name() {
        return PreferredResourceName.fromValue(name);
    }

    /**
     * <p>
     * The type of preferred resource to customize.
     * </p>
     * <note>
     * <p>
     * Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link PreferredResourceName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The type of preferred resource to customize. </p> <note>
     *         <p>
     *         Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
     *         </p>
     * @see PreferredResourceName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * For responses, this returns true if the service returned a value for the IncludeList property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIncludeList() {
        return includeList != null && !(includeList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The preferred resource type values to include in the recommendation candidates. You can specify the exact
     * resource type value, such as m5.large, or use wild card expressions, such as m5. If this isn’t specified, all
     * supported resources are included by default. You can specify up to 1000 values in this list.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIncludeList} method.
     * </p>
     * 
     * @return The preferred resource type values to include in the recommendation candidates. You can specify the exact
     *         resource type value, such as m5.large, or use wild card expressions, such as m5. If this isn’t specified,
     *         all supported resources are included by default. You can specify up to 1000 values in this list.
     */
    public final List<String> includeList() {
        return includeList;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeList property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludeList() {
        return excludeList != null && !(excludeList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The preferred resource type values to exclude from the recommendation candidates. If this isn’t specified, all
     * supported resources are included by default. You can specify up to 1000 values in this list.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeList} method.
     * </p>
     * 
     * @return The preferred resource type values to exclude from the recommendation candidates. If this isn’t
     *         specified, all supported resources are included by default. You can specify up to 1000 values in this
     *         list.
     */
    public final List<String> excludeList() {
        return excludeList;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasIncludeList() ? includeList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeList() ? excludeList() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PreferredResource)) {
            return false;
        }
        PreferredResource other = (PreferredResource) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && hasIncludeList() == other.hasIncludeList()
                && Objects.equals(includeList(), other.includeList()) && hasExcludeList() == other.hasExcludeList()
                && Objects.equals(excludeList(), other.excludeList());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PreferredResource").add("Name", nameAsString())
                .add("IncludeList", hasIncludeList() ? includeList() : null)
                .add("ExcludeList", hasExcludeList() ? excludeList() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "includeList":
            return Optional.ofNullable(clazz.cast(includeList()));
        case "excludeList":
            return Optional.ofNullable(clazz.cast(excludeList()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("includeList", INCLUDE_LIST_FIELD);
        map.put("excludeList", EXCLUDE_LIST_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PreferredResource, T> g) {
        return obj -> g.apply((PreferredResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PreferredResource> {
        /**
         * <p>
         * The type of preferred resource to customize.
         * </p>
         * <note>
         * <p>
         * Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
         * </p>
         * </note>
         * 
         * @param name
         *        The type of preferred resource to customize. </p> <note>
         *        <p>
         *        Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
         *        </p>
         * @see PreferredResourceName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreferredResourceName
         */
        Builder name(String name);

        /**
         * <p>
         * The type of preferred resource to customize.
         * </p>
         * <note>
         * <p>
         * Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
         * </p>
         * </note>
         * 
         * @param name
         *        The type of preferred resource to customize. </p> <note>
         *        <p>
         *        Compute Optimizer only supports the customization of <code>Ec2InstanceTypes</code>.
         *        </p>
         * @see PreferredResourceName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PreferredResourceName
         */
        Builder name(PreferredResourceName name);

        /**
         * <p>
         * The preferred resource type values to include in the recommendation candidates. You can specify the exact
         * resource type value, such as m5.large, or use wild card expressions, such as m5. If this isn’t specified, all
         * supported resources are included by default. You can specify up to 1000 values in this list.
         * </p>
         * 
         * @param includeList
         *        The preferred resource type values to include in the recommendation candidates. You can specify the
         *        exact resource type value, such as m5.large, or use wild card expressions, such as m5. If this isn’t
         *        specified, all supported resources are included by default. You can specify up to 1000 values in this
         *        list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeList(Collection<String> includeList);

        /**
         * <p>
         * The preferred resource type values to include in the recommendation candidates. You can specify the exact
         * resource type value, such as m5.large, or use wild card expressions, such as m5. If this isn’t specified, all
         * supported resources are included by default. You can specify up to 1000 values in this list.
         * </p>
         * 
         * @param includeList
         *        The preferred resource type values to include in the recommendation candidates. You can specify the
         *        exact resource type value, such as m5.large, or use wild card expressions, such as m5. If this isn’t
         *        specified, all supported resources are included by default. You can specify up to 1000 values in this
         *        list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeList(String... includeList);

        /**
         * <p>
         * The preferred resource type values to exclude from the recommendation candidates. If this isn’t specified,
         * all supported resources are included by default. You can specify up to 1000 values in this list.
         * </p>
         * 
         * @param excludeList
         *        The preferred resource type values to exclude from the recommendation candidates. If this isn’t
         *        specified, all supported resources are included by default. You can specify up to 1000 values in this
         *        list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeList(Collection<String> excludeList);

        /**
         * <p>
         * The preferred resource type values to exclude from the recommendation candidates. If this isn’t specified,
         * all supported resources are included by default. You can specify up to 1000 values in this list.
         * </p>
         * 
         * @param excludeList
         *        The preferred resource type values to exclude from the recommendation candidates. If this isn’t
         *        specified, all supported resources are included by default. You can specify up to 1000 values in this
         *        list.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeList(String... excludeList);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<String> includeList = DefaultSdkAutoConstructList.getInstance();

        private List<String> excludeList = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PreferredResource model) {
            name(model.name);
            includeList(model.includeList);
            excludeList(model.excludeList);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(PreferredResourceName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final Collection<String> getIncludeList() {
            if (includeList instanceof SdkAutoConstructList) {
                return null;
            }
            return includeList;
        }

        public final void setIncludeList(Collection<String> includeList) {
            this.includeList = PreferredResourceValuesCopier.copy(includeList);
        }

        @Override
        public final Builder includeList(Collection<String> includeList) {
            this.includeList = PreferredResourceValuesCopier.copy(includeList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder includeList(String... includeList) {
            includeList(Arrays.asList(includeList));
            return this;
        }

        public final Collection<String> getExcludeList() {
            if (excludeList instanceof SdkAutoConstructList) {
                return null;
            }
            return excludeList;
        }

        public final void setExcludeList(Collection<String> excludeList) {
            this.excludeList = PreferredResourceValuesCopier.copy(excludeList);
        }

        @Override
        public final Builder excludeList(Collection<String> excludeList) {
            this.excludeList = PreferredResourceValuesCopier.copy(excludeList);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludeList(String... excludeList) {
            excludeList(Arrays.asList(excludeList));
            return this;
        }

        @Override
        public PreferredResource build() {
            return new PreferredResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
