/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Text extraction encountered one or more page-level errors in the input document.
 * </p>
 * <p>
 * The <code>ErrorCode</code> contains one of the following values:
 * </p>
 * <ul>
 * <li>
 * <p>
 * TEXTRACT_BAD_PAGE - Amazon Textract cannot read the page. For more information about page limits in Amazon Textract,
 * see <a href="https://docs.aws.amazon.com/textract/latest/dg/limits-document.html"> Page Quotas in Amazon
 * Textract</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * TEXTRACT_PROVISIONED_THROUGHPUT_EXCEEDED - The number of requests exceeded your throughput limit. For more
 * information about throughput quotas in Amazon Textract, see <a
 * href="https://docs.aws.amazon.com/textract/latest/dg/limits-quotas-explained.html"> Default quotas in Amazon
 * Textract</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * PAGE_CHARACTERS_EXCEEDED - Too many text characters on the page (10,000 characters maximum).
 * </p>
 * </li>
 * <li>
 * <p>
 * PAGE_SIZE_EXCEEDED - The maximum page size is 10 MB.
 * </p>
 * </li>
 * <li>
 * <p>
 * INTERNAL_SERVER_ERROR - The request encountered a service issue. Try the API request again.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ErrorsListItem implements SdkPojo, Serializable, ToCopyableBuilder<ErrorsListItem.Builder, ErrorsListItem> {
    private static final SdkField<Integer> PAGE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER).memberName("Page")
            .getter(getter(ErrorsListItem::page)).setter(setter(Builder::page))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Page").build()).build();

    private static final SdkField<String> ERROR_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorCode").getter(getter(ErrorsListItem::errorCodeAsString)).setter(setter(Builder::errorCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorCode").build()).build();

    private static final SdkField<String> ERROR_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ErrorMessage").getter(getter(ErrorsListItem::errorMessage)).setter(setter(Builder::errorMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ErrorMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PAGE_FIELD, ERROR_CODE_FIELD,
            ERROR_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer page;

    private final String errorCode;

    private final String errorMessage;

    private ErrorsListItem(BuilderImpl builder) {
        this.page = builder.page;
        this.errorCode = builder.errorCode;
        this.errorMessage = builder.errorMessage;
    }

    /**
     * <p>
     * Page number where the error occurred.
     * </p>
     * 
     * @return Page number where the error occurred.
     */
    public final Integer page() {
        return page;
    }

    /**
     * <p>
     * Error code for the cause of the error.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link PageBasedErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return Error code for the cause of the error.
     * @see PageBasedErrorCode
     */
    public final PageBasedErrorCode errorCode() {
        return PageBasedErrorCode.fromValue(errorCode);
    }

    /**
     * <p>
     * Error code for the cause of the error.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #errorCode} will
     * return {@link PageBasedErrorCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #errorCodeAsString}.
     * </p>
     * 
     * @return Error code for the cause of the error.
     * @see PageBasedErrorCode
     */
    public final String errorCodeAsString() {
        return errorCode;
    }

    /**
     * <p>
     * Text message explaining the reason for the error.
     * </p>
     * 
     * @return Text message explaining the reason for the error.
     */
    public final String errorMessage() {
        return errorMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(page());
        hashCode = 31 * hashCode + Objects.hashCode(errorCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(errorMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ErrorsListItem)) {
            return false;
        }
        ErrorsListItem other = (ErrorsListItem) obj;
        return Objects.equals(page(), other.page()) && Objects.equals(errorCodeAsString(), other.errorCodeAsString())
                && Objects.equals(errorMessage(), other.errorMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ErrorsListItem").add("Page", page()).add("ErrorCode", errorCodeAsString())
                .add("ErrorMessage", errorMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Page":
            return Optional.ofNullable(clazz.cast(page()));
        case "ErrorCode":
            return Optional.ofNullable(clazz.cast(errorCodeAsString()));
        case "ErrorMessage":
            return Optional.ofNullable(clazz.cast(errorMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Page", PAGE_FIELD);
        map.put("ErrorCode", ERROR_CODE_FIELD);
        map.put("ErrorMessage", ERROR_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ErrorsListItem, T> g) {
        return obj -> g.apply((ErrorsListItem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ErrorsListItem> {
        /**
         * <p>
         * Page number where the error occurred.
         * </p>
         * 
         * @param page
         *        Page number where the error occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder page(Integer page);

        /**
         * <p>
         * Error code for the cause of the error.
         * </p>
         * 
         * @param errorCode
         *        Error code for the cause of the error.
         * @see PageBasedErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PageBasedErrorCode
         */
        Builder errorCode(String errorCode);

        /**
         * <p>
         * Error code for the cause of the error.
         * </p>
         * 
         * @param errorCode
         *        Error code for the cause of the error.
         * @see PageBasedErrorCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PageBasedErrorCode
         */
        Builder errorCode(PageBasedErrorCode errorCode);

        /**
         * <p>
         * Text message explaining the reason for the error.
         * </p>
         * 
         * @param errorMessage
         *        Text message explaining the reason for the error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder errorMessage(String errorMessage);
    }

    static final class BuilderImpl implements Builder {
        private Integer page;

        private String errorCode;

        private String errorMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(ErrorsListItem model) {
            page(model.page);
            errorCode(model.errorCode);
            errorMessage(model.errorMessage);
        }

        public final Integer getPage() {
            return page;
        }

        public final void setPage(Integer page) {
            this.page = page;
        }

        @Override
        public final Builder page(Integer page) {
            this.page = page;
            return this;
        }

        public final String getErrorCode() {
            return errorCode;
        }

        public final void setErrorCode(String errorCode) {
            this.errorCode = errorCode;
        }

        @Override
        public final Builder errorCode(String errorCode) {
            this.errorCode = errorCode;
            return this;
        }

        @Override
        public final Builder errorCode(PageBasedErrorCode errorCode) {
            this.errorCode(errorCode == null ? null : errorCode.toString());
            return this;
        }

        public final String getErrorMessage() {
            return errorMessage;
        }

        public final void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }

        @Override
        public final Builder errorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
            return this;
        }

        @Override
        public ErrorsListItem build() {
            return new ErrorsListItem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
